unit Printpag;  {PrintPage Version 4.3 Copyright  W. Murto 1995}

{$DEFINE NORULER}             {THRuler & TVRuler in RULER1.ZIP}
{$DEFINE NOROTATE}            {TRotateLabel in ROTATEL.ZIP}
{To use these components remove 'NO' then Install or Rebuild.}

interface
uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Grids, Printers, StdCtrls, ExtCtrls, Tabs, TabNotBk, Menus, Calendar
  {$IFDEF RULER} , Rulers            {$ENDIF}
  {$IFDEF ROTATE}, Rotatel           {$ENDIF}
  ;

const mrPrint = mrAll + 1;

type
  TPrintPreview = class(TForm)
    MainMenu1: TMainMenu;
    Print1: TMenuItem;
    Cancel1: TMenuItem;
    procedure Print1Click(Sender: TObject);
    procedure Cancel1Click(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
  end;

type
  TPrintPage = class(TComponent)
  private
    { Private declarations }
    fSource : TScrollingWinControl;
    fTags : longint;
    fDest : TCanvas;
    fOnPrintControl : TNotifyEvent;
    fOnExternalPrint : TNotifyEvent;
    fPreviewCaption : string;
    fPreviewing,
    fPreviewMenu,
    fPreviewRulers : boolean;
    RulerOffset : integer;
    fPreviewScale : double;
    fDesignPixelsPerInch : integer;
    fPreviewHeight,
    fPreviewWidth : double;
    fTopOffset, fLeftOffset,
    VOffset, HOffset,
    VScrollPos, HScrollPos,
    fScaleX, fScaleY : integer;
    fScaleRX, fScaleRY : double;
    PDC : HDC;
    procedure SetPreviewRulers(Value: boolean);
    procedure SetPreviewScale(Value: double);
    procedure Paint_Preview(Sender: TObject);
  protected
    { Protected declarations }
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    procedure Print;
    function Preview: integer;
    property Source: TScrollingWinControl read fSource write fSource stored false;
    property Dest: TCanvas read fDest;
    property Previewing: boolean read fPreviewing;
    property OnExternalPrint: TNotifyEvent read fOnExternalPrint write fOnExternalPrint stored false;
    property LeftOffset: integer read fLeftOffset write fLeftOffset stored false;
    property TopOffset: integer read fTopOffset write fTopOffset stored false;
    property LineSize: integer read fScaleY;
    property ScaleRX: double read fScaleRX;
    property ScaleRY: double read fScaleRY;
    property ScaleX: integer read fScaleX;
    property ScaleY: integer read fScaleY;
    function ScaleToPrinter(R:TRect):TRect;
  published
    { Published declarations }
    property PrintTags: longint read fTags write fTags;
    property PreviewCaption: string read fPreviewCaption write fPreviewCaption;
    property PreviewMenu: boolean read fPreviewMenu write fPreviewMenu;
    property PreviewRulers: boolean read fPreviewRulers write SetPreviewRulers;
    property PreviewScale: double read fPreviewScale write SetPreviewScale;
    property DesignPixelsPerInch: integer read fDesignPixelsPerInch write fDesignPixelsPerInch;
    property PerviewHeight: double read fPreviewHeight write fPreviewHeight;
    property PerviewWidth: double read fPreviewWidth write fPreviewWidth;
    property OnUpdatePrintStatus: TNotifyEvent read fOnPrintControl write fOnPrintControl;
  private
    { more Private declarations - the Print/Paint stuff }
    procedure DrawHRuler(R: TRect);
    procedure DrawVRuler(R: TRect);
    procedure PrintLabel(ALabel: TLabel);
    procedure PrintMemo(AMemo: TMemo);
    procedure PrintEdit(AEdit: TEdit);
    procedure PrintComboBox(ACombo: TComboBox);
    procedure PrintShape(AShape:TShape);
    procedure PrintGrid(TheGrid:TObject);
    procedure PrintCheck(ACheck: TCheckBox);
    procedure PrintRadio(ARadio: TRadioButton);
    procedure PrintBevel(ABevel: TBevel);
    procedure PrintTabSet(ATabSet: TTabSet);
    procedure PrintImage(AImage: TImage);
    {$IFDEF RULER}
    procedure PrintHRuler(Ruler: THRuler);
    procedure PrintVRuler(Ruler: TVRuler);
    {$ENDIF}
    {$IFDEF ROTATE}
    procedure PrintRotate(ARotate: TRotateLabel);
    {$ENDIF}
    procedure PrintGroup(AGroup: TGroupBox);
    Procedure PrintPanel(APanel: TPanel);
    Procedure PrintNotebook(ANotebook: TNotebook);
    Procedure PrintTabNotebook(ATabNotebook: TTabbedNotebook);
    procedure PrintControl(AControl: TObject);
  end;

procedure Register;

implementation

{$R *.DFM}

var
  PrintPreview: TPrintPreview;

procedure Register;
begin
  RegisterComponents('Samples', [TPrintPage]);
end;

procedure TPrintPreview.Print1Click(Sender: TObject);
begin
  ModalResult := mrPrint;
end;

procedure TPrintPreview.Cancel1Click(Sender: TObject);
begin
  ModalResult := mrCancel;
end;

    { PrintPage Private declarations }
procedure TPrintPage.SetPreviewRulers(Value: boolean);
begin
  fPreviewRulers := Value;
  if Value then RulerOffset := 32 else RulerOffset := 0;
end;

procedure TPrintPage.SetPreviewScale(Value: double);
begin
  if (Value > 0.9) and (Value < 4.1) then fPreviewScale := Value;
end;

    { preview form onpaint set to this in the preview function }
procedure TPrintPage.Paint_Preview(Sender: TObject);
var I, ROffset : integer;
begin
  if not fPreviewing then exit;
  try
    VOffset := RulerOffset; HOffset := RulerOffset;
    if fPreviewRulers then
    begin
      ROffset :=  trunc(RulerOffset * fScaleRX);
      DrawHRuler(Rect(ROffset, 0, PrintPreview.ClientWidth, ROffset));
      DrawVRuler(Rect(0, ROffset, ROffset, PrintPreview.ClientHeight));
    end;
    VScrollPos := fSource.VertScrollBar.Position;
    HScrollPos := fSource.HorzScrollBar.Position;
    for I := 0 to fSource.ControlCount-1 do
    if (fSource.Controls[I].Visible) and (fSource.Controls[I].Tag >= 0) then
      if (fTags = 0) or (fSource.Controls[I].Tag and fTags = fTags) then
        PrintControl(fSource.Controls[I]);
  except
    on Exception do fPreviewing := false;
  end;
end; {Paint_Preview}

    { Public declarations }
constructor TPrintPage.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  fPreviewCaption := 'Print Preview';
  fPreviewScale := 3.0;
  fDesignPixelsPerInch := 96;
  fPreviewHeight := 10.5;
  fPreviewWidth := 8.0;
end; {create}

procedure TPrintPage.Print;
var I : integer;
begin
  if not Assigned(fSource) then exit;
  fPreviewing := false;
  VOffset := fTopOffset; HOffset := fLeftOffset;
  VScrollPos := fSource.VertScrollBar.Position;
  HScrollPos := fSource.HorzScrollBar.Position;
  Printer.BeginDoc;
  try
    fDest := Printer.Canvas;
    PDC := Printer.Canvas.Handle;
    fScaleRX := WinProcs.GetDeviceCaps(PDC, LOGPIXELSX) / fDesignPixelsPerInch;
    fScaleRY := WinProcs.GetDeviceCaps(PDC, LOGPIXELSY) / fDesignPixelsPerInch;
    fScaleX := Trunc(fScaleRX);
    fScaleY := Trunc(fScaleRY);
    for I := 0 to fSource.ControlCount-1 do {components with a neg. tag won't print}
    if (fSource.Controls[I].Visible) and (fSource.Controls[I].Tag >= 0) then
      if (fTags = 0) or (fSource.Controls[I].Tag and fTags = fTags) then
        PrintControl(fSource.Controls[I]);
  finally
    Printer.EndDoc;
  end;
end;   {Print}

function TPrintPage.Preview: integer;
var LSize, SSize : integer;
begin
  Result := mrCancel;
  if not Assigned(fSource) then exit;
  PrintPreview := TPrintPreview.Create(nil);
  try
    fPreviewing := true;
    PrintPreview.Caption := fPreviewCaption;
    PrintPreview.Print1.Visible := fPreviewMenu;
    PrintPreview.Cancel1.Visible := fPreviewMenu;
    PrintPreview.OnPaint := Paint_Preview;
    fDest := PrintPreview.Canvas;
    PDC := fDest.Handle;
    fScaleRX := 1/fPreviewScale;
    fScaleRY := fScaleRX;
    fScaleX := 1; fScaleY := 1;
    LSize := trunc(fDesignPixelsPerInch * fPreviewHeight * fScaleRX);
    LSize := LSize + trunc(RulerOffset * fScaleRX);
    SSize := trunc(fDesignPixelsPerInch * fPreviewWidth * fScaleRX);
    SSize := SSize + trunc(RulerOffset * fScaleRX);
    PrintPreview.ClientHeight := LSize;
    PrintPreview.ClientWidth := LSize;
    if Printer.Orientation = poLandscape then PrintPreview.ClientHeight := SSize
      else PrintPreview.ClientWidth := SSize;
    Result := PrintPreview.ShowModal;
  finally
    PrintPreview.Free;
    PrintPreview := nil;
    fPreviewing := false;
  end;
end;   {preview}

function TPrintPage.ScaleToPrinter(R:TRect):TRect;
begin
  Result.Top := Trunc((R.Top + VScrollPos + VOffset) * fScaleRY);
  Result.Left := Trunc((R.Left + HScrollPos + HOffset) * fScaleRX);
  Result.Bottom := Trunc((R.Bottom + VScrollPos + VOffset) * fScaleRY);
  Result.Right := Trunc((R.Right + HScrollPos + HOffset) * fScaleRX);
end;

    { more Private declarations - the Print/Paint stuff }
procedure TPrintPage.DrawHRuler(R:TRect);
var a12th, N, Y : word;
    RX : double;
begin
  a12th := fDesignPixelsPerInch div 12;
  fDest.Font.Size := 10;
  if fPreviewing then
  begin
    fDest.Font.Name := 'Arial';
    fDest.Font.Size := trunc(fDest.Font.Size / fPreviewScale);
  end;
  PDC := fDest.Handle;
  fDest.Rectangle(R.Left,R.Top,R.Right,R.Bottom);
  N := 0;
  RX := R.Left;
  Y := R.Top;
  with fDest do
    while trunc(RX) < R.Right do
    begin
      MoveTo(trunc(RX), Y + fScaleY);
      LineTo(trunc(RX), Y + (trunc(6 * fScaleRY) * (1 + byte(N mod 3 = 0) +
        byte(N mod 6 = 0) +
        byte(N mod 12 = 0))));
      if (N > 0) and (N mod 12 = 0) and (PenPos.X < (R.Right - a12th div 2)) then
        TextOut(PenPos.X+trunc(3*fScaleRX), Y+trunc(9*fScaleRY), IntToStr(N div 12));
      N := N + 1;
      RX := RX + a12th * fScaleRX;
    end;
end;

procedure TPrintPage.DrawVRuler(R:TRect);
var a6th, N, X : word;
    RY : double;
begin
  a6th := fDesignPixelsPerInch div 6;
  fDest.Font.Size := 10;
  if fPreviewing then
  begin
    fDest.Font.Name := 'Arial';
    fDest.Font.Size := trunc(fDest.Font.Size / fPreviewScale);
  end;
  PDC := fDest.Handle;
  fDest.Rectangle(R.Left,R.Top,R.Right,R.Bottom);
  N := 0;
  X := R.Left;
  RY := R.Top;
  with fDest do
    while trunc(RY) < R.Bottom do
    begin
      MoveTo(X + fScaleX, trunc(RY));
      LineTo(X + (trunc(6 * fScaleRX) * (2 + byte(N mod 3 = 0) +
        byte(N mod 6 = 0))),trunc(RY));
      if (N > 0) and (N mod 6 = 0) then
        TextOut(X+trunc(12*fScaleRX), PenPos.Y-trunc(16*fScaleRY), IntToStr(N div 6));
      N := N + 1;
      RY := RY + a6th * fScaleRY;
    end;
end;

procedure TPrintPage.PrintLabel(ALabel: TLabel);
const
  Alignments: array[TAlignment] of Word = (DT_LEFT, DT_RIGHT, DT_CENTER);
var C : array[0..255] of char;
    CLen : integer;
    Format : Word;
    R: TRect;
begin
  fDest.Font := ALabel.Font;
  if fPreviewing then
  begin
    fDest.Font.Name := 'Arial';
    fDest.Font.Size := trunc(fDest.Font.Size / fPreviewScale);
  end;
  PDC := fDest.Handle; {so DrawText knows about font}
  R := ScaleToPrinter(ALabel.BoundsRect);
  R.Right := R.Right + fScaleX*3;
  Format := DT_EXPANDTABS or DT_WORDBREAK or Alignments[ALabel.Alignment];
  CLen := ALabel.GetTextBuf(C,255);
  WinProcs.DrawText(PDC, C, CLen, R, Format);
end; {label}

procedure TPrintPage.PrintMemo(AMemo: TMemo);
const
  Alignments: array[TAlignment] of Word = (DT_LEFT, DT_RIGHT, DT_CENTER);
var C : Pchar;
    CLen : integer;
    Format : Word;
    R: TRect;
begin
  fDest.Font := AMemo.Font;
  if fPreviewing then
  begin
    fDest.Font.Name := 'Arial';
    fDest.Font.Size := trunc(fDest.Font.Size / fPreviewScale);
  end;
  PDC := fDest.Handle;
  R := ScaleToPrinter(AMemo.BoundsRect);
  if AMemo.BorderStyle = bsSingle then
    begin
      fDest.Rectangle(R.Left,R.Top,R.Right,R.Bottom);
      R.Left := R.Left + fScaleX + fScaleX;
      R.Right := R.Right - fScaleX - fScaleX;
      R.Top:=R.Top + fScaleY*3;
    end;
  R.Bottom := R.Bottom + fDest.Font.Height;
  Format := DT_EXPANDTABS;
  if AMemo.WordWrap then Format := Format or DT_WORDBREAK;
  Format := Format or Alignments[AMemo.Alignment];
  CLen := AMemo.GetTextLen;
  inc(CLen);
  GetMem(C, CLen);
  AMemo.GetTextBuf(C, CLen);
  WinProcs.DrawText(PDC, C, -1, R, Format);
  FreeMem(C, CLen);
end; {memo}

procedure TPrintPage.PrintEdit(AEdit: TEdit);
var C : array[0..255] of char;
    CLen : integer;
    Format : Word;
    R: TRect;
begin
  fDest.Font := AEdit.Font;
  if fPreviewing then
  begin
    fDest.Font.Name := 'Arial';
    fDest.Font.Size := trunc(fDest.Font.Size / fPreviewScale);
  end;
  PDC := fDest.Handle;
  R := ScaleToPrinter(AEdit.BoundsRect);
  if AEdit.BorderStyle = bsSingle then
    fDest.Rectangle(R.Left,R.Top,R.Right,R.Bottom);
  R.Left := R.Left + fScaleX + fScaleX;
  Format := DT_SINGLELINE or DT_VCENTER;
  CLen := AEdit.GetTextBuf(C,255);
  WinProcs.DrawText(PDC, C, CLen, R, Format);
end; {edit}

procedure TPrintPage.PrintComboBox(ACombo: TComboBox);
var C : array[0..255] of char;
    CLen : integer;
    Format : Word;
    R: TRect;
begin
  fDest.Font := ACombo.Font;
  if fPreviewing then
  begin
    fDest.Font.Name := 'Arial';
    fDest.Font.Size := trunc(fDest.Font.Size / fPreviewScale);
  end;
  PDC := fDest.Handle;
  R := ScaleToPrinter(ACombo.BoundsRect);
  fDest.Rectangle(R.Left,R.Top,R.Right,R.Bottom);
  R.Left := R.Left + fScaleX + fScaleX;
  Format := DT_SINGLELINE or DT_VCENTER;
  CLen := ACombo.GetTextBuf(C,255);
  WinProcs.DrawText(PDC, C, CLen, R, Format);
end; {combo}

procedure TPrintPage.PrintShape(AShape:TShape);
var H, W, S : integer;
    R : TRect;
begin
  fDest.Pen := AShape.Pen;
  fDest.Pen.Width :=  fDest.Pen.Width * fScaleY;
  fDest.Brush := AShape.Brush;
  R := ScaleToPrinter(AShape.BoundsRect);
  W := R.Right - R.Left; H := R.Bottom - R.Top;
  if W < H then S := W else S := H;
  if AShape.Shape in [stSquare, stRoundSquare, stCircle] then
  begin
    Inc(R.Left, (W - S) div 2);
    Inc(R.Top, (H - S) div 2);
    W := S;
    H := S;
  end;
  case AShape.Shape of
    stRectangle, stSquare:
      fDest.Rectangle(R.Left, R.Top, R.Left + W, R.Top + H);
    stRoundRect, stRoundSquare:
      fDest.RoundRect(R.Left, R.Top, R.Left + W, R.Top + H, S div 4, S div 4);
    stCircle, stEllipse:
      fDest.Ellipse(R.Left, R.Top, R.Left + W, R.Top + H);
  end;
end; {Shape}

procedure TPrintPage.PrintGrid(TheGrid:TObject);
var J, K : integer;
    Q, R : TRect;
    Format : Word;
    C : array[0..255] of char;
    CLen : integer;
    AGrid : TDrawGrid;
begin
  AGrid := TDrawGrid(TheGrid);
  fDest.Font := AGrid.Font;
  if fPreviewing then
  begin
    fDest.Font.Name := 'Arial';
    fDest.Font.Size := trunc(fDest.Font.Size / fPreviewScale);
  end;
  PDC := fDest.Handle;
  Format := DT_SINGLELINE or DT_VCENTER;
  Q := AGrid.BoundsRect;
  fDest.Pen.Width := AGrid.GridLineWidth * fScaleY;
  for J := 0 to AGrid.ColCount - 1 do
    for K:= 0 to AGrid.RowCount - 1 do
    begin
      R := AGrid.CellRect(J, K);
      if R.Right > R.Left then
      begin
        R.Left := R.Left + Q.Left;
        R.Right := R.Right + Q.Left + AGrid.GridLineWidth;
        R.Top := R.Top + Q.Top;
        R.Bottom := R.Bottom + Q.Top + AGrid.GridLineWidth;
        R := ScaleToPrinter(R);
        if (J < AGrid.FixedCols) or (K < AGrid.FixedRows) then
          fDest.Brush.Color := AGrid.FixedColor
        else
        begin
          fDest.Brush.Style := bsClear;
          WinProcs.SetBKColor(fDest.Handle, ColorToRGB(clWhite));
        end;
        if AGrid.GridLineWidth > 0 then  {print grid lines or not}
          fDest.Rectangle(R.Left,R.Top,R.Right,R.Bottom);
        C[0] := Chr(0);
        if TheGrid is TStringGrid then
        begin
          StrPCopy(C, TStringGrid(TheGrid).Cells[J,K]);
          R.Left := R.Left + fScaleX + fScaleX;
        end;
        if TheGrid is TCalendar then
        begin
          StrPCopy(C, TCalendar(TheGrid).CellText[J,K]);
          Format := Format or DT_CENTER;
        end;
        WinProcs.DrawText(PDC, C, StrLen(C), R, Format);
      end;
    end;
end; {Grid}

procedure TPrintPage.PrintCheck(ACheck: TCheckBox);
var R, BR : TRect;
    W, H : integer;
    C : array[0..255] of char;
    CLen : integer;
    Format : Word;
begin
  fDest.Font := ACheck.Font;
  if fPreviewing then
  begin
    fDest.Font.Name := 'Arial';
    fDest.Font.Size := trunc(fDest.Font.Size / fPreviewScale);
  end;
  PDC := fDest.Handle;
  W := trunc(12 * fScaleRX); H := trunc(12 * fScaleRY);
  R := ScaleToPrinter(ACheck.BoundsRect);
  BR := R;
  BR.Top := R.Top + ((R.Bottom - R.Top) div 2) - (H div 2);
  BR.Bottom := BR.Top + H;
  if ACheck.Alignment = taLeftJustify then
    begin
      BR.Right := R.Right; BR.Left := R.Right - W;
      R.Right := R.Right - W - fScaleX - fScaleX;
    end
    else
    begin
      BR.Right := R.Left + w; BR.Left := R.Left;
      R.Left := R.Left + W + fScaleX + fScaleX;
    end;
  fDest.Rectangle(BR.Left, BR.Top, BR.Right, BR.Bottom);
  if ACheck.Checked then with fDest do
  begin
    fDest.Pen.Width := 2*fScaleY;
    MoveTo(BR.Left+fScaleX, BR.Top + H div 2);
    LineTo(BR.Left + W div 2 - fScaleX, BR.Bottom-2*fScaleY);
    LineTo(BR.Right-fScaleX, BR.Top+fScaleY);
  end;
  Format := DT_SINGLELINE or DT_VCENTER;
  CLen := ACheck.GetTextBuf(C,255);
  WinProcs.DrawText(PDC, C, CLen, R, Format);
end; {Check}

procedure TPrintPage.PrintRadio(ARadio: TRadioButton);
var R, BR : TRect;
    W, H, CutX, CutY : integer;
    C : array[0..255] of char;
    CLen : integer;
    Format : Word;
begin
  fDest.Font := ARadio.Font;
  if fPreviewing then
  begin
    fDest.Font.Name := 'Arial';
    fDest.Font.Size := trunc(fDest.Font.Size / fPreviewScale);
  end;
  PDC := fDest.Handle;
  W := trunc(12 * fScaleRX); H := trunc(12 * fScaleRY);
  CutX := W div 3; CutY := H div 3;
  R := ScaleToPrinter(ARadio.BoundsRect);
  BR := R;
  BR.Top := R.Top + ((R.Bottom - R.Top) div 2) - (H div 2);
  BR.Bottom := BR.Top + H;
  if ARadio.Alignment = taLeftJustify then
    begin
      BR.Right := R.Right; BR.Left := R.Right - W;
      R.Right := R.Right - W - fScaleX - fScaleX;
    end
    else
    begin
      BR.Right := R.Left + w; BR.Left := R.Left;
      R.Left := R.Left + W + fScaleX * 3;
    end;
  fDest.Ellipse(BR.Left, BR.Top, BR.Right, BR.Bottom);
  if ARadio.Checked then with fDest do
  begin
    Brush.Color := clBlack;
    Ellipse(BR.Left+CutX, BR.Top+CutY, BR.Right-CutX, BR.Bottom-CutY);
    Brush.Style := bsClear;
    WinProcs.SetBKColor(Handle, ColorToRGB(clWhite));
  end;
  Format := DT_SINGLELINE or DT_VCENTER;
  CLen := ARadio.GetTextBuf(C,255);
  WinProcs.DrawText(PDC, C, CLen, R, Format);
end; {Radio}

procedure TPrintPage.PrintBevel(ABevel: TBevel);
var R : TRect;
    AShape : TBevelShape;
begin
  R := ScaleToPrinter(ABevel.BoundsRect);
  AShape := ABevel.Shape;
  with fDest do
    case AShape of
      bsBox, bsFrame: Rectangle(R.Left,R.Top,R.Right,R.Bottom);
      bsTopLine: PolyLine([Point(R.Left,R.Top),Point(R.Right,R.Top)]);
      bsBottomLine: PolyLine([Point(R.Left,R.Bottom),Point(R.Right,R.Bottom)]);
      bsLeftLine: PolyLine([Point(R.Left,R.Top),Point(R.Left,R.Bottom)]);
      bsRightLine: PolyLine([Point(R.Right,R.Top),Point(R.Right,R.Bottom)]);
    end;
end; {bevel}

procedure TPrintPage.PrintTabSet(ATabSet: TTabSet);
var R : TRect;
begin
  if ATabSet.TabIndex < 0 then exit;
  fDest.Font := ATabSet.Font;
  fDest.Font.Style := fDest.Font.Style + [fsBold];
  if fPreviewing then
  begin
    fDest.Font.Name := 'Arial';
    fDest.Font.Size := trunc(fDest.Font.Size / fPreviewScale);
  end;
  PDC := fDest.Handle;
  R := ScaleToPrinter(ATabSet.BoundsRect);
  with fDest , ATabSet do
  begin
    TextOut(R.Left + trunc(15*fScaleRX), R.Top, Tabs[TabIndex]);
    MoveTo(R.Left+fScaleX,R.Top);
    R.Left := R.Left + trunc(10*fScaleRX);
    LineTo(R.Left, R.Top);
    R.Left := R.Left + trunc(5*fScaleRX);
    R.Bottom := R.Top + trunc(3*fScaleRY) - fDest.Font.Height;
    LineTo(R.Left, R.Bottom);
    R.Left := R.Left + TextWidth(Tabs[TabIndex]);
    LineTo(R.Left, R.Bottom);
    R.Left := R.Left + trunc(5*fScaleRX);
    LineTo(R.Left, R.Top);
    LineTo(R.Right-fScaleX, R.Top);
  end;
end; {tabset}

procedure TPrintPage.PrintImage(AImage: TImage);
var R : TRect;
    Info: PBitmapInfo;
    InfoSize: Integer;
    Image: Pointer;
    ImageSize: Longint;
begin
  if not(AImage.Picture.Graphic is TBitmap) then exit;  {bitmap only}
  R := ScaleToPrinter(AImage.BoundsRect);
  if fPreviewing then
  begin
    fDest.Rectangle(R.Left,R.Top,R.Right,R.Bottom);
    fDest.Font.Size := 7;
    fDest.TextRect(R, R.Left, R.Top, ' Image');
  end
  else
    with AImage.Picture.Bitmap do
    begin
      GetDIBSizes(Handle, InfoSize, ImageSize);
      Info := MemAlloc(InfoSize);
      try
        Image := MemAlloc(ImageSize);
        try
          GetDIB(Handle, Palette, Info^, Image^);
          with Info^.bmiHeader do
            StretchDIBits(fDest.Handle, R.Left, R.Top, R.Right-R.Left,
              R.Bottom-R.Top, 0, 0, biWidth, biHeight, Image, Info^,
              DIB_RGB_COLORS, SRCCOPY);
        finally
          FreeMem(Image, ImageSize);
        end;
      finally
        FreeMem(Info, InfoSize);
      end;
    end;
end; {image}

{$IFDEF RULER}
procedure TPrintPage.PrintHRuler(Ruler: THRuler);
var R: TRect;
begin
  R := ScaleToPrinter(Ruler.BoundsRect);
  DrawHRuler(R);
end; {HRuler}

procedure TPrintPage.PrintVRuler(Ruler: TVRuler);
var R: TRect;
begin
  R := ScaleToPrinter(Ruler.BoundsRect);
  DrawVRuler(R);
end; {VRuler}
{$ENDIF}

{$IFDEF ROTATE}
procedure TPrintPage.PrintRotate(ARotate: TRotateLabel);
var R: TRect;
    LogRec: TLOGFONT;
    OldFont, NewFont: HFONT;
    midX, midY, H, W, X, Y: integer;
    DegToRad, CosAngle, SinAngle: double;
    P1, P2, P3, P4: TPoint;
begin
  R := ScaleToPrinter(ARotate.BoundsRect);
  fDest.Font := ARotate.Font;
  if fPreviewing then
  begin
    fDest.Font.Name := 'Arial';
    fDest.Font.Size := trunc(fDest.Font.Size / fPreviewScale);
  end;
  PDC := fDest.Handle;
  GetObject(fDest.Font.Handle, SizeOf(LogRec), @LogRec);
  LogRec.lfEscapement := ARotate.Angle*10;
  LogRec.lfOutPrecision := OUT_TT_ONLY_PRECIS;
  NewFont := CreateFontIndirect(LogRec);
  OldFont := SelectObject(fDest.Handle,NewFont);
  midX := (R.Right - R.Left) div 2 + R.Left;
  midY := (R.Bottom - R.Top) div 2 + R.Top;
  DegToRad := PI / 180;
  CosAngle := cos(ARotate.Angle * DegToRad);
  SinAngle := sin(ARotate.Angle * DegToRad);
  W := fDest.TextWidth(ARotate.Caption);
  H := fDest.TextHeight(ARotate.Caption);
  X := midX - trunc(W/2*CosAngle) - trunc(H/2*SinAngle);
  Y := midY + trunc(W/2*SinAngle) - trunc(H/2*CosAngle);
  if not ARotate.Transparent then
  begin
    W := W+7*fScaleX; H := H+5*fScaleY;
    P1.X := midX - trunc(W/2*CosAngle) - trunc(H/2*SinAngle);
    P1.Y := midY + trunc(W/2*SinAngle) - trunc(H/2*CosAngle);
    P2.X := midX + trunc(W/2*CosAngle) - trunc(H/2*SinAngle);
    P2.Y := midY - trunc(W/2*SinAngle) - trunc(H/2*CosAngle);
    P3.X := midX + trunc(W/2*CosAngle) + trunc(H/2*SinAngle);
    P3.Y := midY - trunc(W/2*SinAngle) + trunc(H/2*CosAngle);
    P4.X := midX - trunc(W/2*CosAngle) + trunc(H/2*SinAngle);
    P4.Y := midY + trunc(W/2*SinAngle) + trunc(H/2*CosAngle);
    fDest.PolyLine([P1, P2, P3, P4, P1]);
  end;
  fDest.TextOut(X, Y, ARotate.Caption);
  NewFont := SelectObject(fDest.Handle,OldFont);
  DeleteObject(NewFont);
end; {Rotate}
{$ENDIF}

procedure TPrintPage.PrintGroup(AGroup: TGroupBox);
var I : integer;
    R, F : TRect;
begin
  R := ScaleToPrinter(AGroup.BoundsRect);
  fDest.Font := AGroup.Font;
  if fPreviewing then
  begin
    fDest.Font.Name := 'Arial';
    fDest.Font.Size := trunc(fDest.Font.Size / fPreviewScale);
  end;
  PDC := fDest.Handle;
  VOffset := VOffset + AGroup.BoundsRect.Top;
  HOffset := HOffset + AGroup.BoundsRect.Left;
  F := R; F.Bottom := F.Bottom - fScaleY;
  F.Top := F.Top - (fDest.Font.Height div 2) + fScaleY;
  F.Left := F.Left + fScaleX; F.Right := F.Right - fScaleX;
  with fDest do
  begin
    if AGroup.Caption = '' then Rectangle(F.Left,F.Top,F.Right,F.Bottom)
    else
    begin
      TextOut(R.Left+trunc(8*fScaleRX), R.Top, AGroup.Caption);
      MoveTo(F.Left+TextWidth(AGroup.Caption)+trunc(10*fScaleRX), F.Top);
      LineTo(F.Right, F.Top); LineTo(F.Right, F.Bottom);
      LineTo(F.Left, F.Bottom); LineTo(F.Left, F.Top);
      LineTo(F.Left+trunc(4*fScaleRX), F.Top);
    end;
  end;
  for I := 0 to AGroup.ControlCount-1 do
    if (AGroup.Controls[I].Visible) and (AGroup.Controls[I].Tag >= 0) then
      if (fTags = 0) or (AGroup.Controls[I].Tag and fTags = fTags) then
        PrintControl(AGroup.Controls[I]);
  VOffset := VOffset - AGroup.BoundsRect.Top;
  HOffset := HOffset - AGroup.BoundsRect.Left;
end; {group}

Procedure TPrintPage.PrintPanel(APanel: TPanel);
var I : integer;
    R : TRect;
begin
  R := ScaleToPrinter(APanel.BoundsRect);
  VOffset := VOffset + APanel.BoundsRect.Top;
  HOffset := HOffset + APanel.BoundsRect.Left;
  if APanel.BorderStyle = bsSingle then
  begin
    fDest.PolyLine([Point(R.Left, R.Bottom-fScaleY),
                   Point(R.Left, R.Top),
                   Point(R.Right-fScaleX, R.Top)]);
    fDest.Pen.Width := 2*fScaleY;
    fDest.PolyLine([Point(R.Right-fScaleX, R.Top+fScaleY),
                   Point(R.Right-fScaleX, R.Bottom-fScaleY),
                   Point(R.Left+fScaleX, R.Bottom-fScaleY)]);
    fDest.Pen.Width := fScaleY;
  end;
  for I := 0 to APanel.ControlCount-1 do
    if (APanel.Controls[I].Visible) and (APanel.Controls[I].Tag >= 0) then
      if (fTags = 0) or (APanel.Controls[I].Tag and fTags = fTags) then
        PrintControl(APanel.Controls[I]);
  VOffset := VOffset - APanel.BoundsRect.Top;
  HOffset := HOffset - APanel.BoundsRect.Left;
end; {panel}

Procedure TPrintPage.PrintNotebook(ANotebook: TNotebook);
var I : integer;
    APage : TPage;
begin
  VOffset := VOffset + ANotebook.BoundsRect.Top;
  HOffset := HOffset + ANotebook.BoundsRect.Left;
  APage := ANotebook.Pages.Objects[ANotebook.PageIndex] as TPage;
  for I := 0 to APage.ControlCount-1 do
    if (APage.Controls[I].Visible) and (APage.Controls[I].Tag >= 0) then
      if (fTags = 0) or (APage.Controls[I].Tag and fTags = fTags) then
        PrintControl(APage.Controls[I]);
  VOffset := VOffset - ANotebook.BoundsRect.Top;
  HOffset := HOffset - ANotebook.BoundsRect.Left;
end; {notebook}

Procedure TPrintPage.PrintTabNotebook(ATabNotebook: TTabbedNotebook);
var I : integer;
    R : TRect;
    APage : TTabPage;
begin
  APage := ATabNotebook.Pages.Objects[ATabNotebook.PageIndex] as TTabPage;
  VOffset := VOffset + ATabNotebook.BoundsRect.Top + APage.BoundsRect.Top;
  HOffset := HOffset + ATabNotebook.BoundsRect.Left + APage.BoundsRect.Left;
  R := ScaleToPrinter(APage.ClientRect);
  fDest.Font := ATabNotebook.TabFont;
  fDest.Font.Style := fDest.Font.Style + [fsBold];
  if fPreviewing then
  begin
    fDest.Font.Name := 'Arial';
    fDest.Font.Size := trunc(fDest.Font.Size / fPreviewScale);
  end;
  PDC := fDest.Handle;
  with fDest , ATabNotebook do
  begin
    TextOut(R.Left + trunc(15*fScaleRX),
            R.Top-trunc(3*fScaleRY)+fDest.Font.Height, Pages[PageIndex]);
    fDest.Pen.Width := 2*fScaleY;
    PolyLine([Point(R.Right,R.Top+fScaleY),
              Point(R.Right,R.Bottom),
              Point(R.Left+fScaleX,R.Bottom)]);
    fDest.Pen.Width := fScaleY;
    MoveTo(R.Left,R.Bottom);
    LineTo(R.Left,R.Top);
    R.Left := R.Left + trunc(10*fScaleRX);
    LineTo(R.Left, R.Top);
    R.Left := R.Left + trunc(5*fScaleRX);
    LineTo(R.Left, R.Top - trunc(6*fScaleRY) + fDest.Font.Height);
    R.Left := R.Left + TextWidth(Pages[PageIndex]);
    LineTo(R.Left, R.Top - trunc(6*fScaleRY) + fDest.Font.Height);
    R.Left := R.Left + trunc(5*fScaleRX);
    LineTo(R.Left, R.Top);
    LineTo(R.Right, R.Top);
  end;
  for I := 0 to APage.ControlCount-1 do
    if (APage.Controls[I].Visible) and (APage.Controls[I].Tag >= 0) then
      if (fTags = 0) or (APage.Controls[I].Tag and fTags = fTags) then
        PrintControl(APage.Controls[I]);
  VOffset := VOffset - ATabNotebook.BoundsRect.Top - APage.BoundsRect.Top;
  HOffset := HOffset - ATabNotebook.BoundsRect.Left - APage.BoundsRect.Left;
end; {tabnotebook}

procedure TPrintPage.PrintControl(AControl: TObject);
begin
  fDest.Pen.Width := fScaleY;
  fDest.Pen.Color := clBlack;
  fDest.Pen.Style := psSolid;
  fDest.Brush.Style := bsClear;
  WinProcs.SetBKColor(fDest.Handle, ColorToRGB(clWhite));
  if Assigned(fOnExternalPrint) then fOnExternalPrint(AControl);
  if not fPreviewing then
    if Assigned(fOnPrintControl) then fOnPrintControl(AControl);

  if (AControl is TCustomLabel) {$IFDEF ROTATE} and
  not(AControl is TRotateLabel) {$ENDIF}
                                   then PrintLabel(TLabel(AControl));
  if (AControl is TCustomMemo)     then PrintMemo(TMemo(AControl));
  if (AControl is TCustomEdit) and
  not(AControl is TCustomMemo)     then PrintEdit(TEdit(AControl));
  if (AControl is TCustomComboBox) then PrintComboBox(TComboBox(AControl));
  if (AControl is TShape)          then PrintShape(TShape(AControl));
  if (AControl is TStringGrid) or
     (AControl is TCalendar)       then PrintGrid(AControl);
  if (AControl is TCustomCheckBox) then PrintCheck(TCheckBox(AControl));
  if (AControl is TRadioButton)    then PrintRadio(TRadioButton(AControl));
  if (AControl is TBevel)          then PrintBevel(TBevel(AControl));
  if (AControl is TTabSet)         then PrintTabSet(TTabSet(AControl));
  if (AControl is TImage)          then PrintImage(TImage(AControl));
  {$IFDEF RULER}
  if (AControl is THRuler)         then PrintHRuler(THRuler(AControl));
  if (AControl is TVRuler)         then PrintVRuler(TVRuler(AControl));
  {$ENDIF}
  {$IFDEF ROTATE}
  if (AControl is TRotateLabel)    then PrintRotate(TRotateLabel(AControl));
  {$ENDIF}
  if (AControl is TCustomGroupBox) then PrintGroup(TGroupBox(AControl));
  if (AControl is TPanel)          then PrintPanel(TPanel(AControl));
  if (AControl is TNotebook)       then PrintNotebook(TNotebook(AControl));
  if (AControl is TTabbedNotebook) then PrintTabNotebook(TTabbedNotebook(AControl));
end;  {printcontrol}

end.
