unit Streams;

(*********************************************
TSmartFileStream -> TFileStream

Provides a more user-friendly way of reading and writing from files by defining
a set of read and write methods specific to various data types.

READ METHODS:

function ReadBoolean
function ReadByte
function ReadInteger - Returns a longint
function ReadFloat - Returns an extended
function ReadQuotedString - Double quotes as delimiters
function ReadVarString - The string length is stored in the file preceding the string data
function ReadFixedString - The string length is supplied as a parameter
function ReadMemo - Returns a TStrings

WRITE METHODS:

procedure WriteBoolean
procedure WriteByte
procedure WriteInteger - Writes a longint
procedure WriteFloat - Writes an extended
procedure WriteString - Writes up to dynamic string length
procedure WriteFixedString - Writes the number of bytes specified
procedure WriteVarString - Writes the length of the string as first byte
procedure WriteCRLF
procedure WriteQuotedString - Double quotes around string
procedure WriteMemo
*********************************************)

interface

uses
  Classes, WinTypes, DB, DBTables, SysUtils;

type

  TSmartFileStream = class( TFileStream )
  private
     sString: string;
     str: TStrings;
     FEOF: boolean;
  protected
  public
     destructor Destroy; override;

     function ReadBoolean: boolean;
     function ReadByte: byte;
     function ReadInteger: longint;
     function ReadFloat: extended;
     function ReadQuotedString: string;
     function ReadLn: string;
     function ReadVarString: string;
     function ReadFixedString( const n: byte ): string;
     function ReadMemo: TStrings;

     procedure WriteBoolean( const bArg: boolean );
     procedure WriteByte( const bArg: byte );
     procedure WriteInteger( const nArg: longint );
     procedure WriteFloat( const xArg: extended );
     procedure WriteLn( const sArg: string );
     procedure WriteString( const sArg: string );
     procedure WriteFixedString( const sArg: string; const n: integer );
     procedure WriteVarString( const sArg: string );
     procedure WriteCRLF;
     procedure WriteQuotedString( const sArg: string );
     procedure WriteMemo( strArg: TStrings );

     property EOF: boolean read FEOF;
  end;

implementation

destructor TSmartFileStream.Destroy;
begin
  if str <> nil then
     begin
        str.Clear;
        str.Free;
     end;
end;

(*********************************************
Read methods.
*********************************************)
function TSmartFileStream.ReadBoolean: boolean;
var
  bArg: boolean;
begin
 Read( bArg, SizeOf( boolean ) );
 Result := bArg;
end;

function TSmartFileStream.ReadByte: byte;
var
  bArg: byte;
begin
  Read( bArg, SizeOf( byte ) );
  Result := bArg;
end;

function TSmartFileStream.ReadInteger: longint;
var
  nArg: longint;
begin
  Read( nArg, SizeOf( longint ) );
  Result := nArg;
end;

function TSmartFileStream.ReadFloat: extended;
var
  xArg: extended;
begin
  Read( xArg, SizeOf( extended ) );
  Result := xArg;
end;

function TSmartFileStream.ReadQuotedString: string;
var
  c: char;
  i: integer;
  sItem: string;
begin
  sItem := '';
  i := 0;
  Read( c, 1 );
  if c = '"' then
     repeat
        Read( c, 1 );
        if c <> '"' then
           begin
              Inc( i );
              sItem[i] := c;
           end;
     until ( c = '"');
  sItem[0] := Chr( i );
  Result := sItem;
end;

function TSmartFileStream.ReadFixedString( const n: byte ): string;
var
  sItem: string;
begin
  Read( sItem[1], n );
  sItem[0] := Chr( n );
  Result := sItem;
end;

function TSmartFileStream.ReadVarString: string;
var
  sItem: string;
  nBytes: byte;
begin
  nBytes := ReadByte;
  Read( sItem[1], nBytes );
  sItem[0] := Chr( nBytes );
  Result := sItem;
end;

function TSmartFileStream.ReadMemo: TStrings;
var
  nBytes: integer;
  lpBuffer: PChar;
begin
  if str = nil then
     str := TStringList.Create;
  nBytes := ReadInteger;
  if nBytes > 0 then
     begin
        lpBuffer := StrAlloc( nBytes + 1 );
        Read( lpBuffer^, nBytes );
        str.SetText( lpBuffer );
        StrDispose( lpBuffer );
     end;
  Result := str;
end;

(*********************************************
Write methods.
*********************************************)
procedure TSmartFileStream.WriteBoolean( const bArg: boolean );
begin
  Write( bArg, SizeOf( boolean ) );
end;

procedure TSmartFileStream.WriteByte( const bArg: byte );
begin
  Write( bArg, SizeOf( byte ) );
end;

procedure TSmartFileStream.WriteInteger( const nArg: longint );
begin
  Write( nArg, SizeOf( longint ) );
end;

procedure TSmartFileStream.WriteFloat( const xArg: extended );
begin
  Write( xArg, SizeOf( extended ) );
end;

(*********************************************
This one doesn't have a corresponding read method
because the length is unknown.
*********************************************)
procedure TSmartFileStream.WriteString( const sArg: string );
begin
  Write( sArg[1], Length( sArg ) );
end;

(*********************************************
Length is stored as a byte prior to string data.
*********************************************)
procedure TSmartFileStream.WriteVarString( const sArg: string );
var
  n: byte;
begin
  n := Length( sArg );
  WriteByte( n );
  Write( sArg[1], n );
end;

(*********************************************
Length is supplied as parameter.
*********************************************)
procedure TSmartFileStream.WriteFixedString( const sArg: string; const n: integer );
begin
  Write( sArg[1], n );
end;

procedure TSmartFileStream.WriteCRLF;
begin
  WriteString( #13#10 );
end;

procedure TSmartFileStream.WriteQuotedString( const sArg: string );
var
  sQuote: string[1];
begin
  sQuote := '"';
  Write( sQuote[1], 1 );
  Write( sArg[1], Length( sArg ) );
  Write( sQuote[1], 1 );
end;

(*********************************************
Length is stored as an integer before data.
*********************************************)
procedure TSmartFileStream.WriteMemo( strArg: TStrings );
var
  nBytes: integer;
begin
  nBytes := StrLen( strArg.GetText );
  WriteInteger( nBytes );
  if nBytes > 0 then
     Write( strArg.GetText^, nBytes );
end;

(*********************************************
Readln/Writeln support.
*********************************************)
procedure TSmartFileStream.WriteLn( const sArg: string );
begin
  WriteString( sArg );
  WriteCRLF;
end;

function TSmartFileStream.ReadLn: string;
var
  nRead, i: longint;
  c: char;
begin
  FEOF := FALSE;
  i := 0;
  while TRUE do
     begin
        nRead := Read( c, 1 );
        if nRead = 0 then
           begin
              FEOF := TRUE;
              Break;
           end;
        if c = #13 then
           begin
              Read( c, 1 );
              Break;
           end
        else
           begin
              Inc( i );
              if i <= 255 then
                 sString[i] := c;
           end;
     end;
  sString[0] := Chr( i );
  if i > 0 then
     Result := sString
  else
     Result := '';
end;

end.
