{ DataBase Check GRID
  indicate a data column matches a select criteria by placing a bitmap
  in the first (fixed) column.
  3/26/96 Gary Brookman 75462,3610
}
unit Dbcgrid;

interface
{$R DBCGRID.RES}
uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, Grids, DBGrids, Db, DBTables;

type

   TMatchOperators = ( moEqual, moNotEqual, moGreaterThan, moLessThan,
    moGTorEqual, moLTorEqual, moNull, moNotNull );

   TDBCGrid = class(TDBGrid)
   private                                { Private declarations }
   FGlyph         : TBitmap;
   FCheckCol      : TFieldDataLink;       { column that determines if checked }
   FCheckCriteria : string;               { match criteria for check  }
   FFieldIndex    : integer;              { relative field pos for CheckCol }
   FOperator      : TMatchOperators;
   FCaseSensitive : boolean;
   procedure SetGlyph( Value : TBitmap );
   function GetCheckCol : string;
   procedure SetCheckCol( const Value : string );
   procedure SetCheckCriteria( const Value : string );
   procedure DataChange( Sender : TObject );
   function GetCheckColValue( ACol, ARow: Longint ): string;
   procedure SetOperator( Value: TMatchOperators );
   procedure SetCaseSensitive( Value: boolean );
   function MatchCriteria( sVar : string ) : boolean;
   procedure ValidateCheckCol;
   function GetSourceColField(ACol: Integer): TField;

  protected                               { Protected declarations }
   procedure DrawCell(ACol, ARow: Longint; ARect: TRect;
             AState: TGridDrawState); override;
   procedure Scroll(Distance: Integer); override;
   procedure LayoutChanged; override;
   procedure Notification( AComponent : TComponent;
                            Operation : TOperation ); override;
   procedure LinkActive(Value: Boolean); override;

  public                                  { Public declarations }
    constructor Create( AOwner : TComponent ); override;
    destructor Destroy; override;

  published                               { Published declarations }
   property Glyph : TBitmap read FGlyph write SetGlyph;
   property DataField : string read GetCheckCol write SetCheckCol;
   property CheckCriteria : string read FCheckCriteria write SetCheckCriteria;
   property Operator : TMatchOperators read FOperator write SetOperator
            default moEqual;
   property CaseSensitive : boolean read FCaseSensitive write SetCaseSensitive
            default False;
  end;

EFieldNotSelected = class( Exception );   { custom exception }
procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Extend', [ TDBCGrid ]);
end;
{ =================================================================== }
procedure TDBCGrid.SetGlyph( Value : TBitmap );
begin
  FGlyph.Assign( Value );
  if FGlyph.Handle = 0 then
     FGlyph.Handle := LoadBitMap( HInstance, 'CHECK_1' ); { set default }
  Invalidate;
end;
{ =================================================================== }
constructor TDBCGrid.Create( AOwner : TComponent );
begin
  inherited Create( AOwner );
  FGlyph := TBitmap.Create;
  FGlyph.Handle := LoadBitMap( HInstance, 'CHECK_1' ); { set default }
  FCheckCriteria := '';
  FCheckCol := TFieldDataLink.Create;               { Create DataLink }
  FCheckCol.Control := Self;
  FCheckCol.OnDataChange := DataChange;             { Assign Event Handler }
  FCheckCol.DataSource := DataLink.DataSource;      { same as grid }
  FCheckCol.FieldName := '';
  FOperator := moEqual;
  FCaseSensitive := False;
end;
{ =================================================================== }
destructor TDBCGrid.Destroy;
begin
  FGlyph.Free;
  FCheckCol.Free;
  FCheckCol := nil;
  inherited Destroy;
end;
{ =================================================================== }
procedure TDBCGrid.DrawCell(ACol, ARow: Longint; ARect: TRect; AState: TGridDrawState);
Var
Rectangle : TRect;
begin
   if ( ARow < FixedRows ) or                 { fixed rows have no data }
      ( not Datalink.Active ) then            { no source, no data }
      inherited DrawCell( ACol, ARow, ARect, AState )
   else if ( ACol = 0 ) and ( FCheckCol.Field <> nil )
      and ( MatchCriteria( GetCheckColValue( FFieldIndex, ARow -FixedRows ) ) )
      then
      begin
      Canvas.StretchDraw( ARect, FGlyph );
      end
   else
     if not ( ( ACol = 0 ) and
        ( MatchCriteria( GetCheckColValue( FFieldIndex, ARow - FixedRows ) ) ) )
        then
            inherited DrawCell( ACol, ARow, ARect, AState );
end;
{ =================================================================== }
function TDBCGrid.GetCheckCol : string;
begin
  Result := FCheckCol.FieldName;
end;
{ =================================================================== }
procedure TDBCGrid.SetCheckCol( const Value : string );
begin
   if Value <> FCheckCol.FieldName then
      begin
      FCheckCol.FieldName := Value;
      FCheckCol.DataSource := DataLink.DataSource;      { same as grid }
      ValidateCheckCol;
      LayoutChanged;
      end;
end;
{ =================================================================== }
procedure TDBCGrid.SetCheckCriteria( const Value : string );
begin
  if Value <> FCheckCriteria then
     begin
     FCheckCriteria := Value;
     LayoutChanged;
     end;
end;
{ =================================================================== }
{ FCheckCol.OnDataChange must be set to this, even though it doesn't
  do anything.  If FCheckCol.OnDataChange = nil the value doesn't change }
procedure TDBCGrid.DataChange( Sender : TObject );
begin
  if FCheckCol.Field = nil then
     Exit;
end;
{ =================================================================== }
function TDBCGrid.GetCheckColValue( ACol, ARow: Longint ): string;
var
  OldActive: Integer;
  Indicator: Integer;
  Highlight: Boolean;
  CheckField : TField;
begin
      Result := '';
      OldActive := DataLink.ActiveRecord;
      try
        DataLink.ActiveRecord := ARow;
        CheckField := GetSourceColField( ACol );
        if CheckField <> nil then
           Result := CheckField.AsString;    { ### future typed returns?}
      finally
        DataLink.ActiveRecord := OldActive;
      end;
end;
{ =================================================================== }
procedure TDBCGrid.Scroll( Distance: Integer );
var
i : integer;
InvRect: TRect;
begin
   { redraw check marks when we scroll }
   if FCheckCol.Field <> nil then
      for i := TopRow to VisibleRowCount do
          begin
          InvRect := CellRect( 0, i );    { what if multiple fixed? ### }
          InvalidateRect( Handle, @InvRect, False );
          end;
   inherited Scroll( Distance );
end;
{ =================================================================== }
procedure TDBCGrid.LayoutChanged;
begin
   if csLoading in ComponentState then Exit;
   if not HandleAllocated then Exit;

   if not (dgIndicator in Options) then
      begin
      Options := Options + [dgIndicator];  { ### change this to set 1 fixec col }
      end;
   inherited LayoutChanged;
end;
{ =================================================================== }
procedure TDBCGrid.SetOperator( Value: TMatchOperators );
begin
  if FOperator <> Value then
     begin
     FOperator := Value;
     Invalidate;
     end;
end;
{ =================================================================== }
procedure TDBCGrid.SetCaseSensitive( Value: boolean );
begin
   if FCaseSensitive <> Value then
      begin
      FCaseSensitive := Value;
      Invalidate;
      end;
end;
{ =================================================================== }
function TDBCGrid.MatchCriteria( sVar : string ) : boolean;
var
sCell, sConst : string;
begin
   if Not FCaseSensitive then
      begin
      sCell := UpperCase( sVar );
      sConst := UpperCase( FCheckCriteria );
      end
   else
      begin
      sCell := sVar;
      sConst := FCheckCriteria;
      end;
   Result := False;
   if FOperator = moEqual then
      begin
      if sCell = sConst then
         Result := True;
      Exit;
      end;
   if FOperator = moNotEqual then
      begin
      if sCell <> sConst then
         Result := True;
      Exit;
      end;
   if FOperator = moGreaterThan then
      begin
      if sCell > sConst then
         Result := True;
      Exit;
      end;
   if FOperator = moLessThan then
      begin
      if sCell < sConst then
         Result := True;
      Exit;
      end;
   if FOperator = moGTorEqual then
      begin
      if sCell >= sConst then
         Result := True;
      Exit;
      end;
   if FOperator = moLTorEqual then
      begin
      if sCell <= sConst then
         Result := True;
      Exit;
      end;
   if FOperator = moNull then       { not truly null but zero len str }
      begin
      if sCell = '' then
         Result := True;
      Exit;
      end;
    if FOperator = moNotNull then    { not truly notnull but not zero len str }
      begin
      if sCell <> '' then
         Result := True;
      Exit;
      end;
end;
{ =================================================================== }
procedure TDBCGrid.ValidateCheckCol;
var
i      : word;
s      : string;
bFound : boolean;
begin
   if (FCheckCol.DataSource = nil) or
      ( DataLink.Dataset.FieldCount < 1 ) then
      Exit;                       { can't do anything else til connected }
   bFound := False;
  { find out what field CheckCol is in DataSource.  Use dataset fields
    so nonvisible columns can be a CheckCol }
  { DataSet could be a TTable, TQuery or TStoredProc }
   for i := 0 to DataLink.DataSet.FieldCount -1 do
      begin
      if UpperCase( FCheckCol.FieldName ) =
         UpperCase( DataLink.DataSet.Fields[i].FieldName ) then
         begin
         FFieldIndex := i;
         bFound := True;
         Break;
         end;
      end;                                { end of for }
(*   if not bFound then
      begin
      if FCheckCol.FieldName = '' then
         s := 'No value for property CheckCol'
      else
         s := Format( 'CheckCol %s is not in grid', [FCheckCol.FieldName]);
      { Raise a Custom Exception
      raise EFieldNotSelected.Create( s ); }
      {MessageDlg( s , mtWarning, [mbOk], 0 ); }
      MessageBeep(0);
      end;                 do I want to do something here?       *)
end;
{ =================================================================== }
procedure TDBCGrid.Notification( AComponent : TComponent; Operation : TOperation );
begin
  inherited Notification( AComponent, Operation );
  if ( Operation = opRemove ) and
     ( DataLink <> nil ) and
     ( AComponent = DataSource ) then
    DataSource := nil;
end;
{ =================================================================== }
procedure TDBCGrid.LinkActive(Value: Boolean);
begin
  inherited LinkActive( Value );
  if Value then
     begin
     FCheckCol.DataSource := DataLink.DataSource; { set same as grid }
     ValidateCheckCol;
     end;
end;
{ =================================================================== }
{ get the field from the DataSet not the grid value }
function TDBCGrid.GetSourceColField(ACol: Integer): TField;
begin
  Result := nil;
  if (ACol >= 0) and Datalink.Active
                 and (ACol < DataLink.Dataset.FieldCount) then
    Result := Datalink.DataSet.Fields[ACol];
end;

end.