unit Finger;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, ExtCtrls, WinSock;

type
  ESocketError = class(Exception);
  EFingerError = class(Exception);
  TFinger = class(TComponent)
  private
    { Private declarations }
    FTimeOut : Integer;
    FQuery : string;
    procedure SetQuery(Value : string);
  protected
    { Protected declarations }
    Timer : TTimer;
    CurTick : Integer;
    TimedOut : boolean;
    MyWSAData : TWSADATA;
    FingerSocket : TSocket;
    FingerPort : word;
    ServerInAddr : u_long;
    ServerIPAddr : string;
    ServerName : string;
    AddInfo : string;
    RecvBuf : array[1..$4000] of Char;
    Canceled : boolean;
    procedure TimerOnTimer(Sender : TObject);
    procedure TimerOn;
    procedure TimerOff;
    procedure OpenSocket;
    procedure CloseSocket;
    function SocketErrorStr(Errno : word) : string;
    procedure ResolveRemoteHost;
    procedure FindFingerService;
    procedure Open;
    procedure Connect;
    procedure Close;
    procedure SendQuery;
    procedure RecvData;
    procedure ReInit;
  public
    { Public declarations }
    RecvLines : TStringList;
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
    procedure Execute;
    procedure Cancel;
  published
    { Published declarations }
    property TimeOut : Integer read FTimeOut write FTimeOut
                          default 40;
    property Query : string read FQuery write SetQuery;
  end;

procedure Register;

implementation

const
  Finger_Port = 79;

procedure Register;
begin
  RegisterComponents('Internet', [TFinger]);
end;

constructor TFinger.Create(AOwner : TComponent);
begin
  inherited Create(AOwner);
  if not (csDesigning in ComponentState) then
  begin
    if WSAStartUp($0101,MyWSADATA)<>0 then
      raise ESocketError.Create('Invalid Winsock Version');
  end;
  RecvLines:=TStringList.Create;
  FTimeOut:=40;
  Canceled:=false;
  TimedOut:=false;
  Timer:=TTimer.Create(Self);
  Timer.Enabled:=false;
  Timer.OnTimer:=TimerOnTimer;
end;

destructor TFinger.Destroy;
begin
  Timer.Free;
  RecvLines.Free;
  if not (csDesigning in ComponentState) then
    WSACleanUp;
  inherited Destroy;
end;

procedure TFinger.SetQuery(Value : string);
var
  i : byte;
begin
  i:=Pos('@',Value);
  if i=0 then
    raise EFingerError.Create('Invalid query string format.'^M^J+
                              'Valid format is user@domain');
  AddInfo:=Copy(Value,1,i-1);
  ServerName:=Copy(Value,i+1,Length(Value)-i);
  FQuery:=Value;
end;

function TFinger.SocketErrorStr(ErrNo : word) : string;
begin
  Result:='WinSock Error No. '+IntToStr(ErrNo);
end;

procedure TFinger.TimerOnTimer(Sender : TObject);
begin
  Dec(CurTick);
  if CurTick=0 then
  begin
    if WSAIsBlocking then
      WSACancelBlockingCall;
    TimerOff;
    TimedOut:=true;
  end;
end;

procedure TFinger.TimerOn;
begin
  Timer.Enabled:=true;
  CurTick:=FTimeOut;
end;

procedure TFinger.TimerOff;
begin
  Timer.Enabled:=false;
end;

procedure TFinger.ResolveRemoteHost;
var
  RemoteHost : PHostEnt;
  Buf : PChar;
  a : array[0..3] of byte;
  i : byte;
begin
  Buf:=StrAlloc(255);
  StrPCopy(Buf,ServerName);
  ServerInAddr:=Inet_Addr(Buf);
  if ServerInAddr=SOCKET_ERROR then
  begin
    StrPCopy(Buf,ServerName);
    TimerOn;
    RemoteHost:=GetHostByName(Buf);
    TimerOff;
    if RemoteHost=nil then
    begin
      StrDispose(Buf);
      if TimedOut then raise EFingerError.Create('Timed Out')
      else
      if Canceled then raise EFingerError.Create('Operation canceled.')
      else
      raise ESocketError.Create(SocketErrorStr(WSAGetLastError));
    end
    else
    begin
      for i:=0 to 3 do
        a[i]:=byte(RemoteHost^.h_addr_list^[i]);
      ServerIPAddr:=IntToStr(a[0])+'.'+IntToStr(a[1])+
        '.'+IntToStr(a[2])+'.'+IntToStr(a[3]);
      StrPCopy(Buf,ServerIPAddr);
      ServerInAddr:=Inet_Addr(Buf);
      if ServerInAddr=SOCKET_ERROR then
      begin
        raise EFingerError.Create('Error resolving remote host');
      end;
    end;
  end;
end;

procedure TFinger.FindFingerService;
var
  PSE : PServEnt;
begin
  PSE:=GetServByName('finger','tcp');
  if PSE=nil then
  begin
    FingerPort:=Finger_PORT;
  end
  else
    FingerPort:=htons(PSE^.s_port);
end;

procedure TFinger.OpenSocket;
begin
  FingerSocket:=Socket(PF_INET,SOCK_STREAM,IPPROTO_IP);
  if FingerSocket=TSocket(INVALID_SOCKET) then
  begin
    raise ESocketError.Create(SocketErrorStr(WSAGetLastError));
  end;
end;

procedure TFinger.Connect;
var
  RemoteAddress : TSockAddr;
begin
  with RemoteAddress do
  begin
    Sin_Family:=PF_INET;
    Sin_Port:=htons(FingerPort);
    Sin_addr:=TInAddr(ServerInAddr);
  end;
  TimerOn;
  if WinSock.Connect(FingerSocket,RemoteAddress,
             SizeOf(RemoteAddress))=SOCKET_ERROR then
  begin
    TimerOff;
    if TimedOut then raise EFingerError.Create('Timed Out')
    else
    if Canceled then raise EFingerError.Create('Operation Canceled.')
    else
    raise ESocketError.Create(SocketErrorStr(WSAGetLastError));
  end;
  TimerOff;
end;

procedure TFinger.ReInit;
begin
  TimedOut:=false;
  Canceled:=false;
  TimerOff;
end;

procedure TFinger.Open;
begin
  ReInit;
  ResolveRemoteHost;
  FindFingerService;
  OpenSocket;
  Connect;
end;

procedure TFinger.CloseSocket;
begin
  if FingerSocket<>TSocket(INVALID_SOCKET) then
  begin
    if WinSock.CloseSocket(FingerSocket)=0 then
    begin
      FingerSocket:=TSocket(INVALID_SOCKET);
    end
      else raise ESocketError.Create(SocketErrorStr(WSAGetLastError));
  end;
end;

procedure TFinger.Close;
begin
  CloseSocket;
end;

procedure TFinger.SendQuery;
var
  Buf : PChar;
begin
  Buf:=StrAlloc(255);
  StrPCopy(Buf,AddInfo+^M^J);
  TimerOn;
  if Winsock.Send(FingerSocket,Buf^,
     StrLen(Buf),0)=SOCKET_ERROR then
  begin
    TimerOff;
    StrDispose(Buf);
    if Canceled then raise EFingerError.Create('Operation Canceled.')
    else
    raise ESocketError.Create(SocketErrorStr(WSAGetLastError));
  end;
  TimerOff;
  StrDispose(Buf);
  if TimedOut then
    raise EFingerError.Create('Timed Out');
end;

procedure TFinger.RecvData;
var
  rc,i : Integer;
  Finished : boolean;
begin
  i:=1;
  FillChar(RecvBuf,$4000,0);
  repeat
    TimerOn;
    rc:=Winsock.recv(FingerSocket,RecvBuf[i],$4000,0);
    TimerOff;
    Finished:=TimedOut or (rc=0) or (rc=SOCKET_ERROR)
              or Canceled;
    Inc(i,rc);
  until Finished;
  RecvLines.SetText(@RecvBuf);
  TimerOff;
  if TimedOut then
    raise EFingerError.Create('Timed Out.');
  if rc=SOCKET_ERROR then
    raise ESocketError.Create(SocketErrorStr(WSAGetLastError));
  if Canceled then
    raise EFingerError.Create('Operation has been canceled.');
end;

procedure TFinger.Cancel;
begin
  Canceled:=true;
  if WSAIsBlocking then
    WSACancelBlockingCall;
end;

procedure TFinger.Execute;
begin
  Open;
  try
    SendQuery;
    RecvData;
  finally
    Close;
  end;
end;

end.
