unit TransBtn;

{$define OffScreen}


interface

uses WinTypes, WinProcs,
     SysUtils,
     Classes, Controls, Graphics, StdCtrls, ExtCtrls;

const
  DefaultSize         = 32;
  DefaultTransparency = True;
  VersionID           = '0.90 (alpha)';

type
  TGlyphModes  = (gmNone, gmDFM, gmFile, gmResource);
  TTransButton =
    class (TGraphicControl)

    public
    constructor Create(Owner: TComponent); override;
    destructor  Destroy;                   override;

    protected
    procedure   Paint;                     override;
    function    GetPalette: hPalette;      override;

    private
    GlyphHeight,
    GlyphWidth:       word;

    TransparentColor: TColor;

    fMask,
    fOriginalGlyph:   TBitMap;


    fGlyph:           TBitMap;
    fGlyphName:       TFileName;
    fGlyphID:         string;
    fGlyphMode:       TGlyphModes;

    fNumGlyphs:       word;

    fEmulateDisabled: boolean;
    fOutlineColor:    TColor;
    fTransparent:     boolean;

    fAllowAllUp,
    fDown,
    fPushed:          boolean;
    fGroupIndex:      word;

    fAutoSize:        boolean;

    fTracking:        boolean;

    fOnUp,
    fOnDown:          TNotifyEvent; {procedure (Sender: TObject) of object;}

    function GetVersion: string;
    procedure SetVersion(NewVersion: string);

    procedure SetMaskAndCtrlSize;
    procedure ClearGlyph;

    procedure SetGlyph(NewBitmap: TBitMap);
    function  GetGlyph: TBitmap;
    function  StoreGlyphP: boolean;
    procedure SetGlyphName(NewGlyphName: TFileName);
    function  GetGlyphName: TFileName;
    function  StoreGlyphNameP: boolean;
    procedure SetGlyphID(ID: string);
    function  GetGlyphID: string;
    function  StoreGlyphIdP: boolean;
    function  NewGlyph(Bitmap: TBitmap): boolean; {code common to all set fns}

    procedure SetEmulateDisabled(EmulateDisabled: boolean);
    procedure SetOutlineColor(OutlineColor: TColor);

    procedure SetNumGlyphs(NewNumGlyphs: word);
    procedure SetTransparent(NewTransparent: boolean);
    procedure SetDown(NewDown: boolean);
    procedure SetAllowAllUp(NewAllowAllUp: boolean);
    procedure SetGroupIndex(NewGroupIndex: word);

    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);   override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer);                       override;
    procedure Click;                                                              override;

    function OnGlyphP(X, Y: integer): boolean;

    published

    {new properties}
    property AutoSize:    boolean   read fAutoSize    write fAutoSize    default True;
             {Must be declared before any of the glyph properties, or they will be loaded
              before a non-default (ie, False) value is set}
    property Transparent: boolean   read fTransparent write SetTransparent
                          default DefaultTransparency;
    property Glyph:       TBitMap   read GetGlyph     write SetGlyph
                          stored    StoreGlyphP;
    property GlyphName:   TFileName read GetGlyphName write SetGlyphName
                          stored    StoreGlyphNameP;
    property GlyphID:     string    read GetGlyphID   write SetGlyphID
                          stored    StoreGlyphIdP;
    property NumGlyphs:   word      read fNumGlyphs   write SetNumGlyphs default 1;

    property OutlineColor: TColor     read    fOutlineColor    write SetOutlineColor
                                      default clBlack;
    property EmulateDisabled: boolean read    fEmulateDisabled write SetEmulateDisabled
                                      default True;

    property AllowAllUp:  boolean read fAllowAllUp write SetAllowAllUp   default False;
    property GroupIndex:  word    read fGroupIndex write SetGroupIndex   default 0;

    property Down:        boolean read fDown       write SetDown         default False;

    property Version: string      read GetVersion  write SetVersion      stored False;

    {new events}
    property OnUp: TNotifyEvent   read fOnUp       write fOnUp;
    property OnDown: TNotifyEvent read fOnDown     write fOnDown;

    {Republished properties}
    property DragCursor;
    property DragMode;
    property ShowHint;
    property ParentShowHint;
    property Visible;
    property Enabled;
    property Height default DefaultSize;
    property Width  default DefaultSize;

    {Republished events}
    property OnClick;
    property OnDblClick;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;

    end;

type
  TExBitmap = record
              Bitmap:  hBitmap;
              Palette: hPalette;
              end;

function LoadBitmapExtended(Instance: THandle; ResourceBitmap: PChar): TExBitmap;

procedure Register;

implementation

uses DsgnIntf, Dialogs, Forms;

const
  InternalSetoff = '(';
  inNoneTxt      = 'None';
  InDfmTxt       = 'Glyph';
  InFileTxt      = 'GlyphName';
  InResTxt       = 'GlyphID';
  InternalClose  = ')';
  InNone         = InternalSetoff + InNoneTxt  + InternalClose;
  InDfm          = InternalSetoff + InDfmTxt  + InternalClose;
  InFile         = InternalSetoff + InFileTxt + InternalClose;
  InRes          = InternalSetoff + InResTxt  + InternalClose;

{GlyphName editor}

type
  TFileNameEditor = class(TPropertyEditor)
                    function GetValue: string;                   override;
                    procedure SetValue(const Value: string);     override;
                    procedure Edit;                              override;
                    function GetAttributes: TPropertyAttributes; override;
                    end;

function TFileNameEditor.GetValue: string;
begin
  Result := GetStrValue;
end;

procedure TFileNameEditor.SetValue(const Value: string);
begin
  SetStrValue(Value);
end;

procedure TFileNameEditor.Edit;
var
  FileDialog: TOpenDialog;
begin
  FileDialog := TOpenDialog.Create(Application);
  try
    with FileDialog do
      begin
      Title      := 'Choose a glyph file';
      Filter     := 'Glyph files|*.bmp';
      DefaultExt := 'bmp';
      FileName   := GetStrValue;
      if FileName[1] = InternalSetoff then FileName := '';
      Options    := [ofFileMustExist, ofHideReadOnly];
      if Execute then SetStrValue(FileName);
      end;
  finally
    FileDialog.Free;
  end;
end;

function TFileNameEditor.GetAttributes: TPropertyAttributes;
begin
  Result := [paDialog, paMultiSelect];
end;

{TTransButton}

const
  cmNotSrcAnd = $00220326;

{setup/teardown}

  constructor TTransButton.Create(Owner: TComponent);
  begin
    Height           := DefaultSize; GlyphHeight := DefaultSize;
    Width            := DefaultSize; GlyphWidth  := DefaultSize;
    fAutoSize        := True;
    fNumGlyphs       := 1;
    fTransparent     := DefaultTransparency;
    fGlyph           := TBitmap.Create;
    fOriginalGlyph   := TBitmap.Create;
    fMask            := TBitmap.Create;
    fEmulateDisabled := True;
    fOutlineColor    := clBlack; {Not strictly necessary, as clBlack=0, but ...}
    Inherited Create(Owner);
  end;

  destructor TTransButton.Destroy;
  begin
    try
      fGlyph.Free;
      fMask.Free;
      fOriginalGlyph.Free;
    except
      on Exception do ;
    end;
    Inherited Destroy;
  end;

  type
    pBitmap = ^ TBitmap;

  procedure ResetBitmaps(Bitmaps: array of pBitmap); near;
  var
    Index: word;
  begin
    for Index := Low(Bitmaps) to High(Bitmaps) do
      begin
      Bitmaps[Index]^.Free;
      Bitmaps[Index]^ := TBitmap.Create;
      {ReleasePalette/ReleaseHandle/Assign causes resource & memory leaks}
      end;
  end;

{drawing}

  function  TTransButton.GetPalette: hPalette;
  begin
    if fGlyph.Empty
      then Result := Inherited GetPalette
      else Result := fGlyph.Palette;
  end;

  function CreateDisabledBitmapAlaDelphi(FOriginal: TBitmap; OutlineColor: TColor): TBitmap;
  { This is the code from the Delphi BUTTONS unit to create a disabled }
  { version of a button's glyph                                        }
  var
    TmpImage,
    MonoBmp: TBitmap;
    IRect:   TRect;
  begin
    IRect    := Rect(0, 0, fOriginal.Width, fOriginal.Height);
    TmpImage := TBitmap.Create;
    TmpImage.Width  := fOriginal.Width;
    TmpImage.Height := fOriginal.Height;
    MonoBmp  := TBitmap.Create;
    try
      with MonoBmp do
        begin
        Assign(FOriginal);
        Canvas.Brush.Color := OutlineColor;
        if Monochrome then
          begin
          Canvas.Font.Color := clWhite;
          Monochrome := False;
          Canvas.Brush.Color := clWhite;
          end;
        Monochrome := True;
        end;
      with TmpImage.Canvas do
        begin
        Brush.Color := clBtnFace;
        FillRect(IRect);
        Brush.Color := clBlack;
        Font.Color  := clWhite;
        CopyMode    := MergePaint;
        Draw(IRect.Left + 1, IRect.Top + 1, MonoBmp);
        CopyMode := SrcAnd;
        Draw(IRect.Left, IRect.Top, MonoBmp);
        Brush.Color := clBtnShadow;
        Font.Color  := clBlack;
        CopyMode    := SrcPaint;
        Draw(IRect.Left, IRect.Top, MonoBmp);
        CopyMode    := SrcCopy;
        end;
      Result := TmpImage;
    finally
      MonoBmp.Free;
    end;
  end;

  function MaxW(A, B: word): word; near;
  begin
    if A > B then Result := A else Result := B;
  end;

  function MinW(A, B: word): word; near;
  begin
    if A < B then Result := A else Result := B;
  end;

  procedure TTransButton.Paint;
  type
    TBtnModes = (BtnUp, BtnDisabled, BtnDown, BtnStayDown);
  var
    Src, Dst: TRect;
    Offset:   word;
    fMode:    TBtnModes;
    Fake:     record
              Disable, Down:    boolean;
              Glyph:            TBitmap;
              end;
    Image,
    DrawGlyph: TBitmap;
  begin
    if fGlyph.Empty
      then begin
           Canvas.Brush.Style := bsClear; {*Don't* fill the rectangle}
           Canvas.Pen.Style   := psDot;
           Canvas.Rectangle(0, 0, Width, Height);
           end
      else begin {Normal drawing}
           if not Enabled
             then fMode := BtnDisabled
             else if {Enabled and} (not (fDown or fPushed))
               then fMode := BtnUp
               else if {Enabled and (fDown or fPushed) and} (fPushed or (fGroupIndex = 0))
                 then fMode := BtnDown
                 else fMode := BtnStayDown;
           {Default emulations}
             Fake.Disable := False;
             Fake.Down    := False;
           case fMode of {Set actual emulations, as necessary}
             BtnDisabled: if fNumGlyphs < 2 then
                            begin
                            fMode := BtnUp;
                            Fake.Disable := fEmulateDisabled;
                            end;
             BtnDown:     if fNumGlyphs < ord(fMode) + 1 then
                            begin
                            Fake.Down := True;
                            fMode := BtnUp;
                            end;
             BtnStayDown: begin
                          Fake.Down := fNumGlyphs < 3;
                          case fNumGlyphs of
                            1, 2: fMode := BtnUp;
                            3:    fMode := BtnDown;
                            end;
                          end;
             end;
           Offset := ord(fMode) * GlyphWidth;
           Dst := Rect(0, 0, MinW(Width, GlyphWidth), MinW(Height, GlyphHeight));
           Src := Dst;
           Inc(Src.Left,  Offset);
           Inc(Src.Right, Offset);
           if Fake.Down then
             begin
             Inc(Dst.Top);    Inc(Dst.Right);
             Inc(Dst.Bottom); Inc(Dst.Left);
             end;
           if Fake.Disable
             then begin
                  Fake.Glyph := CreateDisabledBitmapAlaDelphi(fOriginalGlyph, fOutlineColor);
                  if fTransparent then {Turn the transparent color black}
                    begin
                    Fake.Glyph.Canvas.CopyMode := cmNotSrcAnd;
                    Fake.Glyph.Canvas.Draw(0, 0, fMask);
                    end;
                  DrawGlyph  := Fake.Glyph;
                  end
             else DrawGlyph  := fGlyph;
           {Get a screen image}
{$ifdef OffScreen}
             Image := TBitmap.Create;
             with Image do
               begin
               Width  := Self.Width;
               Height := Self.Height;
               Canvas.CopyMode := cmSrcCopy;
               Canvas.CopyRect(Self.ClientRect, Self.Canvas, Self.ClientRect);
{$endif}
           {Draw the button on the image}
               if fTransparent
                 then begin
                      Canvas.CopyMode := cmSrcAnd;    Canvas.CopyRect(Dst, fMask.Canvas, Src);
                      Canvas.CopyMode := cmSrcPaint;
                      end
                 else Canvas.CopyMode := cmSrcCopy;
               Canvas.CopyRect(Dst, DrawGlyph.Canvas, Src);
{$ifdef OffScreen}
               end;
           {Write the changes to the screen}
             Canvas.CopyMode := cmSrcCopy; Canvas.Draw(0, 0, Image);
           {Cleanup}
             Image.Free;
{$endif}
             if Fake.Disable then Fake.Glyph.Free;
           end; {not fGlyph.Empty - ie, normal drawing}
  end;

{bitmap set/get}

  procedure TTransButton.SetMaskAndCtrlSize;
  begin
    ResetBitmaps([@ fMask, @ fOriginalGlyph]);
    if fTransparent and (not fGlyph.Empty) then
      begin
      fOriginalGlyph.Assign(fGlyph); {Copy the glyph to fOriginalGlyph}
      {Create fMask}
        TransparentColor := fGlyph.Canvas.Pixels[0, 0];
        fMask.Handle := CreateBitmap (fGlyph.Width, fGlyph.Height, 1, 1, nil); {a mono bitmap}
        fMask.Canvas.CopyMode := cmMergeCopy; {PSa - Pattern AND Source}
        SetBkColor(fGlyph.Canvas.Handle, TransparentColor);
        fMask.Canvas.Draw(0, 0, fGlyph); {fMask := Source=TC}
      {Apply fMask to fGlyph - turn transparent parts black}
        fGlyph.Canvas.CopyMode := cmNotSrcAnd;
        fGlyph.Canvas.Draw(0, 0, fMask);
      end;
    {Set ctrl size}
      if fGlyph.Empty
        then begin
             GlyphHeight := DefaultSize;
             GlyphWidth  := DefaultSize;
             end
        else begin
             GlyphHeight := fGlyph.Height;
             GlyphWidth  := fGlyph.Width div fNumGlyphs;
             end;
    if AutoSize then
      begin
      Height := GlyphHeight;
      Width  := GlyphWidth;
      end;
    Invalidate;
  end;

  function TTransButton.NewGlyph(Bitmap: TBitmap): boolean; {code common to all set fns}
  begin
    try
      ResetBitmaps([@ fGlyph]);
      fGlyph.Assign(Bitmap); 
      SetMaskAndCtrlSize;
      Result := True;
    except
      on Exception do begin
                      ClearGlyph;
                      Result     := False;
                      end;
    end;
  end;

  procedure TTransButton.ClearGlyph;
  begin
    ResetBitmaps([@ fGlyph, @ fMask, @ fOriginalGlyph]);
    fGlyphMode := gmNone;
    GlyphHeight := DefaultSize;
    GlyphWidth  := DefaultSize;
    if AutoSize then
      begin
      Height := GlyphHeight;
      Width  := GlyphWidth;
      end;
  end;

  procedure TTransButton.SetGlyph(NewBitmap: TBitmap);
  begin                          messagebeep(0);
    if not NewBitMap.Empty
      then begin
           if NewGlyph(NewBitmap) then fGlyphMode := gmDFM;
           end
      else ClearGlyph;
  end;

  procedure TTransButton.SetGlyphName(NewGlyphName: TFileName);
  var
    Bitmap: TBitmap;
  begin
    if NewGlyphName = ''
      then ClearGlyph
      else try
             if ExtractFileExt(NewGlyphName) = '' then
               begin
               if NewGlyphName[Length(NewGlyphName)] <> '.' then
                 NewGlyphName := NewGlyphName + '.';
               NewGlyphName := NewGlyphName + 'bmp';
               end;
             Bitmap := TBitmap.Create;
             try
               Bitmap.LoadFromFile(NewGlyphName);
               if NewGlyph(Bitmap) then
                 begin
                 fGlyphName := NewGlyphName;
                 fGlyphMode := gmFile;
                 end;
             finally
               Bitmap.Free;
             end;
           except
             on Exception do {nothing} ;
           end;
  end;

  function Quad2Pal(RGB: TRgbQuad): TPaletteEntry; near;
  begin
    Result.peRed   := RGB.rgbRed;
    Result.peGreen := RGB.rgbGreen;
    Result.peBlue  := RGB.rgbBlue;
    Result.peFlags := 0;
  end;

  function LoadBitmapExtended(Instance: THandle; ResourceBitmap: PChar): TExBitmap;
  type
    aRgbQuad       = array[0..255] of TRgbQuad;
    aPaletteEntry  = array[0..255] of TPaletteEntry;
    paRgbQuad      = ^ aRgbQuad;
    paPaletteEntry = ^ aPaletteEntry;
  var
    Found, Loaded: THandle;
    Header:        pBitmapInfoHeader;
    DC:            hDC;
    oBitmap:       hBitmap;
    dPalette:      hPalette;
    BitCount,
    Colors,
    Idx:           word;
    Quad:          paRgbQuad;
    Pal:           paPaletteEntry;
    PaletteSpec:   record
                   case byte of
                     1: (Log: TLogPalette);
                     2: (Hdr: TLogPalette;
                         Pad: array[1..255] of TPaletteEntry)
                     end;
  begin
    Result.Bitmap  := 0;
    Result.Palette := 0;
    Found  := FindResource(Instance, ResourceBitmap, rt_Bitmap);
    if Found <> 0 then
      begin
      Loaded := LoadResource(Instance, Found);
      if Loaded <> 0 then
        try
          Header := LockResource(Loaded);
          if Header <> Nil then
            try
              if not ((Header^.biSize in [ SizeOf(TBitmapInfoHeader),
                                           SizeOf(TBitmapCoreHeader)]) and
                      (Header^.biBitCount in [1, 4, 8])) then EXIT;
              {Invalid header, or a 24-bit DIB}

              BitCount := Header^.biBitCount;
              Colors   := 1 shl BitCount;
              if (Header^.biClrUsed <> 0) and (Header^.biClrUsed < Colors) then
                Colors := Header^.biClrUsed;
              PaletteSpec.Log.palVersion    := $0300;
              PaletteSpec.Log.palNumEntries := Colors;
              pointer(Quad) := PChar(Header) + Header^.biSize;
                               {color table starts right after the header}
              pointer(Pal)  := @ PaletteSpec.Log.palPalEntry ;
              for Idx := 0 to PaletteSpec.Log.palNumEntries - 1 do
                Pal^[Idx] := Quad2Pal( Quad^[Idx] );
              Result.Palette := CreatePalette(PaletteSpec.Log);

              DC := CreateCompatibleDC(0); {Screen DC}
              if DC = 0 then EXIT else
                try
                  oBitmap  := SelectObject(DC, CreateBitmap( 1, 1, 1,
                                                             GetDeviceCaps(DC, BitsPixel),
                                                             Nil));
                  if BitCount = 8 then
                    begin
                    dPalette := SelectPalette(DC, Result.Palette, False);
                    RealizePalette(DC);
                    end;
                  Result.Bitmap :=
                    CreateDIBitmap( DC, Header^, cbm_Init,
                                    PChar(Header) + Header^.biSize +
                                      Colors * SizeOf(TRGBQuad),
                                    pBitmapInfo(Header)^, dib_RGB_Colors );
                finally
                  if BitCount = 8 then
                    begin
                    SelectPalette(DC, dPalette, False); {Not sure this is necessary ....}
                    end;
                  DeleteObject(SelectObject(DC, oBitmap));
                  DeleteDC(DC);
                end;
            finally
              UnlockResource(Loaded);
            end;
        finally
          FreeResource(Loaded);
        end;
      end;
    if (Result.Bitmap = 0) and (Result.Palette <> 0) then {free palette resource ....}
      begin
      DeleteObject(Result.Palette);
      Result.Palette := 0;
      end;
  end;

  procedure TTransButton.SetGlyphID(ID: string);
  var
    Resource:      record
                   Handles: TExBitmap;
                   Bitmap:  TBitmap;
                   end;
    Number, Error: word;
    IsId:          boolean;
    pID:           PChar;
    zID:           array[0..255] of char;
  begin
    IsId := False;
    if ID = ''
      then ClearGlyph
      else begin
           Val(ID, Number, Error);
           if Error = 0
             then pID := MakeIntResource(Number)
             else pID := StrPCopy(zID, ID);
           Resource.Handles := LoadBitmapExtended(hInstance, pID);
           if Resource.Handles.Bitmap = 0 {Not found?}
             then begin
                  IsID := csDesigning in ComponentState;
                  {Allow a GlyphID that's not present at designtime}
                  if IsId then ClearGlyph;
                  end
             else begin
                  Resource.Bitmap := TBitmap.Create;
                  try
                    Resource.Bitmap.Handle  := Resource.Handles.Bitmap;
                    Resource.Bitmap.Palette := Resource.Handles.Palette;
                    IsId := NewGlyph(Resource.Bitmap);
                  finally
                    Resource.Bitmap.Free;
                  end;
                  end;
           if IsID
             then begin
                  fGlyphID   := ID;
                  fGlyphMode := gmResource;
                  end
             else ClearGlyph;
           end;
  end;


  function TTransButton.GetGlyph: TBitmap;
  begin
    if Transparent
      then Result := fOriginalGlyph
      else Result := fGlyph;
  end;

  function TTransButton.GetGlyphName: TFileName;
  begin
    case fGlyphMode of
      gmDFM:      Result := InDFM;
      gmFile:     Result := fGlyphName;
      gmResource: Result := InRes;
      else        Result := InNone;
      end;
  end;

  function TTransButton.GetGlyphID: string;
  begin
    case fGlyphMode of
      gmDFM:      Result := InDFM;
      gmFile:     Result := InFile;
      gmResource: Result := fGlyphID;
      else        Result := InNone;
      end;
  end;


  function  TTransButton.StoreGlyphP: boolean;
  begin
    Result := fGlyphMode = gmDFM;
  end;

  function  TTransButton.StoreGlyphNameP: boolean;
  begin
    Result := fGlyphMode = gmFile;
  end;

  function  TTransButton.StoreGlyphIdP: boolean;
  begin
    Result := fGlyphMode = gmResource;
  end;

{misc properties}

  procedure TTransButton.SetTransparent(NewTransparent: boolean);
  begin
    if fTransparent <> NewTransparent then
      begin
      if fTransparent then
        begin
        ResetBitmaps([@ fGlyph]);
        fGlyph.Assign(fOriginalGlyph);
        end;
      fTransparent := NewTransparent;
      SetMaskAndCtrlSize;
      end;
  end;

  procedure TTransButton.SetNumGlyphs(NewNumGlyphs: word);
  begin
    if NewNumGlyphs < 1
      then NewNumGlyphs := 1
      else if NewNumGlyphs > 4 then NewNumGlyphs := 4;
    if NewNumGlyphs <> fNumGlyphs then
      begin
      fNumGlyphs := NewNumGlyphs;
      SetMaskAndCtrlSize;
      end;
  end;

  procedure TTransButton.SetGroupIndex(NewGroupIndex: word);
  begin
    fGroupIndex := NewGroupIndex;
    AllowAllUp  := AllowAllUp; {Force all in group to have this ctrl's AAU property,
                                ala SpeedButtons}
  end;

  procedure TTransButton.SetAllowAllUp(NewAllowAllUp: boolean);
  var
    Index: word;
  begin
    if GroupIndex = 0
      then fAllowAllUp := NewAllowAllUp
      else for Index := 0 to Parent.ControlCount - 1 do
        if Parent.Controls[Index] is TTransButton then
          with TTransButton(Parent.Controls[Index]) do
            if GroupIndex = Self.GroupIndex then fAllowAllUp := NewAllowAllUp;
  end;

  function  TTransButton.GetVersion: string;
  begin
    Result := VersionID;
  end;

  procedure TTransButton.SetVersion(NewVersion: string);
  begin
  end;

  procedure TTransButton.SetDown(NewDown: boolean);
  var
    Index:  word;
  begin
    if fDown <> NewDown then
      begin
      fDown := NewDown;
      Invalidate;
      if fDown and (GroupIndex <> 0) then
        for Index := 0 to Parent.ControlCount - 1 do
          if (Parent.Controls[Index] <> Self) and
             (Parent.Controls[Index] is TTransButton) then
            with TTransButton(Parent.Controls[Index]) do
              if (GroupIndex = Self.GroupIndex) and Down then
                begin
                Down := False;
                BREAK; {There can never be more than one down at once ...}
                end;
      if NewDown
        then begin if Assigned(fOnDown) then fOnDown(Self) end
        else begin if Assigned(fOnUp)   then fOnUp(Self)   end;
      {Give toggles a chance to affect underlying state}
      end;
  end;

  procedure TTransButton.SetEmulateDisabled(EmulateDisabled: boolean);
  begin
    fEmulateDisabled := EmulateDisabled;
    if (not Enabled) and (fNumGlyphs = 1) then {this setting has immediate visible effect}
      Invalidate;
  end;

  procedure TTransButton.SetOutlineColor(OutlineColor: TColor);
  begin
    fOutlineColor := OutlineColor;
    if (not Enabled) and (fNumGlyphs = 1) then {this setting has immediate visible effect}
      Invalidate;
  end;

{event handlers}

  function TTransButton.OnGlyphP(X, Y: integer): boolean;
  begin
    Result := PtInRect(ClientRect, Point(X, Y)) and
              ((not fTransparent) or (fMask.Canvas.Pixels[X, Y] = 0));
  end;

  procedure TTransButton.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
  begin
    if (Button = mbLeft) and OnGlyphP(X, Y) and
       ((not Down) or (AllowAllUp and (GroupIndex <> 0))) then
      begin
      fTracking := True;
      fPushed   := True;
      Invalidate;
      end;
    Inherited MouseDown(Button, Shift, X, Y);
  end;

  procedure TTransButton.MouseMove(Shift: TShiftState; X, Y: Integer);
  var
    WasPushed: boolean;
  begin
    if fTracking then
      begin
      WasPushed := fPushed;
      fPushed   := OnGlyphP(X, Y);
      if fPushed <> WasPushed then Invalidate;
      end;
    Inherited MouseMove(Shift, X, Y);
  end;

  procedure TTransButton.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
  begin
    if fPushed then Invalidate; {refresh when done}
    fTracking := False;
    fPushed   := False;
    Inherited MouseUp(Button, Shift, X, Y);
  end;

  procedure TTransButton.Click;
  begin
    fTracking := False; {whether or not ....}
    if fPushed then
      begin
      fPushed := False;
      if GroupIndex = 0
        then Invalidate
        else if Down
          then Down := not AllowAllUp
          else Down := True;
      Inherited Click; {Call the user event handler}
      end;
  end;

{registration}

procedure Register;
begin
  RegisterComponents('Additional', [TTransButton]);
  RegisterPropertyEditor(TypeInfo(TFileName), TTransButton, '', TFileNameEditor);
end;

end.
