{+------------------------------------------------------------
 | Unit Drives
 |
 | Version: 2.1  Last modified: 04/25/95, 21:49:56
 | Author : P. Below  CIS [100113,1101]
 | Project: Common utilities
 | Language: Delphi 16Bit
 |           THIS UNIT WILL NOT WORK WITH WIN32!
 | Units required (only for the Implementation section):
 |   Windows: Wintypes (Borland),
 |            WinProcs (Borland), DPMI (own);
 | Description:
 |   This Unit collects routines to gather information about
 |   the disk drives on a system in both DOS and Windows. You can
 |   build a drive map with a single procedure call. In addition
 |   to drive type checking the volume name, serial number, and
 |   FAT type for local fixed disks or the netshare name for
 |   networked drives are also retrieved.
 |   Low-level routines for all the subfunctions involved in
 |   building the drive map are exported, too. You can thus get
 |   the media info or disk paramter block for floppy disk or other
 |   removable media drives "by hand", if necessary.
 |   Most of the low level stuff uses DOS IOCTL functions, even
 |   for the Windows version ( GetDriveType is just to limited ).
 |   CD-ROM identification uses the MSCDEX int 2Fh interface.
 |
 | Copyright (C) 1994,1995 by Peter Below
 |   This code is released to the public domain, use it as you see fit
 |   in your own programs (at your own risk, of course) but you should
 |   include a note of the origin of this code in your program.
 +------------------------------------------------------------}
Unit Drives;

Interface

Const
  DRIVE_CDROM = 5;
  DRIVE_RAM   = 6;

  Min_DriveNums = 1;  (* drive a: *)
  Max_DriveNums = 26; (* drive z: *)
Type
  TDriveTypes = ( FLOPPY, HARDDISK, REMAPPED, REMOTE, CD_ROM,
                  RAMDISK, SUBSTITUTED, SHARED, INVALID );
  TDTypeSet   = SET OF TDriveTypes;
  TDriveNums  = Min_DriveNums..Max_DriveNums;
                (* range A: To Z: of drive letters *)
  TDriveSet   = SET OF TDriveNums;
  TDeviceName = String[127];
  TDiskInfo   = Record (* this is a DOS structure used by GetMediaID
                          but modified to allow for Pascal Strings *)
                  InfoLevel: WORD;
                  serialNo: LongInt;
                  volName : String[11];
                  FATType : String[8];
                End;
  TDriveInfo  = Record
                  Flags: TDTypeSet;
                  Case Boolean OF
                    TRUE:  (* For network drives *)
                      (DevName: TDeviceName);
                    FALSE: (* For all other drives *)
                      (Info: TDiskInfo);
                End;
  TDriveMap   = ARRAY [TDriveNums] OF TDriveInfo;
  PDriveMap   = ^TDriveMap;

  LP_DPB = ^DPB;
  DPB = Record  (* Disk Parameter Block as per MSDOS Programmer's Reference *)
    dpbDrive       : BYTE;
    dpbUnit        : BYTE;
    dpbSectorSize  : WORD;
    dpbClusterMask : BYTE;
    dpbClusterShift: BYTE;
    dpbFirstFAT    : WORD;
    dpbFATCount    : BYTE;
    dpbRootEntries : WORD;
    dpbFirstSector : WORD;
    dpbMaxCluster  : WORD;
    dpbFATSize     : WORD;
    dpbDirSector   : WORD;
    dpbDriverAddr  : Pointer;
    dpbMedia       : BYTE;
    dpbFirstAccess : BYTE;
    dpbNextDPB     : LP_DPB;
    dpbNextFree    : WORD;
    dpbFreeCnt     : WORD;
  End;

  DWord = LongInt;
  TDeviceParams = Record
    (* see GetDeviceParams function for detailed description *)
    dpSpecFunc : Byte;    (* function selection for $440D/$60 *)
    dpDevType  : Byte;    (* drive type *)
    dpDevAttr  : Word;    (* device attributes *) 
    dpCylinders: Word;    (* number of cylinders *)
    dpMediaType: Byte;    (* media type *)
                          (* start of bios parameter block *)
    dpBytesPerSec: Word;  (* bytes per sector *)
    dpSecPerClust: Byte;  (* sectors per cluster *)
    dpResSectors : Word;  (* number of reserved sectors *)
    dpFATs       : Byte;  (* number of file allocation tables *)
    dpRootDirEnts: Word;  (* number of root directory entries *)
    dpSectors    : Word;  (* total number of sectors *)
    dpMedia      : Byte;  (* media descriptor *)
    dpFATSecs    : Word;  (* number of sectors per FAT *)
    dpSecPerTrack: Word;  (* number of sectors per track *)
    dpHeads      : Word;  (* number of heads *)
    dpHiddenSecs : DWord; (* number of hidden sectors *)
    dpHugeSectors: DWord; (* number of sectors if dpSectors = 0 *)
    dpReserved   : array [ 0..5 ] of Byte; 
  End;

  TDriveState= ( DS_NO_DISK, DS_UNFORMATTED_DISK, DS_EMPTY_DISK,
               DS_DISK_WITH_FILES );

Function DriveState( n: TDriveNums ): TDriveState;
Procedure MyGetDriveType( n: TDriveNums; Var f: TDTypeSet );
Procedure GetDriveInfo( n: TDriveNums; Var di: TDriveInfo );
Procedure BuildDriveMap( Var DMap: TDriveMap    );

Function MSCDexIsloaded: Boolean;
Function DriveIsCDROM( n: TDriveNums ): Boolean;
Function DriveIsRamdisk( n: TDriveNums ): Boolean;
Function GetDiskParameterBlock( drive: TDriveNums ): LP_DPB;
Function GetLastdrive: TDriveNums;

{ the following functions map directly to DOS IOCTL calls }
Function MediumIsRemovable( n: TDriveNums ): TDriveTypes;
Function DriveIsShared( n: TDriveNums ): Boolean;
Function DriveIsRemote( n: TDriveNums ): Boolean;
Function DriveIsRemapped( n: TDriveNums ): Boolean;
Function DriveIsSubstituted( n: TDriveNums ): Boolean; 
Function GetMediaID ( Drive: Word; Var info: TDiskInfo ): Boolean;
Function GetDriveMapping( n: TDriveNums ): TDriveNums;
Procedure MapLogicalDrive( n: TDriveNums );
Function GetDeviceParams( drive: Byte; Var dp: TDeviceParams; 
                          getDefaults: Boolean ): Word;
Function GetUnittype( drive: Byte ): byte;

{ The InSitu.. procs convert a pascal string to a C string and vice
  versa using the originals memory ( p points to this ) }
Function InSituPasToPchar( p: PString ): Pchar;
Function InSituPcharToPas( p: PChar ): PString;

Implementation

Uses SysUtils, Wintypes, WinProcs, DPMI;

{ Converts a pascal string pointed to by p to a zero-terminated string
  without requiring any new memory.  Returns p }
Function InSituPasToPchar( p: PString ): PChar;
  Var
    len: Integer;
  Begin
    len := Length( p^ );
    Move( p^[ 1 ], p^[ 0 ], len );
    p^[ len+1 ] := #0;
    Result := PChar(p);
  End; { InSituPasToPchar }

{ Converts a zero-terminated string pointed to by p to a standard Pascal
  string without requiring any new memory.  Returns p }
Function InSituPcharToPas( p: PChar ): PString;
  Var
    len: Integer;
  Begin
    len := StrLen( p );
    Move( p[ 0 ], p[ 1 ], len );
    If len > 255 Then
      len := 255;
    p^ := Chr( len );
    Result := PString( p );
  End; (* InSituPcharToPas *)

{************************************************************
 * Function MediumIsRemovable
 *
 * Parameters:
 *  n: the drive number to check, 1= A:, 2=B: etc.
 * Returns:
 *  one of the three drive types INVALID, FLOPPY or HARDDISK
 * Description:
 *  This function uses DOS IOCTL function $4408 to check a drive
 *  for removable media. It can also be used to check for a drives
 *  existance.
 *
 *Created: 02/26/95 20:49:03 by P. Below
 ************************************************************}
Function MediumIsRemovable( n: TDriveNums ): TDriveTypes;
  Var
    res: Word;
  Begin
    asm
      mov ax, $4408   (* IOCTL is drive changeable function *)
      mov bl, n
      call Dos3Call
      mov res, ax
      { Our handling of the result makes the following asumptions here:
        If the function succeeds (carry flag clear), the return in ax
        will be 0 for removable and 1 for fixed medium.
        If the function fails (carry flag set), the error code in ax
        will be 1, if the device driver does not know how to handle the
        function (in which case we assume a fixed disk, also ax=1, safe bet
        according to MS docs) or it will be $F, if the drive is invalid.
       }
    End; { asm }
    Case res Of
      0: Result := FLOPPY;
      1: Result := HARDDISK
    Else
      Result := INVALID;
    End; { Case }
  End; (* MediumIsRemovable *)

{************************************************************
 * Function DriveIsShared
 *
 * Parameters:
 *  n: the drive number to check, 1= A:, 2=B: etc.
 * Returns:
 *  TRUE, if the drive is Shared, FALSE if it is local or invalid.
 * Description:
 *  This function uses DOS IOCTL function $4409 to check whether
 *  a drive is remote or local.
 *
 *Created: 02/26/95 21:12:32 by P. Below
 ************************************************************}
Function DriveIsShared( n: TDriveNums ): Boolean; Assembler;
  Asm
    mov ax, $4409   (* IOCTL is drive remote function *)
    mov bl, n
    call Dos3Call
    mov ax, False   (* assume error, in which case we return false *)
    jc @error
    and dx, $200  (* Shared drives have bit 9 set *)
    jz @error
    inc ax
  @error:
  End; (* DriveIsShared *)

{************************************************************
 * Function DriveIsSubstituted
 *
 * Parameters:
 *  n: the drive number to check, 1= A:, 2=B: etc.
 * Returns:
 *  TRUE, if the drive is substituted, FALSE if it is local or invalid.
 * Description:
 *  This function uses DOS IOCTL function $4409 to check whether
 *  a drive is remote or local.
 *
 *Created: 02/26/95 21:12:32 by P. Below
 ************************************************************}
Function DriveIsSubstituted( n: TDriveNums ): Boolean; Assembler;
  Asm
    mov ax, $4409   (* IOCTL is drive remote function *)
    mov bl, n
    call Dos3Call
    mov ax, False   (* assume error, in which case we return false *)
    jc @error
    and dx, $8000  (* substituted drives have bit 15 set *)
    jz @error
    inc ax
  @error:
  End; (* DriveIsSubstituted *)

{************************************************************
 * Function DriveIsRemote
 *
 * Parameters:
 *  n: the drive number to check, 1= A:, 2=B: etc.
 * Returns:
 *  TRUE, if the drive is remote, FALSE if it is local or invalid.
 * Description:
 *  This function uses DOS IOCTL function $4409 to check whether
 *  a drive is remote or local.
 *
 *Created: 02/26/95 21:12:32 by P. Below
 ************************************************************}
Function DriveIsRemote( n: TDriveNums ): Boolean; Assembler;
  Asm
    mov ax, $4409   (* IOCTL is drive remote function *)
    mov bl, n
    call Dos3Call
    mov ax, False   (* assume error, in which case we return false *)
    jc @error
    and dx, $1000  (* remote drives have bit 12 set *)
    jz @error
    inc ax
  @error:
  End; (* DriveIsRemote *)

{************************************************************
 * Function DriveIsRemapped
 *
 * Parameters:
 *  n: the drive number to check, 1= A:, 2=B: etc.
 * Returns:
 *  TRUE, if the drive can be remapped, FALSE if not or if it is invalid.
 * Description:
 *  This function uses DOS IOCTL function $440E to check whether
 *  a drive can be remapped.
 *
 *Created: 02/26/95 21:21:46 by P. Below
 ************************************************************}
Function DriveIsRemapped( n: TDriveNums ): Boolean; Assembler;
  Asm
    mov ax, $440E   (* IOCTL get logical drive mapping function *)
    mov bl, n
    call Dos3Call
    jc @error
    cmp al, 0       (* if carry not set, ax returns last drive number *)
    je @error       (* of the mapped drive, or 0, if the block device has *)
    mov ax, True    (* only one drive assigned to it.  *)
    jmp @done
  @error:
    mov ax, false
  @done:
  End; (* DriveIsRemapped *)

{************************************************************
 * Function GetDriveMapping
 *
 * Parameters:
 *  n: the drive number to check, 1= A:, 2=B: etc.
 * Returns:
 *  The logical drive number the drive is mapped to, or the number
 *  passed in n, if the drive is not mapped or is invalid.
 * Description:
 *  This function uses DOS IOCTL function $440E to check whether
 *  a drive is remapped.
 * Error Conditions:
 *  none
 *Created: 02/26/95 21:21:46 by P. Below
 ************************************************************}
Function GetDriveMapping( n: TDriveNums ): TDriveNums; Assembler;
  Asm
    mov ax, $440E   (* IOCTL get logical drive mapping function *)
    mov bl, n
    call Dos3Call
    jc @error       (* if no error *)
    or  al, al      (* check return, 0 means not remapped *)
    jnz @done       (* if remapped, return mapped drive number *)
  @error:
    mov al, n       (* else return original drive number *)
    xor ah, ah
  @done:
  End; (* GetDriveMapping *)

{************************************************************
 * Procedure MapLogicalDrive
 *
 * Parameters:
 *  n: the physical drive number to map, 1= A:, 2=B: etc.
 * Description:
 *  Uses DOS IOCTL function $440F to map the physical drive
 *  passed to the next logical drive number the block device
 *  driver supports. Does nothing for drives that are not
 *  mappable or invalid.
 *  Use it with n=1 to map the floppy drive on a single floppy
 *  system between A: and B:, use GetDriveMapping to check the
 *  current mapping.
 *
 *Created: 02/26/95 21:51:34 by P. Below
 ************************************************************}
Procedure MapLogicalDrive( n: TDriveNums ); assembler;
  Asm
     mov AX, $440F
     mov BL, n
     call Dos3Call
  End ;

Type
  TMediaID = Record  (* This is the MS-DOS original MediaID structure *)
    wInfoLevel: WORD;
    dwSerialNumber: LongInt;
    VolLabel: ARRAY [0..10] of Char;
    FileSysType: ARRAY [0..7] of Char;
  End;
  PMediaID = ^TMediaID;

{************************************************************
 * Function GetMediaID
 *
 * Parameters:
 *  Drive: 0 = default drive, 1 =A: etc.
 *  Info : Record to receive the media id info, filled with 0
 *         if function fails.
 * Returns:
 *  True if successful, False if error.
 * Description:
 *  This function uses DOS IOCTL function $440D, subfunction $0866
 *  to read the media ID record from the disks bootsector. For
 *  Windows, this requires use of DPMI.
 *
 *Created: 02/26/95 21:44:02 by P. Below
 ************************************************************}
Function GetMediaID ( Drive: Word; Var info: TDiskInfo ): Boolean;
  Var
    RealModeReg: TRealModeReg;
    dwGlobalDosBuffer: LongInt;
    lpRMMediaID: PMediaID;
  Begin
    GetMediaID := FALSE;
    FillChar( info, Sizeof( info ), 0 );
    { Get a real mode addressable buffer For the MediaID structure }

    dwGlobalDosBuffer := GlobalDosAlloc(sizeof(TMediaID));
    If (dwGlobalDosBuffer <> 0) Then Begin

      { Now initialize the real mode register structure }
      FillChar(RealModeReg, sizeof(RealModeReg), 0);
      RealModeReg.rmEAX := $440D;           { IOCTL For Block Device }
      RealModeReg.rmEBX := LongInt(Drive);  { 0 = default, 1 = A, 2 = B, etc. }
      RealModeReg.rmECX := $0866;           { Get Media ID }
      RealModeReg.rmDS  := HIWORD(dwGlobalDosBuffer);  { *real mode segment* }

      { Now simulate the real mode interrupt }
      If RealInt($21, RealModeReg) and        { int simulation ok?}
         ((RealModeReg.rmCPUFlags and $0001)=0) { carry clear? }
      Then Begin
         lpRMMediaID := PMEDIAID( MakeLong(0, LOWORD(dwGlobalDosBuffer)));
         info.InfoLevel := lpRMMediaID^.wInfoLevel;
         info.serialNo  := lpRMMediaID^.dwSerialNumber;
         StrMove( @info.volName, lpRMMediaID^.VolLabel, 11 );
         InSituPCharToPas( @info.volName );
         StrMove( @info.FATType, lpRMMediaID^.FileSysType, 8 );
         InSituPCharToPas( @info.FATType );
         GetMediaID := TRUE;
      End;
      GlobalDosFree(LOWORD(dwGlobalDosBuffer));
    End;
  End;


{************************************************************
 * Function MSCDExIsLoaded
 *
 * Parameters:
 *  none
 * Returns:
 *  True, if MSCDEX is loaded, False otherwise
 * Description:
 *  Uses the MSCDEX Int $2F interface, function $00.
 *
 *Created: 02/26/95 21:55:17 by P. Below
 ************************************************************}
Function MSCDExIsLoaded: Boolean; assembler;
  Asm
    mov AX, $1500   (* MSCDEX installed check *)
    xor BX, BX
    int $2F
    xor ax, ax      (* set default return value To false *)
    or  BX, BX      (* returns bx <> 0 If MSCDEX installed *)
    jz  @no_mscdex
    mov al, TRUE
  @no_mscdex:
  End;

{************************************************************
 * Function DriveIsCDROM
 *
 * Parameters:
 *  n: the drive number to check, 1= A:, 2=B: etc.
 * Returns:
 *  True, if the drive is a CD-ROM, False otherwise.
 * Description:
 *  Uses the MSCDEX Int $2F interface, function $0B.
 *  It is not necessary to check for the presence of
 *  MSCDEX first.
 *
 *Created: 02/26/95 21:57:06 by P. Below
 ************************************************************}
Function DriveIsCDROM( n: TDriveNums ): Boolean; assembler;
  Asm
    mov ax, $150B (* MSCDEX check drive Function *)
    mov cl, n
    xor ch, ch
    dec cx        (* 0 = A: etc.*)
    xor bx, bx
    int $2F
    cmp bx, $ADAD (* is MSCDEX present? *)
    jne @no_cdrom
    or  ax, ax
    jz  @no_cdrom
    mov ax, True
    jmp @done
  @no_cdrom:
    mov ax, False
  @done:
  End;

Procedure Beautify( Var s: string );
  (* internal procedure, remove a dot from the volume name,
     padd to 11 chars with blanks *)
  Var
    i: Integer;
  Begin
    i := Pos( '.', s );
    If i <> 0 Then
      Delete( s, i, 1 );

    (* padd To 11 chars with blanks *)

    While Length(s) < 11 Do
      s:= s + ' ';
  End ;

{************************************************************
 * Procedure GetNetworkShareName  [ NOT EXPORTED! ]
 *
 * Parameters:
 *  n: the drive number, 1= A:, 2=B: etc.,  should be a network drive!
 *  name: array of char to take the device name
 * Description:
 *  This is a internal helper procedure, it does not check its
 *  parameters. name will return an empty string, if the drive is
 *  not a network drive.
 *
 *Created: 02/26/95 22:07:49 by P. Below
 ************************************************************}
Procedure GetNetworkShareName( n: TDriveNums; Var name: TDeviceName );
  Var
    Param: ARRAY [0..16] OF CHAR;
    i    : Integer;
  Begin
    Param[0] := Chr( n - 1 + Ord('A'));
    Param[1] := ':';
    Param[2] := #0;
    name [0] := #0;

    i := Sizeof( name );
    WNetGetConnection( @Param, @name, @i );
    InSituPCharToPas( @name );
  End ;

{************************************************************
 * Procedure GetDiskInfo
 *
 * Parameters:
 *  n: the drive number, 1= A:, 2=B: etc.
 *  info: record to take the drive info
 * Description:
 *  Calls GetMediaID to read the info block from the disk boot
 *  sector, then searches for the volume name. Will return with
 *  a serial number of 0, if GetMediaID is not supported or the
 *  drive is invalid or contains no disk.
 *
 *Created: 02/26/95 22:11:35 by P. Below
 ************************************************************}
Procedure GetDiskInfo( n: TDriveNums; Var info: TDiskInfo );
  Var fake: Boolean;
      oldsettings: Word;
  Procedure DoItOldStyle;
    Var
      dinfo: TSearchRec;
      s: string[8];
    Begin
      s := '@:\*.*';
      s[1] := CHR( Ord( s[1] ) + n );
      If FindFirst( s, faVolumeID , dinfo ) = 0 Then Begin
        Beautify( dinfo.Name );
        info.volName := dinfo.Name;
      End;
      If fake Then Begin
        info.FATType := 'FAT12   ';
        info.serialNo := 0;
      End;
    End ;
  Begin
    FillChar( info, Sizeof( info ), 0 );
    oldsettings := SetErrorMode( SEM_FAILCRITICALERRORS );
    (* check the DOS version  *)
    If Hi( HiWord( GetVersion )) >= 4 Then
      fake := NOT GetMediaID( n, info )
    Else
      fake := TRUE;
    (* we get the volume label thru the old-style method of directory
       search because pre-DOS 5.x may not have it in the boot sector
       and even later versions may either not have it (If the disk was
       formatted by something other than DOS Format) or it may have been
       changed with LABEL, which does not change the boot sector entry!
    *)
    DoItOldStyle;
    SetErrorMode( oldsettings );
  End ;

{************************************************************
 * Procedure GetDriveName
 *
 * Parameters:
 *  n: the drive number, 1= A:, 2=B: etc.
 *  di: a drive info record that MUST have its flags field
 *      already filled by MyGetDriveType!
 * Description:
 *  Tries to obtain the volume info or netshare name for the
 *  passed drive. Default names are used of the info cannot be
 *  safely obtained because the drive handles removable media.
 *
 *Created: 02/26/95 22:22:28 by P. Below
 ************************************************************}
Procedure GetDriveName( n: TDriveNums; Var di: TDriveInfo );
  Begin
    FillChar( di.Info, SIZEOF( di.Info ), 0 );
    If INVALID IN di.Flags Then
      di.DevName:= ''
    Else
      If (FLOPPY IN di.Flags) OR (CD_ROM IN di.Flags) Then
      (* don't try To get the volume name For removable media *)
        If (REMOTE IN di.Flags) Then
          di.DevName := ' -UNKNOWN- '
        Else
          di.Info.volName :=' -UNKNOWN- '

      Else
       If (REMOTE IN di.Flags) Then Begin
         GetNetworkShareName( n, di.DevName );
         If Length(di.DevName) = 0 Then
           di.DevName := ' -NETWORK- ';
       End
       Else
         GetDiskInfo( n, di.Info )
  End;

{************************************************************
 * Function GetDiskParameterBlock
 *
 * Parameters:
 *  n: the drive number, 1= A:, 2=B: etc.
 * Returns:
 *  a pointer to the disk parameter bloc, or Nil, if the function
 *  fails.
 * Description:
 *  Uses DOS int $21, function $32. This function fails in network
 *  drives and it tries to actually read the disk. So If you try to
 *  use it on drive that handles removable media, make preparations
 *  to trap critical errors!
 *
 *Created: 02/26/95 22:33:14 by P. Below
 ************************************************************}
Function GetDiskParameterBlock( drive: TDriveNums ): LP_DPB; Assembler;
  (* return a far pointer to the requested drives disk parameter block.
     This call is appearendly supported by the windows dos extender,
     we get a valid selector back in ds. *)
  Asm
      push ds
      mov DL, drive
      mov AH, $32
      call Dos3Call
      cmp AL, $0FF
      jne @valid
      (* run into an error somewhere, return nil *)
      xor ax, ax
      mov dx, ax
      jmp @done
    @valid:
      mov ax, bx
      mov dx, ds
    @done:
      pop ds
  End;

{************************************************************
 * Function DriveIsRamdisk
 *
 * Parameters:
 *  n: the drive number, 1= A:, 2=B: etc. MUST be a fixed disk!
 * Returns:
 *  True, if the disk in question has only one FAT copy, False
 *  otherwise.
 * Description:
 *  Tries to read the disks parameter block and checks the number
 *  of FATs present. One FAT is taken to be a sign for a RAM disk.
 *  This check is not bomb-proof!
 * Error Conditions:
 *  If you call this function for a drive handling removable media,
 *  be prepared to trap critical errors!
 *
 *Created: 02/26/95 22:35:42 by P. Below
 ************************************************************}
Function DriveIsRamdisk( n: TDriveNums ): Boolean;
  Var
    pDPB: LP_DPB;
  Begin
    DriveIsRamdisk := FALSE;
    pDPB := GetDiskParameterBlock( n );
    If pDPB <> NIL Then
      If pDPB^.dpbFATCount = 1 Then
        DriveIsRamdisk := TRUE;
  End;

{************************************************************
 * Procedure MyGetDriveType
 *
 * Parameters:
 *  n: the drive number, 1= A:, 2=B: etc.
 *  f: set of flags that describes the drive filled by this procedure
 * Description:
 *  Tries to determine for the specified drive, whether it is valid,
 *  holds removable media, is remote, remapped, a CD-ROM or RAM-disk.
 *
 *Created: 02/26/95 22:48:32 by P. Below
 ************************************************************}
Procedure MyGetDriveType( n: TDriveNums; Var f: TDTypeSet );
  Var
    dt: word;
  Begin
    f := [];
    Include( f, MediumIsRemovable( n ));
    If ( n=2 ) and DriveIsRemapped( 1 ) Then Begin
      Include( f, REMAPPED );
    End; { If }
    If not ( INVALID In f ) Then Begin
      If DriveIsRemote( n ) Then
        Include( f, REMOTE );
      If DriveIsSubstituted( n ) Then 
        Include( f, SUBSTITUTED );
      If DriveIsShared( n ) Then 
        Include( f, SHARED );
      If (REMOTE IN f) and DriveIsCDROM( n ) Then
        Include( f, CD_ROM );
      If ([HARDDISK, CD_ROM, REMOTE] * f)= [HARDDISK] Then
        If DriveIsRamdisk( n ) Then
          Include( f, RAMDISK );
    End; { If }
  End;

{************************************************************
 * Procedure GetDriveInfo
 *
 * Parameters:
 *  n: the drive number, 1= A:, 2=B: etc.
 *  di: record to take the drive type and volume/netshare info
 * Description:
 *  Uses other routines in this Unit do obtain info on the drive
 *  in question.
 *
 *Created: 02/26/95 22:53:43 by P. Below
 ************************************************************}
Procedure GetDriveInfo( n: TDriveNums; Var di: TDriveInfo );
  Begin
    MyGetDriveType( n, di.Flags );
    GetDriveName( n, di );
  End;

{************************************************************
 * Procedure BuildDriveMap
 *
 * Parameters:
 *  DMap: array of TDriveInfo records to take the info on all
 *        drives on the system.
 * Description:
 *  Uses other routines from this Unit to build a map of all
 *  drives on the system with drive letters in the range
 *  A..Z ( logical drives 1..26 ). The map contains for each
 *  drive a set of flags describing the drive type and also,
 *  if the drive is valid and does not handle removable media,
 *  the media id info ( volume name, serial number, FAT type ) or
 *  netshare name.
 *
 *Created: 02/26/95 22:55:50 by P. Below
 ************************************************************}
Procedure BuildDriveMap( Var DMap: TDriveMap );
  Var
    n : TDriveNums;
  Begin
    (* build a drive properties map For all possible drives.
       CAVEAT! DR-DOS 6.x has a bug that will fail IOCTL calls For
               drive letters > P:! (Thank's To Ray Tackett [76416,276]
               For this info) *)
    For n := Min_DriveNums To Max_DriveNums Do Begin
      MyGetDriveType( n, DMap[n].Flags );
      GetDriveName( n, DMap[n] );
    End;
  End ;

{************************************************************
 * Function GetLastdrive
 *
 * Parameters:
 *  none
 * Returns:
 *  the logical drive number ( 1=A: etc. ) for the last valid
 *  drive on the system.
 * Description:
 *  Uses DOS int $21, function $0E. On some systems this may just
 *  return the LASTDRIVE setting in CONFIG.SYS!
 *
 *Created: 02/26/95 22:59:12 by P. Below
 ************************************************************}
Function GetLastdrive: TDriveNums; assembler;
  asm
    mov ah, $19   (* get current drive *)
    call Dos3Call
    mov ah, $E    (* set current To same *)
    mov dl, al
    call Dos3Call
    sub ah, ah
  End;

{************************************************************
 * Function DriveState
 *
 * Parameters:
 *  n: drive number of drive to check
 * Returns:
 *  state of the drive (disk, no disk, unformatted disk)
 * Description:
 *  This function can be used to detect whether a drive supporting
 *  changeable media has a usable medium inside. It traps all
 *  critical errors internally. Note that the value returned by
 *  FindFirst for no disk seems to be system dependent. This routine
 *  has been checked on WfW 3.11 and Win NT 3.5. Better run a few
 *  checks on Win95 if you want to use this routine on that platform
 *  as well!
 * Error Conditions:
 *  none
 *
 *Created: 10/04/95 14:50:43 by P. Below
 ************************************************************}
Function DriveState( n: TDriveNums ): TDriveState;
Var
  mask: String[6];
  sRec: TSearchRec;
  oldMode: Cardinal;
  retcode: Integer;
Begin
  oldMode := SetErrorMode( SEM_FAILCRITICALERRORS );
  mask:= '?:\*.*';
  mask[1] := Chr( n + Ord( '@' ));
  {$I-}  { don't raise exceptions if we fail }
  retcode := FindFirst( mask, faAnyfile, SRec );
  FindClose( SRec );
  {$I+}
  Case retcode Of
     0: Result := DS_DISK_WITH_FILES;  { found at least one file }
   -18: Result := DS_EMPTY_DISK;  { found no files but otherwise ok }
   -21, -3: Result := DS_NO_DISK; { DOS ERROR_NOT_READY on WinNT,
                                    ERROR_PATH_NOT_FOUND on Win 3.1 }
  Else
    Result := DS_UNFORMATTED_DISK;{ gave -1785 on my system! }
  End;
  SetErrorMode( oldMode );
End; { DriveState }

{************************************************************
 * Function GetUnittype
 *
 * Parameters:
 *  drive: drive to get the type for, 0: current drive, 
 *         1: drive A:, 2: drive B: etc.
 * Returns:
 *    unit code of requested block device
 *    0: 320/360 KByte 5.25" floppy
 *    1: 1,2 MByte 5.25" floppy
 *    2: 720 KByte 3.5" floppy
 *    3: 8" single density floppy
 *    4: 8" double density floppy
 *    5: fixed disk
 *    6: tape
 *    7: other, normally 1.44 MByte 3.5" floppy
 *    8: read/write optical disk
 *    9: 2.88 MByte 3.5" floppy
 *    $FF: error
 *  
 * Description:
 *  Uses an IOCTL function to get the _default_ unit type of the
 *  specified drive. This involves no disk access, it can be used
 *  on a drive without a disk in it without causing an exception.
 *  Please note that device types for more exotic devices like CD-ROMS
 *  and MOs are not always returned as expected ( often as type 5 ). 
 *  This depends totally on the device driver in use.
 * Error Conditions:
 *  invalid drive or out of memory errors will cause a return of
 *  $FF. 
 *
 *Created: 10/04/95 15:14:35 by P. Below
 ************************************************************}
Function GetUnittype( drive: Byte ): byte;
  Var
    pMem: ^TDeviceParams;
  Begin
    Result := $FF;  { assume error }
    New( pMem );
    try
      If GetDeviceParams( drive, pMem^, True ) = 0 Then
        Result := pMem^.dpDevType;
    finally
      Dispose( pMem );
    end;
  End;

  { TDeviceParams description:
  
    dpDevType
    Specifies the device type. This field can be one of the following values: 

    00h 320/360K
    01h 1.2 MB
    02h 720K
    03h 8-inch, single-density
    04h 8-inch, double-density
    05h Hard disk
    06h Tape drive
    07h 1.44 MB
    08h Read/write optical
    09h 2.88 MB

    dpDevAttr
    Specifies device attributes. This field can contain some combination 
    of the following values: 
  
    Bit Meaning

    0 0 = The medium is removable.
      1 = The medium is not removable.
    1 0 = Disk change-line is not supported (no door-lock support).
      1 = Disk change-line is supported (door-lock support).
    All other bits are reserved and must be zero. 

    dpCylinders
    Specifies the maximum number of cylinders that the physical device can
    support. This information is set by the device. 

    dpMediaType
    Specifies which medium the drive currently accepts (for drives that 
    accept more than one media type). For a 1.2-MB drive, if bit 0 is clear, 
    it indicates that the drive accepts quad-density, 1.2-MB disks (the 
    default media type); if bit 0 is set, the drive accepts double-density, 
    320/360K disks. 

    dpBytesPerSec
    Specifies the number of bytes per sector. 

    dpSecPerClust
    Specifies the number of sectors in a cluster. The sectors must be 
    consecutive, and the number must be a power of 2. 

    dpResSectors
    Specifies the number of reserved sectors on the drive, beginning with 
    sector 0. Typically, this value is 1 (for the startup sector), unless 
    the disk-drive manufacturers software reserves additional sectors. 

    dpFATs
    Specifies the number of file allocation tables (FATs) following the 
    reserved sectors. Most versions of MS-DOS maintain one or more copies 
    of the primary FAT and use the extra copies to recover data on the disk 
    if the first FAT is corrupted. 

    dpRootDirEnts
    Specifies the maximum number of entries in the root directory. 

    dpSectors
    Specifies the number of sectors on the drive. If the size of the drive 
    is greater than 32 MB, this field is set to zero and the number of 
    sectors is specified by the dpHugeSectors field. 

    dpMedia
    Specifies the media descriptor, a value that identifies the type of 
    media in a drive. Some device drivers use the media descriptor to 
    determine quickly whether the removable medium in a drive has changed. 
    MS-DOS passes the media descriptor to the device driver so that programs 
    can check the media type. Also, the first byte in the FAT is often (but 
    not always) identical to the media descriptor. 
    Following is a list of the most commonly used media descriptors and 
    their corresponding media: 
  
    Value Type of medium

    0F0h  3.5-inch, 2 sides, 18 sectors/track (1.44 MB); 3.5-inch, 2 sides, 
          36 sectors/track (2.88 MB); 5.25-inch, 2 sides, 15 sectors/track 
          (1.2 MB). This value is also used to describe other media types.
    0F8h  Hard disk, any capacity.
    0F9h  3.5-inch, 2 sides, 9 sectors/track, 80 tracks/side (720K); 
          5.25-inch, 2 sides, 15 sectors/track, 80 tracks/side (1.2 MB).
    0FAh  5.25-inch, 1 side, 8 sectors/track, (320K).
    0FBh  3.5-inch, 2 sides, 8 sectors/track (640K).
    0FCh  5.25-inch, 1 side, 9 sectors/track, 40 tracks/side (180K).
    0FDh  5.25-inch, 2 sides, 9 sectors/track, 40 tracks/side (360K). 
          This value is also used for 8-inch disks.
    0FEh  5.25-inch, 1 side, 8 sectors/track, 40 tracks/side (160K). 
          This value is also used for 8-inch disks.
    0FFh  5.25-inch, 2 sides, 8 sectors/track, 40 tracks/side (320K).

    dpFATsecs
    Specifies the number of sectors occupied by each FAT. 

    dpSecPerTrack
    Specifies the number of sectors on a single track. 

    dpHeads
    Specifies the number of read/write heads on the drive. 

    dpHiddenSecs
    Specifies the number of hidden sectors on the drive. 

    dpHugeSectors
    Specifies the number of sectors if the dpSectors field is zero. 
    This value supports drives larger than 32 MB. 

    NOTE! The list above comes from the MS-DOS
          Programmer's Reference for DOS 6.0
    }

{************************************************************
 * Function GetDeviceParams
 *
 * Parameters:
 *  drive: drive to get the parameters for, 0: current drive, 
 *         1: drive A:, 2: drive B: etc.
 *  dp   : device parameter block to fill
 *  geDefaults: True if to get the default parameters ( no disk access ),
 *              False if the get the parameters for the disk in drive
 *              ( involves a disk access )
 * Returns:
 *  0 if no errors, else DOS error code, which can be one of
 *    1:  ERROR_INVALID_FUNCTION
 *    2:  ERROR_FILE_NOT_FOUND  ( invalid drive specified )
 *    5:  ERROR_ACCESS_DENIED
 * Description:
 *  Uses DOS IOCTL function $440D, subfunction $60 ( Get Device Parameters )
 *  to query the requested block device. This function is supported by the
 *  Windows DOS extender, so no DPMI stuff is needed.
 * Error Conditions:
 *  handled by DOS
 *
 *Created: 09/06/95 12:04:00 by P. Below
 ************************************************************}
Function GetDeviceParams( drive: Byte; Var dp: TDeviceParams; 
                          getDefaults: Boolean ): Word;
  Begin
    Result := 0;  { assume no error }
    FillChar( dp, Sizeof( dp ), 0 );
    If not getDefaults Then
      dp.dpSpecFunc := 1;    (* gets params for current disk in drive! *)
    asm
      push ds       { save ds }
      lds dx, dp    { point ds:dx at dp }
      mov ax, $440D { IOCTL generic request function }
      mov bl, drive { drive number, 1= A:, 2=B: etc. }
      mov cx, $860  { get device parameters subfunction }
      call Dos3Call { execute }
      pop ds        { restore ds }
      jnc @ok
      mov Result, ax
    @ok:
    end;
  End;


End.
