{===EZDSLBSE==========================================================

Part of the EZ Delphi Structures Library--the base class and node
store routines.

EZDSLBSE is Copyright (c) 1993, 1995 by  Julian M. Bucknall

VERSION HISTORY
18Jun95 JMB 1.00 conversion of EZStrucs to Delphi
======================================================================}
{ Copyright (c) 1993, 1995, Julian M. Bucknall. All Rights Reserved   }

unit EZDSLBse;

{Declare the compiler defines}
{$I EZDSLDEF.INC}

{------Changeable compiler switches-----------------------------------}
{$A+   Word align variables }
{$F+   Force Far calls }
{$K+   Use smart callbacks
{$N+   Allow coprocessor instructions }
{$P+   Open parameters enabled }
{$Q+   Integer overflow checking }
{$R+   Range checking }
{$S+   Stack checking }
{$T-   @ operator is NOT typed }
{$U-   Non Pentium safe FDIV }
{$Z-   No automatic word-sized enumerations}
{---------------------------------------------------------------------}

interface

{$R EZDSLCts.RES}

uses
  SysUtils,
  WinTypes,
  WinProcs,
  Classes,
  EZDSLCts,
  EZDSLSup;

const
  skMaxLevels = 16;       {Maximum levels in a skip list}

type
  TAbstractContainer = class;
  TAbstractContainerClass = class of TAbstractContainer;
  PNode = ^TNode;

  {$IFDEF MaxIs64KItems}
  TContainerCount = word;
  {$ELSE}
  TContainerCount = longint;
  {$ENDIF}

  TChild = (CLeft, CRight);
    {-Binary trees: flags for left and right children}
  TTraversalType = (ttPreOrder, ttInOrder, ttPostOrder, ttLevelOrder);
    {-Binary trees: methods of traversing their nodes}

  TListCursor = longint;
    {-Cursor for TDList and TSkipList (double linked & skip lists)}
  TTreeCursor = longint;
    {-Cursor for TBinTree and descendants (binary trees)}

  TDisposeDataProc = procedure (aData : pointer);
    {-Data disposal procedure type for containers}
  TCompareFunc = function (Data1, Data2 : pointer) : integer;
    {-Data comparison procedure type for containers}
  TDupDataFunc = function (aData : pointer) : pointer;
    {-Data duplication procedure type for containers}

  TIterator = function (C : TAbstractContainer;
                        aData : pointer;
                        ExtraData : pointer) : boolean;
    {-Iterator function called by Iterate for each item, must return
      true to continue iterating, false to stop}



  {--Internal object type definitions--}
  TNode = record
    {-Internal definition of a node}
    Data : pointer;
    case byte of                           {For...}
      0 : (Link : PNode);                  {Stacks, Queues, Deques, Lists}
      1 : (FLink, BLink : PNode);          {Doubly-linked lists}
      2 : (Size   : word;                  {Skip lists}
           Lvls   : word;
           BkLink : PNode;
           FwLink : array [0..pred(skMaxLevels)] of PNode);
      3 : (TLink : array [TChild] of PNode;{Trees}
           case byte of
             0 : (PKC : longint);          {Binary Trees}
             1 : (PLink : PNode))          {Heaps}
  end;

  TNodeStore = class
    {-Internal object that maintains suballocation of nodes}
    private
      RefCount  : word;
      NodeSize  : word;
      Block     : PNode;
      NodeStack : PNode;
      SpareNodeCount : longint;

    protected
      procedure GrowSpareNodeStack;

    public
      constructor Create(aNodeSize : word);
      destructor Destroy; override;

      function  Alloc : PNode;
      procedure Dealloc(aNode : PNode);
  end;


  {--Container object types--}
  TAbstractContainer = class(TPersistent)
    {-Ancestor object: methods will be overridden}
    private
      FCompare     : TCompareFunc;
      FDisposeData : TDisposeDataProc;
      FDupData     : TDupDataFunc;

      NS       : TNodeStore;

    protected
      FCount       : TContainerCount;
      FIsDataOwner : boolean;
      NodeSize     : word;
      InDone   : boolean;

      procedure SetCompare(NewFunc : TCompareFunc);
      procedure SetDisposeData(NewProc : TDisposeDataProc);
      procedure SetDupData(NewFunc : TDupDataFunc);

      procedure acDisposeNode(aNode : PNode); virtual;
      function  acNewNode(aData : pointer) : PNode; virtual;

    public
      {constructor/destructor}
      constructor Create(DataOwner : boolean); virtual;
      constructor Clone(Source : TAbstractContainer;
                        DataOwner : boolean; NewCompare : TCompareFunc); virtual; abstract;
      destructor Destroy; override;

      {methods}
      procedure Empty; virtual; abstract;
      function  IsEmpty : boolean;

      {properties}
      property Count : TContainerCount
         read FCount;

      property IsDataOwner : boolean
         read FIsDataOwner;

      property Compare : TCompareFunc
         read FCompare
         write SetCompare;

      property DisposeData : TDisposeDataProc
         read FDisposeData
         write SetDisposeData;

      property DupData : TDupDataFunc
         read FDupData
         write SetDupData;
  end;

implementation

const
  ListInitialised : boolean = false;
  MinNodeSize = 8;
  MaxNodeSize = 32;
  NodeSizeDelta = 4;
  NumNodeSizes = succ((MaxNodeSize - MinNodeSize) div NodeSizeDelta);

var
  NodeStoreList : array [0..pred(NumNodeSizes)] of TNodeStore;

{===NodeStore helper routines=========================================}
function GetNodeStore(Size : word) : TNodeStore;
  var
    Index : word;
  begin
    if (Size < MinNodeSize) then
      Size := MinNodeSize
    else if (Size > MaxNodeSize) then
      Size := MaxNodeSize;
    Size := (pred(Size + NodeSizeDelta) div NodeSizeDelta) * NodeSizeDelta;

    if not ListInitialised then
      begin
        FillChar(NodeStoreList, sizeof(NodeStoreList), 0);
        ListInitialised := true;
      end;

    Index := (Size - MinNodeSize) div NodeSizeDelta;

    Result := NodeStoreList[Index];
    if not Assigned(Result) then
      begin
        Result := TNodeStore.Create(Size);
        NodeStoreList[Index] := Result;
      end;
    inc(Result.RefCount);
  end;
{--------}
procedure FreeNodeStore(NS : TNodeStore);
  var
    Index : word;
  begin
    if Assigned(NS) then
      begin
        dec(NS.RefCount);
        if (NS.RefCount = 0) then
          begin
            Index := (NS.NodeSize - MinNodeSize) div NodeSizeDelta;
            NS.Destroy;
            NodeStoreList[Index] := nil;
          end;
      end;
  end;
{=====================================================================}


{=TNodeStore==========================================================
A node warehouse.

A node warehouse stores nodes for TAbstractContainer descendants.
Because the size of a node for a given container is fixed, the
TNodeStore can preallocate a single block of them, and dole them out
singly to the requesting container (ie suballocate the larger block
into smaller nodes). When a node is finished with, it is returned to
the store and will be doled out again. The node store manages two
structures: a very simple linked list of node blocks and a simple
stack of used nodes. The node block is 128 nodes large. This extra
effort is well rewarded, compared with allocating nodes when and
where needed from the heap manager, this is noticeably faster (15-
20% faster, dependent on the number of allocations/frees of nodes).

The node warehouses are stored in a simple array as a global resource.
For each node size there will be one node warehouse. There can be many
containers attached to each warehouse, the count is held in the
RefCount field. Every time a container gets attached to a node
warehouse RefCount is incremented, every time one is unlinked the
RefCount is decremented. If it reaches zero, it is freed. Node
warehouses are allocated with GetNodeStore and freed with
FreeNodeStore.

18Jun95 JMB
======================================================================}
const
  NumNodes = 128; {Best if it is a power of two}
{--------}
constructor TNodeStore.Create(aNodeSize : word);
  begin
    NodeSize := aNodeSize;
    GrowSpareNodeStack;
  end;
{--------}
destructor TNodeStore.Destroy;
  var
    Temp : PNode;
  begin
    while Assigned(Block) do
      begin
        Temp := Block;
        Block := Temp^.Link;
        SafeFreeMem(Temp, NodeSize * NumNodes);
      end;
  end;
{--------}
function TNodeStore.Alloc : PNode;
  begin
    if (SpareNodeCount = 0) then
      GrowSpareNodeStack;
    Result := NodeStack;
    NodeStack := Result^.Link;
    dec(SpareNodeCount);
  end;
{--------}
procedure TNodeStore.Dealloc(aNode : PNode);
  begin
    if Assigned(aNode) then
      begin
        {$IFDEF DEBUG}
        FillChar(aNode^, NodeSize, $CC);
        {$ENDIF}
        aNode^.Link := NodeStack;
        NodeStack := aNode;
        inc(SpareNodeCount);
      end;
  end;
{--------}
procedure TNodeStore.GrowSpareNodeStack;
  var
    i : integer;
    Temp : PNode;
    Node : PNode;
    WalkerNode : PChar absolute Node; {for pointer arithmetic}
  begin
    SafeGetMem(Temp, NodeSize * NumNodes);
    Temp^.Link := Block;
    Block := Temp;
    Node := Block;
    WalkerNode := WalkerNode + NodeSize; {alters Node}
    for i := 1 to pred(NumNodes) do
      begin
        Node^.Link := NodeStack;
        NodeStack := Node;
        WalkerNode := WalkerNode + NodeSize; {alters Node}
      end;
    inc(SpareNodeCount, pred(NumNodes));
  end;
{---------------------------------------------------------------------}

{---Data object routines----------------------------------------------}
function  EZAbstractCompare(Data1, Data2 : pointer) : integer;
  begin
    RaiseError(escNoCompare);
  end;
{--------}
procedure EZAbstractDisposeData(aData : pointer);
  begin
    RaiseError(escNoDisposeData);
  end;
{--------}
function  EZAbstractDupData(aData : pointer) : pointer;
  begin
    RaiseError(escNoDupData);
  end;
{---------------------------------------------------------------------}


{=TAbstractContainer==================================================}
constructor TAbstractContainer.Create(DataOwner : boolean);
  begin
    FIsDataOwner := DataOwner;
    FCompare := EZAbstractCompare;
    if DataOwner then
         FDisposeData := EZAbstractDisposeData
    else FDisposeData := EZNoDisposeData;
    FDupData := EZAbstractDupData;
    if (NodeSize <> 0) then
      NS := GetNodeStore(NodeSize);
  end;
{--------}
destructor TAbstractContainer.Destroy;
  begin
    InDone := true;
    Empty;
    FreeNodeStore(NS);
  end;
{--------}
function  TAbstractContainer.IsEmpty : boolean;
  begin
    Result := (FCount = 0);
  end;
{--------}
procedure TAbstractContainer.SetCompare(NewFunc : TCompareFunc);
  begin
    if IsEmpty then
      if Assigned(NewFunc) then
           FCompare := NewFunc
      else FCompare := EZAbstractCompare;
  end;
{--------}
procedure TAbstractContainer.SetDisposeData(NewProc : TDisposeDataProc);
  begin
    if IsEmpty then
      if not IsDataOwner then
        FDisposeData := EZNoDisposeData
      else if Assigned(NewProc) then
        FDisposeData := NewProc
      else
        FDisposeData := EZAbstractDisposeData;
  end;
{--------}
procedure TAbstractContainer.SetDupData(NewFunc : TDupDataFunc);
  begin
    if IsEmpty then
      if Assigned(NewFunc) then
           FDupData := NewFunc
      else FDupData := EZAbstractDupData;
  end;
{--------}
procedure TAbstractContainer.acDisposeNode(aNode : PNode);
  begin
    {$IFDEF DEBUG}
    Assert(Assigned(aNode), ascFreeNilNode);
    Assert((NodeSize <> 0), ascFreeNodeSize0);
    {$ENDIF}
    NS.Dealloc(aNode);
    if (FCount > 0) then
      dec(FCount);
  end;
{--------}
function  TAbstractContainer.acNewNode(aData : pointer) : PNode;
  begin
    {$IFDEF DEBUG}
    Assert((NodeSize <> 0), ascNewNodeSize0);
    {$ENDIF}
    {$IFDEF MaxIs64KItems}
    if (Count = $FFFF) then
      RaiseError(escTooManyItems);
    {$ENDIF}
    Result := NS.Alloc;
    FillChar(Result^, NodeSize, 0);
    inc(FCount);
    Result^.Data := aData;
  end;
{---------------------------------------------------------------------}

end.
