{
Program:      FINDEXP1.PAS (FINDEXP1.DFM)
Description:  Source Find Expert version 1.3
              With grep (file contain) search feature added.
Programmer:   Todd Miller (Borland International/Delphi Team)
Installation: Install FINDEXP1.PAS as a new component under
              Options|Install Components|Add.
}

unit findexp1;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, Buttons, StdCtrls, ExptIntf, ToolIntf, FileCtrl, ExtCtrls;

const
  ExpertName = 'File Find Expert 1.3';

type
  TFindExpDialog = class(TForm)
    SearchItem: TEdit;
    SearcgItemLabel: TLabel;
    FoundList: TListBox;
    SearchCriteria: TRadioGroup;
    CancelButton: TBitBtn;
    LoadButton: TBitBtn;
    FoundListLabel: TLabel;
    SearchButton: TBitBtn;
    SearchMethodRadioGrp: TRadioGroup;
    Panel1: TPanel;
    IgnoreCase: TCheckBox;
    FilterTypeCombo: TFilterComboBox;
    UserDefinedSearchPath: TEdit;
    AlternatePathLabel: TLabel;
    procedure SearchCriteriaClick(Sender: TObject);
    procedure CancelButtonClick(Sender: TObject);
    procedure LoadButtonClick(Sender: TObject);
    procedure SearchButtonClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormShow(Sender: TObject);
    function  SearchFile(Const AFile: String): Boolean;
    procedure SearchMethodRadioGrpClick(Sender: TObject);
    procedure UserDefinedSearchPathExit(Sender: TObject);
  private
    BaseDelphiDir : String;
    SearchDir     : String;
    procedure GrepSearch(NewPath : string);
    procedure RecurseDirTree(NewPath : string);
  end;

var
  FindExpDialog: TFindExpDialog;

Type
  SourceFindExpert = class(TIExpert)
  public
    function GetStyle: TExpertStyle; override;
    function GetIDString: string; override;
    function GetName: string; override;
    function GetComment: string; override;
    function GetGlyph: HBITMAP; override;
    function GetState: TExpertState; override;
    function GetMenuText: string; override;
    { Launch the Expert }
    procedure Execute; override;
  end;

  procedure Register;

implementation

uses IniFiles;

{$R *.DFM}

procedure Register;
begin
  RegisterLibraryExpert(SourceFindExpert.Create);
end;

{-------- Method of the SourceFindExpert class --------}

function SourceFindExpert.GetStyle: TExpertStyle;
begin
  Result := esStandard
end;

function SourceFindExpert.GetIDString: String;
begin
  Result := 'Todd Miller.Source Find Expert'
end;

function SourceFindExpert.GetComment: String;
begin
  Result := '' { not needed for esStandard }
end;

function SourceFindExpert.GetGlyph: HBITMAP;
begin
  Result := 0 { not needed for esStandard }
end;

function SourceFindExpert.GetName: String;
begin
  Result := ExpertName
end;

function SourceFindExpert.GetState: TExpertState;
begin
  Result := [esEnabled]
end;

function SourceFindExpert.GetMenuText: String;
begin
  Result := ExpertName
end;

procedure SourceFindExpert.Execute;
begin
  if not Assigned(FindExpDialog) then
    FindExpDialog := TFindExpDialog.Create(Application);
  FindExpDialog.Show;
  FindExpDialog.SetFocus
end;

{-------- Method of the TFindExpDialog class --------}

procedure TFindExpDialog.SearchCriteriaClick(Sender: TObject);
begin
  if SearchCriteria.ItemIndex <> 5 then begin
     UserDefinedSearchPath.Visible := False;
     AlternatePathLabel.Visible := False;
     SearchItem.SetFocus;
     case SearchCriteria.ItemIndex of
       0: SearchDir := BaseDelphiDir + 'SOURCE\VCL\';
       1: SearchDir := BaseDelphiDir + 'SOURCE\';
       2: SearchDir := BaseDelphiDir + 'DEMOS\';
       3: SearchDir := BaseDelphiDir + 'DOC\';
       4: SearchDir := BaseDelphiDir;
     end
  end else begin
     AlternatePathLabel.Visible := True;
     UserDefinedSearchPath.Visible := True;
     UserDefinedSearchPath.Text := 'C:\';
     UserDefinedSearchPath.SetFocus;
  end;

  Caption := ExpertName + ' - ' + SearchDir;
  FoundList.Items.Clear;
  LoadButton.Enabled := False;

end;

procedure TFindExpDialog.CancelButtonClick(Sender: TObject);
begin
  Close;
end;

procedure TFindExpDialog.LoadButtonClick(Sender: TObject);
var
FileName : String;
begin
  FileName := FoundList.Items[FoundList.ItemIndex];
  if (FoundList.ItemIndex = -1) or not Assigned(ToolServices)then
      Exit;

  if UpperCase(ExtractFileExt(FileName)) = '.DPR' then begin
     if ToolServices.CloseProject then begin
        ToolServices.OpenProject(FileName)
     end;
  end else
    ToolServices.OpenFile(FileName);

  Close {done with the dialog}
end;

function TFindExpDialog.SearchFile(Const AFile: String): Boolean;
Var
  Stream:       TFileStream;
  FileString:   PChar;
  SearchString: PChar;
  BufSize:      Word;
begin
  Result := False;
  if Length(SearchItem.Text) = 0 then Exit;

  Stream := TFileStream.Create(AFile, fmOpenRead);
  try
  GetMem(SearchString, Length(SearchItem.Text) + 1);
  if SearchString <> Nil then begin
     StrPCopy(SearchString, SearchItem.Text);
     if IgnoreCase.Checked then StrUpper(SearchString);

     if Stream.Size > 0 then begin
        {The buffer size max is 64K - 16 bytes - this means that a
        word that is on the 64K boundary may not be caught by this
        algorithm - this is compensated for below}
        BufSize := $FFF0;
        if BufSize > (Stream.Size + 1) then
           BufSize := Stream.Size + 1;
        GetMem(FileString, BufSize + 1);
        if FileString <> Nil then begin
           Repeat
             FileString[Stream.Read(FileString^,BufSize - 1)]:=#0;
             if IgnoreCase.Checked then StrUpper(FileString);

             {If a match is found in any one of the buffer reads return True}
             Result := StrPos(FileString, SearchString) <> Nil;
             {Reset the stream pointer back by the size of the search
             string to compensate for the 64k boundary issue mentioned
             above}
             if Stream.Position < Stream.Size then
                Stream.Position := Stream.Position - Length(SearchItem.Text);
           Until Result or (Stream.Position >= Stream.Size);
          FreeMem(FileString, BufSize + 1);
        end; {if FileString <> Nil...}
        FreeMem(SearchString, Length(SearchItem.Text) + 1);
     end; {if Stream.Size > 0...}
  end; {if SearchString <> Nil...}
  finally
    Stream.Free;
  end; {Try..Finally}
end;

procedure TFindExpDialog.GrepSearch(NewPath : string);
var
  SearchRec:  TSearchRec;
  ReturnCode: Integer;
begin
  if FindFirst(NewPath + '*.*', faDirectory, SearchRec) = 0 then begin
     repeat
     {Check to make sure it is a normal file}
     with SearchRec do begin
       if Name[1] <> '.' then
       if (Attr and faDirectory > 0) then
          GrepSearch(NewPath + Name + '\')
       else
         {Check too see if the file is of the right file type}
         if (ExtractFileExt(Name) = ExtractFileExt(FilterTypeCombo.Mask)) or
            (FilterTypeCombo.Mask = '*.*' ) then
              if SearchFile(NewPath + Name) then begin
                 FoundList.Items.Add(NewPath + Name);
                 Application.ProcessMessages;
                 LoadButton.Enabled := True;
              end; {if SearchFile(NewPath + Name)...}
    end;  {with SearchRec...}
    until FindNext(SearchRec) <> 0;
  end; {if FindFirst...}
end;

procedure TFindExpDialog.RecurseDirTree(NewPath : string);
var
  SearchRec:  TSearchRec;
  ReturnCode: Integer;
begin
 {The default extension is .PAS if one isn't present}
 with SearchItem do
    if Pos('.', Text) = 0 then Text := UpperCase(Text) + '.PAS';
     {Look for the search file name in the current directory}
    if FindFirst(NewPath + SearchItem.Text, faAnyFile, SearchRec) = 0 then begin
       repeat
       {Check to make sure it is a normal file}
       if (SearchRec.Attr and (faHidden or faSysFile or FaVolumeID)) = 0 then begin
           FoundList.Items.Add(NewPath + SearchRec.Name);
           LoadButton.Enabled := True;
           Application.ProcessMessages;
       end;
     ReturnCode := FindNext(SearchRec);
     until ReturnCode <> 0;
   end;
  {Go through the directories now}
  if FindFirst(NewPath + '*.*', faDirectory, SearchRec) = 0 then begin
     repeat
       if (SearchRec.Attr and faDirectory > 1)
          and (SearchRec.Name <> '.')
          and (SearchRec.Name <> '..') then
          RecurseDirTree(NewPath + SearchRec.Name + '\');
     until FindNext(SearchRec) <> 0;
  end;
end;

procedure TFindExpDialog.SearchButtonClick(Sender: TObject);
begin
  Screen.Cursor := crHourGlass;
  LoadButton.Enabled := False;
  FoundList.Items.Clear;
  FoundList.Update;

  {Determine wheather this is a Grep searhc or a file find search}
  if SearchMethodRadioGrp.ITemIndex = 0 then
     GrepSearch(SearchDir)
  else
     RecurseDirTree(SearchDir);

  {select the first found item by default}
  if FoundList.Items.Count > 0 then
     FoundList.ItemIndex := 0
  else
    FoundList.Items.Add('No File(s) Found');

  Screen.Cursor := crDefault;
end;

procedure TFindExpDialog.FormCreate(Sender: TObject);
Const
  DelphiINIFileName = {$IFDEF WIN32} 'DELPHI32.INI';
                      {$ELSE}        'DELPHI.INI'; {$ENDIF}
var
  DelphiIni: TIniFile;
begin
  DelphiIni := TIniFile.Create(DelphiINIFileName);
  try
  with DelphiIni do begin
    BaseDelphiDir := UpperCase(ReadString('Library', 'ComponentLibrary', 'ERROR'));
    if BaseDelphiDir <> 'ERROR' then begin
       BaseDelphiDir := Copy(BaseDelphiDir, 1, Pos('BIN', BaseDelphiDir) - 1);
       {This sets up the default as the VCL directory}
       SearchDir := BaseDelphiDir + 'SOURCE\VCL\';
    end else begin
      MessageDlg('Delphi''s INI file could not be found - program terminating',mtError, [mbOk], 0);
      Application.terminate;
    end;
  end;
  finally
    DelphiIni.Free;
  end;
end;

procedure TFindExpDialog.FormShow(Sender: TObject);
begin
  SearchItem.SetFocus;
end;

procedure TFindExpDialog.SearchMethodRadioGrpClick(Sender: TObject);
begin
  if SearchMethodRadioGrp.ITemIndex = 0 then begin
     SearcgItemLabel.Caption := 'Search String';
     IgnoreCase.Enabled := True;
     FilterTypeCombo.Visible := True;
  end else begin
     SearcgItemLabel.Caption := 'File Name';
     IgnoreCase.Enabled := False;
     FilterTypeCombo.Visible := False;
  end;
  SearchItem.SetFocus;
  {SearchItem.Text := '';}
  FoundList.Items.Clear;
  LoadButton.Enabled := False;
end;

procedure TFindExpDialog.UserDefinedSearchPathExit(Sender: TObject);
var
  PathLen : Byte;
  TempPath : String;
begin
  TempPath := UserDefinedSearchPath.Text;
  PathLen := Length( TempPath );
  if TempPath[PathLen] <> '\' then
     UserDefinedSearchPath.Text := UpperCase(TempPath) + '\'
  else
    UserDefinedSearchPath.Text := UpperCase(TempPath);
  SearchDir := UserDefinedSearchPath.Text;
  Caption := ExpertName + ' - ' + SearchDir;
end;

end.
