unit Main;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, Buttons, StdCtrls, ExtCtrls, Options;
type
  Direction = (Up, Down);
type
  TForm1 = class(TForm)
    PanelToolBar: TPanel;
    PanelClient: TPanel;
    SpeedButtonPgDn: TSpeedButton;
    ComboBoxDir: TComboBox;
    Label1: TLabel;
    SpeedButtonDisplay: TSpeedButton;
    SpeedButtonOptions: TSpeedButton;
    SpeedButton2: TSpeedButton;
    SpeedButtonAbout: TSpeedButton;
    procedure DisplayAll(Sender: TObject);
    procedure BitBtnClearClick(Sender: TObject);
    procedure SpeedBtnScrollDown(Sender: TObject);
    procedure PanelClientResize(Sender: TObject);
    procedure SpeedButtonOptionsClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure SpeedButtonAboutClick(Sender: TObject);
    procedure ComboBoxDirChange(Sender: TObject);
    procedure ComboBoxDirKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure FormShow(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure SpeedButton2Click(Sender: TObject);
  private
    { Private declarations }

    OptionsList:TStringList; { Saved in Glyph.Ini }
    function  CreateNewSpeedBtn(location: TPoint; size: integer;
                                BmpFile: string) : boolean;
    procedure Display;
    procedure Initialize;
    function  Position(OldPosition : TPoint): TPoint;
    procedure FormAdjustSize;
    function  NewPage(d: Direction; var SearchRec: TSearchRec): integer;
    function  DirExists(const S : String) : boolean;
    function  UpdateIniFile(const INIFileName: string;
                            OptionsList: TStringList): boolean;
  public
    { Public declarations }
    FormOptions : TFormOptions;

    { ini parameters }
    iButSize:     byte; { buts size }
    iNumOfGlyphs: integer;
    iSetLimit:    integer;
    sDelphiDir:   string;

    IniFile:      string;
    procedure ClearOldButs;
    procedure DeleteOldButs;
    function  ReadIniFile(const INIFileName: string;
                          var OptionsList: TStringList): boolean;
    function  CreateIniFile(const INIFileName: string;
                            OptionsList: TStringList): boolean;
    procedure DynamicSpeedButtonClick(Sender: TObject);
  end;

var
  Form1: TForm1;

implementation
const
  DELTA = 2;
var
  pLocation :   TPoint;
  BmpPath :     string;
  iRow:         byte;
  SearchRec :   TSearchRec;

function TForm1.CreateNewSpeedBtn(location: TPoint; size: integer;
                                   BmpFile: string) : boolean;
const
  iCount : Integer = 0;
var
  GlyphBtn: TSpeedButton;
begin
  GlyphBtn := TSpeedButton.Create(PanelClient);
  GlyphBtn.Parent := PanelClient;
  with GlyphBtn do
  begin
    Left := location.x;
    Top := location.y;
    Width := size;
    Height := size;
    Caption := '';
    TabOrder := iCount;
    Tag := iCount;
    OnClick := DynamicSpeedButtonClick;
    if BmpFile <> '' then
    try
      NumGlyphs := iNumOfGlyphs;
      Glyph.LoadFromFile(BmpFile);
      Hint := ExtractFileName(BmpFile);
      ShowHint := TRUE;
      Result := TRUE;
    except
      MessageDlg('Error! - Unable to load ' + BmpFile + ' into button',
                 mtError, [mbOK], 0);
      Result := FALSE;
    end;
    Visible := TRUE;
  end;
  Inc(iCount, 1);
end;

{$R *.DFM}

procedure TForm1.Initialize;
begin
  pLocation.x := DELTA;
  pLocation.y := DELTA;
  iRow := 0;
  BmpPath := 'c:\delphi\images\buttons';
end;

procedure TForm1.DisplayAll(Sender: TObject);
begin
  Screen.Cursor := CrHourGlass;
  Initialize;
  FormAdjustSize;
  DeleteOldButs;
  if not DirExists(ComboBoxDir.Text) then
  begin
    Screen.Cursor := CrDefault;
    MessageDlg('Directory '+ComboBoxDir.Text+' does not exist',
                mtWarning, [mbOK], 0);
    ComboBoxDir.Items.Delete(ComboBoxDir.ItemIndex);
  end
  else
  begin
    Display;
    SpeedButtonPgDn.Enabled := TRUE;
    SpeedButtonDisplay.Caption := '&Display';
  end;
  Screen.Cursor := CrDefault;
end;

procedure TForm1.Display;
var
  iResult, iCount: integer;
begin
  iResult := findfirst(ComboBoxDir.Text + '\*.bmp', faAnyFile, SearchRec);
  if iResult <> 0 then
    MessageDlg( 'No files found in ' + ComboBoxDir.Text, mtError, [mbOK], 0);
  iCount := 0;
  while (iResult = 0) and (iCount < iSetLimit) do
  begin
    pLocation := Position(pLocation);
    CreateNewSpeedBtn(pLocation, iButSize, ComboBoxDir.Text
                                           +'\' + SearchRec.Name);
    Application.ProcessMessages;
    iResult := FindNext(SearchRec);
    Inc(iCount, 1);
  end;
end;

{ returns new position for a button based on the old one }
function TForm1.Position(OldPosition: TPoint) : TPoint;
var
  iScreenWidth, iOffset: integer;
begin
  iOffset := PanelClient.BorderWidth + DELTA;
  { calc x coord }
  iScreenWidth := PanelClient.Width - 2*PanelClient.BorderWidth;
  if(pLocation.x >= iScreenWidth - iButSize - 2*iOffset) or
    (iRow = 0) then
  begin
    pLocation.x := iOffset; { start new row }
    iRow := iRow + 1;
  end
  else
    pLocation.x := OldPosition.x + iButSize;
  { calculate y coord }
  pLocation.y := (iRow-1)*iButSize + iOffset;
  Position := pLocation;
end;

procedure TForm1.FormAdjustSize;
var
  iNumOfButs_x, iNumOfButs_y: byte;
  iCaption_y, a, b: integer;
begin
  iCaption_y := GetSystemMetrics(sm_cyCaption);
  a := GetSystemMetrics(sm_cxFrame);
  b := GetSystemMetrics(sm_cyFrame);
  { adjust horizontal size }
  iNumOfButs_x := (PanelClient.Width - 2*PanelClient.BorderWidth) div
                  iButSize;
  Width := iButSize*iNumOfButs_x + 2*(PanelClient.BorderWidth + a) + 4;
  { adjust vertical size }
  if iSetLimit mod iNumOfButs_x <> 0 then
    iNumOfButs_y := iSetLimit div iNumOfButs_x + 1
  else
    iNumOfButs_y := iSetLimit div iNumOfButs_x;
  Height := iNumOfButs_y*iButSize +
            2*(PanelClient.BorderWidth + b) +
            PanelToolBar.Height + iCaption_y + 3;
end;

procedure TForm1.DeleteOldButs;
var
  i,n: integer;
begin
  n := PanelClient.ComponentCount;
  for i:= n - 1 downto 0 do
    (PanelClient.Components[i] as TSpeedButton).Free;
end;

procedure TForm1.ClearOldButs;
var
  i,n: integer;
begin
  n := PanelClient.ComponentCount;
  for i:= n - 1 downto 0 do
    (PanelClient.Components[i] as TSpeedButton).Glyph := nil;
end;

procedure TForm1.BitBtnClearClick(Sender: TObject);
begin
  ClearOldButs;
end;

procedure TForm1.SpeedBtnScrollDown(Sender: TObject);
var
  i: integer;
begin
  i := NewPage(Down, SearchRec);
  if i = -18 then  { hit test }
    SpeedButtonPgDn.Enabled := FALSE;
  SpeedButtonDisplay.Caption := '&Reset';
end;

function  TForm1.NewPage(d: Direction;
                         var SearchRec: TSearchRec): integer;
var
  i, n: integer;
begin
  i := 0;
  n := PanelClient.ComponentCount;
  Screen.Cursor := CrHourGlass;
  repeat
    (PanelClient.Components[i] as TSpeedButton).Glyph.LoadFromFile(
                          ComboBoxDir.Text + '\' + SearchRec.Name);
    (PanelClient.Components[i] as TSpeedButton).Hint :=
                                                    SearchRec.Name;
    Result := FindNext(SearchRec);
    Inc(i, 1);
  until (i > (n - 1)) or (Result = -18);
  Screen.Cursor := CrDefault;
end;

procedure TForm1.PanelClientResize(Sender: TObject);
var
  i,n: integer;
begin
  pLocation.x := DELTA;
  pLocation.y := DELTA;
  n := PanelClient.ComponentCount;
  iRow := 0;
  for i:= 0 to n - 1 do
  begin
    pLocation := Position(pLocation);
    (PanelClient.Components[i] as TSpeedButton).Left := pLocation.x;
    (PanelClient.Components[i] as TSpeedButton).Top := pLocation.y;
  end;
  PanelClient.Align := alClient;
  Application.ProcessMessages;
  PanelClient.Align := alNone;
  FormAdjustSize;
  PanelClient.Align := alClient;
end;

procedure TForm1.SpeedButtonOptionsClick(Sender: TObject);
begin
  FormOptions := TFormOptions.Create(Self);
  if FormOptions.ShowModal = mrOK then
    DisplayAll(nil);
  FormOptions.Free;
end;

procedure TForm1.FormCreate(Sender: TObject);
var
  WinDir: string;
begin
  Initialize;
  SpeedButtonPgDn.Enabled := FALSE;
  ComboBoxDir.ItemIndex := 0;
  iNumOfGlyphs := 2;
  iSetLimit := 20;
  iButSize := 45;
  OptionsList := TStringList.Create; { ini file }
  GetWindowsDirectory(@WinDir[1], 254);
  WinDir[0] := char(StrLen(@WinDir[1]));
  IniFile := WinDir + '\' + 'GlyphPro.Ini';
  if not FileExists(IniFile) then
    CreateIniFile(IniFile, OptionsList);
  ReadIniFile(IniFile, OptionsList);
  ComboBoxDir.Text := ComboBoxDir.Items.Strings[0];
  DisplayAll(nil);
end;

procedure TForm1.SpeedButtonAboutClick(Sender: TObject);
begin
  MessageDlg('Glyph Pro prodused by '#13#10 +
             'Alex Tuller'#13#10 +
             'CIS 102247,747'#13#10 +
             'Stamford, US'#13#10#10'   Enjoy!',
             mtInformation, [mbOK], 0);
  ComboBoxDir.SetFocus;
  ComboBoxDir.SelLength := 0;
end;

function TForm1.DirExists(const S : String) : boolean;
var
  OldMode: Word;
  OldDir: String;
begin
  Result := True;
  GetDir(0, OldDir); { save old dir for return }
  OldMode := SetErrorMode(SEM_FAILCRITICALERRORS);{if drive empty ,
                                                    except}
  try
    try
      ChDir(S);
    except
      on EInOutError do Result := FALSE;
    end;
  finally
    ChDir(OldDir); { return to old dir }
    SetErrorMode(OldMode); { restore old error mode }
  end;
end;

procedure TForm1.ComboBoxDirChange(Sender: TObject);
begin
  ClearOldButs;
  SpeedButtonPgDn.Enabled := FALSE;
  SpeedButtonDisplay.Caption := '&Display';
  DisplayAll(nil);
end;

procedure TForm1.ComboBoxDirKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin { pgDn respond }
  if (Key = 34) and (SpeedButtonPgDn.Enabled = TRUE) then
  begin
    Key := 29;
    SpeedBtnScrollDown(nil)  { pgDown Pressed }
  end
  else if (Key = 33) then
  begin
    Key := 29;
    DisplayAll(nil);         { PgUp Pressed }
  end;
end;

procedure TForm1.FormShow(Sender: TObject);
begin
  ComboBoxDir.SetFocus;
  ComboBoxDir.SelLength := 0;
end;

function TForm1.CreateIniFile(const INIFileName: string;
                              OptionsList: TStringList): boolean;
var i : integer;
begin
  with OptionsList do
  begin
    Add('MainFormLeft='+IntToStr(Form1.Left));
    Add('MainFormTop='+IntToStr(Form1.Top));
    Add('MainFormHeight='+IntToStr(Form1.Height));
    Add('MainFormWidth='+IntToStr(Form1.Width));
    Add('ButtonSize='+IntToStr(iButSize));
    Add('NumberOfGlyphs='+IntToStr(iNumOfGlyphs));
    Add('ButtonsPerPage='+IntToStr(iSetLimit));
    Add('DelphiDir='+'C:\DELPHI');
    Add('GlyphDir1=' + BmpPath);
    for i := 2 to 15 do
      Add('GlyphDir'+IntToStr(i)+'=');
    SaveToFile(INIFileName);
  end;

end;

function TForm1.UpdateIniFile(const INIFileName: string;
                              OptionsList: TStringList): boolean;
var
  i : integer;
begin
  with OptionsList do
  begin
    Values['MainFormLeft'] := IntToStr(Form1.Left);
    Values['MainFormTop'] := IntToStr(Form1.Top);
    Values['MainFormHeight'] := IntToStr(Form1.Height);
    Values['MainFormWidth'] := IntToStr(Form1.Width);
    Values['ButtonSize'] := IntToStr(iButSize);
    Values['NumberOfGlyphs'] := IntToStr(iNumOfGlyphs);
    Values['ButtonsPerPage'] := IntToStr(iSetLimit);
    Values['DelphiDir'] := sDelphiDir;
    for i:= 1 to 15 do
      Values['GlyphDir'+IntToStr(i)] :=
                             ComboBoxDir.Items.Strings[i - 1];
    SaveToFile(INIFileName);
  end;
end;

function TForm1.ReadIniFile(const INIFileName: string;
                            var OptionsList: TStringList): boolean;
var
  i: integer;
  s: string;
begin
  with OptionsList do
  begin
    LoadFromFile(INIFileName);
    Form1.Left := StrToInt(Values['MainFormLeft']);
    Form1.Top := StrToInt(Values['MainFormTop']);
    Form1.Height := StrToInt(Values['MainFormHeight']);
    Form1.Width := StrToInt(Values['MainFormWidth']);
    iButSize := StrToInt(OptionsList.Values['ButtonSize']);
    iNumOfGlyphs := StrToInt(OptionsList.Values['NumberOfGlyphs']);
    iSetLimit := StrToInt(OptionsList.Values['ButtonsPerPage']);
    sDelphiDir := OptionsList.Values['DelphiDir'];
    for i := 1 to 15 do
    begin
      s := Values['GlyphDir'+IntToStr(i)];
      if s <> '' then
        ComboBoxDir.Items.Add(s);
    end;
    end;
end;

procedure TForm1.FormDestroy(Sender: TObject);
begin
  UpdateIniFile(IniFile, OptionsList);
  OptionsList.Free;
end;

procedure TForm1.DynamicSpeedButtonClick(Sender: TObject);
var
  s: string;
begin
  s := UpperCase(ComboBoxDir.Text+ '\' +
                                    (Sender as TSpeedButton).Hint);
  MessageDlg(s,mtInformation,[mbOK],0);
  ComboBoxDir.SetFocus;
  ComboBoxDir.SelLength := 0;
end;

procedure TForm1.SpeedButton2Click(Sender: TObject);
begin
  MessageDlg('GlyphPro supports:'#13#10+
             '- Ini File'#13#10+
             '- Up to 15 Search Directoris'#13#10+
             '- Form Resizing'#13#10+
             '- Customizable Number of Buttons per Page,'#13#10+
             '  Glyphs per Button and Button Sizes'#13#10+
             '- PgUp, PgDn Key'#13#10,
             mtInformation, [mbOK], 0);
  ComboBoxDir.SetFocus;
  ComboBoxDir.SelLength := 0;
end;

end.


