unit ClockFrm;
{====================================================================}
{ Program: Ledclock                                                  }
{ Author:  Sean R. Malloy                                            }
{ Copyright 1995 by Sean R. Malloy                                   }
{                                                                    }
{ This program creates a binary clock in the upper left corner of    }
{ the Windows display. The program displays three columns of LEDs;   }
{ the leftmost column is hours, the middle column is minutes, and    }
{ the rightmost column is seconds. Each column displays the value of }
{ that part of the time in a binary format, low-order bit at the     }
{ bottom. The color of the LEDs can be changed by the popup menu     }
{ accessed by right-clicking anywhere within the program's client    }
{ area.                                                              }
{                                                                    }
{ The compiled version of this program and its source code may be    }
{ freely distributed under the following conditions:                 }
{                                                                    }
{ 1. This copyright notice remains unaltered.                        }
{ 2. No 'compilation copyright' is claimed over any collection       }
{    containing this component or any of its associated files.       }
{ 3. No fee is charged for this component, except to cover the       }
{    cost of the disk by which the component is transferred.         }
{    This prohibition includes all WWW sites and BBSes that          }
{    require payment for download privileges.                        }
{                                                                    }
{ This program uses two custom components:                           }
{     the RoundLed component available as RNDLED11.ZIP               }
{     the TResolutionFix component available as RESFIX.ZIP           }
{ The RoundLed component is copyright 1995 by Martyn Dowsett         }
{     (CIS 100676,1560)                                              }
{ The TResolutionFix component is copyright 1995 by Sean Malloy      }
{     (malloy@cris.com)                                              }
{ Both components are available on the Super Delphi Page at          }
{      http://sunsite.icm.edu.pl/~robert/delphi                      }
{====================================================================}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, Buttons, RoundLed, Menus, ExtCtrls, IniFiles,
  Resfix;

const
  WS_SMALLTITLEBAR : longint = $4000;

type
  TButtonArrow = (arwUp, arwDown, arwRestore);

type
  TForm1 = class(TForm)
    HrLED1: TRoundLed;
    HrLED2: TRoundLed;
    HrLED4: TRoundLed;
    HrLED8: TRoundLed;
    HrLED16: TRoundLed;
    HrLED32: TRoundLed;
    MinLED1: TRoundLed;
    MinLED2: TRoundLed;
    MinLED4: TRoundLed;
    MinLED8: TRoundLed;
    MinLED16: TRoundLed;
    MinLED32: TRoundLed;
    SecLED1: TRoundLed;
    SecLED2: TRoundLed;
    SecLED4: TRoundLed;
    SecLED8: TRoundLed;
    SecLED16: TRoundLed;
    SecLED32: TRoundLed;
    Timer1: TTimer;
    PopupMenu1: TPopupMenu;
    Silver1: TMenuItem;
    Red1: TMenuItem;
    Lime1: TMenuItem;
    Yellow1: TMenuItem;
    Blue1: TMenuItem;
    Fuchsia1: TMenuItem;
    Aqua1: TMenuItem;
    White1: TMenuItem;
    ResolutionFix1: TResolutionFix;
    procedure FormCreate(Sender: TObject);
    procedure Timer1Timer(Sender: TObject);
    procedure Silver1Click(Sender: TObject);
    procedure Red1Click(Sender: TObject);
    procedure Lime1Click(Sender: TObject);
    procedure Yellow1Click(Sender: TObject);
    procedure Blue1Click(Sender: TObject);
    procedure Fuchsia1Click(Sender: TObject);
    procedure Aqua1Click(Sender: TObject);
    procedure White1Click(Sender: TObject);
    procedure FormShow(Sender: TObject);
  private
    { Private declarations }
    fMenuUp : boolean;       {state of the system menu}
    TitleBarSize : integer;  {height of the title bar in pixels}
    function TestWinStyle(dwStyleBit : longint) : boolean;
    function HasCaption : boolean;
    function GetTitleBarRect(var rc : TRect) : boolean;
    function GetControlBoxRect(var rc : TRect) : boolean;
    function GetMinButtonRect(var rc : TRect) : boolean;
    function GetMaxButtonRect(var rc : TRect) : boolean;
    function GetButtonRect(nPos : word; var rc : TRect) : boolean;
    function DoMenu : boolean;
    procedure SetupSystemMenu(menu : HMenu);
    procedure DrawControlBox(dc : HDC; fInvert : boolean);
    function DepressMinMaxButton(HitTest : word; var rc : TRect) : boolean;
    procedure DrawButton(dc : HDC; fMin, fDepressed : boolean);
    procedure DrawArrow(dc : HDC; const rc : TRect; style : TButtonArrow);
    function DrawCaption(fSysMenu, fMin, fMax, fActive : boolean) : boolean;
  protected
    { Protected declarations }
    procedure WndProc(var Message : TMessage); override;
    procedure WMNCCreate(var Message : TWMNCCreate); message WM_NCCREATE;
    procedure WMNCCalcSize(var Message : TWMNCCalcSize); message WM_NCCALCSIZE;
    procedure WMNCHitTest(var Message : TWMNCHitTest); message WM_NCHITTEST;
    procedure WMNCLButtonDblClk(var Message : TWMNCLButtonDblClk); message WM_NCLBUTTONDBLCLK;
    procedure WMNCLButtonDown(var Message : TWMNCLButtonDown); message WM_NCLBUTTONDOWN;
    procedure WMSysChar(var Message : TWMSysChar); message WM_SYSCHAR;
    procedure WMCommand(var Message : TMessage); message WM_COMMAND;
    procedure WMKeyDown(var Message : TWMKeyDown); message WM_KEYDOWN;
    procedure WMKeyUp(var Message : TWMKeyUp); message WM_KEYUP;
    procedure WMSysKeyDown(var Message : TWMSysKeyDown); message WM_SYSKEYDOWN;
    procedure WMSysKeyUp(var Message : TWMSysKeyUp); message WM_SYSKEYUP;
    procedure WMGetMinMaxInfo(var Message : TWMGetMinMaxInfo); message WM_GETMINMAXINFO;
  public
    { Public declarations }
  end;

var
  Form1: TForm1;

implementation

{$R *.DFM}

var
  lstHr,lstMin : integer;
  ClockIni : TIniFile;
  ClockColor : longint;

function TForm1.TestWinStyle(dwStyleBit : longint) : boolean;
begin
  Result := ((GetWindowLong(Handle, GWL_STYLE) and dwStyleBit) <> 0);
end;

function TForm1.HasCaption : boolean;
begin
  Result := TestWinStyle(WS_SMALLTITLEBAR);
end;

function TForm1.GetTitleBarRect(var rc : TRect) : boolean;
begin
  Result := false;  {Initially assume no caption.}
  if HasCaption then begin
    GetWindowRect(Handle, rc);
    if TestWinStyle(WS_THICKFRAME) then  {Adjust for borders.}
      InflateRect(rc, -GetSystemMetrics(SM_CXFRAME),
                      -GetSystemMetrics(SM_CYFRAME))
    else if TestWinStyle(DS_MODALFRAME) then
      InflateRect(rc, -(GetSystemMetrics(SM_CXDLGFRAME)
                      + GetSystemMetrics(SM_CXBORDER)),
                      -(GetSystemMetrics(SM_CYDLGFRAME)
                      + GetSystemMetrics(SM_CYBORDER)))
    else if TestWinStyle(WS_BORDER) then
      InflateRect(rc, -GetSystemMetrics(SM_CXBORDER),
                      -GetSystemMetrics(SM_CYBORDER));
    rc.bottom := rc.top + TitleBarSize;
    Result := true;
  end else
    SetRectEmpty(rc);
end;

function TForm1.GetControlBoxRect(var rc : TRect) : boolean;
begin
  Result := false;  {Initially assume no control box.}
  if GetTitleBarRect(rc) then begin
    if TestWinStyle(WS_SYSMENU) then begin
      rc.right := rc.left + TitleBarSize - 1;
      Dec(rc.bottom);
      Result := true;
    end else
      SetRectEmpty(rc);
  end;
end;

function TForm1.GetMinButtonRect(var rc : TRect) : boolean;
begin
  Result := false;  {Initially assume no min. button.}
  if TestWinStyle(WS_MINIMIZEBOX) then begin
    if TestWinStyle(WS_MAXIMIZEBOX) then  {If win has a maximize box,}
      Result := GetButtonRect(2, rc)      {min. box is in position 2.}
    else                                  {Otherwise,                }
      Result := GetButtonRect(1, rc);     {min. box is in position 1.}
  end else
    SetRectEmpty(rc);
end;

function TForm1.GetMaxButtonRect(var rc : TRect) : boolean;
begin
  Result := false;  {Initially assume no max. button.}
  if TestWinStyle(WS_MAXIMIZEBOX) then
    Result := GetButtonRect(1, rc)
  else
    SetRectEmpty(rc);
end;

function TForm1.GetButtonRect(nPos : word; var rc : TRect) : boolean;
begin
  Result := false;  {Initially assume no button.}
  if GetTitleBarRect(rc) then begin
    Dec(rc.right, TitleBarSize * (nPos - 1));
    rc.left := rc.right - TitleBarSize + 1;
    Result := true;
  end;
end;

function TForm1.DoMenu : boolean;
var
  dc   : HDC;
  rc   : TRect;
  pt   : TPoint;
  menu : HMenu;
begin
  Result := false;  {Initially assume no menu}
  if TestWinStyle(WS_SYSMENU) then begin
    dc := GetWindowDC(Handle);
    if dc <> 0 then begin
      {Invert the control box}
      DrawControlBox(dc, true);
      {Pop up the mock-system menu}
      pt := Point(0, -1);
      GetWindowRect(Handle, rc);
      {Convert coordinates to screen coords. using functions in WinProcs unit}
      {("WinProcs" must be given to avoid calling TForm1's ClientToScreen() )}
      WinProcs.ClientToScreen(Handle, pt);
      WinProcs.ClientToScreen(Handle, rc.BottomRight);
      menu := GetSystemMenu(Handle, false);
      SetupSystemMenu(menu);
      TrackPopupMenu(menu, 0, pt.x, pt.y, 0, Handle, @rc);
      DrawControlBox(dc, false);
      ReleaseDC(Handle, dc);
    end;
    Result := true;
  end;
end;

procedure TForm1.SetupSystemMenu(menu : HMenu);
var
  wMove    : word;
  wSize    : word;
  wMinBox  : word;
  wMaxBox  : word;
  wRestore : word;
begin
  {Initially assume all menu items should be grayed}
  wMove    := MF_GRAYED;
  wSize    := MF_GRAYED;
  wMinBox  := MF_GRAYED;
  wMaxBox  := MF_GRAYED;
  wRestore := MF_GRAYED;
  {Now check the window styles, etc.}
  if not (IsIconic(Handle) or IsZoomed(Handle)) then begin
    if TestWinStyle(WS_CAPTION) then
      wMove := MF_ENABLED;
    if TestWinStyle(WS_THICKFRAME) then
      wSize := MF_ENABLED;
  end;
  if TestWinStyle(WS_MINIMIZEBOX) then
    wMinBox := MF_ENABLED;
  if TestWinStyle(WS_MAXIMIZEBOX) or IsIconic(Handle) then
    wMaxBox := MF_ENABLED;
  if IsZoomed(Handle) then
    wRestore := MF_ENABLED;
  EnableMenuItem(menu, SC_MOVE,     wMove);
  EnableMenuItem(menu, SC_SIZE,     wSize);
  EnableMenuItem(menu, SC_MINIMIZE, wMinBox);
  EnableMenuItem(menu, SC_MAXIMIZE, wMaxBox);
  EnableMenuItem(menu, SC_RESTORE,  wRestore);
end;

procedure TForm1.DrawControlBox(dc : HDC; fInvert : boolean);
var
  rc    : TRect;
  rcBox : TRect;
begin
  {Prepare to draw the control box}
  if dc <> 0 then begin
    {Calculate size and position of control box in window coords.}
    GetControlBoxRect(rcBox);
    GetWindowRect(Handle, rc);
    OffsetRect(rcBox, -rc.left, -rc.top);
    {Calculate separator line to right of control box}
    rc := rcBox;
    rc.left := rc.right;
    Inc(rc.right);
    {Fill control box area}
    SetBkColor(dc, ColorToRGB(clSilver));
    ExtTextOut(dc, 0, 0, ETO_OPAQUE, @rcBox, nil, 0, nil);
    {Draw separator line}
    SetBkColor(dc, ColorToRGB(clWindowFrame));
    ExtTextOut(dc, 0, 0, ETO_OPAQUE, @rc, nil, 0, nil);
    {If title bar isn't too small, draw the lil' horizontal doo-hickey}
    if TitleBarSize > 4 then begin
      rc := rcBox;
      rc.top    := rcBox.top + ((TitleBarSize - 1) div 2);
      rc.bottom := rc.top + 3;
      rc.left   := rc.left + 3;
      rc.right  := rc.right - 1;
      SetBKColor(dc, ColorToRGB(clGray));
      ExtTextOut(dc, 0, 0, ETO_OPAQUE, @rc, nil, 0, nil);
      OffsetRect(rc, -1, -1);
      SetBkColor(dc, ColorToRGB(clBlack));
      ExtTextOut(dc, 0, 0, ETO_OPAQUE, @rc, nil, 0, nil);
      InflateRect(rc, -1, -1);
      SetBkColor(dc, ColorToRGB(clWhite));
      ExtTextOut(dc, 0, 0, ETO_OPAQUE, @rc, nil, 0, nil);
    end;
  end;
  {Invert the control box if needed}
  if fInvert then
    InvertRect(dc, rcBox);
end;

function TForm1.DepressMinMaxButton(HitTest : word; var rc : TRect) : boolean;
var
  msg        : TMsg;
  fDepressed : boolean;
  fDone      : boolean;
begin
  fDone := false;  {we've only just begun}
  fDepressed := true;  {initially draw button in down state}
  DrawButton(0, (HitTest = HTMINBUTTON), fDepressed);
  SetCapture(Handle);  {collect all mouse events until WM_LBUTTONUP}
  while not fDone do begin  {loop until the button is released}
    if PeekMessage(msg, 0, WM_MOUSEFIRST, WM_MOUSELAST, PM_REMOVE) then begin
      case msg.message of
      WM_LBUTTONUP:
        begin
          if fDepressed then
            DrawButton(0, (HitTest = HTMINBUTTON), not fDepressed);
          ReleaseCapture;
          Result := PtInRect(rc, msg.pt);
          fDone  := true;
        end;
      WM_MOUSEMOVE:
        if PtInRect(rc, msg.pt) then begin
          if not fDepressed then begin
            fDepressed := true;
            DrawButton(0, (HitTest = HTMINBUTTON), fDepressed);
          end;
        end else begin
          if fDepressed then begin
            fDepressed := false;
            DrawButton(0, (HitTest = HTMINBUTTON), fDepressed);
          end;
        end;
      end;
    end;
  end;
end;

procedure TForm1.DrawButton(dc : HDC; fMin, fDepressed : boolean);
const
  THRESHOLD = 20; {if button less than 20 pixels, use one line of shadow}
var
  rc          : TRect;
  rcButton    : TRect;
  fDC         : boolean;  {did we have to create the DC here?   }
  nOffset     : word;     {displacement by button shadow/hilight}
  n           : integer;
begin
  if TitleBarSize >= THRESHOLD then
    nOffset := 2
  else
    nOffset := 1;
  if dc = 0 then begin
   fDC := true;
   dc  := GetWindowDC(Handle);
  end else
    fDC := false;
  if dc <> 0 then begin
    {Get size & position of button, and convert to window coordinates}
    if fMin then
      GetMinButtonRect(rcButton)
    else
      GetMaxButtonRect(rcButton);
    GetWindowRect(Handle, rc);
    OffsetRect(rcButton, -rc.left, -rc.top);
    {Draw vertical separator line to the left of button}
    rc := rcButton;
    rc.right := rc.left;
    Dec(rc.left);
    Dec(rcButton.bottom);
    SetBkColor(dc, ColorToRGB(clWindowFrame));
    ExtTextOut(dc, 0, 0, ETO_OPAQUE, @rc, nil, 0, nil);
    {Fill button area}
    SetBkColor(dc, ColorToRGB(clSilver));
    ExtTextOut(dc, 0, 0, ETO_OPAQUE, @rcButton, nil, 0, nil);
    if not fDepressed then begin
      {Draw button hilight (left & top)}
      SetBkColor(dc, ColorToRGB(clWhite));
      {Left edge}
      rc := rcButton;
      rc.right := rc.left + 1;
      ExtTextOut(dc, 0, 0, ETO_OPAQUE, @rc, nil, 0, nil);
      {Top edge}
      rc := rcButton;
      rc.bottom := rc.top + 1;
      ExtTextOut(dc, 0, 0, ETO_OPAQUE, @rc, nil, 0, nil);
      {Draw button shadow (right & bottom)}
      SetBkColor(dc, ColorToRGB(clGray));
      {Right edge}
      rc := rcButton;
      rc.left := rc.right - 1;
      ExtTextOut(dc, 0, 0, ETO_OPAQUE, @rc, nil, 0, nil);
      if TitleBarSize > THRESHOLD then begin
        Dec(rc.left);
        Inc(rc.top);
        ExtTextOut(dc, 0, 0, ETO_OPAQUE, @rc, nil, 0, nil);
      end;
      {Bottom edge}
      rc := rcButton;
      rc.top := rc.bottom - 1;
      ExtTextOut(dc, 0, 0, ETO_OPAQUE, @rc, nil, 0, nil);
      if TitleBarSize > THRESHOLD then begin
        Dec(rc.top);
        Inc(rc.left);
        ExtTextOut(dc, 0, 0, ETO_OPAQUE, @rc, nil, 0, nil);
      end;
      {Adjust rcButton to everything inside the shadows/hilights}
      Inc(rcButton.left);
      Inc(rcButton.top);
      Dec(rcButton.right,  nOffset);
      Dec(rcButton.bottom, nOffset);
    end else begin
      {Draw depressed state}
      SetBkColor(dc, ColorToRGB(clGray));
      {Left edge}
      rc := rcButton;
      rc.right := rc.left + nOffset;
      ExtTextOut(dc, 0, 0, ETO_OPAQUE, @rc, nil, 0, nil);
      {Top edge}
      rc := rcButton;
      rc.bottom := rc.top + nOffset;
      ExtTextOut(dc, 0, 0, ETO_OPAQUE, @rc, nil, 0, nil);
      {Adjust rcButton to everything inside the shadows/hilights}
      Inc(rcButton.left, 2 * nOffset);
      Inc(rcButton.top,  2 * nOffset);
    end;
    {Draw the arrows, restricting their size for larger than normal title bars.}
    {A good maximum size for arrows is (SM_CYCAPTION div 2).                   }
    n := (GetSystemMetrics(SM_CYCAPTION) div 2)
         - (rcButton.right - rcButton.left);
    if n < 1 then
      InflateRect(rcButton, (n div 2) - 1, (n div 2) - 1);
    if fMin then
      DrawArrow(dc, rcButton, arwDown)
    else if IsZoomed(Handle) then
      DrawArrow(dc, rcButton, arwRestore)
    else
      DrawArrow(dc, rcButton, arwUp);
    {Release the DC if we created in this procedure}
    if fDC then
      ReleaseDC(Handle, dc);
  end;
end;


procedure TForm1.DrawArrow(dc : HDC; const rc : TRect; style : TButtonArrow);
var
  row     : integer;
  xTip    : integer;
  yTip    : integer;
  rcArrow : TRect;
  nMax    : integer;
begin
  nMax := (rc.bottom - rc.top) shr 1;
  {The arrow is drawn as a series of horizontal lines}
  SetBkColor(dc, ColorToRGB(clBlack));
  xTip := rc.left + ((rc.right - rc.left + 1) shr 1);
  case style of
  arwUp:
    begin
      yTip := rc.top + ((rc.bottom - rc.top - 1) shr 2);
      for row := 1  to nMax do begin
        rcArrow.left   := xTip - row;
        rcArrow.right  := xTip + row - 1;
        rcArrow.top    := yTip + row;
        rcArrow.bottom := rcArrow.top + 1;
        ExtTextOut(dc, 0, 0, ETO_OPAQUE, @rcArrow, nil, 0, nil);
      end;
    end;
  arwDown:
    begin
      yTip := rc.bottom - ((rc.bottom - rc.top - 1) shr 2);
      for row := nMax downto 1 do begin
        rcArrow.left   := xTip - row;
        rcArrow.right  := xTip + row - 1;
        rcArrow.top    := yTip - row;
        rcArrow.bottom := rcArrow.top + 1;
        ExtTextOut(dc, 0, 0, ETO_OPAQUE, @rcArrow, nil, 0, nil);
      end;
    end;
  arwRestore:
    begin
      yTip := rc.top + ((rc.bottom - rc.top - 1) shr 3) - 2;
      for row := 1  to nMax do begin
        rcArrow.left   := xTip - row;
        rcArrow.right  := xTip + row - 1;
        rcArrow.top    := yTip + row;
        rcArrow.bottom := rcArrow.top + 1;
        ExtTextOut(dc, 0, 0, ETO_OPAQUE, @rcArrow, nil, 0, nil);
      end;
      Inc(yTip, (nMax + 1) * 2);
      for row := nMax downto 1 do begin
        rcArrow.left   := xTip - row;
        rcArrow.right  := xTip + row - 1;
        rcArrow.top    := yTip - row;
        rcArrow.bottom := rcArrow.top + 1;
        ExtTextOut(dc, 0, 0, ETO_OPAQUE, @rcArrow, nil, 0, nil);
      end;
    end;
  end;
end;

function TForm1.DrawCaption(fSysMenu, fMin, fMax, fActive : boolean) : boolean;
const
  THRESHOLD = 20; {if caption >= 20 pixels, use bold font}
var
  dc        : HDC;
  rc        : TRect;
  rcCap     : TRect;
  rgbText   : TColor;
  rgbBkGrnd : TColor;
  hbrCap    : HBrush;
  lpsz      : PChar;
  textlen   : word;
  lf        : TLogFont;
  font      : HFont;
  size      : TSize;
  cx, cy    : integer;
begin
  dc := GetWindowDC(Handle);
  if dc <> 0 then begin
    {Determine colors}
    if fActive then begin
      rgbText   := ColorToRGB(clCaptionText);
      rgbBkGrnd := ColorToRGB(clActiveCaption);
    end else begin
      rgbText   := ColorToRGB(clInactiveCaptionText);
      rgbBkGrnd := ColorToRGB(clInactiveCaption);
    end;
    {Calculate titlebar rectangle in window coords.}
    GetTitleBarRect(rcCap);
    GetWindowRect(Handle, rc);
    OffsetRect(rcCap, -rc.left, -rc.top);
    {Calculate horizontal separator line below titlebar}
    SetRect(rc, rcCap.left, rcCap.bottom - 1, rcCap.right, rcCap.bottom);
    {Draw separator line}
    SetBkMode(dc, TRANSPARENT);
    SelectObject(dc, GetStockObject(NULL_BRUSH));
    SelectObject(dc, GetStockObject(NULL_PEN));
    SetBkColor(dc, ColorToRGB(clWindowFrame));
    ExtTextOut(dc, 0, 0, ETO_OPAQUE, @rc, nil, 0, nil);
    {Shrink caption area to avoid overlapping control box & min/max buttons}
    if fSysMenu then
      Inc(rcCap.left, TitleBarSize);
    if fMax then
      Dec(rcCap.right, TitleBarSize);
    if fMin then
      Dec(rcCap.right, TitleBarSize);
    {Draw caption background (don't use ExtTextOut, we may need dithered colors)}
    hbrCap := CreateSolidBrush(rgbBkGrnd);
    hbrCap := SelectObject(dc, hbrCap);  {swap brushes with device context }
    SelectObject(dc, GetStockObject(NULL_PEN));
    Rectangle(dc, rcCap.left, rcCap.top, rcCap.right + 1, rcCap.bottom);
    hbrCap := SelectObject(dc, hbrCap);  {swap brushes back again          }
    DeleteObject(hbrCap);                {delete the brush we created      }
    {Draw caption text here}
    textlen := GetWindowTextLength(Handle);     {Get length of caption text}
    lpsz := GlobalAllocPtr(GHND, textlen + 2);  {Allocate a text buffer    }
    if lpsz <> nil then begin
      GetWindowText(Handle, lpsz, textlen + 1); {Copy in caption text      }
      rgbText := SetTextColor(dc, rgbText);     {Swap text colors with dc  }
      {Prepare logical font structure to get a font to use.}
      FillChar(lf, sizeof(TLogFont), #0);       {Clear the font structure  }
      lf.lfHeight  := -(TitleBarSize - 3);
      lf.lfCharSet := ANSI_CHARSET;
      lf.lfQuality := DEFAULT_QUALITY;
      lf.lfClipPrecision := CLIP_LH_ANGLES or CLIP_STROKE_PRECIS;
      if TitleBarSize >= THRESHOLD then
        lf.lfWeight := FW_BOLD;
      {Use small fonts for caption since it looks more like "System" than Arial}
      lf.lfPitchAndFamily := FF_SWISS;
      font := CreateFontIndirect(lf);
      font := SelectObject(dc, font);        {swap font with device context}
      {Calculate centering for caption text}
      GetTextExtentPoint(dc, lpsz, textlen, size);
      cx := rcCap.left + ((rcCap.right - rcCap.left - size.cx) div 2);
      cy := rcCap.top + ((rcCap.bottom - rcCap.top - size.cy) div 2);
      if rcCap.left > cx then
        cx := rcCap.left;                  {limit starting position of text}
      {Draw caption text}
      ExtTextOut(dc, cx, cy, ETO_CLIPPED, @rcCap, lpsz, textlen, nil);
      font := SelectObject(dc, font);                 {swap font back again}
      DeleteObject(font);                             {delete it           }
      {Clean up device context & free memory}
      rgbText := SetTextColor(dc, rgbText);           {swap back text color}
      GlobalFreePtr(lpsz);                            {deallocate buffer   }
    end;
    {Draw control box, min button, and max button as needed}
    if fSysMenu then
      DrawControlBox(dc, false);
    if fMin then
      DrawButton(dc, true, false);
    if fMax then
      DrawButton(dc, false, false);
    Result := true;
    ReleaseDC(Handle, dc);
  end;
end;

procedure TForm1.WndProc(var Message : TMessage);
var
        fActive  : boolean;
begin
  with Message do begin
    case msg of
    WM_NCPAINT, WM_NCACTIVATE:
      begin
        inherited WndProc(Message);
        if HasCaption and not IsIconic(Handle) then begin
          if msg = WM_NCPAINT then
            fActive := (Handle = GetActiveWindow)
          else
            fActive := (wparam <> 0);
          DrawCaption(TestWinStyle(WS_SYSMENU),
                      TestWinStyle(WS_MINIMIZEBOX),
                      TestWinStyle(WS_MAXIMIZEBOX),
                      fActive);
        end;
      end;
    else
      inherited WndProc(Message);
    end; { case msg of }
  end; { with Message do }
end;

procedure TForm1.WMNCCreate(var Message : TWMNCCreate);
var
  dwStyle : longint;
begin
  fMenuUp := false;  {System menu not initially showing}
  TitleBarSize := (GetSystemMetrics(SM_CYCAPTION) div 2) + 1;
  dwStyle := GetWindowLong(Handle, GWL_STYLE);
  dwStyle := dwStyle or WS_SMALLTITLEBAR;
  if (dwStyle and WS_DLGFRAME) = WS_DLGFRAME then
    dwStyle := dwStyle and not longint(WS_DLGFRAME);
  SetWindowLong(Handle, GWL_STYLE, dwStyle);
  inherited;  {Call default processing.}
end;

procedure TForm1.WMNCCalcSize(var Message : TWMNCCalcSize);
begin
  inherited;  {Call default processing.}
  if HasCaption and not IsIconic(Handle) then
    Inc(Message.CalcSize_Params^.rgrc[0].top, TitleBarSize);
end;

procedure TForm1.WMNCHitTest(var Message : TWMNCHitTest);
var
  rcCap  : TRect;
  rcMenu : TRect;
  rcMin  : TRect;
  rcMax  : TRect;
begin
  inherited;  {Call default processing.}
  if (Message.Result = HTNOWHERE) and HasCaption and not IsIconic(Handle) then begin
    GetTitleBarRect(rcCap);
    if PtInRect(rcCap, Message.Pos) then begin
      Message.Result := HTCAPTION;
      GetControlBoxRect(rcMenu);
      GetMinButtonRect(rcMin);
      GetMaxButtonRect(rcMax);
      if PtInRect(rcMenu, Message.Pos) then
        Message.Result := HTSYSMENU
      else if PtInRect(rcMin, Message.Pos) then
        Message.Result := HTMINBUTTON
      else if PtInRect(rcMax, Message.Pos) then
        Message.Result := HTMAXBUTTON;
    end;
  end;
  if Message.Result <> HTSYSMENU then
    fMenuUp := false;  {Indicate the system menu is not showing}
end;

procedure TForm1.WMNCLButtonDblClk(var Message : TWMNCLButtonDblClk);
begin
  if (Message.HitTest = HTSYSMENU) and HasCaption and not IsIconic(Handle) then
    SendMessage(Handle, WM_CLOSE, 0, 0)
  else
    inherited;  {Call default processing.}
end;

procedure TForm1.WMNCLButtonDown(var Message : TWMNCLButtonDown);
var
  rc : TRect;
  pt : TPoint;
begin
  if HasCaption and not IsIconic(Handle) then begin
    case Message.HitTest of
    HTSYSMENU:
      if not fMenuUp and DoMenu then
        fMenuUp := true
      else
        fMenuUp := false;
    HTMINBUTTON:
      begin
        pt := Point(Message.XCursor, Message.YCursor);
        GetMinButtonRect(rc);
        if DepressMinMaxButton(Message.HitTest, rc) then
          SendMessage(Handle, WM_SYSCOMMAND, SC_MINIMIZE, longint(pt));
      end;
    HTMAXBUTTON:
      begin
        pt := Point(Message.XCursor, Message.YCursor);
        GetMaxButtonRect(rc);
        if DepressMinMaxButton(Message.HitTest, rc) then begin
          if IsZoomed(Handle) then
            SendMessage(Handle, WM_SYSCOMMAND, SC_RESTORE, longint(pt))
          else
            SendMessage(Handle, WM_SYSCOMMAND, SC_MAXIMIZE, longint(pt));
        end;
      end;
    else
      inherited;  {Call default processing.}
    end;
  end else
    inherited;  {Call default processing.}
end;

procedure TForm1.WMSysChar(var Message : TWMSysChar);
begin
  if HasCaption and (Message.CharCode = VK_SPACE) then
    DoMenu
  else
    inherited;  {Call default processing.}
end;

procedure TForm1.WMCommand(var Message : TMessage);
begin
  if Message.wParam >= $F000 then
    PostMessage(Handle, WM_SYSCOMMAND, Message.wParam, Message.lParam);
  inherited;  {Call default processing.}
end;

procedure TForm1.WMKeyDown(var Message : TWMKeyDown);
var
  dwStyle : longint;
begin
  dwStyle := GetWindowLong(Handle, GWL_STYLE);
  SetWIndowLong(Handle, GWL_STYLE, dwStyle and not longint(WS_SYSMENU));
  inherited;  {Call default processing.}
  SetWindowLong(Handle, GWL_STYLE, dwStyle);
end;

procedure TForm1.WMKeyUp(var Message : TWMKeyUp);
var
  dwStyle : longint;
begin
  dwStyle := GetWindowLong(Handle, GWL_STYLE);
  SetWIndowLong(Handle, GWL_STYLE, dwStyle and not longint(WS_SYSMENU));
  inherited;  {Call default processing.}
  SetWindowLong(Handle, GWL_STYLE, dwStyle);
end;

procedure TForm1.WMSysKeyDown(var Message : TWMSysKeyDown);
var
  dwStyle : longint;
begin
  dwStyle := GetWindowLong(Handle, GWL_STYLE);
  SetWIndowLong(Handle, GWL_STYLE, dwStyle and not longint(WS_SYSMENU));
  inherited;  {Call default processing.}
  SetWindowLong(Handle, GWL_STYLE, dwStyle);
end;

procedure TForm1.WMSysKeyUp(var Message : TWMSysKeyUp);
var
  dwStyle : longint;
begin
  dwStyle := GetWindowLong(Handle, GWL_STYLE);
  SetWIndowLong(Handle, GWL_STYLE, dwStyle and not longint(WS_SYSMENU));
  inherited;  {Call default processing.}
  SetWindowLong(Handle, GWL_STYLE, dwStyle);
end;

procedure TForm1.WMGetMinMaxInfo(var Message : TWMGetMinMaxInfo);
var
  nX     : integer;
  cy     : integer;
  rcBox  : TRect;
  rcMin  : TRect;
  rcMax  : TRect;
begin
  if HasCaption and TestWinStyle(WS_THICKFRAME) then begin
    cy := GetSystemMetrics(SM_CYFRAME);
    {The following functions return empty rects. if box/button doesn't exist}
    GetControlBoxRect(rcBox);
    GetMinButtonRect(rcMin);
    GetMaxButtonRect(rcMax);
    nX := (rcBox.right - rcBox.left) +
          (rcMin.right - rcMin.left) +
          (rcMax.right - rcMax.left);
    with Message.MinMaxInfo^.ptMinTrackSize do begin
      x := nX + 2 * TitleBarSize;
      y := TitleBarSize + 2 * cy - 1;
    end;
  end;
  with Message.MinMaxInfo^ do
  begin
    ptMaxSize.X:= 80;
    ptMaxSize.Y:= 150;
    ptMinTrackSize.X:=20;
    ptMinTrackSize.Y:=50;
    ptMaxTrackSize.X:=80;
    ptMaxTrackSize.Y:=150;
  end;
  Message.Result:= 0;
end;

procedure ReColor(newcolor : TLedOnColors);
begin
  with Form1 do
  begin
    HrLED1.LedOnColor:=   newcolor;
    HrLED2.LedOnColor:=   newcolor;
    HrLED4.LedOnColor:=   newcolor;
    HrLED8.LedOnColor:=   newcolor;
    HrLED16.LedOnColor:=  newcolor;
    HrLED32.LedOnColor:=  newcolor;
    MinLED1.LedOnColor:=  newcolor;
    MinLED2.LedOnColor:=  newcolor;
    MinLED4.LedOnColor:=  newcolor;
    MinLED8.LedOnColor:=  newcolor;
    MinLED16.LedOnColor:= newcolor;
    MinLED32.LedOnColor:= newcolor;
    SecLED1.LedOnColor:=  newcolor;
    SecLED2.LedOnColor:=  newcolor;
    SecLED4.LedOnColor:=  newcolor;
    SecLED8.LedOnColor:=  newcolor;
    SecLED16.LedOnColor:= newcolor;
    SecLED32.LedOnColor:= newcolor;
  end;
  ClockColor:= ClockIni.ReadInteger('LEDClock','LEDColor',-1);
  if (Ord(newcolor)<>ClockColor) then
    ClockIni.WriteInteger('LEDClock','LEDColor',Ord(newcolor));
end;

procedure TForm1.FormCreate(Sender: TObject);
begin
  Form1.Width:= SecLED1.Left + SecLED1.Width + 2;
  Form1.Height:= SecLED1.Top + SecLED1.Height + TitleBarSize + 2;
  Form1.Top:= 0;
  Form1.Left:= 0;
  lstHr:= -1;
  lstMin:= -1;
  ClockIni:= TIniFile.Create('WIN.INI');
  ClockColor:= ClockIni.ReadInteger('LEDClock','LEDColor',-1);
  if (ClockColor<0) then
  begin
    ClockColor:= 1;
    ClockIni.WriteInteger('LEDClock','LEDColor',ClockColor);
  end;
  case ClockColor of
    0: ReColor(lcSilver);
    1: ReColor(lcRed);
    2: ReColor(lcLime);
    3: ReColor(lcYellow);
    4: ReColor(lcBlue);
    5: ReColor(lcFuchsia);
    6: ReColor(lcAqua);
    7: ReColor(lcWhite);
  end;
end;

function NewLedState(check:integer):TLedStatus;
begin
  if (check<>0) then Result:= LsON
             else Result:= LsOff;
end;

procedure TForm1.Timer1Timer(Sender: TObject);
var
  hr,min,sec : integer;
  tmp : Real;
begin
  tmp:= 24 * Frac(Now);
  hr:= Trunc(tmp);
  tmp:= 60 * (tmp - hr);
  min:= Trunc(tmp);
  sec:= Trunc(60 * (tmp - min));
  if (lstHr<>hr) then
  begin
    HrLED1.LedStatus:=  NewLedState((hr AND 1));
    HrLED2.LedStatus:=  NewLedState((hr AND 2));
    HrLED4.LedStatus:=  NewLedState((hr AND 4));
    HrLED8.LedStatus:=  NewLedState((hr AND 8));
    HrLED16.LedStatus:= NewLedState((hr AND 16));
    HrLED32.LedStatus:= NewLedState((hr AND 32));
    lstHr:= hr;
  end;
  if (lstMin<>min) then
  begin
    MinLED1.LedStatus:=  NewLedState((min AND 1));
    MinLED2.LedStatus:=  NewLedState((min AND 2));
    MinLED4.LedStatus:=  NewLedState((min AND 4));
    MinLED8.LedStatus:=  NewLedState((min AND 8));
    MinLED16.LedStatus:= NewLedState((min AND 16));
    MinLED32.LedStatus:= NewLedState((min AND 32));
    lstMin:= min;
  end;
  SecLED1.LedStatus:=  NewLedState((sec AND 1));
  SecLED2.LedStatus:=  NewLedState((sec AND 2));
  SecLED4.LedStatus:=  NewLedState((sec AND 4));
  SecLED8.LedStatus:=  NewLedState((sec AND 8));
  SecLED16.LedStatus:= NewLedState((sec AND 16));
  SecLED32.LedStatus:= NewLedState((sec AND 32));
end;

procedure TForm1.Silver1Click(Sender: TObject);
begin
  ReColor(lcSilver);
end;

procedure TForm1.Red1Click(Sender: TObject);
begin
  ReColor(lcRed);
end;

procedure TForm1.Lime1Click(Sender: TObject);
begin
  ReColor(lcLime);
end;

procedure TForm1.Yellow1Click(Sender: TObject);
begin
  ReColor(lcYellow);
end;

procedure TForm1.Blue1Click(Sender: TObject);
begin
  ReColor(lcBlue);
end;

procedure TForm1.Fuchsia1Click(Sender: TObject);
begin
  ReColor(lcFuchsia);
end;

procedure TForm1.Aqua1Click(Sender: TObject);
begin
  ReColor(lcAqua);
end;

procedure TForm1.White1Click(Sender: TObject);
begin
  ReColor(lcWhite);
end;

procedure TForm1.FormShow(Sender: TObject);
begin
  ResolutionFix1.FixResolution(Sender);
  Form1.Width:= SecLED1.Left + SecLED1.Width + 2;
  Form1.Height:= SecLED1.Top + SecLED1.Height + TitleBarSize + 2;
end;

end.
