unit MakeDBF;

{
                       MakeDBF unit for Delphi
                             Version 1.01
             Copyright 1995 Absolute Computer Consulting,
                         All Rights Reserved.

  Description:
  ------------

   This unit is designed to create a .DBF file header.  It will
   create a table with any number of fields.  It is designed to
   create a FoxPro / dBase table without memos.

  Written by:
  -----------

  Mark Klaamas, Absolute Computer Consulting
  13 Apollo Court
  Halifax, Nova Scotia
  Canada

  Usage:
  ------

    var
      DBF : tCreateDBF;
    begin
      DBF := tCreateDBF.Create;

      try
        with DBF do
        begin

          (* Add fields to the database *)
          AddField('test1', 'C', 10, 0);
          AddField('test2', 'C', 20, 0);
          AddField('test3', 'N', 4, 0);
          AddField('test4', 'N', 6, 2);
          AddField('test5', 'D', 8, 0);
          AddField('test6', 'L', 1, 0);

          (* set the table name *)
          FileName := 'test.dbf';

          (* create the table file. *)
          if not WriteTable then
            MessageDlg('Error occured', mtError, [mbOk], 0);

        end;
      finally
        DBF.Free;
      end;

   end;


  Disclaimer and License:
  -----------------------

    This unit and the associated files can be freely used and
    distributed in commercial and private environments, provided this
    notice is not modified in any way without my expressed written
    consent.  The MakeDBF unit is released as is.  No warranty is
    implied and we are not responsible for any problems that might
    arise from the use of this unit.  You use this unit
    entirely at your own risk.

    Bugs (a.k.a. undocumented features)

    If you find any, please let us know, and we will attempt to fix
    them.  Note that bugs often depend on glitches in the system.
    Sometimes it helps to re-boot and try again...

    Feel free to contact us if you have any questions, comments or
    suggestions at klaamasm@tuns.ca

  Revision History:


  20 July 1995:  (v1.01) Forced field names to uppercase.  This is to be
                 consistant with FoxPro naming for fields.  Forced a date
                 field to have a length of 8.  Only Numeric fields can
                 be assigned decimal places.

                 Fixed WriteTable routine to correctly detect if the file
                 was created correctly or not.

  11 July 1995:  (v1.00) First release.  Memos will not be supported
                 unless there are requests :)

                 Supports following field types: Character, Numeric,
                 Logical, Date.  More will be added on request. }

interface

{ --------------------------------------------------------------------------- }

uses
  SysUtils, Classes;

{ --------------------------------------------------------------------------- }

TYPE

  tCreateDBF = class(tObject)
  private
    fFileName : STRING;
    dbfType   : BYTE;                  { database type                        }
    Update    : array[0..2] of byte;   { date last updated.                   }
    NumRecs   : Longint;               { number of records. set to 0 :)       }
    Rec1Pos   : WORD;                  { position of record 1                 }
    RecLen    : WORD;                  { length of record including delete flg}
    resv      : array[0..19] of char;  { reserved for furture use             }
    SubRecs   : tList;                 { list of sub records (fields/columns) }
  public
    constructor Create;
    destructor  Free;
    procedure   AddField(Name : string; fldtype : char; fldsize : byte; dec : byte);
    procedure   UpdateTable;
    function    WriteTable : boolean;
    procedure   ClearAll;

    property    FileName : string read fFileName write fFileName;
  end;

  {
    Define a sub-record (aka database field/column) class.  Make it a
    class instead of a record so we can add it to a tList without
    doing pointer magic.  It will also make it easier to dispose of
    memory since a tObject "knows" how to dispose of itself.
  }

  tDBFSubRec = class(tObject)
  public
    Name      : array[0..10] of char;  { Field name                           }
    FldType   : char;                  { Field type                           }
    FldDisp   : longint;               { Field displacement in record.        }
    FldLen    : byte;                  { Size of field.                       }
    FldDec    : byte;                  { Number of decimal places.            }
    resv      : array[0..13] of char;  { reserved.                            }
  END;

{ --------------------------------------------------------------------------- }

implementation

{ --------------------------------------------------------------------------- }

const
  ValidTypes = 'CNLD';                 { Character, Numeric, Logical, Date    }

{ --------------------------------------------------------------------------- }

constructor tCreateDBF.Create;
const
  D : pChar = #95#12#28;               { fake an update date.                 }
begin
  fFileName := '';
  dbfType   := 3;                      { FoxPro / dBase 4 table, no memos.    }
  StrCopy(pChar(@update), D);          { set false update date ;)             }
  NumRecs   := 0;                      { set the number of records to 0!      }
  FillChar(resv, sizeof(resv), #0);
  SubRecs   := tList.Create;           { create a new instance of a tList.    }
end; { constructor tCreateDBF.Create }

{ --------------------------------------------------------------------------- }

destructor tCreateDBF.Free;
begin

  SubRecs.Free;                        { Delete the SubRecs and Free Memory.  }

end; { destructor tCreateDBF.Free }

{ --------------------------------------------------------------------------- }

procedure tCreateDBF.AddField(Name : string; fldtype : char; fldsize : byte; dec : byte);
var
  TmpSubRec : tDBFSubRec;
begin

                                       { validate the field type.             }
  fldtype := UpCase(fldtype);
  if pos(fldtype, ValidTypes) = 0 then exit;

  TmpSubRec := tDBFSubRec.Create;      { create a new field instance.         }

  Name := UpperCase(Name);             { convert field name to UPPERCASE      }
  Move(Name[1], TmpSubRec.Name, Length(Name));
  TmpSubRec.FldType := fldtype;

  with TmpSubRec do
  begin
    FldDisp := 0;                      { ******  updated later.               }

    if fldtype = 'D' then
      FldLen := 8                      { date fields are 8 characters wide.   }
    else
      FldLen  := fldsize;

    if fldtype = 'N' then              { only numeric has decimal places.     }
      FldDec  := dec
    else
      FldDec  := 0;

    fillchar(resv, sizeof(resv), #0);
  end;

  SubRecs.Add(TmpSubRec);              { add new field to the list.           }

end; { procedure tCreateDBF.AddField }

{ --------------------------------------------------------------------------- }

procedure tCreateDBF.UpdateTable;
var
  x       : integer;
  NumFld  : word;
  RecSize : word;
  TmpFld  : tDBFSubRec;
begin

  RecSize := 0;                        { account for the delete flag in size  }
  NumFld := SubRecs.Count;

  Rec1Pos := 32 + (32*NumFld) + 1;     { record 1 start position              }

  for x := 0 to (NumFld - 1) do
  begin

    TmpFld := SubRecs.Items[x];

    with TmpFld do
    begin
      FldDisp := RecSize + 1;          { update field displacement in record. }
      RecSize := RecSize + FldLen;
    end;

  end; { for }

  RecSize := RecSize + 1;              { account for the delete flag.         }
  RecLen  := RecSize;                  { set the record length.               }

end; { procedure tCreateDBF.UpdateTable }

{ --------------------------------------------------------------------------- }

function tCreateDBF.WriteTable : boolean;
var
  OutFile  : file;
  x        : word;
  TmpFld   : tDBFSubRec;
  HeadEnd  : char;
  fSize    : longint;
begin
  UpdateTable;
  WriteTable := TRUE;
  AssignFile(OutFile, fFileName);

  fSize := 0;                          { define the initial file size.        }
  {$I-}
  if IOResult <> 0 then ;              { clear IOResult                       }
  {

    We need to clear the IOresult because if there was an IOResult that was
    set before we called this method it would cause this method to fail even
    if it worked!

    A little know fact that I didn't know about: according to the manual

    "If an I/O error occurs and I/O-checking is off ( (*$I-*) ), all subsequent
    I/O operations are ignored until a call is made to IOResult. Calling
    IOResult clears the internal error flag."

    However, from testing, if a previous procedure sets IOResult, this procedure
    will run, and work, but still report the IOResult from the previous
    procedure.

  }
  ReWrite(OutFile, 1);

  BlockWrite(OutFile, dbfType, sizeof(dbfType));
  BlockWrite(OutFile, Update,  sizeof(Update));
  BlockWrite(OutFile, NumRecs, sizeof(NumRecs));
  BlockWrite(OutFile, Rec1Pos, sizeof(Rec1Pos));
  BlockWrite(OutFile, RecLen,  sizeof(RecLen));
  BlockWrite(OutFile, resv,    sizeof(resv));

  for x := 0 to (SubRecs.Count - 1) do
  begin
    TmpFld := SubRecs.Items[x];
    BlockWrite(OutFile, TmpFld.Name,    sizeof(TmpFld.Name));
    BlockWrite(OutFile, TmpFld.FldType, sizeof(TmpFld.FldType));
    BlockWrite(OutFile, TmpFld.FldDisp, sizeof(TmpFld.FldDisp));
    BlockWrite(OutFile, TmpFld.FldLen,  sizeof(TmpFld.FldLen));
    BlockWrite(OutFile, TmpFld.FldDec,  sizeof(TmpFld.FldDec));
    BlockWrite(OutFile, TmpFld.resv,    sizeof(TmpFld.resv));
  end; { for }

  HeadEnd := #13;
  BlockWrite(OutFile, HeadEnd, 1);
  CloseFile(OutFile);

  if IOResult <> 0 then WriteTable := FALSE;
  {I+}
end; { function tCreateDBF.WriteTable : boolean }

{ --------------------------------------------------------------------------- }

procedure tCreateDBF.ClearAll;
begin

  SubRecs.Free;                        { Delete the SubRecs and Free Memory.  }
  SubRecs := tList.Create;

end; { procedure tCreateDBF.ClearAll }

{ --------------------------------------------------------------------------- }

end.
