unit Pntcell;

(*************************************************************
Implements a cell-based game world
*************************************************************)

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, ExtCtrls, DynArray;

type

  TCellRange = 1..100;

  TImageCell = class( TObject )
  private
  protected
  public
     sName: string;
     bm: TBitmap;
     constructor Create;
     destructor Destroy; override;
  end;

  TCellPaintBox = class( TPaintBox )
  private
     FCellsWidth: TCellRange;
     FCellsHeight: TCellRange;
     FMouseX, FMouseY: TCellRange;
     FClickX, FClickY: TCellRange;
     nAllocated: word;
     xMultX, xMultY: real;
     matrix: TDynamicArray;
     procedure AllocateMatrix;
     function LookupCell( const sName: string ): integer;
  protected
     lstCells: TList;
     procedure Click; override;
     procedure MouseMove( Shift: TShiftState; X, Y: Integer ); override;
     procedure SetCellWidth( const n: TCellRange );
     procedure SetCellHeight( const n: TCellRange );
     procedure SetCellMatrix( X, Y: TCellRange; const sName: string );
     function GetCellMatrix( X, Y: TCellRange ): string;
  public
     constructor Create( AOwner: TComponent ); override;
     destructor Destroy; override;
     procedure AddImage( const sName: string; bm: TBitmap );
     procedure ClearImages;
     procedure Paint; override;
     procedure PaintStandardCell( const x, y: integer );
     procedure PaintCell( const x, y: integer; bm: TBitmap; const bStretch: boolean );
     function GetPixelCoords( pt: TPoint ): TPoint;

     procedure ScrollLeft;
     procedure ScrollUp;
     procedure ScrollRight;
     procedure ScrollDown;

     property CellMatrix[X, Y: TCellRange]: string read GetCellMatrix write SetCellMatrix;
     property MouseX: TCellRange read FMouseX;
     property MouseY: TCellRange read FMouseY;
     property ClickX: TCellRange read FClickX;
     property ClickY: TCellRange read FClickY;
  published
     property CellsWidth: TCellRange read FCellsWidth write SetCellWidth default 1;
     property CellsHeight: TCellRange read FCellsHeight write SetCellHeight default 1;
  end;

procedure Register;

implementation

var
  xScreenX, xScreenY: single;

constructor TCellPaintBox.Create( AOwner: TComponent );
begin
  inherited Create( AOwner );
  matrix := TDynamicArray.Create2D( 1, 1 );
  nAllocated := 1;
  CellsWidth := 1;
  CellsHeight := 1;
  lstCells := TList.Create;
end;

destructor TCellPaintBox.Destroy;
begin
  ClearImages;
  lstCells.Free;
  inherited Destroy;
end;

(*************************************************************
Plumbing
*************************************************************)
procedure TCellPaintBox.SetCellWidth( const n: TCellRange );
begin
  FCellsWidth := n;
  AllocateMatrix;
end;

procedure TCellPaintBox.SetCellHeight( const n: TCellRange );
begin
  FCellsHeight := n;
  AllocateMatrix;
end;

procedure TCellPaintBox.AllocateMatrix;
begin
  matrix.Dim2D( FCellsWidth, FCellsHeight );
  nAllocated := FCellsWidth * FCellsHeight;
end;

function TCellPaintBox.LookupCell( const sName: string ): integer;
var
  i: integer;
begin
  Result := -1;
  for i := 0 to lstCells.Count - 1 do
     if TImageCell( lstCells.Items[i] ).sName = sName then
        begin
           Result := i;
           Break;
        end;
end;

procedure TCellPaintBox.SetCellMatrix( X, Y: TCellRange; const sName: string );
begin
  matrix.Value2D[X, Y] := LookupCell( sName );
end;

function TCellPaintBox.GetCellMatrix( X, Y: TCellRange ): string;
var
  nIndex: integer;
begin
  nIndex := matrix.Value2D[X, Y];
  Result := TImageCell( lstCells.Items[nIndex] ).sName;
end;

(*************************************************************
The paint handler draws the cells, then returns control to
user-defined handler.
*************************************************************)
procedure TCellPaintBox.Paint;
var
  X, Y: integer;
begin
  if ( FCellsWidth > 0 ) and ( FCellsHeight > 0 ) then
     begin
        xMultX := Width / FCellsWidth;
        xMultY := Height / FCellsHeight;
        for X := 1 to FCellsWidth do
           for Y := 1 to FCellsHeight do
              PaintStandardCell( x, y );
     end;
  inherited Paint;
end;

procedure TCellPaintBox.PaintStandardCell( const x, y: integer );
var
  nIndex: integer;
begin
  nIndex := matrix.Value2D[X, Y];
  if ( nIndex >= 0 ) and ( nIndex < lstCells.Count ) then
     PaintCell( x, y, TImageCell( lstCells.Items[matrix.Value2D[X, Y]] ).bm, TRUE );
end;

(*********************************************
Paint a single cell.
*********************************************)
procedure TCellPaintBox.PaintCell( const x, y: integer; bm: TBitmap; const bStretch: boolean );
var
  rectCell, rectDummy, rectClip: TRect;
begin
  if bStretch then
     begin
        rectCell.Left := Round( ( X - 1 ) * xMultX );
        rectCell.Top := Round( ( Y - 1 ) * xMultY );
        rectCell.Right := Round( ( X - 1 ) * xMultX + xMultX );
        rectCell.Bottom := Round( ( Y - 1 ) * xMultY + xMultY );
     end
  else
     begin

{ Adjust to screen dimensions }
        rectCell.Left := Round( ( X - 1 ) * xMultX + ( xMultX / 2 ) - ( bm.Width / 2 * xScreenX ) );
        rectCell.Right := Round( rectCell.Left + ( bm.Width * xScreenX ) );
        rectCell.Top := Round( ( Y - 1 ) * xMultY + ( xMultY / 2 ) - ( bm.Height / 2 * xScreenY ) );
        rectCell.Bottom := Round( rectCell.Top + ( bm.Height * xScreenY ) );
     end;
  if IntersectRect( rectDummy, rectCell, Canvas.ClipRect ) > 0 then
     Canvas.StretchDraw( rectCell, bm );
end;

(*************************************************************
Track mouse position as it's moved.
*************************************************************)
procedure TCellPaintBox.MouseMove( Shift: TShiftState; X, Y: Integer );
begin
  if ( xMultX > 0 ) and ( xMultY > 0 ) then
     begin
        FMouseX := Trunc( X / xMultX ) + 1;
        FMouseY := Trunc( Y / xMultY ) + 1;
     end;
  inherited MouseMove( Shift, X, Y );
end;

procedure TCellPaintBox.Click;
begin
  FClickX := FMouseX;
  FClickY := FMouseY;
  inherited Click;
end;

(*************************************************************
Add an image to the cell list.
*************************************************************)
procedure TCellPaintBox.AddImage( const sName: string; bm: TBitmap );
var
  cell: TImageCell;
begin
  if LookupCell( sName ) = -1 then
     begin
        cell := TImageCell.Create;
        cell.sName := sName;
        cell.bm.Assign( bm );
        lstCells.Add( cell );
     end;
end;

(*************************************************************
Clear images in prepearation for relaoding.
*************************************************************)
procedure TCellPaintBox.ClearImages;
var
  i: integer;
begin
  for i := 0 to lstCells.Count - 1 do
     TImageCell( lstCells.Items[i] ).Free;
  lstCells.Clear;
end;

(*********************************************
Get pixels coords from logical coords.
*********************************************)
function TCellPaintBox.GetPixelCoords( pt: TPoint ): TPoint;
begin
  Result.X := Round( xMultX * pt.X );
  Result.Y := Round( xMultY * pt.Y );
end;

(*********************************************
Scrolling Routines
*********************************************)
procedure TCellPaintBox.ScrollLeft;
var
  x, y: integer;
begin
  for x := 1 to CellsWidth - 1 do
     for y := 1 to CellsHeight do
        matrix.Value2D[x, y] := matrix.Value2D[x + 1, y];
end;

procedure TCellPaintBox.ScrollUp;
var
  x, y: integer;
begin
  for y := 1 to CellsHeight - 1 do
     for x := 1 to CellsWidth do
        matrix.Value2D[x, y] := matrix.Value2D[x, y + 1];
end;

procedure TCellPaintBox.ScrollRight;
var
  x, y: integer;
begin
  for x := CellsWidth downto 2 do
     for y := 1 to CellsHeight do
        matrix.Value2D[x, y] := matrix.Value2D[x - 1, y];
end;

procedure TCellPaintBox.ScrollDown;
var
  x, y: integer;
begin
  for y := CellsHeight downto 2 do
     for x := 1 to CellsWidth do
        matrix.Value2D[x, y] := matrix.Value2D[x, y - 1];
end;

(*************************************************************
TImageCell
*************************************************************)
constructor TImageCell.Create;
begin
  bm := TBitmap.Create;
end;

destructor TImageCell.Destroy;
begin
  bm.Free;
  inherited Destroy;
end;

procedure Register;
begin
  RegisterComponents( 'SilCmd', [TCellPaintBox] );
end;

initialization

  xScreenX := Screen.Width / 640.0;
  xScreenY := Screen.Height / 480.0;

end.
