{==============================================================================}
{= RzStrEdt - String List Property Editor and Dialog                          =}
{=                                                                            =}
{= This unit file contains a replacement string list property editor and its  =}
{= associated dialog box.  After installing this unit into the Delphi IDE,    =}
{= this editor will be used to edit all string list properties.  To install   =}
{= this editor, follow the same rules used to install a custom component.     =}
{=                                                                            =}
{= Copyright  1995 by Raize Software Solutions, Inc.                         =}
{==============================================================================}

unit RzStrEdt;

interface

  uses
    SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
    Forms, Dialogs, ExtCtrls, RzPanel, StdCtrls, Buttons, IniFiles, DsgnIntf,
    Menus;

  type
    TRzStrListEditorDlg = class( TForm )
      BtnOk         : TBitBtn;
      BtnCancel     : TBitBtn;
      BtnHelp       : TBitBtn;
      PnlToolbar    : TPanel;
      BtnTab        : TSpeedButton;
      BtnOpen       : TSpeedButton;
      BtnSave       : TSpeedButton;
      BtnCut        : TSpeedButton;
      BtnCopy       : TSpeedButton;
      BtnPaste      : TSpeedButton;
      BtnUndo       : TSpeedButton;
      BtnFont       : TSpeedButton;
      PnlStringList : TPanel;
      EdtStrings    : TMemo;
      LblCount      : TLabel;
      ChkDefault    : TCheckBox;
      Label1        : TLabel;
      LblRow        : TLabel;
      Label3        : TLabel;
      LblCol        : TLabel;
      MnuEdit       : TPopupMenu;
      MnuUndo       : TMenuItem;
      MnuCut        : TMenuItem;
      MnuCopy       : TMenuItem;
      MnuPaste      : TMenuItem;
      MnuSep1       : TMenuItem;
      MnuOpen       : TMenuItem;
      MnuSave       : TMenuItem;
      MnuSep2       : TMenuItem;
      MnuTab        : TMenuItem;
      DlgOpen       : TOpenDialog;
      DlgSave       : TSaveDialog;
      DlgFont       : TFontDialog;
      LblCopyright: TLabel;
      BtnCopyright: TButton;
      procedure FormCreate( Sender : TObject );
      procedure BtnOkClick( Sender : TObject );
      procedure BtnFontClick( Sender : TObject );
      procedure BtnUndoClick( Sender : TObject );
      procedure BtnCutClick( Sender : TObject );
      procedure BtnCopyClick( Sender : TObject );
      procedure BtnPasteClick( Sender : TObject );
      procedure BtnOpenClick( Sender : TObject );
      procedure BtnSaveClick( Sender : TObject );
      procedure BtnTabClick( Sender : TObject );
      procedure EdtStringsChange( Sender : TObject );
      procedure EdtStringsKeyDown( Sender : TObject; var Key : Word;
                                   Shift : TShiftState );
      procedure EdtStringsKeyUp( Sender : TObject; var Key : Word;
                                 Shift : TShiftState );
      procedure EdtStringsClick( Sender : TObject);
      procedure BtnCopyrightClick(Sender: TObject);
    private
      SingleLine    : string[ 15 ];
      MultipleLines : string[ 15 ];
      DelphiIni     : TIniFile;
      procedure UpdateClipboardStatus;
    end;

  type
    TRzStringListProperty = class( TPropertyEditor )
      function GetAttributes : TPropertyAttributes; override;
      function GetValue: string; override;
      procedure Edit; override;
    end;

  procedure Register;

implementation

{$R *.DFM}

  uses
    LibConst, LibHelp, ClipBrd;

  const                               { Constants used to determine font style }
    fsBoldMask      = 8;
    fsItalicMask    = 4;
    fsUnderlineMask = 2;
    fsStrikeOutMask = 1;

  procedure TRzStrListEditorDlg.FormCreate(Sender: TObject);
  var
    StyleBits  : Byte;
  begin
    HelpContext := hcDStringListEditor;

    DlgOpen.HelpContext := hcDStringListLoad;
    DlgSave.HelpContext := hcDStringListSave;
    SingleLine := LoadStr( srLine );
    MultipleLines := LoadStr( srLines );

    DelphiIni := TIniFile.Create( 'DELPHI.INI' );
    with EdtStrings.Font do
    begin
      Name := DelphiIni.ReadString( 'Raize.StrListEditor', 'FontName',
                                    'MS Sans Serif' );
      Size := DelphiIni.ReadInteger( 'Raize.StrListEditor', 'FontSize', 8 );
      Color := DelphiIni.ReadInteger( 'Raize.StrListEditor', 'FontColor', clBlack );
      StyleBits := DelphiIni.ReadInteger( 'Raize.StrListEditor', 'FontStyle', fsBoldMask );
      Style := [];
      if StyleBits and fsBoldMask = fsBoldMask then
        Style := Style + [ fsBold ];
      if StyleBits and fsItalicMask = fsItalicMask then
        Style := Style + [ fsItalic ];
      if StyleBits and fsUnderlineMask = fsUnderlineMask then
        Style := Style + [ fsUnderline ];
      if StyleBits and fsStrikeOutMask = fsStrikeOutMask then
        Style := Style + [ fsStrikeOut ];
    end;
    UpdateClipboardStatus;
  end; {= TRzStrListEditorDlg.FormCreate =}


  procedure TRzStrListEditorDlg.BtnOkClick(Sender: TObject);
  var
    StyleBits : Byte;
  begin
    if ChkDefault.Checked then
    begin
      with EdtStrings.Font do
      begin
        DelphiIni.WriteString( 'Raize.StrListEditor', 'FontName', Name );
        DelphiIni.WriteInteger( 'Raize.StrListEditor', 'FontSize', Size );
        DelphiIni.WriteInteger( 'Raize.StrListEditor', 'FontColor', Color );

        if fsBold in Style then
          StyleBits := fsBoldMask;
        if fsItalic in Style then
          StyleBits := StyleBits + fsItalicMask;
        if fsUnderline in Style then
          StyleBits := StyleBits + fsUnderlineMask;
        if fsStrikeOut in Style then
          StyleBits := StyleBits + fsStrikeOutMask;
        DelphiIni.WriteInteger( 'Raize.StrListEditor', 'FontStyle', StyleBits );
      end;
    end;
    DelphiIni.Free;
  end; {= TRzStrListEditorDlg.BtnOkClick =}


  procedure TRzStrListEditorDlg.BtnFontClick(Sender: TObject);
  begin
    DlgFont.Font := EdtStrings.Font;
    if DlgFont.Execute then
    begin
      EdtStrings.Font := DlgFont.Font;
    end;
  end; {= TRzStrListEditorDlg.BtnFontClick =}


  procedure TRzStrListEditorDlg.BtnUndoClick(Sender: TObject);
  begin
    EdtStrings.Perform( wm_Undo, 0, 0 );
  end; {= TRzStrListEditorDlg.BtnUndoClick =}


  procedure TRzStrListEditorDlg.BtnCutClick(Sender: TObject);
  begin
    EdtStrings.CutToClipboard;
    UpdateClipboardStatus;
  end; {= TRzStrListEditorDlg.BtnCutClick =}


  procedure TRzStrListEditorDlg.BtnCopyClick(Sender: TObject);
  begin
    EdtStrings.CopyToClipboard;
    UpdateClipboardStatus;
  end; {= TRzStrListEditorDlg.BtnCopyClick =}


  procedure TRzStrListEditorDlg.BtnPasteClick(Sender: TObject);
  begin
    EdtStrings.PasteFromClipboard;
  end; {= TRzStrListEditorDlg.BtnPasteClick =}


  procedure TRzStrListEditorDlg.BtnOpenClick(Sender: TObject);
  begin
    if DlgOpen.Execute then
      EdtStrings.Lines.LoadFromFile( DlgOpen.FileName );
  end; {= TRzStrListEditorDlg.BtnOpenClick =}


  procedure TRzStrListEditorDlg.BtnSaveClick(Sender: TObject);
  begin
    if DlgSave.Execute then
      EdtStrings.Lines.SaveToFile( DlgSave.FileName );
  end; {= TRzStrListEditorDlg.BtnSaveClick =}


  procedure TRzStrListEditorDlg.BtnTabClick(Sender: TObject);
  begin
    EdtStrings.Perform( wm_Char, vk_tab, 0 );
  end; {= TRzStrListEditorDlg.BtnTabClick =}


  procedure TRzStrListEditorDlg.EdtStringsChange(Sender: TObject);
  var
    Count    : Integer;
    LineText : PString;
  begin
    Count := EdtStrings.Lines.Count;
    if Count = 1 then
      LineText := @SingleLine
    else
      LineText := @MultipleLines;
    LblCount.Caption := Format( '%d %s', [ Count, LineText^ ] );
    EdtStringsClick( Sender );
  end; {= TRzStrListEditorDlg.EdtStringsChange =}


  procedure TRzStrListEditorDlg.EdtStringsKeyDown(Sender: TObject;
    var Key: Word; Shift: TShiftState);
  begin
    EdtStringsClick( Sender );
    if Key = vk_Escape then
      BtnCancel.Click;
  end; {= TRzStrListEditorDlg.EdtStringsKeyDown =}


  procedure TRzStrListEditorDlg.EdtStringsKeyUp(Sender: TObject;
    var Key: Word; Shift: TShiftState);
  begin
    EdtStringsClick( Sender );
  end; {= TRzStrListEditorDlg.EdtStringsKeyUp =}


  procedure TRzStrListEditorDlg.EdtStringsClick(Sender: TObject);
  var
    X, Y : LongInt;
  begin
    Y := EdtStrings.Perform( em_LineFromChar, EdtStrings.SelStart, 0 );
    X := EdtStrings.SelStart - EdtStrings.Perform( em_LineIndex, Y, 0 );
    LblRow.Caption := IntToStr( Y + 1 );
    LblCol.Caption := IntToStr( X + 1 );
    UpdateClipboardStatus;
  end; {= TRzStrListEditorDlg.EdtStringsClick =}


  procedure TRzStrListEditorDlg.UpdateClipboardStatus;
  var
    HasText      : Boolean;
    HasSelection : Boolean;
  begin
    HasSelection := EdtStrings.SelLength <> 0;
    BtnCut.Enabled := HasSelection;
    MnuCut.Enabled := HasSelection;
    BtnCopy.Enabled := HasSelection;
    MnuCopy.Enabled := HasSelection;
    HasText := Clipboard.HasFormat( cf_Text );
    BtnPaste.Enabled := HasText;
    MnuPaste.Enabled := HasText;
  end; {= TRzStrListEditorDlg.UpdateClipboardStatus =}


  procedure TRzStrListEditorDlg.BtnCopyrightClick(Sender: TObject);
  begin
    LblCopyright.Visible := True;
  end; {= TRzStrListEditorDlg.BtnCopyrightClick =}



  {===================================}
  {== TRzStringListProperty Methods ==}
  {===================================}

  function TRzStringListProperty.GetAttributes: TPropertyAttributes;
  begin
    Result := inherited GetAttributes + [paDialog] - [paSubProperties];
  end; {= TRzStringListProperty.GetAttributes =}


  function TRzStringListProperty.GetValue : string;
  begin
    FmtStr( Result, '(%s)', [ GetPropType^.Name ] );
  end; {= TRzStringListProperty.GetValue =}


  procedure TRzStringListProperty.Edit;
  var
    Dlg : TRzStrListEditorDlg;
  begin
    with TRzStrListEditorDlg.Create( Application ) do
    begin
      try
        EdtStrings.Lines := TStringList( GetOrdValue );
        EdtStringsClick( nil );
        if ( ShowModal = mrOK ) and ( EdtStrings.Modified ) then
          SetOrdValue( Longint( EdtStrings.Lines ) );
      finally
        Free;
      end;
    end; { with }
  end; {= TRzStringListProperty.Edit =}


  {========================}
  {== Register Procedure ==}
  {========================}

  procedure Register;
  begin
    RegisterPropertyEditor( TypeInfo( TStrings ), nil, '',
                            TRzStringListProperty );
  end;

end.
