/////////////////////////////////////////////////////////////////
//                                                             //
// CLSPRITE.CPP: Member functions of the ClSprite class.       //
//                                                             //
/////////////////////////////////////////////////////////////////

#define STRICT
#include <windows.h>
#include "clsprite.h"
    
/////////////////////////////////////////////////////////////////
// Constructors, destructors, and overloaded operators:        //
/////////////////////////////////////////////////////////////////

// default constructor:    
ClSprite::ClSprite ()
   {
   mHBMImage = 0;
   mHBMMask = 0;
   mHBMSave = 0;
   mX = mY = 0;
   mWidth = mHeight = 0;
   }                

// copy constructor:   
ClSprite::ClSprite (const ClSprite &OtherSprite)
   {
   mHBMImage = OtherSprite.mHBMImage;
   mHBMMask = OtherSprite.mHBMMask;
   mHeight = OtherSprite.mHeight;
   mWidth = OtherSprite.mWidth;
   mX = OtherSprite.mX;
   mY = OtherSprite.mY;
                    
   if (OtherSprite.mHBMSave == 0)
      // old object has no "save" bitmap
      mHBMSave = 0;
   else
      {              
      // old object has a "save" bitmap; create a new "save"
      // bitmap for new object:
      HDC HDCScreen = GetDC (NULL);
      mHBMSave = CreateCompatibleBitmap
         (HDCScreen,
         OtherSprite.mWidth,
         OtherSprite.mHeight);
      ReleaseDC (NULL, HDCScreen);         
      }            
   }   
                                           
// destructor:                                           
ClSprite::~ClSprite ()
   {
   if (mHBMSave != 0)
      DeleteObject (mHBMSave);
   }
              
// overload = operator:              
ClSprite & ClSprite::operator= (const ClSprite &OtherSprite)
   {
   if (&OtherSprite == this)  // prevent attempt to assign
      return *this;           // an object to itself
      
   mHBMImage = OtherSprite.mHBMImage;
   mHBMMask = OtherSprite.mHBMMask;
   mHeight = OtherSprite.mHeight;
   mWidth = OtherSprite.mWidth;
   mX = OtherSprite.mX;
   mY = OtherSprite.mY;
                                                    
   // if target object already has a "save" bitmap, delete it:
   if (mHBMSave != 0)
      {
      DeleteObject (mHBMSave);
      mHBMSave = 0;
      }   
      
   if (OtherSprite.mHBMSave != 0)
      {
      // source object has a "save" bitmap; create a new "save"
      // bitmap for target object:
      HDC HDCScreen = GetDC (NULL);
      mHBMSave = CreateCompatibleBitmap
         (HDCScreen,
         OtherSprite.mWidth,
         OtherSprite.mHeight);
      ReleaseDC (NULL, HDCScreen);         
      }
      
   return *this;                  
   }

/////////////////////////////////////////////////////////////////
// Other member functions:                                     //
/////////////////////////////////////////////////////////////////
   
void ClSprite::GetCoord (RECT *Rect)
// returns the current sprite coordinates
   {
   Rect->left = mX;
   Rect->top = mY;
   Rect->right = mX + mWidth;
   Rect->bottom = mY + mHeight;
   }

BOOL ClSprite::Hide (HDC HDc)
// makes the sprite invisible
   {
   HDC HMemDC;
   int Result;

   if (mHBMSave == 0)  // return error if object not initialized
      return (FALSE);

   // erase the sprite by restoring the saved screen
   // graphics at the current sprite position:                    
   HMemDC = CreateCompatibleDC (HDc);
   if (HMemDC == 0)
      return (FALSE);
   SelectObject (HMemDC, mHBMSave);

   Result = BitBlt
      (HDc,
      mX,
      mY,
      mWidth,
      mHeight,
      HMemDC,
      0,
      0,
      SRCCOPY);

   DeleteDC (HMemDC);

   if (Result == 0)
      return (FALSE);

   return (TRUE);
   }

BOOL ClSprite::Initialize (HBITMAP HBMMask, HBITMAP HBMImage)
// assigns the bitmaps containing the sprite image and
// prepares the sprite object for a new animation sequence; may
// be called more than once for a given object
   {
   HDC HDCScreen;
   HBITMAP HBMSave;
   BITMAP ImageBM;
   BITMAP MaskBM;
   int Result;

   // get and compare the sizes of the mask and image bitmaps:
   
   Result = GetObject (HBMMask, sizeof (BITMAP), &MaskBM);
   if (Result == 0)
      return (FALSE);

   Result = GetObject (HBMImage, sizeof (BITMAP), &ImageBM);
   if (Result == 0)
      return (FALSE);
                     
   // return an error code if sizes are unequal:                     
   if (MaskBM.bmWidth != ImageBM.bmWidth ||
       MaskBM.bmHeight != ImageBM.bmHeight)
      return (FALSE);
   
   // create the "save" bitmap for saving and restoring screen
   // graphics:
   HDCScreen = GetDC (NULL);
   HBMSave = CreateCompatibleBitmap
      (HDCScreen,
      MaskBM.bmWidth,    // same size as mask and image bitmaps
      MaskBM.bmHeight);
   ReleaseDC (NULL, HDCScreen);
   if (HBMSave == NULL)
      return (FALSE);
      
   // delete prior "save" bitmap, if any:   
   if (mHBMSave != 0)
      DeleteObject (mHBMSave);
                                         
   // function is successful; now assign values to data members:                                         
   mHBMSave = HBMSave;
   mHBMMask = HBMMask;
   mHBMImage = HBMImage;
   mWidth = MaskBM.bmWidth;
   mHeight = MaskBM.bmHeight;
   mX = mY = 0;
   
   return (TRUE);
   }

HBITMAP ClSprite::MakeMask (HBITMAP HBMImage)
// returns a handle for a mask bitmap that cooresponds to the
// image bitmap whose handle is assigned to the HBMImage 
// parameter
   {
   HBITMAP HBMMask;
   HDC HMemDCImage, HMemDCMask;
   BITMAP ImageBM;
   int Result;
   int Row, Col;
   
   // get size of image bitmap:
   Result = GetObject (HBMImage, sizeof (BITMAP), &ImageBM);
   if (Result == 0)
      return 0;
                               
   // create memory device context for image bitmap:                               
   HMemDCImage = CreateCompatibleDC (NULL);
   if (HMemDCImage == 0)
      return 0;
   SelectObject (HMemDCImage, HBMImage);

   // create mask bitmap:   
   HBMMask = CreateCompatibleBitmap
      (HMemDCImage,
      ImageBM.bmWidth,  // same size as image bitmap
      ImageBM.bmHeight);
   if (HBMMask == 0)
      {
      DeleteDC (HMemDCImage);
      return 0;
      }    

   // create memory device context for mask bitmap:             
   HMemDCMask = CreateCompatibleDC (NULL);
   if (HMemDCMask == 0)
      {
      DeleteDC (HMemDCImage);
      DeleteObject (HBMMask);
      return 0;
      }
   SelectObject (HMemDCMask, HBMMask);
   
   // assign each pixel in mask bitmap according to the value
   // of the corresponding pixel in image bitmap:
   for (Row = 0; Row < ImageBM.bmHeight; ++Row)
      for (Col = 0; Col < ImageBM.bmWidth; ++Col)
         if (GetPixel (HMemDCImage, Col, Row) == RGB (0,0,0))
            SetPixel (HMemDCMask, Col, Row, RGB (255,255,255));
         else
            SetPixel (HMemDCMask, Col, Row, RGB (0,0,0));             

   DeleteDC (HMemDCImage);
   DeleteDC (HMemDCMask);                                       
   
   return (HBMMask);
   }
   
BOOL ClSprite::MoveTo (HDC HDc, int X, int Y)
// moves the sprite to a new position
   {
   HBITMAP HBMHold;
   HDC HMemDC;
   HDC HMemDCHold;
   RECT RectNew;
   RECT RectOld;
   RECT RectUnion;

   // (1) create temporary hold bitmap:
                                      
   // calculate coordinates of entire affected screen area:
   RectOld.left = mX;
   RectOld.top = mY;
   RectOld.right = mX + mWidth;
   RectOld.bottom = mY + mHeight;

   RectNew.left = X;
   RectNew.top = Y;
   RectNew.right = X + mWidth;
   RectNew.bottom = Y + mHeight;

   UnionRect (&RectUnion, &RectOld, &RectNew);
   RectUnion.left -= RectUnion.left % 8;
   
   HBMHold = CreateCompatibleBitmap
      (HDc,
      RectUnion.right - RectUnion.left,
      RectUnion.bottom - RectUnion.top);
   if (HBMHold == 0)
      return (FALSE);
                           
   // (2) copy affected area of screen into hold bitmap:                           
   
   HMemDCHold = CreateCompatibleDC (HDc);
   SelectObject (HMemDCHold, HBMHold);

   BitBlt
      (HMemDCHold,
      0,
      0,
      RectUnion.right - RectUnion.left,
      RectUnion.bottom - RectUnion.top,
      HDc,
      RectUnion.left,
      RectUnion.top,
      SRCCOPY);

   // (3) erase sprite in hold bitmap:
   
   HMemDC = CreateCompatibleDC (HDc);
   SelectObject (HMemDC, mHBMSave);
   
   BitBlt
      (HMemDCHold,
      mX - RectUnion.left,
      mY - RectUnion.top,
      mWidth,
      mHeight,
      HMemDC,
      0,
      0,
      SRCCOPY);

   // (4) save screen graphics at new sprite position:
   
    BitBlt
      (HMemDC,
      0,
      0,
      mWidth,
      mHeight,
      HMemDCHold,
      X - RectUnion.left,
      Y - RectUnion.top,
		SRCCOPY);

   // (5) transfer mask bitmap:
   
   SelectObject (HMemDC, mHBMMask);
   BitBlt
      (HMemDCHold,
      X - RectUnion.left,
      Y - RectUnion.top,
      mWidth,
      mHeight,
      HMemDC,
      0,
      0,
      SRCAND);

   // (6) transfer image bitmap:
   
   SelectObject (HMemDC, mHBMImage);
   BitBlt
      (HMemDCHold,
      X - RectUnion.left,
      Y - RectUnion.top,
      mWidth,
      mHeight,
      HMemDC,
      0,
      0,
      SRCINVERT);

   // (7) copy hold bitmap back to screen:
   
   BitBlt
      (HDc,
      RectUnion.left,
      RectUnion.top,
      RectUnion.right - RectUnion.left,
      RectUnion.bottom - RectUnion.top,
      HMemDCHold,
      0,
      0,
      SRCCOPY);

   // delete the memory device contexts:
   DeleteDC (HMemDCHold);
   DeleteDC (HMemDC);

   // (8) delete hold bitmap:
   DeleteObject (HBMHold);
          
   // (9) save coordinates of new sprite position:          
   mX = X;
   mY = Y;

   return (TRUE);
   }

BOOL ClSprite::Redraw (HDC HDc)
// redraws the sprite at its current location
   {
   HDC HMemDC;
   register int Result;

   if (mHBMSave == 0)  // return error if object not initialized
      return (FALSE);

   HMemDC = CreateCompatibleDC (HDc);
   if (HMemDC == 0)
      return (FALSE);

   // transfer mask bitmap to screen:
   SelectObject (HMemDC, mHBMMask);
   Result = BitBlt
      (HDc,
      mX,
      mY,
      mWidth,
      mHeight,
      HMemDC,
      0,
      0,
      SRCAND);
   if (Result == 0)
      {
      DeleteDC (HMemDC);
      return (FALSE);
      }

   // transfer image bitmap to screen:
   SelectObject (HMemDC, mHBMImage);
   Result = BitBlt
      (HDc,
      mX,
      mY,
      mWidth,
      mHeight,
      HMemDC,
      0,
      0,
      SRCINVERT);

   DeleteDC (HMemDC);

   if (Result == 0)
      return (FALSE);

   return (TRUE);
   }

BOOL ClSprite::Start (HDC HDc, int X, int Y)
// draws the sprite at its initial position in an animation
// sequence
   {
   HDC HMemDC;
   int Result;

   if (mHBMSave == 0)  // return error if object not initialized
      return (FALSE);

   // save the screen graphics at the sprite position:
   
   HMemDC = CreateCompatibleDC (HDc);
   if (HMemDC == 0)
      return (FALSE);
   if (SelectObject (HMemDC, mHBMSave) == 0)
      return (FALSE);
      
   Result = BitBlt
      (HMemDC,
      0,
      0,
      mWidth,
      mHeight,
      HDc,
      X,
      Y,
      SRCCOPY);
   if (Result == 0)
      {
      DeleteDC (HMemDC);
      return (FALSE);
      }

   // transfer mask bitmap to screen:
   
   SelectObject (HMemDC, mHBMMask);
   Result = BitBlt
      (HDc,
      X,
      Y,
      mWidth,
      mHeight,
      HMemDC,
      0,
      0,
      SRCAND);
   if (Result == 0)
      {
      DeleteDC (HMemDC);
      return (FALSE);
      }

   // transfer image bitmap to screen:
   
   SelectObject (HMemDC, mHBMImage);
   Result = BitBlt
      (HDc,
      X,
      Y,
      mWidth,
      mHeight,
      HMemDC,
      0,
      0,
      SRCINVERT);

   DeleteDC (HMemDC);

   if (Result == 0)
      return (FALSE);
   
   // save coordinates of initial sprite position:
   mX = X;
   mY = Y;

   return (TRUE);
   }
