unit Timeedit;

(*********************************************
TTimeEdit -> TEdit

PROPERTIES:

Time - Set or Get the time associated with the control.
*********************************************)

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, Buttons, IniFiles, Spin;

type

  TToken = ( tkHour, tkMinute, tkAMPM );

  TTimeEdit = class( TEdit )
  private
     FTime: TDateTime;
     FSpin: TSpinButton;
     Token: TToken;
     sSep: string[1];
     function GetCursorPos: integer;
  protected
     procedure CreateParams( var Params: TCreateParams ); override;
     procedure WMSize( var Message: TWMSize ); message WM_SIZE;
     procedure DoEnter; override;
     procedure DoExit; override;
     procedure SetTime( const t: TDateTime );
     procedure MouseUp( Button: TMouseButton; ShiftState: TShiftState; X, Y: integer ); override;
     procedure SelectHour;
     procedure SelectMinute;
     procedure SelectAMPM;
     procedure ChangeValue( const k: integer );
     procedure UpClick (Sender: TObject); virtual;
     procedure DownClick (Sender: TObject); virtual;
     procedure KeyPress( var Key: char ); override;
  public
     constructor Create( AOwner: TComponent ); override;
     destructor Destroy; override;
     property Time: TDateTime read FTime write SetTime;
  published
  end;

procedure Register;

implementation

constructor TTimeEdit.Create( AOwner: TComponent );
var
  ini: TIniFile;
begin
  inherited Create( AOwner );

  FSpin := TSpinButton.Create( self );
  FSpin.Visible := TRUE;
  FSpin.Parent := self;
  FSpin.Cursor := crArrow;
  FSpin.OnUpClick := UpClick;
  FSpin.OnDownClick := DownClick;

{ Get international time seperator }
  ini := TIniFile.Create( 'WIN.INI' );
  sSep := ini.ReadString( 'intl', 'sTime', ':' );
  ini.Free;

  FTime := 0.0;
end;

(*********************************************
Make sure edit field doesn't draw over children.
*********************************************)
procedure TTimeEdit.CreateParams( var Params: TCreateParams );
begin
  inherited CreateParams( Params );
  Params.Style := Params.Style or WS_CLIPCHILDREN;
end;

destructor TTimeEdit.Destroy;
begin
  FSpin.Free;
  inherited Destroy;
end;

procedure TTimeEdit.SetTime( const t: TDateTime );
begin
  if t <> FTime then
     begin
        FTime := t;
        Text := FormatDateTime( 't', FTime );
     end;
end;

(*********************************************
Determine which "token" the user is on and
highlight only that token.
*********************************************)
procedure TTimeEdit.MouseUp( Button: TMouseButton; ShiftState: TShiftState; X, Y: integer );
var
  nPos: integer;
begin
  nPos := GetCursorPos;
  if nPos <= Pos( sSep, Text ) then
     SelectHour
  else if nPos <= Pos( ' ', Text ) then
     SelectMinute
  else
     SelectAMPM;
  inherited MouseUp( Button, ShiftState, X, Y );
end;

(*********************************************
Select the hour portion of the field.
*********************************************)
procedure TTimeEdit.SelectHour;
var
  nPos: integer;
begin
  Token := tkHour;
  nPos := Pos( sSep, Text );
  if nPos = 0 then
     nPos := Length( Text );
  SendMessage( Handle, em_SetSel, 0, ( nPos - 1 ) * 65536 );
end;

(*********************************************
Select the minute portion of the field.
*********************************************)
procedure TTimeEdit.SelectMinute;
var
  nPos1, nPos2: integer;
begin
  Token := tkMinute;
  nPos1 := Pos( sSep, Text );
  nPos2 := Pos( ' ', Text );
  if nPos2 = 0 then
     nPos2 := Length( Text );
  SendMessage( Handle, em_SetSel, 0, nPos1 + ( ( nPos2 - 1 ) * 65536 ) );
end;

(*********************************************
Select the AM/PM portion of the field.
*********************************************)
procedure TTimeEdit.SelectAMPM;
var
  nPos: integer;
begin
  Token := tkAMPM;
  nPos := Pos( ' ', Text );
  if nPos = 0 then
     nPos := Length( Text );
  SendMessage( Handle, em_SetSel, 0, nPos + ( ( Length( Text ) ) * 65536 ) );
end;

(*********************************************
Return the position of the first character after
the selected block.
*********************************************)
function TTimeEdit.GetCursorPos: integer;
begin
  Result := SendMessage( Handle, cb_GetEditSel, 0, 0 ) div 65536;
end;

(*********************************************
Resize child controls.
*********************************************)
procedure TTimeEdit.WMSize( var Message: TWMSize );
begin
  FSpin.Height := Height;
  FSpin.Top := 0;
  FSpin.Width := Height div 2;
  FSpin.Left := Width - FSpin.Width;
  FSpin.Refresh;
  if FTime = 0 then
     Time := Now;
end;

(*********************************************
Increment or decrement a token ... this is tied
to the pressing of a spin button.
*********************************************)
procedure TTimeEdit.ChangeValue( const k: integer );
var
  nHour, nMinute: integer;
  nDummy: word;
begin
  DoExit;
  DecodeTime( Time, word( nHour ), word( nMinute ), nDummy, nDummy );
  case Token of
     tkHour:
        nHour := nHour + k;
     tkMinute:
        nMinute := nMinute + k;
     tkAMPM:
        nHour := nHour + ( 12 * k );
  end;

  if nMinute >=60 then
     begin
        nMinute := nMinute - 60;
        Inc( nHour );
     end;
  if nHour >= 24 then
     nHour := nHour - 24;

  if nMinute < 0 then
     begin
        nMinute := nMinute + 60;
        Dec( nHour );
     end;
  if nHour < 0 then
     nHour := nHour + 24;

  Time := EncodeTime( nHour, nMinute, 0, 0 );
  SetFocus;
  case Token of
     tkHour:
        SelectHour;
     tkMinute:
        SelectMinute;
     tkAMPM:
        SelectAMPM;
  end;
end;

(*********************************************
These 2 procedures are hooked into the
TTimerSpeedButtons.
*********************************************)
procedure TTimeEdit.UpClick;
begin
  ChangeValue( 1 );
end;

procedure TTimeEdit.DownClick;
begin
  ChangeValue( -1 );
end;

(*********************************************
Highlight the hour when the field is entered.
*********************************************)
procedure TTimeEdit.DoEnter;
begin
  inherited DoEnter;
  SelectHour;
end;

(*********************************************
Check validity of input when the field is
exited.
!! Would be nicer to check valid dates without
having to throw an exception !!
*********************************************)
procedure TTimeEdit.DoExit;
var
  tOld: TDateTime;
  t: TDateTime;
begin
  inherited DoExit;
  tOld := Time;
try
  t := StrToDateTime( FormatDateTime( 'mm/dd/yy', Now ) + ' ' + Text );
  Time := t;
except
  FTime := 0;
  Time := tOld;
  SetFocus;
end;
end;

(*********************************************
Mask out invalid key presses.
*********************************************)
procedure TTimeEdit.KeyPress( var key: char );
begin
  if key <> #8 then
     begin
        if key = sSep[1] then
           begin
              SelectMinute;
              key := #0;
           end
        else if key = ' ' then
           begin
              SelectAMPM;
              key := #0;
           end;
        if ( key < '0' ) or ( key > '9' ) then
           begin
              if key in ['a','m','p','A','M','P'] then
                 begin
                    if Token <> tkAMPM then
                       SelectAMPM;
                 end
              else
                 key := #0;
           end;
     end;
end;

procedure Register;
begin
  RegisterComponents('Additional', [TTimeEdit]);
end;

end.
