#include "windows.h"
#include "winfract.h"
#include "mathtool.h"
#include "math.h"

static char MTClassName[] =     "FFWMathTools";
static char MTWindowTitle[] =   "Math Tools";
HWND hFractalWnd, hMathToolsWnd, hCoordBox;
HANDLE hThisInst;
long FAR PASCAL MTWndProc(HWND, WORD, WORD, DWORD);
int MTWindowOpen = 0, CoordBoxOpen = 0, KillCoordBox = 0;
static double Pi =  3.14159265359;

WORD CoordFormat = IDM_RECT;
WORD AngleFormat = IDM_DEGREES;

extern unsigned xdots, ydots;
extern double xxmin, yymin, xxmax, yymax, delxx, delyy;

void CheckMathTools(void) {
   if(KillCoordBox) {
      DestroyWindow(hCoordBox);
      KillCoordBox = 0;
   }
}

BOOL RegisterMathWindows(HANDLE hInstance) {
    WNDCLASS  wc;

    hThisInst = hInstance;

    wc.style = CS_OWNDC;
    wc.lpfnWndProc = MTWndProc;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = 0;
    wc.hInstance = hInstance;
    wc.hIcon = LoadIcon(hInstance, "MathToolIcon");
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground = GetStockObject(BLACK_BRUSH);
    wc.lpszMenuName =  "MathToolsMenu";
    wc.lpszClassName = MTClassName;

    return (RegisterClass(&wc));
}

BOOL OpenMTWnd(void) {
    hMathToolsWnd = CreateWindow(
        MTClassName,
        MTWindowTitle,
        WS_OVERLAPPEDWINDOW | WS_HSCROLL | WS_VSCROLL,
        CW_USEDEFAULT, CW_USEDEFAULT,
        xdots, ydots,
        hFractalWnd,
        NULL,
        hThisInst,
        NULL
    );
    if(!hMathToolsWnd)
        return(FALSE);

    ShowWindow(hMathToolsWnd, SW_SHOWNORMAL);
    UpdateWindow(hMathToolsWnd);

    return(TRUE);
}

void MathToolBox(HWND hWnd) {
    hFractalWnd = hWnd;
    if(MTWindowOpen)
       DestroyWindow(hMathToolsWnd);
    else {
      if(!OpenMTWnd())
         MessageBox(hWnd, "Error Opening Math Tools Window", NULL,
                    MB_ICONEXCLAMATION | MB_OK);
    }
}

BOOL FAR PASCAL CoordBoxDlg(HWND hDlg, WORD Message, WORD wp, DWORD dp) {
   HMENU hDlgMenu;

   hDlgMenu = GetMenu(hDlg);
   switch(Message) {
      case WM_INITDIALOG:
         CoordBoxOpen = 1;
         CheckMenuItem(GetMenu(hFractalWnd), IDM_COORD, MF_CHECKED);
         hCoordBox = hDlg;
         return(TRUE);
      case WM_CLOSE:
         KillCoordBox = 1;
         break;
      case WM_DESTROY:
         CheckMenuItem(GetMenu(hFractalWnd), IDM_COORD, MF_UNCHECKED);
         CoordBoxOpen = 0;
         break;
      case WM_COMMAND:
         CheckMenuItem(hDlgMenu, AngleFormat, MF_UNCHECKED);
         CheckMenuItem(hDlgMenu, CoordFormat, MF_UNCHECKED);
         switch(wp) {
            case IDM_RADIANS:
            case IDM_GRAD:
            case IDM_DEGREES:
               AngleFormat = wp;
               break;
            case IDM_POLAR:
            case IDM_RECT:
            case IDM_PIXEL:
               CoordFormat = wp;
               break;
         }
         CheckMenuItem(hDlgMenu, AngleFormat, MF_CHECKED);
         CheckMenuItem(hDlgMenu, CoordFormat, MF_CHECKED);
         if(CoordFormat == IDM_POLAR) {
            SetDlgItemText(hDlg, ID_X_NAME, "|z|");
            SetDlgItemText(hDlg, ID_Y_NAME, "\xD8");
            EnableMenuItem(hDlgMenu, IDM_DEGREES, MF_ENABLED);
            EnableMenuItem(hDlgMenu, IDM_RADIANS, MF_ENABLED);
            EnableMenuItem(hDlgMenu, IDM_GRAD, MF_ENABLED);
         }
         else {
            SetDlgItemText(hDlg, ID_X_NAME, "x");
            SetDlgItemText(hDlg, ID_Y_NAME, "y");
            EnableMenuItem(hDlgMenu, IDM_DEGREES, MF_DISABLED | MF_GRAYED);
            EnableMenuItem(hDlgMenu, IDM_RADIANS, MF_DISABLED | MF_GRAYED);
            EnableMenuItem(hDlgMenu, IDM_GRAD, MF_DISABLED | MF_GRAYED);
         }
   }
   return(FALSE);
}

void UpdateCoordBox(DWORD dw) {
   unsigned xPixel, yPixel;
   double xd, yd, Angle, Modulus;
   char xStr[40], yStr[40];

   xPixel = (unsigned)dw;
   yPixel = (unsigned)(dw >> 16);
   xd = xxmin + (delxx * xPixel);
   yd = yymax - (delyy * yPixel);
   switch(CoordFormat) {
      case IDM_PIXEL:
         sprintf(xStr, "%d", xPixel);
         sprintf(yStr, "%d", yPixel);
         break;
      case IDM_RECT:
         sprintf(xStr, "%+.8g", xd);
         sprintf(yStr, "%+.8g", yd);
         break;
      case IDM_POLAR:
         Modulus = (xd*xd) + (yd*yd);
         if(Modulus > 1E-20) {
            Modulus = sqrt(Modulus);
            Angle = atan2(yd, xd);
            switch(AngleFormat) {
               case IDM_DEGREES:
                  Angle = (Angle / Pi) * 180;
                  break;
               case IDM_GRAD:
                  Angle = (Angle / Pi) * 200;
               case IDM_RADIANS:
                  break;
            }
         }
         else {
            Modulus = 0.0;
            Angle = 0.0;
         }
         sprintf(xStr, "%+.8g", Modulus);
         sprintf(yStr, "%+.8g", Angle);
         break;
   }
   SetDlgItemText(hCoordBox, ID_X_COORD, xStr);
   SetDlgItemText(hCoordBox, ID_Y_COORD, yStr);
}

void CoordinateBox(HWND hWnd) {
   FARPROC lpCoordBox;

   hFractalWnd = hWnd;
   if(CoordBoxOpen)
      DestroyWindow(hCoordBox);
   else {
      if(lpCoordBox = MakeProcInstance(CoordBoxDlg, hThisInst)) {
         if(CreateDialog(hThisInst, "CoordBox", hWnd, lpCoordBox))
            return;
      }
      MessageBox(hWnd, "Error Opening Coordinate Box",
                 NULL, MB_ICONEXCLAMATION | MB_OK);
   }
}

long FAR PASCAL MTWndProc(HWND hWnd, unsigned Message, WORD wParam, DWORD lParam) {
   switch (Message) {
      case WM_CREATE:
         CheckMenuItem(GetMenu(hFractalWnd), IDM_MATH_TOOLS, MF_CHECKED);
         MTWindowOpen = 1;
         break;
      case WM_COMMAND:
         switch(wParam) {
            case IDM_EXIT:
               DestroyWindow(hWnd);
               break;
         }
         break;
      case WM_DESTROY:
         CheckMenuItem(GetMenu(hFractalWnd), IDM_MATH_TOOLS, MF_UNCHECKED);
         MTWindowOpen = 0;
         break;
      default:
         return(DefWindowProc(hWnd, Message, wParam, lParam));
    }
    return(0L);
}
