/****************************************************************************


    PROGRAM: winfract.c

    PURPOSE: Windows-specific main-driver code for Fractint for Windows
             (look in MAINFRAC.C for the non-windows-specific code)

    Copyright (C) 1990 The Stone Soup Group.  Fractint for Windows 
    may be freely copied and distributed, but may not be sold.
    
    We are, of course, copyrighting the code we wrote to implement
    Fractint-for-Windows, and not the routines we lifted directly
    or indirectly from Microsoft's Windows 3.0 Software Development Kit.

****************************************************************************/

#include "windows.h"
#include "winfract.h"
#include "mathtool.h"
#include "fractype.h"
#include "fractint.h"
#include "select.h"
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <search.h>
#include <string.h>
#include <time.h>

LPSTR win_lpCmdLine;

HANDLE hInst;

HWND hwnd;                               /* handle to main window */
HWND hWndCopy;				 /* Copy of hWnd */

#define PALETTESIZE 256               /* dull-normal VGA                    */
HANDLE hpal;                          /* palette handle                     */
PAINTSTRUCT ps;                       /* paint structure                    */
HDC hDC;                              /* handle to device context           */
HDC hMemoryDC;                        /* handle to memory device context    */
BITMAP Bitmap;                        /* bitmap structure                   */

HANDLE  hPal;          /* Handle to the application's logical palette  */
HANDLE  hLogPal;       /* Temporary Handle */
LPLOGPALETTE pLogPal;  /* pointer to tha application's logical palette */
int     iNumColors;    /* Number of colors supported by device	       */
int     iRasterCaps;   /* Raster capabilities			               */
int     iPalSize;      /* Size of Physical palette		               */

BOOL	win_systempaletteused = FALSE;	/* flag system palette set */
extern int win_syscolorindex[21];
extern DWORD win_syscolorold[21];
extern DWORD win_syscolornew[21];

extern int CoordBoxOpen;
extern HWND hCoordBox;

#define EXE_NAME_MAX_SIZE  128

BOOL       bHelp = FALSE;      /* Help mode flag; TRUE = "ON"*/
HCURSOR    hHelpCursor;        /* Cursor displayed when in help mode*/
char       szHelpFileName[EXE_NAME_MAX_SIZE+1];    /* Help file name*/

void MakeHelpPathName(char*);  /* Function deriving help file path */

unsigned char far win_dacbox[256][3];
extern unsigned char dacbox[256][3];

BOOL bTrack = FALSE;                  /* TRUE if user is selecting a region */
BOOL zoomflag = FALSE;                /* TRUE is a zoom-box selected */
RECT Rect;

int Shape = SL_BLOCK;            /* shape to use for the selection rectangle */

/* pointers to various dialog-box routines */
FARPROC lpProcAbout;
FARPROC lpOpenDlg;
FARPROC lpSelectFractal;
FARPROC lpSelectFracParams;
FARPROC lpSelectImage;
FARPROC lpSelectDoodads;
FARPROC lpSelectCycle;
FARPROC lpSaveAsDlg;
FARPROC lpProcStatus;
FARPROC lpSelect3D;
FARPROC lpSelect3DPlanar;
FARPROC lpSelect3DSpherical;

extern unsigned char readname[], FileName[], FormFileName[];
extern unsigned char LFileName[], LName[];
extern unsigned char DialogTitle[], DefSpec[], DefExt[];

HBITMAP hBitmap, oldBitmap, oldoldbitmap;              /* working bitmaps */

HANDLE hDibInfo;		/* handle to the Device-independent bitmap */
LPBITMAPINFO pDibInfo;		/* pointer to the DIB info */
HANDLE hpixels;			/* handle to the DIB pixels */
unsigned char huge *pixels;	/* the device-independent bitmap  pixels */
extern int bytes_per_pixelline;	/* pixels/line / pixels/byte */
extern long win_bitmapsize;     /* size of the DIB in bytes */

HANDLE hClipboard1, hClipboard2, hClipboard3; /* handles to clipboard info */
LPSTR lpClipboard1, lpClipboard2;            /* pointers to clipboard info */

int last_written_y = -2;        /* last line written */
int screen_to_be_cleared = 1;	/* flag that the screen is to be cleared */
int time_to_act = 0;		/* time to take some sort of action? */
int time_to_restart = 0;        /* time to restart?  */
int time_to_reinit = 0;         /* time to reinitialize?  */
int time_to_quit = 0;           /* time to quit? */
int time_to_save = 0;		/* time to save the file? */
int time_to_print = 0;		/* time to print the file? */
int time_to_load = 0;		/* time to load a new file? */
int time_to_cycle = 0;          /* time to begin color-cycling? */

int win_3dspherical = 0;          /* spherical 3D? */
int win_display3d, win_overlay3d; /* 3D flags */

extern int win_cycledir, win_cyclerand;

extern int calc_status;

int xdots, ydots, colors, maxiter;
int ytop, ybottom, xleft, xright;
int xposition, yposition, win_xoffset, win_yoffset, xpagesize, ypagesize;
int win_xdots, win_ydots;
extern int fractype;
extern double param[4];
extern double xxmin, xxmax, yymin, yymax, xx3rd, yy3rd;
double jxxmin, jxxmax, jyymin, jyymax, jxx3rd, jyy3rd;
extern int frommandel, bitshift, biomorph;

extern char str[255];

int cpu, fpu;			/* cpu, fpu flags */

extern int win_release;

char *win_choices[100];
int win_numchoices, win_choicemade;

extern int onthelist[];
extern int CountFractalList;
extern int CurrentFractal;
int MaxFormNameChoices = 80;
char FormNameChoices[80][21];
char FormName[30];
extern char	ifsfilename[];    /* IFS code file */
extern char     ifs3dfilename[];  /* IFS 3D code file */
double far *temp_array;
HANDLE htemp_array;

/****************************************************************************

    FUNCTION: WinMain(HANDLE, HANDLE, LPSTR, int)

    PURPOSE: calls initialization function, processes message loop

****************************************************************************/

int PASCAL WinMain(hInstance, hPrevInstance, lpCmdLine, nCmdShow)
HANDLE hInstance;
HANDLE hPrevInstance;
LPSTR lpCmdLine;
int nCmdShow;
{
    int Return;

    win_lpCmdLine = lpCmdLine;

    if (!hPrevInstance)
        if (!InitApplication(hInstance))
            return (FALSE);

    if (!InitInstance(hInstance, nCmdShow))
        return (FALSE);

    fractint_main();            /* fire up the main Fractint code */
    if(htemp_array) {
        GlobalUnlock(htemp_array);
        GlobalFree(htemp_array);
    }
    DestroyWindow(hWndCopy);    /* stop everything when it returns */
    
    return(0);                  /* we done when 'fractint_main' returns */
}

/****************************************************************************

    FUNCTION: InitApplication(HANDLE)

    PURPOSE: Initializes window data and registers window class

****************************************************************************/

BOOL InitApplication(hInstance)
HANDLE hInstance;
{
    WNDCLASS  wc;

    wc.style = CS_VREDRAW | CS_HREDRAW;
    wc.lpfnWndProc = MainWndProc;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = 0;
    wc.hInstance = hInstance;
    wc.hIcon = LoadIcon(hInstance, "FracIcon");
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground = GetStockObject(WHITE_BRUSH); 
    wc.lpszMenuName =  "WinFracMenu";
    wc.lpszClassName = "FractintForWindowsV0010";

    return (RegisterClass(&wc) && RegisterMathWindows(hInstance));
}


/****************************************************************************

    FUNCTION:  InitInstance(HANDLE, int)

    PURPOSE:  Saves instance handle and creates main window

****************************************************************************/
BOOL InitInstance(hInstance, nCmdShow)
    HANDLE          hInstance;
    int             nCmdShow;
{
    DWORD WinFlags;
    WORD  DeviceCaps;
    int iLoop, jLoop;
    DWORD ThisColor;
    
    float temp;
    char tempname[40];

    /* so, what kind of a computer are we on, anyway? */
    WinFlags = GetWinFlags();
    cpu = 88;                             /* determine the CPU type */
    if (WinFlags & WF_CPU186) cpu = 186;
    if (WinFlags & WF_CPU286) cpu = 286;
    if (WinFlags & WF_CPU386) cpu = 386;
    if (WinFlags & WF_CPU486) cpu = 386;
    fpu = 0;                              /* determine the FPU type */
    if (WinFlags & WF_80x87)  fpu = 87;
    if ((WinFlags & WF_CPU486) || (fpu && (cpu == 386))) fpu = 387;

    hInst = hInstance;

    temp = win_release / 100.0;
    sprintf(tempname,"Fractint for Windows - Vers %5.2f", temp);

    hwnd = CreateWindow(
        "FractintForWindowsV0010",
        tempname,
        WS_OVERLAPPEDWINDOW | WS_HSCROLL | WS_VSCROLL,
        160, 120,  /* initial locn instead of CW_USEDEFAULT, CW_USEDEFAULT, */
        320, 240,  /* initial size instead of CW_USEDEFAULT, CW_USEDEFAULT, */
        NULL,
        NULL,
        hInstance,
        NULL
    );

    if (!hwnd) {  /* ?? can't create the initial window! */
        return (FALSE);
        }

    /* This program doesn't run in "real" mode, so shut down right
       now to keep from mucking up Windows */
    if (!((WinFlags & WF_STANDARD) || (WinFlags & WF_ENHANCED))) {
        MessageBox (
            GetFocus(),
            "This program requires Standard\nor 386-Enhanced Mode",
            "Fractint For Windows",
            MB_ICONSTOP | MB_OK);
        return(FALSE);
        }
    
    xdots   = 200;                   /* for now, let's start with a tiny image */
    ydots   = 150;
    win_xdots = xdots;
    win_ydots = ydots;
    maxiter = 150;                   /* and a few iterations */
    xposition = yposition = 0;       /* dummy up a few pointers */
    xpagesize = ypagesize = 2000;
    set_win_offset();
        
    ShowWindow(hwnd, nCmdShow);
    UpdateWindow(hwnd);

    /* let's ensure that we have at lease 40K of free memory */
    {
    HANDLE temphandle;
    if (!(temphandle = GlobalAlloc(GMEM_FIXED,40000L)) ||
        !(htemp_array = GlobalAlloc(GMEM_FIXED, sizeof(double) * 8001))) {
        MessageBox (
            GetFocus(),
            "There isn't enough available\nmemory to run Winfract.",
            "Fractint For Windows",
            MB_ICONSTOP | MB_OK);
        return(FALSE);
        }
        GlobalLock(temphandle);
        GlobalUnlock(temphandle);
        GlobalFree(temphandle);
        temp_array = (double far *)GlobalLock(htemp_array);
    }

   MakeHelpPathName(szHelpFileName);

    /* so, what kind of a display are we using, anyway? */
    hDC = GetDC(NULL);
    iPalSize    = GetDeviceCaps (hDC, SIZEPALETTE);
    iRasterCaps = GetDeviceCaps (hDC, RASTERCAPS);
    iRasterCaps = (iRasterCaps & RC_PALETTE) ? TRUE : FALSE;
    if (iRasterCaps)
       iNumColors = GetDeviceCaps(hDC, SIZEPALETTE);
    else
       iNumColors = GetDeviceCaps(hDC, NUMCOLORS);
    ReleaseDC(NULL,hDC);

    /* fudges for oddball stuff (is there any?) */
    /* also, note that "true color" devices return -1 for NUMCOLORS */
    colors = iNumColors;
    if (colors < 2 || colors > 16) colors = 256;
    if (colors > 2 && colors < 16) colors = 16;

     /* Allocate enough memory for a logical palette with
      * PALETTESIZE entries and set the size and version fields
      * of the logical palette structure.
      */
     if (!(hLogPal = GlobalAlloc (GMEM_FIXED,
        (sizeof (LOGPALETTE) +
        (sizeof (PALETTEENTRY) * (PALETTESIZE)))))) {
        MessageBox (
            GetFocus(),
            "There isn't enough available\nmemory to run Winfract..",
            "Fractint For Windows",
            MB_ICONSTOP | MB_OK);
        return(FALSE);
          }
    pLogPal = (LPLOGPALETTE)GlobalLock(hLogPal); 
    pLogPal->palVersion    = 0x300;
    pLogPal->palNumEntries = PALETTESIZE;
    /* fill in intensities for all palette entry colors */
    for (iLoop = 0; iLoop < PALETTESIZE; iLoop++) {
        pLogPal->palPalEntry[iLoop].peRed   = iLoop;
        pLogPal->palPalEntry[iLoop].peGreen = 0;
        pLogPal->palPalEntry[iLoop].peBlue  = 0;
        pLogPal->palPalEntry[iLoop].peFlags = PC_EXPLICIT;
        }
    /* flip the ugly red color #1 with the pretty blue color #4 */
    if (iNumColors < 0 || iNumColors > 4) {
        pLogPal->palPalEntry[1].peRed = 4;
        pLogPal->palPalEntry[4].peRed = 1;
        }
    /*  create a logical color palette according the information
        in the LOGPALETTE structure. */
    hPal = CreatePalette ((LPLOGPALETTE) pLogPal) ;
    hDC = GetDC(GetFocus());
    SetMapMode(hDC,MM_TEXT);
    SelectPalette (hDC, hPal, 1);
    RealizePalette(hDC);
    /* get the "real" colors */
    jLoop = iNumColors;
    if (jLoop < 0) jLoop = -1;
    for (iLoop = 0; iLoop < PALETTESIZE; iLoop++){
        if (++jLoop >= iNumColors)
            if (iNumColors > 0)
                jLoop = 0;
        ThisColor = GetNearestColor (hDC, PALETTEINDEX (jLoop) );
        win_dacbox[iLoop][0] = dacbox[iLoop][0] =
            ((BYTE)GetRValue (ThisColor)) >> 2;
        win_dacbox[iLoop][1] = dacbox[iLoop][1] =
            ((BYTE)GetGValue (ThisColor)) >> 2;
        win_dacbox[iLoop][2] = dacbox[iLoop][2] =
            ((BYTE)GetBValue (ThisColor)) >> 2;
        }
    ReleaseDC(GetFocus(),hDC);

    /* allocate a device-independent bitmap header */
    if (!(hDibInfo = GlobalAlloc(GMEM_FIXED,
        sizeof(BITMAPINFOHEADER)+256*sizeof(PALETTEENTRY)))) {
        MessageBox (
            GetFocus(),
            "There isn't enough available\nmemory to run Winfract...",
            "Fractint For Windows",
            MB_ICONSTOP | MB_OK);
        return(FALSE);
        }
    pDibInfo = (LPBITMAPINFO)GlobalLock(hDibInfo);
    /* fill in the header */
    pDibInfo->bmiHeader.biSize  = (long)sizeof(BITMAPINFOHEADER);
    pDibInfo->bmiHeader.biWidth  = win_xdots;
    pDibInfo->bmiHeader.biHeight = win_ydots;
    pDibInfo->bmiHeader.biSizeImage = (DWORD)win_xdots * win_ydots;
    pDibInfo->bmiHeader.biPlanes = 1;
    pDibInfo->bmiHeader.biBitCount = 8;
    pDibInfo->bmiHeader.biCompression = BI_RGB;
    pDibInfo->bmiHeader.biXPelsPerMeter = 0L;
    pDibInfo->bmiHeader.biYPelsPerMeter = 0L;
    pDibInfo->bmiHeader.biClrUsed = 0L;
    pDibInfo->bmiHeader.biClrImportant = 0L;
    default_dib_palette();

    /* allocate and lock a pixel array for the initial bitmap */
    hpixels = (HANDLE) 0;
    pixels = (char huge *) NULL;
    if (clear_screen(0))
        return(TRUE);

    MessageBox (
        GetFocus(),
        "There isn't enough available\nmemory to run Winfract....",
        "Fractint For Windows",
         MB_ICONSTOP | MB_OK);

    return (FALSE);

}

/****************************************************************************

    FUNCTION: MainWndProc(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages

****************************************************************************/

void lmemcpy(  LPSTR to,  LPSTR from, long len)
{
long i;

for (i = 0; i < len; i++)
  to[i] = from[i];
}

long FAR PASCAL MainWndProc(hWnd, message, wParam, lParam)
HWND hWnd;				/* handle to main window */
unsigned message;
WORD wParam;
LONG lParam;
{

    RECT tempRect;

    int IOStatus;                           /* result of file i/o      */
    int Return;
    int i, fchoice;

    switch (message) {
 
       case WM_KEYDOWN:
           switch (wParam) {
               case VK_F1:
               /* F1, shifted F1 bring up the Help Index */
                   WinHelp(hWnd,szHelpFileName,HELP_INDEX,0L);
                   break;
               
               case VK_SPACE:
               /* Space/etc.. toggles Color-cycling parameters */
                   if (time_to_cycle == 1)
                       time_to_cycle = 0;
                   else 
                       if (win_oktocycle())
                            time_to_cycle = 1;
                   break;
               case VK_RIGHT:
               case VK_ADD: 
                   if (win_oktocycle()) {
                       time_to_cycle = 1;
                       win_cycledir = -1;
                       }
                   break;
               case VK_LEFT:
               case VK_SUBTRACT: 
                   if (win_oktocycle()) {
                       time_to_cycle = 1;
                       win_cycledir = 1;
                       }
                   break;
               case VK_RETURN: 
                   if (win_oktocycle()) {
                       time_to_cycle = 1;
                       win_cyclerand = 2;
                       }
                   break;
               }
           break;

        case WM_LBUTTONDOWN:           /* message: left mouse button pressed */

            /* Start selection of region */

            bTrack = TRUE;
            SetRectEmpty(&Rect);
            StartSelection(hWnd, MAKEPOINT(lParam), &Rect,
                (wParam & MK_SHIFT) ? (SL_EXTEND | Shape) : Shape);
            break;

        case WM_MOUSEMOVE:                        /* message: mouse movement */

            /* Update the selection region */

            if (bTrack)
                UpdateSelection(hWnd, MAKEPOINT(lParam), &Rect, Shape);
            if(CoordBoxOpen)
                UpdateCoordBox(lParam);
            break;

        case WM_LBUTTONUP:            /* message: left mouse button released */

            if (bTrack) {

               /* End the selection */

               EndSelection(MAKEPOINT(lParam), &Rect);

               ClearSelection(hWnd, &Rect, Shape);
               
               if (abs(Rect.bottom - Rect.top ) > 5 &&
                   abs(Rect.right  - Rect.left) > 5) {
                   
                   float win_aspect, zoom_aspect, rel_aspect;

                   ytop    = Rect.top;
                   ybottom = Rect.bottom;
                   xleft   = Rect.left;
                   xright  = Rect.right;
                   xleft    = xleft   + win_xoffset;
                   ytop     = ytop    + win_yoffset;
                   xright   = xright  + win_xoffset;
                   ybottom  = ybottom + win_yoffset;
                   
                   /* adjust the zoom-box for aspect ratio */
                   win_aspect = (float)ydots/(float)xdots;
                   zoom_aspect = (float)(ybottom-ytop+1)/(float)(xright-xleft+1);
                   rel_aspect = win_aspect/zoom_aspect;
                   
                   if (rel_aspect > 1.0 && rel_aspect < 1.50) {
                       ybottom = ytop + ((xright-xleft+1)*win_aspect) - 1;
                       if (ybottom >= ydots) ybottom = ydots-1;
                       }
                   if (rel_aspect < 1.0 && rel_aspect > 0.66) {
                       xright = xleft + ((ybottom-ytop+1)/win_aspect) - 1;
                       if (xright >= xdots) xright = xdots-1;
                       }
                   
                   zoomflag = TRUE;
                   time_to_restart = 1;
                   time_to_cycle = 0;
                   calc_status = 0;
                   }

               bTrack = FALSE;

            }
            break;

        case WM_RBUTTONUP:
            {
            int xx, yy;
            xx = LOWORD(lParam);
            yy = HIWORD(lParam);
            if (xx >= xdots || yy >= ydots) 
                break;
            if (fractalspecific[fractype].tojulia != NOFRACTAL
                && param[0] == 0.0 && param[1] == 0.0) {
                /* switch to corresponding Julia set */
                fractype = fractalspecific[fractype].tojulia;
                param[0] = xxmin + (xxmax - xxmin) * xx / xdots;
                param[1] = yymax - (yymax - yymin) * yy / ydots;
                jxxmin = xxmin; jxxmax = xxmax;
                jyymax = yymax; jyymin = yymin;
                jxx3rd = xx3rd; jyy3rd = yy3rd;
                frommandel = 1;
                xxmin = fractalspecific[fractype].xmin;
                xxmax = fractalspecific[fractype].xmax;
                yymin = fractalspecific[fractype].ymin;
                yymax = fractalspecific[fractype].ymax;
                xx3rd = xxmin;
                yy3rd = yymin;
                if(biomorph != -1 && bitshift != 29) {
                   xxmin *= 3.0;
                   xxmax *= 3.0;
                   yymin *= 3.0;
                   yymax *= 3.0;
                   xx3rd *= 3.0;
                   yy3rd *= 3.0;
                   }
                calc_status = 0;
                }
            else if (fractalspecific[fractype].tomandel != NOFRACTAL) {
                /* switch to corresponding Mandel set */
                fractype = fractalspecific[fractype].tomandel;
                if (frommandel) {
                    xxmin = jxxmin;  xxmax = jxxmax;
                    yymin = jyymin;  yymax = jyymax;
                    xx3rd = jxx3rd;  yy3rd = jyy3rd;
                    }
                else {
                    double ccreal,ccimag;
                    ccreal = (fractalspecific[fractype].xmax - fractalspecific[fractype].xmin) / 2;
                    ccimag = (fractalspecific[fractype].ymax - fractalspecific[fractype].ymin) / 2;
                    xxmin = xx3rd = param[0] - ccreal;
                    xxmax = param[0] + ccreal;
                    yymin = yy3rd = param[1] - ccimag;
                    yymax = param[1] + ccimag;
                    }
                frommandel = 0;
                param[0] = 0;
                param[1] = 0;
                calc_status = 0;
                }
            else {
                buzzer(2); /* can't switch */
                break;
                }

            ytop    = 0;
            ybottom = ydots-1;
            xleft   = 0;
            xright  = xdots-1;

            time_to_restart  = 1;
            time_to_cycle = 0;
            calc_status = 0;
            }
            break;

        case WM_CREATE:

            /* the scroll bars are hard-coded to 100 possible values */
            xposition = yposition = 0;      /* initial scroll-bar positions */
            SetScrollRange(hWnd,SB_HORZ,0,100,FALSE);
            SetScrollRange(hWnd,SB_VERT,0,100,FALSE);
            SetScrollPos(hWnd,SB_HORZ,xposition,TRUE);
            SetScrollPos(hWnd,SB_VERT,yposition,TRUE);
            break;

        case WM_SIZE:
            xpagesize = LOWORD(lParam);		/* remember the window size */
            ypagesize = HIWORD(lParam);
            set_win_offset();
            break;
        case WM_HSCROLL:
            switch (wParam) {
               case SB_LINEDOWN:       xposition += 1; break;
               case SB_LINEUP:         xposition -= 1; break;
               case SB_PAGEDOWN:       xposition += 10; break;
               case SB_PAGEUP:         xposition -= 10; break;
               case SB_THUMBPOSITION:  xposition = LOWORD(lParam);
               default:                break;
               }
            if (xposition > 100) xposition = 100;
            if (xposition <     0) xposition = 0;
            if (xposition != GetScrollPos(hWnd,SB_HORZ)) {
               SetScrollPos(hWnd,SB_HORZ,xposition,TRUE);
               InvalidateRect(hWnd,NULL,TRUE);
               }
            set_win_offset();
           break;
        case WM_VSCROLL:
            switch (wParam) {
               case SB_LINEDOWN:       yposition += 1; break;
               case SB_LINEUP:         yposition -= 1; break;
               case SB_PAGEDOWN:       yposition += 10; break;
               case SB_PAGEUP:         yposition -= 10; break;
               case SB_THUMBPOSITION:  yposition = LOWORD(lParam);
               default:                break;
               }
            if (yposition > 100) yposition = 100;
            if (yposition <     0) yposition = 0;
            if (yposition != GetScrollPos(hWnd,SB_VERT)) {
               SetScrollPos(hWnd,SB_VERT,yposition,TRUE);
               InvalidateRect(hWnd,NULL,TRUE);
               }
            set_win_offset();
            break;
        case WM_CLOSE:
            goto GlobalExit;

        case WM_COMMAND:
            switch (wParam) {
                case IDM_COORD:
                    CoordinateBox(hWnd);
                    break;
                case IDM_ABOUT:
                    lpProcAbout = MakeProcInstance(About, hInst);
                    DialogBox(hInst, "AboutBox", hWnd, lpProcAbout);
                    FreeProcInstance(lpProcAbout);
                    break;

               case IDM_HELP_INDEX:
                   WinHelp(hWnd,szHelpFileName,HELP_INDEX,0L);
                   break;

               case IDM_HELP_KEYBOARD:
		   WinHelp(hWnd,szHelpFileName,HELP_KEY,(DWORD)(LPSTR)"keys");
                   break;

               case IDM_HELP_HELP:
		   WinHelp(hWnd,"WINHELP.HLP",HELP_INDEX,0L);
                   break;
                   
                case IDM_COPY:
                   /* allocate the memory for the BITMAPINFO structure 
                      (followed by the bitmap bits) */
                   if (!(hClipboard1 = GlobalAlloc(GMEM_FIXED,
                   sizeof(BITMAPINFOHEADER)+colors*sizeof(PALETTEENTRY)
                   + win_bitmapsize))) {
                        cant_clip();
                        return(TRUE);
                        }
                    if (!(lpClipboard1 = 
                        (LPSTR) GlobalLock(hClipboard1))) {
                        GlobalFree(hClipboard1);
                        cant_clip();
                        return(TRUE);
                        }
                    rgb_dib_palette();
                    lmemcpy((LPSTR)lpClipboard1, (LPSTR)pDibInfo,
                        sizeof(BITMAPINFOHEADER)+colors*sizeof(RGBQUAD)
                        );
                    lpClipboard1 += 
                        (sizeof(BITMAPINFOHEADER))+
                        (colors*sizeof(RGBQUAD));
                    lmemcpy((LPSTR)lpClipboard1, (LPSTR)pixels,
                        win_bitmapsize);
                    GlobalUnlock(hClipboard1);

                   /* allocate the memory for the palette info */
                   if (!lpClipboard2) {
                       if (!(hClipboard2 = GlobalAlloc (GMEM_FIXED,
                           (sizeof (LOGPALETTE) +
                           (sizeof (PALETTEENTRY) * (PALETTESIZE)))))) {
                            GlobalFree(hClipboard1);
                            cant_clip();
                            return(TRUE);
                            }
                        if (!(lpClipboard2 = 
                            (LPSTR) GlobalLock(hClipboard2))) {
                            GlobalFree(hClipboard1);
                            GlobalFree(hClipboard2);
                            cant_clip();
                            return(TRUE);
                            }
                       }
                   /* fill in the palette info */
                   lpClipboard2[0] = 0;
                   lpClipboard2[1] = 3;
                   lpClipboard2[2] = 0;
                   lpClipboard2[3] = 1;
                   lmemcpy((LPSTR)&lpClipboard2[4],
                        (LPSTR)&pDibInfo->bmiColors[0],
                        PALETTESIZE*sizeof(RGBQUAD)
                        );
                   hClipboard3 = CreatePalette ((LPLOGPALETTE) lpClipboard2);

                    if (OpenClipboard(hWnd)) {
                        EmptyClipboard();
                        SetClipboardData(CF_PALETTE, hClipboard3);
                        SetClipboardData(CF_DIB, hClipboard1);
                        CloseClipboard();
                        }

                    default_dib_palette();

                    break;

                case IDM_NEW:
                case IDM_OPEN:
                case IDM_3D:
                case IDM_3DOVER:
                    win_display3d = 0;
                    win_overlay3d = 0;
                    /* Call OpenDlg() to get the filename */
                    strcpy(DialogTitle,"Select a File to Open");
                    strcpy(FileName, readname);
                    strcpy(DefSpec,"*.gif");
                    strcpy(DefExt,".gif");
                    lpOpenDlg = MakeProcInstance((FARPROC) OpenDlg, hInst);
                    Return = DialogBox(hInst, "Open", hWnd, lpOpenDlg);
                    FreeProcInstance(lpOpenDlg);
                    if (Return && (wParam == IDM_3D || wParam == IDM_3DOVER)) {
                        lpSelect3D = MakeProcInstance(
                            (FARPROC) Select3D, hInst);
                        Return = DialogBox(hInst, "Select3D",
                             hWnd, lpSelect3D);
                        FreeProcInstance(lpSelect3D);
                        if (Return && !win_3dspherical) {
                            lpSelect3DPlanar = MakeProcInstance(
                                (FARPROC) Select3DPlanar, hInst);
                            Return = DialogBox(hInst, "Select3DPlanar",
                                 hWnd, lpSelect3DPlanar);
                            FreeProcInstance(lpSelect3DPlanar);
                            }
                        if (Return && win_3dspherical) {
                            lpSelect3DSpherical = MakeProcInstance(
                                (FARPROC) Select3DSpherical, hInst);
                            Return = DialogBox(hInst, "Select3DSpherical",
                                 hWnd, lpSelect3DSpherical);
                            FreeProcInstance(lpSelect3DSpherical);
                            }
                        }
                    if (Return) {
                        strcpy(readname,FileName);
                        time_to_load = 1;
                        time_to_cycle = 0;
                        if (wParam == IDM_3D || wParam == IDM_3DOVER)
                            win_display3d = 1;
                        if (wParam == IDM_3DOVER)
                            win_overlay3d = 1;
                        }
                    break;

                case IDM_MAPIN:
                case IDM_MAPOUT:
                    if (!win_oktocycle())
                        break;
                    /* Call OpenDlg() to get the filename */
                    strcpy(DialogTitle,"Select a File to Open");
                    strcpy(FileName, "default");
                    if (wParam == IDM_MAPOUT)
                        strcpy(FileName, "mymap");
                    strcpy(DefSpec,"*.map");
                    strcpy(DefExt,".map");
                    lpOpenDlg = MakeProcInstance((FARPROC) OpenDlg, hInst);
                    Return = DialogBox(hInst, "Open", hWnd, lpOpenDlg);
                    FreeProcInstance(lpOpenDlg);
                    if (Return && wParam == IDM_MAPIN) {
                        ValidateLuts(FileName);
                        spindac(0,1);
                        }
                    if (Return && wParam == IDM_MAPOUT) {
                        FILE *dacfile;
			dacfile = fopen(FileName,"w");
			if (dacfile == NULL) {
				break;
				}
			fprintf(dacfile,"  0   0   0\n");
			for (i = 1; i < 256; i++)
				fprintf(dacfile, "%3d %3d %3d\n",
				dacbox[i][0] << 2,
				dacbox[i][1] << 2,
				dacbox[i][2] << 2);
			fclose(dacfile);
                        }
                    break;

                case IDM_SAVE:
                case IDM_SAVEAS:
                    /* Call the SaveAsDlg() function to get the new filename */
                    strcpy(DialogTitle,"Select a File to Save");
                    strcpy(FileName, readname);
                    strcpy(DefSpec,"*.gif");
                    strcpy(DefExt,".gif");
                    lpSaveAsDlg = MakeProcInstance(SaveAsDlg, hInst);
                    Return = DialogBox(hInst, "SaveAs", hWnd, lpSaveAsDlg);
                    FreeProcInstance(lpSaveAsDlg);
                    Return = time_to_save; /* ?? ugly klooge */
                    if (Return) {
                        strcpy(readname,FileName);
                        time_to_save = 1;
                        time_to_cycle = 0;
                        }
                    break;                                  /* User canceled */

                case IDM_PRINT:
                    time_to_print = 1;
                    time_to_cycle = 0;
                    break;

                case IDM_FORMULA:
                    strcpy(DialogTitle,"Select a Fractal Formula");
                    win_numchoices = CountFractalList;
                    win_choicemade = 0;
                    CurrentFractal = fractype;
                    for (i = 0; i < win_numchoices; i++) {
                        win_choices[i] = fractalspecific[onthelist[i]].name;
                        if (onthelist[i] == fractype ||
                            fractalspecific[onthelist[i]].tofloat == fractype)
                            win_choicemade = i;
                        }
                    lpSelectFractal = MakeProcInstance(SelectFractal, hInst);
                    Return = DialogBox(hInst, "SelectFractal",
                        hWnd, lpSelectFractal);
                    FreeProcInstance(lpSelectFractal);
                    fchoice = win_choicemade;
                    if (Return && onthelist[fchoice] == IFS) {
                        /* obtain the IFS filename */
                        strcpy(DialogTitle,"Select an IFS Filename");
                        strcpy(FileName, FormFileName);
                        strcpy(DefSpec,"*.ifs");
                        strcpy(DefExt,".ifs");
                        lpOpenDlg = MakeProcInstance((FARPROC) OpenDlg, hInst);
                        Return = DialogBox(hInst, "Open", hWnd, lpOpenDlg);
                        FreeProcInstance(lpOpenDlg);
                        if (Return) {
                            strcpy(ifsfilename, FileName);
                            ifsgetfile();
                            }
                        }
                    if (Return && onthelist[fchoice] == IFS3D) {
                        /* obtain the IFS filename */
                        strcpy(DialogTitle,"Select an IFS Filename");
                        strcpy(FileName, FormFileName);
                        strcpy(DefSpec,"*.ifs");
                        strcpy(DefExt,".ifs");
                        lpOpenDlg = MakeProcInstance((FARPROC) OpenDlg, hInst);
                        Return = DialogBox(hInst, "Open", hWnd, lpOpenDlg);
                        FreeProcInstance(lpOpenDlg);
                        if (Return) {
                            strcpy(ifs3dfilename, FileName);
                            ifs3dgetfile();
                            }
                        }
                    if (Return && (onthelist[fchoice] == FORMULA || 
                        onthelist[fchoice] == FFORMULA)) {
                        /* obtain the formula filename */
                        strcpy(DialogTitle,"Select a Formula File");
                        strcpy(FileName, FormFileName);
                        strcpy(DefSpec,"*.frm");
                        strcpy(DefExt,".frm");
                        lpOpenDlg = MakeProcInstance((FARPROC) OpenDlg, hInst);
                        Return = DialogBox(hInst, "Open", hWnd, lpOpenDlg);
                        FreeProcInstance(lpOpenDlg);
                        if (Return) {
                            strcpy(FormFileName, FileName);
                            get_formula_names();
                            strcpy(DialogTitle,"Select a Formula");
                            win_choicemade = 0;
                            lpSelectFractal = MakeProcInstance(SelectFractal, hInst);
                            Return = DialogBox(hInst, "SelectFractal",
                                hWnd, lpSelectFractal);
                            FreeProcInstance(lpSelectFractal);
                            if (Return) 
                                Return = parse_formula_names();
                            }
                        }
                    if (Return && (onthelist[fchoice] == LSYSTEM)) {
                        /* obtain the lsystem formula filename */
                        strcpy(DialogTitle,"Select an Lsystem File");
                        strcpy(FileName, LFileName);
                        strcpy(DefSpec,"*.l");
                        strcpy(DefExt,".l");
                        lpOpenDlg = MakeProcInstance((FARPROC) OpenDlg, hInst);
                        Return = DialogBox(hInst, "Open", hWnd, lpOpenDlg);
                        FreeProcInstance(lpOpenDlg);
                        if (Return) {
                            strcpy(LFileName, FileName);
                            get_lsys_name();
                            strcpy(DialogTitle,"Select a Formula");
                            win_choicemade = 0;
                            lpSelectFractal = MakeProcInstance(SelectFractal, hInst);
                            Return = DialogBox(hInst, "SelectFractal",
                                hWnd, lpSelectFractal);
                            FreeProcInstance(lpSelectFractal);
                            if (Return) {
                                strcpy(LName, win_choices[win_choicemade]);
                                Return = !LLoad();
                                }
                            }
                        }
                    if (Return) {
                        CurrentFractal = onthelist[fchoice];
                        lpSelectFracParams = MakeProcInstance(SelectFracParams,
                            hInst);
                        Return = DialogBox(hInst, "SelectFracParams",
                            hWnd, lpSelectFracParams);
                        FreeProcInstance(lpSelectFracParams);
                        }
                    if (Return) {
			restoredac();
                        time_to_reinit = 1;
                        time_to_cycle = 0;
                        calc_status = 0;
                        }
                    break;

                case IDM_IMAGE:
			lpSelectImage = MakeProcInstance(SelectImage, hInst);
			Return = DialogBox(hInst, "SelectImage",
				hWnd, lpSelectImage);
			FreeProcInstance(lpSelectImage);
			if (Return) {
				restoredac();
				time_to_restart = 1;
                                time_to_cycle = 0;
                                calc_status = 0;
				}
                        break;

                case IDM_DOODADS:
			lpSelectDoodads = MakeProcInstance(SelectDoodads, hInst);
			Return = DialogBox(hInst, "SelectDoodads",
				hWnd, lpSelectDoodads);
			FreeProcInstance(lpSelectDoodads);
			if (Return) {
				restoredac();
				time_to_restart = 1;
                                time_to_cycle = 0;
                                calc_status = 0;
				}
                        break;

                case IDM_CYCLE:
                    if (!win_oktocycle())
                        break;
                    lpSelectCycle = MakeProcInstance(SelectCycle, hInst);
                    Return = DialogBox(hInst, "SelectCycle",
                    hWnd, lpSelectCycle);
                    FreeProcInstance(lpSelectCycle);
                    break;

                case IDM_MATH_TOOLS:
                    MathToolBox(hWnd);
                    break;

                case IDS_STATUS:
                    lpProcStatus = MakeProcInstance(Status, hInst);
                    DialogBox(hInst, "ShowStatus", hWnd, lpProcStatus);
                    FreeProcInstance(lpProcStatus);
                    break;

                GlobalExit:
                case IDM_EXIT:
                    time_to_quit = 1;
                    time_to_cycle = 0;
                    ValidateRect(hWnd, NULL);
                    hWndCopy = hWnd;
                    /* the main routine will actually call 'DestroyWindow()' */
                    break;

                case IDC_EDIT:
                    if (HIWORD (lParam) == EN_ERRSPACE) {
                        MessageBox (
                              GetFocus ()
                            , "Out of memory."
                            , "Fractint For Windows"
                            , MB_ICONSTOP | MB_OK
                        );
                    }
                    break;

            } 
            break;

        case WM_PAINT:
            if (screen_to_be_cleared && last_written_y < 0) {
                 /* an empty window */
                 screen_to_be_cleared = 0;
                 GetUpdateRect(hWnd, &tempRect, TRUE);
                 hDC = BeginPaint(hWnd,&ps);
                 if (last_written_y == -2)
                     last_written_y = -1;
                 else
                     BitBlt(hDC, 0, 0, xdots, ydots,
                         NULL, 0, 0, BLACKNESS);
                 ValidateRect(hWnd, &tempRect);
                 EndPaint(hWnd,&ps);
                 break;
                 }
            screen_to_be_cleared = 0;
            GetUpdateRect(hWnd, &tempRect, FALSE);
            hDC = BeginPaint(hWnd,&ps);
            if (last_written_y >= 0) {
              int i, j, top, bottom, left, right, xcount, ycount;
              /* bit-blit the invalidated bitmap area */
              int fromleft, fromtop, fromright, frombottom;
              long firstpixel;
              top    = tempRect.top;
              bottom = tempRect.bottom;
              left   = tempRect.left;
              right  = tempRect.right;
              if (bottom >  ydots) bottom = ydots;
              if (right  >= xdots) right  = xdots-1;
              if (top    >  ydots) top    = ydots;
              if (left   >= xdots) left   = xdots;
              fromleft  = left  + win_xoffset;
              fromright = right + win_xoffset;
              fromtop    = top    + win_yoffset;
              frombottom = bottom + win_yoffset;
              xcount = fromright - fromleft + 1;
              ycount = frombottom - fromtop;
              firstpixel = win_ydots - frombottom;
              firstpixel = firstpixel * bytes_per_pixelline;
              if (left < xdots && top < ydots) {
                  SelectPalette (hDC, hPal, 0);
                  RealizePalette(hDC);
                  SetMapMode(hDC,MM_TEXT);
                  SetDIBitsToDevice(hDC,
                       left, top,
                       xcount, ycount,
                       fromleft, 0,
                       0, ydots,
                       &pixels[firstpixel], (LPBITMAPINFO)pDibInfo,
                       DIB_PAL_COLORS);
                  }
              } 
            ValidateRect(hWnd, &tempRect);
            EndPaint(hWnd,&ps);
            break;

        case WM_DESTROY:
            /* delete the handle to the logical palette if it has any
               color entries and quit. */
            if (pLogPal->palNumEntries)
            DeleteObject (hPal);
            time_to_quit = 1;
            time_to_cycle = 0;
            WinHelp(hWnd,szHelpFileName,HELP_QUIT,0L);
            if (win_systempaletteused) 
                win_stop_cycling();
            PostQuitMessage(0);
            break;

        case WM_ACTIVATE:
            if (!wParam) {  /* app. is being de-activated */
                if (win_systempaletteused) 
                    win_stop_cycling();
                break;
                }
        case WM_QUERYNEWPALETTE:
            /* If palette realization causes a palette change,
             * we need to do a full redraw.
             */
             if (last_written_y >= 0) {
                 hDC = GetDC (hWnd);
                 SelectPalette (hDC, hPal, 0);
                 i = RealizePalette(hDC);
                 ReleaseDC (hWnd, hDC);
                 if (i) {
                     InvalidateRect (hWnd, (LPRECT) (NULL), 1);
                     return 1;
                     }
                 else
                     return FALSE;
                 }
             else
                 return FALSE;
             break;

        case WM_PALETTECHANGED:
            if (wParam != hWnd){
                if (last_written_y >= 0) {
                    hDC = GetDC (hWnd);
                    SelectPalette (hDC, hPal, 0);
                    i = RealizePalette (hDC);
                    if (i)
                        UpdateColors (hDC);
                    else
                        InvalidateRect (hWnd, (LPRECT) (NULL), 1);
                    ReleaseDC (hWnd, hDC);
                    }
                }
            break;

        default:
            return (DefWindowProc(hWnd, message, wParam, lParam));
    }
    return (NULL);
}

win_oktocycle()
{
if (!(iRasterCaps) || iNumColors < 16) {
    stopmsg(0,
        "I'm sorry, but color-cycling and\ncolor maps require a palette-based\nvideo driver."
         );
    return(0);
    }
return(1);
}

extern int win_animate_flag;

win_stop_cycling()
{
HDC hDC;                      /* handle to device context           */

hDC = GetDC(GetFocus());
SetSystemPaletteUse(hDC,SYSPAL_STATIC);
ReleaseDC(GetFocus(),hDC);

time_to_cycle = 0;
win_animate_flag = 0;
restoredac();
win_systempaletteused = FALSE;
SetSysColors(COLOR_ENDCOLORS,win_syscolorindex,win_syscolorold);
}

mono_dib_palette()
{
int i, j, k;		/* fill in the palette index values */
    for (i = 0; i < 128; i = i+2) {
        pDibInfo->bmiColors[i  ].rgbBlue      =   0;
        pDibInfo->bmiColors[i  ].rgbGreen     =   0;
        pDibInfo->bmiColors[i  ].rgbRed       =   0;
        pDibInfo->bmiColors[i  ].rgbReserved  =   0;
        pDibInfo->bmiColors[i+1].rgbBlue      = 255;
        pDibInfo->bmiColors[i+1].rgbGreen     = 255;
        pDibInfo->bmiColors[i+1].rgbRed       = 255;
        pDibInfo->bmiColors[i+1].rgbReserved  =   0;
        }
}

default_dib_palette()
{
int i, j, k;		/* fill in the palette index values */
int far *palette_values;	/* pointer to palette values */

    palette_values = (int far *)&pDibInfo->bmiColors[0];
    k = 0;
    for (i = 0; i < 256; i++) {
        palette_values[i] = k;
        if (++k >= iNumColors)
           if (iNumColors > 0)
               k = 0;
        }
}

rgb_dib_palette()
{
int i;		/* fill in the palette index values */

    for (i = 0; i < 256; i++) {
        pDibInfo->bmiColors[i].rgbRed       = dacbox[i][0] << 2;
        pDibInfo->bmiColors[i].rgbGreen     = dacbox[i][1] << 2;
        pDibInfo->bmiColors[i].rgbBlue      = dacbox[i][2] << 2;
        pDibInfo->bmiColors[i].rgbReserved  =   0;
        }
}


/****************************************************************************

   FUNCTION:   MakeHelpPathName

   PURPOSE:    Winfract assumes that the .HLP help file is in the same
               directory as the Winfract executable.This function derives
               the full path name of the help file from the path of the
               executable.

****************************************************************************/

void MakeHelpPathName(szFileName)
char * szFileName;
{
   char *  pcFileName;
   int     nFileNameLen;

   nFileNameLen = GetModuleFileName(hInst,szFileName,EXE_NAME_MAX_SIZE);
   pcFileName = szFileName + nFileNameLen;

   while (pcFileName > szFileName) {
       if (*pcFileName == '\\' || *pcFileName == ':') {
           *(++pcFileName) = '\0';
           break;
       }
   nFileNameLen--;
   pcFileName--;
   }

   if ((nFileNameLen+13) < EXE_NAME_MAX_SIZE) {
       lstrcat(szFileName, "winfract.hlp");
   }

   else {
       lstrcat(szFileName, "?");
   }

   return;
}

set_win_offset()
{
win_xoffset = ((long)xposition*((long)xdots-(long)xpagesize))/100L;
win_yoffset = ((long)yposition*((long)ydots-(long)ypagesize))/100L;
if (win_xoffset+xpagesize > xdots) win_xoffset = xdots-xpagesize;
if (win_yoffset+ypagesize > ydots) win_yoffset = ydots-ypagesize;
if (xpagesize >= xdots) win_xoffset = 0;
if (ypagesize >= ydots) win_yoffset = 0;
}


/*
  Read a formula file, picking off the formula names.
  Formulas use the format "  name = { ... }  name = { ... } "
*/

int get_formula_names()	 /* get the fractal formula names */
{
   int numformulas, i;
   FILE *File;
   char msg[81], tempstring[201];

   FormName[0] = 0;		/* start by declaring failure */
   for (i = 0; i < MaxFormNameChoices; i++) {
      FormNameChoices[i][0] = 0;
      win_choices[i] = FormNameChoices[i];
      }

   if((File = fopen(FormFileName, "rt")) == NULL) {
      sprintf("I Can't find %s", FormFileName);
      stopmsg(1,msg);
      return(-1);
   }

   numformulas = 0;
   while(fscanf(File, " %20[^ \n\t({]", FormNameChoices[numformulas]) != EOF) {
      int c;

      while(c = getc(File)) {
	 if(c == EOF || c == '{' || c == '\n')
	    break;
      }
      if(c == EOF)
	 break;
      else if(c != '\n'){
	 numformulas++;
	 if (numformulas >= MaxFormNameChoices) break;
skipcomments:
	 if(fscanf(File, "%200[^}]", tempstring) == EOF) break;
	 if (getc(File) != '}') goto skipcomments;
	 if (stricmp(FormNameChoices[numformulas-1],"") == 0 ||
	     stricmp(FormNameChoices[numformulas-1],"comment") == 0)
		 numformulas--;
      }
   }
   fclose(File);
   win_numchoices = numformulas;
   qsort(FormNameChoices,win_numchoices,21,strcmp);
   return(0);
}

int parse_formula_names()	 /* parse a fractal formula name */
{

   strcpy(FormName, win_choices[win_choicemade]);

   if (RunForm(FormName)) {
       FormName[0] = 0; 	/* declare failure */
       stopmsg(0,"Can't Parse that Formula");
       return(0);
       }

return(1);
}

/* --------------------------------------------------------------------- */

get_lsys_name()		/* get the Lsystem formula name */
{
   int Litem;
   int numformulas, i;
   FILE *File;
   char buf[201];

   for (i = 0; i < MaxFormNameChoices; i++) {
      FormNameChoices[i][0] = 0;
      win_choices[i] = FormNameChoices[i];
      }

   if ((File = fopen(LFileName, "rt")) == NULL) {
      sprintf(buf,"I Can't find %s", LFileName);
      stopmsg(1,buf);
      LName[0] = 0;
      return(-1);
      }

   numformulas = 0;
   while (1) {
      int c;
      FormNameChoices[numformulas][0] = 0;
      if (fscanf(File, " %20[^ \n\t({]", FormNameChoices[numformulas]) == EOF)
	 break;
      while(c = getc(File)) {
	 if(c == EOF || c == '{' || c == '\n')
	    break;
	 }
      if(c == EOF)
	 break;
      else if(c != '\n') {
skipcomments:
	 if(fscanf(File, "%200[^}]", buf) == EOF) break;
	 if (getc(File) != '}') goto skipcomments;
	 if (stricmp(FormNameChoices[numformulas],"") != 0 &&
	     stricmp(FormNameChoices[numformulas],"comment") != 0)
		 if (++numformulas >= MaxFormNameChoices) break;
	 }
      }
   fclose(File);

   win_numchoices = numformulas;
   qsort(FormNameChoices,win_numchoices,21,strcmp);
   return(0);
}

cant_clip()
{
MessageBox (
   GetFocus(),
   "Not Enough Free Memory to Copy to the Clipboard",
   "Fractint For Windows",
    MB_ICONSTOP | MB_OK);
    return(TRUE);
}
