////////////////////////////////////////////////////////////////////////////////
//    FILE: GRAPHICS.CPP
//
//   Simple graphics library for mode 0x13. Defines standart graphic
// functions like lines, rectangles, circles etc. Features predefined
// constructor to change to graphics mode if this module is included.
////////////////////////////////////////////////////////////////////////////////

#define GRAPH_UNIT

#include <io.h>
#include <mstd.h>
#include <mobject.h>
#include <mrect.h>
#include <mfileop.h>
#include <mmem.h>
#include <marray.h>
#include <mlist.h>
#include <mstaque.h>
#include <mgraph.h>

#ifndef NO_GRAPHICS
GRAPH gr;
#endif

extern BYTE bits[];
Stack DCStack;

///////////////////////////// GRAPHICS TYPE //////////////////////////////////
// Constructor and destructor are in the end of file.

/*********************************************/
 BOOL             clipscreen
/*********************************************
 Clips the given point against the viewport.
*/
(int *x, int *y)
{
int ox = *x, oy = *y;

    if (*x > gr.vp.right)
       *x = gr.vp.right;
    if (*x < gr.vp.left)
       *x = gr.vp.left;
    if (*y > gr.vp.bottom)
       *y = gr.vp.bottom;
    if (*y < gr.vp.top)
       *y = gr.vp.top;

    return (ox == *x && oy == *y);
}

/*********************************************/
 void             clipscreen
/*********************************************
 Clips the given rectangle against the viewport.
*/
(RECT *area)
{
    if (area->right > gr.vp.right)
       area->right = gr.vp.right;
    else if (area->right < gr.vp.left)
       area->right = gr.vp.left;
    if (area->left < gr.vp.left)
       area->left = gr.vp.left;
    else if (area->left > gr.vp.right)
       area->left = gr.vp.right;
    if (area->bottom > gr.vp.bottom)
       area->bottom = gr.vp.bottom;
    else if (area->bottom < gr.vp.top)
       area->bottom = gr.vp.top;
    if (area->top < gr.vp.top)
       area->top = gr.vp.top;
    else if (area->top > gr.vp.bottom)
       area->top = gr.vp.bottom;
}

/*********************************************/
 void             clipscreen
/*********************************************
 Clips the given rectangle against the viewport.
*/
(VP *area)
{
    clipscreen ((RECT*)area);
    area->length = area->right - area->left;
    area->width = area->bottom - area->top;
}

/*********************************************/
 void           initgraph (void)
/*********************************************/
{
union REGS in_regs,out_regs;
    in_regs.h.ah = 0;          // Set display adapter...
    in_regs.h.al = 0x13;       // To mode 0x13  (Graphics, 320x200, 256 colors)
    int86 (0x10,&in_regs,&out_regs);
    gr.IsGraphics = TRUE;
}

/*********************************************/
 void           closegraph (void)
/*********************************************/
{
union REGS in_regs,out_regs;
    in_regs.h.ah = 0;          // Set display adapter
    in_regs.h.al = 0x03;       // To mode 0x03  (Text, 80x24, 16 colors)
    int86 (0x10,&in_regs,&out_regs);
    gr.IsGraphics = FALSE;
}

/*********************************************/
 void              setdpy
/*********************************************/
(ADR address, WORD length, WORD width)
{
    gr.dpy.address = address;
    gr.dpy.length = length;
    gr.dpy.width = width;
    setviewport (0,0,length,width);   // Set viewport to whole screen
}

/*********************************************/
 void            setviewport
/*********************************************/
(int left, int top, int right, int bottom)
{
    gr.vp.Set (left, top, right, bottom);
    gr.scr_size = (right - left) * (bottom - top);
}

/*********************************************/
 void            setviewport
/*********************************************/
(const RECT& View)
{
    gr.vp = View;
    gr.scr_size = gr.vp.length * gr.vp.width;
}

/*********************************************/
 void         clearviewport (void)
/*********************************************
 Draws a black bar in place of the viewport.
*/
{
    setfillstyle (SOLID_FILL,0);
    bar (gr.vp.left,gr.vp.top,gr.vp.right,gr.vp.bottom);
}

/*********************************************/
 void         cleardevice (void)
/*********************************************
 Clears the whole screen.
*/
{
    memset (gr.dpy.address, 0x00, gr.dpy.length * gr.dpy.width);
}

#pragma argsused
/*********************************************/
 void           setfillstyle
/*********************************************
 Sets fill color. Pattern is left only for
  compatibility with standart package.
*/
(int new_fill, int new_fill_color)
{
    gr.fill_color = new_fill;      // Just to kill a warning
    gr.fill_color = new_fill_color;
}

/*********************************************/
 int              outtextxy
/*********************************************/
(int x, int y, char *str)
{
    return (gr.font.PrintString (x,y,str));
}

/*********************************************/
 int            getmaxx (void)
/*********************************************/
{
    return (gr.vp.right);
}

/*********************************************/
 int            getmaxy (void)
/*********************************************/
{
    return (gr.vp.bottom);
}

/*********************************************/
 int            getminx (void)
/*********************************************/
{
    return (gr.vp.left);
}

/*********************************************/
 int            getminy (void)
/*********************************************/
{
    return (gr.vp.top);
}

/*********************************************/
 int             getx (void)
/*********************************************/
{
    return (gr.xpos);
}

/*********************************************/
 int             gety (void)
/*********************************************/
{
    return (gr.ypos);
}

/*********************************************/
 void            setcolor
/*********************************************/
(WORD new_color)
{
    if (new_color < MAXCOL)
       gr.color = new_color;
}

/*********************************************/
 int            getcolor (void)
/*********************************************/
{
    return (gr.color);
}

/*********************************************/
 void              moveto
/*********************************************/
(int x, int y)
{
    gr.xpos = x;
    gr.ypos = y;
}

/*********************************************/
 void        swap_fill_draw (void)
/*********************************************/
{
int col;
    col = gr.color;
    gr.color = gr.fill_color;
    gr.fill_color = col;
}

/*********************************************/
 void             lineto
/*********************************************/
(int to_x, int to_y)
{
    line (gr.xpos, gr.ypos, to_x, to_y);
}

/*********************************************/
 void             linerel
/*********************************************/
(int dx, int dy)
{
    line (gr.xpos, gr.ypos, gr.xpos + dx, gr.ypos + dy);
}

/*********************************************/
 void           push_DC (void)
/*********************************************/
{
DC *stel;
    stel = new DC;
    stel->vp = gr.vp;
    stel->color = gr.color;
    stel->fill_color = gr.fill_color;
    stel->dpy = gr.dpy;
    DCStack.Push (stel);
}

/*********************************************/
 void           pop_DC (void)
/*********************************************/
{
DC *stel;
    stel = (DC *) DCStack.Pop();
    if (stel != NULL)
    {
       gr.vp = stel->vp;
       gr.color = stel->color;
       gr.fill_color = stel->fill_color;
       gr.dpy = stel->dpy;
       delete (stel);
    }
}

/*********************************************/
 void            drawpoly
/*********************************************/
(WORD nPoints, POINT *polypts)
{
int n;
    moveto (polypts[0].x, polypts[0].y);
    for (n = 0;n < nPoints; ++ n)
       lineto (polypts[n].x, polypts[n].y);
}

/********************************************/
 void             circle
/********************************************
 Draws a circle with bresenham algorythm.
*/
(int xpos,int ypos,unsigned int radius)
{
int d,dE,dSE,x,y;

    x = 0; y = radius;
    d = 1 - radius;
    dE = 3; dSE = (-2)*radius + 5;

    putpixel (x+xpos,y+ypos);
    putpixel (y+xpos,x+ypos);
    putpixel (x+xpos,(-y)+ypos);
    putpixel ((-y)+xpos,x+ypos);

    while (y > x)
    {
       if (d < 0)             // Select E
       {
	  d += dE;
	  dE += 2;
	  dSE += 2;
	  ++ x;
       }
       else                   // Select SE
       {
	  d += dSE;
	  dE += 2;
	  dSE += 4;
	  ++ x;
	  -- y;
       }

       putpixel (x+xpos,y+ypos);
       putpixel (y+xpos,x+ypos);
       putpixel (y+xpos,(-x)+ypos);
       putpixel (x+xpos,(-y)+ypos);
       putpixel ((-x)+xpos,(-y)+ypos);
       putpixel ((-y)+xpos,(-x)+ypos);
       putpixel ((-y)+xpos,x+ypos);
       putpixel ((-x)+xpos,y+ypos);
    }
}

/********************************************/
 void            fillcircle
/********************************************
 Draws a circle with bresenham algorythm.
*/
(int xpos,int ypos,unsigned int radius)
{
int d,dE,dSE,x,y;

    x = 0; y = radius;
    d = 1 - radius;
    dE = 3; dSE = (-2)*radius + 5;

    swap_fill_draw();

    putpixel (xpos, ypos + radius);
    hor_line (xpos - radius, xpos + radius, ypos);
    putpixel (xpos, ypos - radius);

    while (y > x)
    {
       if (d < 0)             // Select E
       {
	  d += dE;
	  dE += 2;
	  dSE += 2;
	  ++ x;
       }
       else                   // Select SE
       {
	  d += dSE;
	  dE += 2;
	  dSE += 4;
	  ++ x;
	  -- y;
       }
       hor_line (xpos - x, xpos + x, ypos - y);
       hor_line (xpos - x, xpos + x, ypos + y);
       hor_line (xpos - y, xpos + y, ypos + x);
       hor_line (xpos - y, xpos + y, ypos - x);
    }
    swap_fill_draw();
}

//////////////// FLOODFILL defs /////////////////

class Pass : public MObject {
public:
    WORD     xpos;
    WORD     ypos;
public:
	     Pass (WORD x, WORD y) {xpos = x; ypos = y;};
};

#define MAX_PASSES     200
#define LEFT           -1
#define RIGHT          1

/////////////////////////////////////////////////

/*********************************************/
 static WORD    find_diff_color
/*********************************************/
(int dir, WORD xpos, WORD ypos, BYTE OldColor)
{
int x, maxx, minx;
    x = xpos;
    maxx = getmaxx();
    minx = getminx();

    while (getpixel (x, ypos) == OldColor && x >= minx && x < maxx)
       x += dir;

    return (x - dir);
}

/*********************************************/
 static void      scan_line
/*********************************************/
(Stack *passes, WORD left, WORD right, WORD line, BYTE OldColor)
{
Pass *NewPass;
int pos;
BOOL IsGap = FALSE;
BYTE color;

    for (pos = left; pos <= right; ++ pos)
    {
       color = getpixel (pos, line);
       if (color == OldColor && !IsGap)
	  IsGap = TRUE;
       else if (color != OldColor && IsGap)
       {
	  NewPass = new Pass (pos - 1, line);
	  passes->Push (NewPass);
	  IsGap = FALSE;
       }
    }
    if (pos > right && IsGap)
    {
       NewPass = new Pass (right, line);
       passes->Push (NewPass);
       IsGap = FALSE;
    }
}

/*********************************************/
 void             floodfill
/*********************************************
 Fills the area of the same color as the initial point.
*/
(int x,int y)
{
Stack passes (MAX_PASSES);
Pass *APass;
BYTE OldColor;
WORD left,right;

    OldColor = getpixel (x,y);
    APass = new Pass (x,y);
    passes.Push (APass);

    swap_fill_draw();
    while (!passes.IsEmpty() && !passes.IsFull())
    {
       APass = (Pass*) passes.Pop();
       left = find_diff_color (LEFT, APass->xpos, APass->ypos, OldColor);
       right = find_diff_color (RIGHT, APass->xpos, APass->ypos, OldColor);

       hor_line (left, right, APass->ypos);

       if (APass->ypos - 1 >= getminy())
	  scan_line (&passes, left, right, APass->ypos - 1, OldColor);
       if (APass->ypos + 1 < getmaxy())
	  scan_line (&passes, left, right, APass->ypos + 1, OldColor);

       delete APass;
    }
    swap_fill_draw();
}

////////////// End floodfill ///////////////////////////

/*********************************************/
 void        hor_line
/*********************************************/
(int from,int to,int level)
{
ADR address;
int rf, rt;

    rf = min (from, to);
    rt = max (from, to);
    if (level >= gr.vp.bottom || level < gr.vp.top)
       return;
    if (rf >= gr.vp.right)
       return;
    else if (rf < gr.vp.left)
       rf = gr.vp.left;
    if (rt < gr.vp.left)
       return;
    else if (rt >= gr.vp.right)
       rt = gr.vp.right - 1;

    address = (ADR) (gr.dpy.address + level * gr.dpy.length + rf);
    memset (address, (char)gr.color, rt - rf + 1);
    moveto (to, level);
}

/*********************************************/
 void        vert_line
/*********************************************/
(int from,int to,int level)
{
ADR address;
int color, n;
int rf, rt;

    rf = min (from, to);
    rt = max (from, to);
    if (level >= gr.vp.right || level < gr.vp.left)
       return;
    if (rf >= gr.vp.bottom)
       return;
    else if (rf < gr.vp.top)
       rf = gr.vp.top;
    if (rt < gr.vp.top)
       return;
    else if (rt >= gr.vp.bottom)
       rt = gr.vp.bottom - 1;

    color = getcolor();

    address = (ADR) (gr.dpy.address + rf * gr.dpy.length + level);
    for (n = rf; n <= rt; ++ n)
    {
       *address = color;
       address += gr.dpy.length;
    }
    moveto (level, to);
}

/*********************************************/
 void               line
/*********************************************
 Draws a line using bresenham algorythm.
*/
(int x1, int y1, int x2, int y2)
{
int dx,dy,dxabs,dyabs,i,px,py,x,y,sdx,sdy;

    dx = x2 - x1;
    dxabs = abs(dx);
    sdx = sign (dx);
    dy = y2 - y1;
    dyabs = abs(dy);
    sdy = sign (dy);
    x = 0; y = 0;
    px = x1; py = y1;

    if (dxabs >= dyabs)
    {
       for (i = 0;i <= dxabs; ++ i)
       {
	  y += dyabs;
	  if (y >= dxabs)
	  {
	     y -= dxabs;
	     py += sdy;
	  }
	  putpixel (px,py);
	  px += sdx;
       }
    }
    else
    {
       for (i = 0;i <= dyabs; ++ i)
       {
	  x += dxabs;
	  if (x >= dyabs)
	  {
	     x -= dyabs;
	     px += sdx;
	  }
	  putpixel (px,py);
	  py += sdy;
       }
    }
    moveto (x2, y2);
}

/*********************************************/
 void              bar
/*********************************************/
(int left,int top,int right,int bottom)
{
int n;
int rl,rr,rt,rb;
    swap_fill_draw();

    rl = min (left, right);
    rr = max (left, right);
    rt = min (top, bottom);
    rb = max (top, bottom);

    for (n = rt; n <= rb; ++ n)
       hor_line (rl, rr, n);

    swap_fill_draw();
}

/*********************************************/
 void               bar
/*********************************************/
(const RECT& area)
{
int n;
    swap_fill_draw();
    for (n = area.top; n < area.bottom; ++ n)
       hor_line (area.left, area.right - 1, n);
    swap_fill_draw();
}

/*********************************************/
 void        rectangle
/*********************************************/
(int left,int top,int right,int bottom)
{
    hor_line (left,right-1,top);
    hor_line (left+1,right,bottom);
    vert_line (top,bottom-1,right);
    vert_line (top+1,bottom,left);
}

/*********************************************/
 void        rectangle
/*********************************************/
(const RECT& area)
{
    hor_line (area.left, area.right - 1, area.top);
    hor_line (area.left + 1, area.right, area.bottom);
    vert_line (area.top, area.bottom - 1, area.right);
    vert_line (area.top + 1, area.bottom, area.left);
}

/*********************************************/
 BOOL             IsInside
/*********************************************/
(const RECT& area, int x, int y)
{
    if (x >= area.left && x < area.right &&
	y >= area.top && y < area.bottom)
       return (TRUE);
    return (FALSE);
}

/*********************************************/
 void        putpixel
/*********************************************/
(int x,int y)
{
char far *address;
WORD offset;
    if (IsInside (gr.vp, x, y))
    {
       offset = y * gr.dpy.length + x;
       address = (char far*)(gr.dpy.address + offset);
       *address = gr.color;
    }
}

/*********************************************/
 int              getpixel
/*********************************************/
(int x,int y)
{
char far *address;
    if (IsInside (gr.vp, x, y))
    {
       address = (char far*)(gr.dpy.address + y * gr.dpy.length + x);
       return (*address);
    }
    else
       return (-1);
}

/********************************************/
 void        get_rect_back
/********************************************/
(int left, int top, int right, int bottom, BYTE *image)
{
ADR offset;
WORD len, wid, pos = 0;

    clipscreen (&left, &top);
    clipscreen (&right, &bottom);

    offset = gr.dpy.address + (WORD)(top) * gr.dpy.length + left;
    len = right - left;
    wid = top - bottom;

    MoveMem ((void*)offset, (void*)(image + pos), len - 1);
    pos += len - 1;
    MoveMem ((void*)offset, (void*)(image + pos),
	     wid - 1, 1, getmaxx() - len);
    pos += wid - 1;
    MoveMem ((void*)(offset + gr.dpy.length + len), (void*)(image + pos), 1,
	     wid - 1, gr.dpy.length - len);
    offset += (wid * gr.dpy.length + 1);
    pos += wid - 1;
    MoveMem ((void*)offset, (void*)(image + pos), len - 1);
}

/********************************************/
 void        put_rect_back
/********************************************/
(int left,int top,int right,int bottom,BYTE *image)
{
ADR offset;
WORD len, wid, pos = 0;

    clipscreen (&left, &top);
    clipscreen (&right, &bottom);

    offset = gr.dpy.address + (WORD)(top * gr.dpy.length) + left;
    len = right - left;
    wid = top - bottom;

    MoveMem ((void*)(image + pos), (void*)offset, len - 1);
    pos += len - 1;
    MoveMem ((void*)(image + pos), (void*)offset, 1, wid - 1, 0, gr.dpy.length - len);
    pos += wid - 1;
    MoveMem ((void*)(image + pos), (void*)(offset + gr.dpy.length + len),
	     1, wid - 1, 0, gr.dpy.length - len);
    offset += (wid * gr.dpy.length + 1);
    pos += wid - 1;
    MoveMem ((void*)(image + pos), (void*)offset, len - 1);
}

/********************************************/
 Display&  Display :: operator=
/********************************************/
(const Display& dpy)
{
Display * temp;    // Argh! I hate when compiler does not cast...

    address = dpy.address;
    length = dpy.length;
    width = dpy.width;

    temp = this;
    return (*temp);
}

/********************************************/
      MGraph :: MGraph
/********************************************
 Initializes graphics mode, loads default font,
  initializes screen mask, and sets initial environment.
*/
(void)
{
    setdpy ((ADR)(SCREEN_ADDRESS), MAXX, MAXY);
    xpos = 0;ypos = 0;
    if (access ("default.fnt", 0) == 0)
       font.Load ("default.fnt");
    palette.Update();
    setcolor (0);
    setfillstyle (SOLID_FILL,15);
    scr_size = SCREEN_SIZE;
}

/********************************************/
      MGraph :: ~MGraph
/********************************************/
(void)
{
}
