/* FILE: eprint.c
 *  Error Print Functions.
 *  Output Messages to file or stderr
 *
 * if not initialized with eopenf(),
 *     no data is printed when eprintf is called
 *
 * eopenf(name) : opens error file,
 *      if name == NULL or "stderr", stderr is used
 *      (see xcommon.h for platform definition of stderr
 *
 * eclose(void) : closes error file, no further output till next eopenf
 * eflush(void) : flush error file
 * ereopen(void) : reopen's error file closed by eclose, must have been
 *      opened by an eopenf before eclose in order to be reopened
 * eprintf(var args) : same as printf, but output goes to error file
 *      if no file opened, no output will result
 * edumpstr(char*): prints string, non-printing chars will be in hex
 * edump(void*,long): dumps data of long bytes,
 *      format similar to dump program (8 hex bytes   8 hex bytes 16 char)
 * eFlushAlways(int) : param sets internal flag to flush on every eprintf
 *      default is TRUE : always flush
 * eclose2Flush(int) : param sets interanl flag to use close/reopen for flush
 *      default is TRUE : close & reopen for flush
 *      instead of a conventional fflush(), much safer on DOS machines
 *
 */
  
#include <stdlib.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <dos.h>
#include <dir.h>
  
#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif
  
//****************************************
// error printing routines:
//    eopenf(), eprintf(), eflush, edump, edumpstr
// these output to
//
  
static FILE *log_file = NULL;
static char log_path[ MAXPATH ];
  
static int eopened = FALSE;
static int flush_always = TRUE;
static int close2flush = FALSE;
static int echoON = FALSE;    // controls echo to stderr
  
void eFlushAlways(int flag)
{
   if (flag)
      flush_always = TRUE;
   else
      flush_always = FALSE;
}
  
void eClose2Flush(int flag)
{
   if (flag)
      close2flush = TRUE;
   else
      close2flush = FALSE;
}
  
int eopenf( char *fname )
{
   // be careful about closing stdout
   if (log_file && log_file != stdout)
   {
      fclose(log_file);
   }
  
   // check not asking for stdout
   if (fname && stricmp(fname, "stdout") !=0)
   {
      strcpy (log_path, fname);
      log_file = fopen( log_path, "wt");
   } else
   {
      // requested strerr or NULL
      strcpy(log_path,"stdout");
      log_file = stdout;
   }
  
   if ( !log_file )
   {
      //fprintf( stdout, "Unable to open log file %s!\n", log_path );
      return( FALSE );
   }
  
   eopened = TRUE;
   return( TRUE );
}
  
void eclose( void )
{
   if (!log_file) return;
   if (log_file != stdout)
      fclose(log_file);
   log_file = NULL;
}
  
  
void ereopen(void)
{
   if (!eopened || log_file == stdout)
      return;
   if (log_file)
      fclose(log_file);
   log_file = fopen(log_path,"a");
}
  
void eflush( void )
{
   if (!log_file) return;
  
   if (close2flush && log_file != stdout)
   {
      fclose (log_file);
      log_file = fopen(log_path,"a");
   }
   fflush(log_file);
}
  
void eprintf( char *fmt, ... )
{
   va_list     argptr;
  
   if (!log_file) return;
  
   va_start( argptr, fmt );
  
   vfprintf( log_file, fmt, argptr );
   va_end( argptr );
  
   if (flush_always)
      eflush();
}
  
void edumpstr(char *ptr)
{
   char *p;
   long len;
   long i, j, start;
   char hexBuf[4];
  
   // sidestep flushing
   int hold_flushflag;
   hold_flushflag = flush_always;
   flush_always = FALSE;
  
   if (!log_file) return;
   len = strlen(ptr);
  
   start = 0L;
   while (start < len)
   {
      eprintf("%4.4ld: ",start);
      // display each character as character value
      p = ptr + start;
      for (i=start, j=0; j < 16 && i < len; p++,i++, j++)
      {
         if ( ((*p) > ' ') && ((*p) <= '~') )
               // test displayable
            eprintf("   %c", (*p));
         else
         {
            eprintf(" \\");
            sprintf(hexBuf,"%X",((int)(*p) & 0xFF));
            if( strlen( hexBuf ) < 2 )
               eprintf("0");
            eprintf("%s",hexBuf);
         }
      }
      eprintf("\n");
      start = i;
   }
   // restore flush method
   flush_always = hold_flushflag;
}
  
void edump(void*lp, long len)
{
   char * p;
   long i, j, start;
   char Buff[80];
   char stuffBuff[10];
   char tmpBuf[10];
  
   // sidestep flushing
   int hold_flushflag;
  
   if (!log_file) return;
  
   hold_flushflag = flush_always;
   flush_always = FALSE;
  
   eprintf("Dump of %ld=%lx bytes\n",len, len);
   start = 0L;
   while (start < len)
   {
      // start line with pointer position key
      p = (char*)lp + start;
      sprintf(Buff,"%p: ",p);
  
      // display each character as hex value
      for (i=start, j=0; j < 16; p++,i++, j++)
      {
         if (i < len)
         {
            sprintf(tmpBuf,"%X",((int)(*p) & 0xFF));
  
            if (strlen((char *)tmpBuf) < 2)
            {
               stuffBuff[0] = '0';
               stuffBuff[1] = tmpBuf[0];
               stuffBuff[2] = ' ';
               stuffBuff[3] = '\0';
            } else
            {
               stuffBuff[0] = tmpBuf[0];
               stuffBuff[1] = tmpBuf[1];
               stuffBuff[2] = ' ';
               stuffBuff[3] = '\0';
            }
            strcat(Buff, stuffBuff);
         } else
            strcat(Buff," ");
         if (j == 7) // space between groups of 8
            strcat(Buff," ");
      }
  
      // fill out incomplete lines
      for(;j<16;j++)
      {
         strcat(Buff,"   ");
         if (j == 7)
            strcat(Buff," ");
      }
      strcat(Buff,"  ");
  
      // display each character as character value
      for (i=start,j=0,p=(char*)lp+start;
         (i < len && j < 16); p++,i++, j++)
      {
         if ( ((*p) >= ' ') && ((*p) <= '~') )   // test displayable
            sprintf(tmpBuf,"%c", *p);
         else
            sprintf(tmpBuf,"%c", '.');
         strcat(Buff,tmpBuf);
         if (j == 7)   // space between groups of 8
            strcat(Buff," ");
      }
      eprintf("%s\n", Buff);
      start = i;  // next line starting byte
   }
   // restore flush method
   flush_always = hold_flushflag;
}
