/**********************************************************************
     VARMINT's POV ANIMATOR

     Written by:  Eric Jorgensen (1994)

     This is the parse code file for Varmint's POV animator.  
     This was gennerated and compiled with Turbo C++.
     
     Enjoy.

***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <math.h>
#include "parse.h"
#include "animate.h"

typedef struct variable {
  TOKEN tok;
  char name[32];
  char data[80];
  struct variable *next;
} VARIABLE;

void updatevariable(char *name,TOKEN *value);

                                /* globals */
char *bools[] = {"==","!=","<",">","<=",">="};
char mathchar[] = {"+-/*"};
char *typename[] = { "STRING", "NUMBER", "MATH", "FUNCTION", "CLUSTER",
               "WORD", "BOOL",
               "EMPTY", "BADSTRING", "FEWPARENTHESIS", "MANYPARENTHESIS",
               "NOARGS", "UNKNOWN" };
char *parseerr[] = { "No errors", "No arguments to function",
    "Too many parenthesis", "Too few parenthesis", "Syntax error",
    "divide by zero", "Incompatible data type for expression",
    "Undefined word", "Unmatched quote", "General error",
    "Bad function arguments", "Unknown function", "Out of memory",
    "Function Undefined",  "Bad number of arguments",
    "FROMFILE() File access error.", "FROMFILE() File access beyond EOF",
    "FROMFILE() Parse error in input file" };
VARIABLE *varlist = NULL;



/**************************************************************************
  void updatevariable(char *name,TOKEN *value)

  DESCTRIPTION: Updates Value of a dynamic token.  If the token does not
                exist, it is created.

**************************************************************************/
void updatevariable(char *name,TOKEN *value)
{
  VARIABLE *head;
  int i;

  head = varlist;

  while(head) {                          // See if variable exists
    if(!strcmp(head->name,name)) break;
    head = head->next;
  }
                                         // already there?  Update it.
  if(head) {
    strcpy(head->tok.data,value->data);
    head->tok.type = value->type;
  }
  else {                                 // Make a new one.
    head = varlist;
    varlist = (VARIABLE *)malloc(sizeof(VARIABLE));
    i = sizeof(VARIABLE);
    if(!varlist) {
      textcolor(RED);
      cprintf("OUT OF MEMORY\r\n");
      exit(1);
    }
    varlist->tok.data = varlist->data;    // Set token data slot
    strcpy(varlist->tok.data,value->data);// attach values
    varlist->tok.type = value->type;
    strcpy(varlist->name,name);
    varlist->next = head;                // add link to rest of list;
  }
}

/**************************************************************************
  parsetest(void)

  DESCTRIPTION: Test function for the parser.  Have fun.

**************************************************************************/
parsetest(void)
{
  int i,j,k,spot;
  char string1[256] = {""},string2[256];
  TOKEN tok;
  char r;

  tok.data = string2;
  clrscr();

  while(strcmp(string1,"END")) {
    printf("\nEnter a string:");
    gets(string1);
    if(*string1 == 0) break;
    i = povparse(string1,&tok);
    printf("POVPARSE RETURNS:\n\tError: %s [%d]\n\tType = %s\n\tToken = %s\n",
            parseerr[i],i,typename[tok.type],tok.data);

  }

  return 0;
}



/**************************************************************************
  int gettoken(char *instring, TOKEN *token)

  DESCTRIPTION: grabs the first token in the input string

  INPUTS:
    instring    Null terminated string full o' tokens
    token        output token

  RETURNS:  Index value of the character past the end of the current token

**************************************************************************/
int gettoken(char instring[], TOKEN *token)
{
  int i,j,k,nexttok,depth,match;
  char string[32];

                                // Find start
  i=0;
                                // skip white space
  while(instring[i] == '\n' ||  // newline
        instring[i] == ' '  ||  // space
        instring[i] == '\t' ||  // tab
        instring[i] == ','  ||  // comma
        (instring[i] < 32 && instring[i] > 0)) {  // control characters
    i++;
  }

  if(instring[i] == 0) {        // end of string?
    *(token->data) = 0;
    token->type = EMPTY;
    return(0);
  }
                                // ***  BUILD TOKEN  ***


                                // arithmatic character?
  if(instring[i+1] == ' ' || instring[i+1] == '\t') {
    for(j=0; j < strlen(mathchar); j++) {
      if(instring[i] == mathchar[j]) {
        *(token->data) = instring[i];
        *(token->data+1) = 0;
        token->type = MATH;
        return(i+1);
      }
    }
  }
                                // Check for Boolean tokens
  j =0;
  while(instring[i] == '=' ||
        instring[i] == '!' ||
        instring[i] == '>' ||
        instring[i] == '<') {
    string[j] = instring[i];
    i++; j++;
  }
  if(j) {
    string[j] = 0;              // terminate token string
                                // Loop through all possible BOOLS
    for(k = 0; k < 6; k++) {
      if(!strcmp(string,bools[k])) {  // Matched? Set token and go home
        token->type = BOOL;
        strcpy(token->data,bools[k]);
        return(i);
      }
    }
    token->type = UNKNOWN;      // uh-oh.
    return(0);
  }

  if(instring[i] == '\"')  {              // quote?  Must be a string
    j = i+1;
    while(instring[j] != '\"' && instring[j] != 0) {
      *(token->data+j-i-1) = instring[j];
      j++;
    }
    *(token->data+j-i-1) = 0;
    if(instring[j] == 0) {
      token->type = BADSTRING;
      return(0);
    }
    else {
      token->type = STRING;
      return(j+1);
    }
  }
  else if(instring[i] == '(') {         // Cluster start?
    depth = 1;
    j = i+1;
    while(depth && instring[j] != 0) {
      *(token->data+j-i-1) = instring[j];
      if(instring[j] == ')') depth--;
      else if(instring[j] == '(') depth++;
      if(depth) j++;
    }
    *(token->data+j-i-1) = 0;
    if(instring[j] == 0) {
      token->type = FEWPARENTHESIS;
      return(0);
    }
    else {
      token->type = CLUSTER;
      return(j+1);
    }
  }
  else if(instring[i] == '@') {         // Function start?
    j = i+1;
    while(toupper(instring[j]) >= 'A' &&
          toupper(instring[j]) <= 'Z') {
      *(token->data+j-i-1) = toupper(instring[j]);
      j++;
    }
    *(token->data+j-i-1) = 0;
    if(instring[j] == 0) {
      token->type = NOARGS;
      return(0);
    }
    else {
      token->type = FUNCTION;
      return(j);
    }
  }
  else if((instring[i] >= '0' && instring[i] <= '9') || // Number?
          instring[i] == '.' ||
          instring[i] == '-' ) {
    j = i;
    if(instring[j] == '-')  {
      *(token->data+j-i) = instring[j];
      j++;
    }
    while((instring[j] >= '0' && instring[j] <= '9') ||
          instring[j] == '.' ) {
      *(token->data+j-i) = instring[j];
      j++;
    }
    *(token->data+j-i) = 0;
    token->type = NUMBER;
    return(j);
  }
  else if(instring[i] == ')')  {               //  extra parenthesis?
    *(token->data) = 0;
    token->type = MANYPARENTHESIS;
    return(0);
  }
  else if(toupper(instring[i]) >= 'A' &&       // alphabetic character?
          toupper(instring[i]) <= 'Z') {
    j = 0;
    while(instring[i] != '\n' &&  // newline
          instring[i] != ' '  &&  // space
          instring[i] != '\t' &&  // tab
          instring[i] != ','  &&  // comma
          instring[i] > 32) {     // control characters
      *(token->data+j) = instring[i];
      i++;
      j++;
    }
    *(token->data+j)= 0;
    token->type = WORD;
    return(i);
  }

                                        // Anything else must be unknown
  *(token->data) = 0;
  token->type = UNKNOWN;
  return(0);
}



/**************************************************************************
  int povparse(char *instring, TOKEN *outtok)

  DESCTRIPTION:  This function parses a function line in a POV file

  INPUTS:
    instring    Null terminated string to parse (header must already
                be removed)
    outstring    Output of the parsed string in text.

  RETURNS:
    0    No errors
    1   No arguments to function
    2   Too many parenthesis
    3   Too few parenthesis
    4   Syntax error
    5   divide by zero
    6   Incompatible data type for expression
    7   Undefined word
    8   Unmatched quote
    9   General error
    10  Bad function arguments
    11  Unknown function
    12  Out of memory;
    13  Function Undefined
    14   Bad number of arguments
    15   File error
    16   File access beyond EOF
    17   Parse error in input file

  EXAMPLES:

    "glerb"
      glerb

    "gloob" + @NUMTAG(4,45)
      gloob0045

    "glerb" + 45
      glerb45

    45 + "glerb"
      ERROR(6)

    45 + 5.5
      50.5

    + "blah"
      ERROR(4)

**************************************************************************/
int povparse(char *instring, TOKEN *outtok)
{
  TOKEN tok,tok2,tok3;
  double numdata = 0.0;
  TOKENTYPE thistype = UNKNOWN;
  int spot = 0,error = 0,flag = 1;
  char mathfunc = ' ';

  tok.data  = (char *)malloc(256);
  tok2.data = (char *)malloc(256);
  tok3.data = (char *)malloc(256);

  if(!tok.data || !tok2.data || !tok3.data) return(12);

                                        // Process the token string.
  while(flag) {
    spot += gettoken(instring+spot,&tok);// get next token
                                        // process initial errors
    if(tok.type == EMPTY) {
      flag = 0;
      continue;
    }
    else if(tok.type == BADSTRING)           return(8);
    else if(tok.type == FEWPARENTHESIS)     return(2);
    else if(tok.type == MANYPARENTHESIS)     return(3);
    else if(tok.type == NOARGS)             return(1);
    else if(tok.type == UNKNOWN)            return(9);

                                        // process special tokens
    if(tok.type == MATH) {              // Algebra? Get action and continue.
      mathfunc = *tok.data;
      continue;
    }
    else if(tok.type == CLUSTER) {      // Cluster? -> process further
      tok2.type = tok.type;
      strcpy(tok2.data,tok.data);
      error = povparse(tok2.data,&tok);
      if(error) return(error);
    }
    else if(tok.type == FUNCTION) {     // Function? -> process further
      spot += gettoken(instring+spot,&tok2);
      error = dofunc(tok.data,tok2.data,&tok3);
      if(error < 0) return(-error);
      switch(error) {                   // convert dofunc() errors.
        case 0: break;
        case 1:
          return(13);
          break;
        case 2:
          return(14);
          break;
        case 3:
          return(10);
          break;
        case 4:
          return(15);
          break;
        case 5:
          return(16);
          break;
        case 6:
          return(12);
          break;
        case 7:
          return(17);
          break;
        default:
          return(9);
      }

                                        /*  dofunc() errors:
                                            1   Function Undefined
                                            2   Bad number of arguments
                                            3   Bad arguments
                                            4   File error
                                            5   File access beyond EOF
                                            6   Out of memory
                                            7   Parse error in input file
                                        */
      strcpy(tok.data,tok3.data);       // setup output token.
      tok.type = tok3.type;
    }
    else if(tok.type == WORD) {         // Special Word? convert to token
      error = getword(tok.data,&tok);
      if(error) return(7);
    }
    else if(tok.type == BOOL) {         // BOOLEAN- requires special processing
      if(thistype == UNKNOWN) return(4);// syntax error if boolean goes first
      error = povparse(instring + spot,&tok2); // process rest of token string
      if(error) return(error);
      if(thistype != tok2.type) return(6);// must be same data type;
      outtok->type = NUMBER;            // BOOLS return a 0 or 1
      if(thistype == STRING) {          // Strings only work with == and !=
        if(!strcmp(tok.data,"==")) {
          if(strcmp(outtok->data,tok2.data))  strcpy(outtok->data,"0");
          else                                 strcpy(outtok->data,"1");
        }
        else if(!strcmp(tok.data,"!=")) {
          if(!strcmp(outtok->data,tok2.data)) strcpy(outtok->data,"0");
          else                                 strcpy(outtok->data,"1");
        }
        else return(6);
      }
      else {
        if(!strcmp(tok.data,"==")) {
          if(numdata == atof(tok2.data)) strcpy(outtok->data,"1");
          else                            strcpy(outtok->data,"0");
        }
        else if(!strcmp(tok.data,"!=")) {
          if(numdata != atof(tok2.data)) strcpy(outtok->data,"1");
          else                            strcpy(outtok->data,"0");
        }
        else if(!strcmp(tok.data,"<")) {
          if(numdata < atof(tok2.data))  strcpy(outtok->data,"1");
          else                            strcpy(outtok->data,"0");
        }
        else if(!strcmp(tok.data,">")) {
          if(numdata > atof(tok2.data))  strcpy(outtok->data,"1");
          else                            strcpy(outtok->data,"0");
        }
        else if(!strcmp(tok.data,"<=")) {
          if(numdata <= atof(tok2.data)) strcpy(outtok->data,"1");
          else                            strcpy(outtok->data,"0");
        }
        else if(!strcmp(tok.data,">=")) {
          if(numdata >= atof(tok2.data)) strcpy(outtok->data,"1");
          else                            strcpy(outtok->data,"0");
        }
        else return(9);                  // something weid must ave happened
      }
      free(tok.data);
      free(tok2.data);
      free(tok3.data);
      return(0);

    }

    if(thistype == UNKNOWN) {           // Aha!  First Token!
      if(tok.type == NUMBER) {          // Number? Get value.
        thistype = NUMBER;
        numdata = atof(tok.data);
      }
      else if(tok.type == STRING) {     // String? Copy to output token.
        thistype = STRING;
        strcpy(outtok->data,tok.data);
      }
      mathfunc = ' ';                   // Must clear math function since
                                        // this is the first token.
      continue;
    }


                                        // concatentate token
    if(thistype == NUMBER) {
      if(tok.type != NUMBER) return(6); // Bad data type?
      if(mathfunc == '+') numdata += atof(tok.data);
      else if(mathfunc == '-') numdata -= atof(tok.data);
      else if(mathfunc == '*') numdata *= atof(tok.data);
      else if(mathfunc == '/') numdata /= atof(tok.data);
      else return(4);                    // syntax error.
    }
    else {                              // If we are bulding a string,
                                        // then we just ignore the math
                                        // functions.
      strcat(outtok->data,tok.data);
    }

    mathfunc = ' ';                     // clear math function holder;
  }

  if(thistype == NUMBER) sprintf(outtok->data,"%lf",numdata);

  outtok->type = thistype;


  free(tok.data);
  free(tok2.data);
  free(tok3.data);

  return(0);
}



/**************************************************************************
  int getword(char *word, TOKEN *outtok)

  DESCTRIPTION: This function determines if the input word is a keyword
                and then converts it into the appropriate token.

  RETURNS:
    0    No errors
    1   Word Undefined

**************************************************************************/
int getword(char *word, TOKEN *outtok)
{
  VARIABLE *head;

  if(!strcmp(word,"FRAME")) {               // current frame
    sprintf(outtok->data,"%d",frame);
    outtok->type = NUMBER;
    return(0);
  }
  else if(!strcmp(word,"TIME")) {           // current time (in seconds)
    sprintf(outtok->data,"%lf ",current_time);
    outtok->type = NUMBER;
    return(0);
  }
  else if(!strcmp(word,"PI")) {             // The value if PI
    sprintf(outtok->data,"%lf",M_PI);
    outtok->type = NUMBER;
    return(0);
  }
  else {
    head = varlist;                        // Might be a variable
    while(head) {
      if(!strcmp(head->name,word)) break;
      head = head->next;
    }

    if(!head) {                            // Does not exist?
      outtok->type = UNKNOWN;
      *(outtok->data) = 0;
      return(1);                           // Undefined word error
    }
    else {
      outtok->type = head->tok.type;
      strcpy(outtok->data,head->tok.data);
      return(0);
    }
  }
}

/**************************************************************************
  int dofunc(char *func, char *args, TOKEN *outtok)

  DESCTRIPTION: This function processes the input functions and converts
                the output into a token.

  RETURNS:
   <0   -povparse error
    0    No errors
    1   Function Undefined
    2   Bad number of arguments
    3   Bad arguments
    4   File error
    5   File access beyond EOF
    6   Out of memory
    7   Parse error in input file


**************************************************************************/
int dofunc(char *func, char *args, TOKEN *outtok)
{
  double number;
  int error= 0,spot=0,inum1,inum2,i,j,k,numargs = 0,commas = 0;
  char string[256];
  FILE *input;
  TOKEN toklist[10];

  for(i = 0; i < strlen(args); i++) {
    if(*(args+i) == ',') commas = 1;
  }
                                            // Break up argument string
                                            // into a token list.

  if(commas) {                              // Break up only if there are commas
    for(i = 0; i < 10; i++) {
      spot += gettoken(args+spot,outtok);          // grab token
      if(strlen(outtok->data) > 255) {        // too long?
        error = 3;
        break;
      }
      if(outtok->type == EMPTY) break;        // last token?

      strcpy(string,outtok->data);            // make a copy of token

      if(outtok->type == CLUSTER) {           // only preprocess clusters
        if( strcmp(func,"IF") ||  !i ) {         // (except args 2 and 3 of IF's)
          error = povparse(string,outtok);        // process token
          if(error) {                             // problems?
            error = -error;
            break;
          }
        }
      }                                        // Mkae room for token in list.
      toklist[i].data = (char *)malloc(strlen(outtok->data)+1);
      if(!toklist[i].data) {                  // Out of memory?
        error = 6;
        break;
      }
      strcpy(toklist[i].data,outtok->data);
      toklist[i].type = outtok->type;         // copy type.
    }
    numargs = i;
  }
  else {                                      // No commas -> parse whole thing
    toklist[0].data = (char *)malloc(256);
    if(!toklist[0].data) {                  // Out of memory?
      error = 6;
    }
    else {
      error = povparse(args,&toklist[0]);     // process token
      if(error) {                             // problems?
        error = -error;
      }
    }
    numargs = 1;
  }
                                            // Error?  Free memory and bail.
  if(error) {
    for(j = 0; j < numargs; j++) free(toklist[j].data);
    return(error);
  }

  while(1) {                                // Using a while statement
                                            // allows us to use "breaks"

                                              // Various Math functions

    if(!strcmp(func,"SIN")) {                 // Sine?
      if(numargs != 1) {                      // Sanity check
        error = 2; break;
      }
      if(toklist[0].type != NUMBER) {         // Bail if not a number
        error = 3; break;
      }
      number = sin(atof(toklist[0].data));    // Compute the sine
      sprintf(outtok->data,"%lf",number);     // Convert to a string
      outtok->type = NUMBER;                  // Set token type.
    }
    else if(!strcmp(func,"COS")) {
      if(numargs != 1) {
        error = 2; break;
      }
      if(toklist[0].type != NUMBER) {
        error = 3; break;
      }
      number = cos(atof(toklist[0].data));
      sprintf(outtok->data,"%lf",number);
      outtok->type = NUMBER;
    }
    else if(!strcmp(func,"TAN")) {
      if(numargs != 1) {
        error = 2; break;
      }
      if(toklist[0].type != NUMBER) {
        error = 3; break;
      }
      number = tan(atof(toklist[0].data));
      sprintf(outtok->data,"%lf",number);
      outtok->type = NUMBER;
    }
    else if(!strcmp(func,"ASIN")) {
      if(numargs != 1) {
        error = 2; break;
      }
      if(toklist[0].type != NUMBER) {
        error = 3; break;
      }
      number = asin(atof(toklist[0].data));
      sprintf(outtok->data,"%lf",number);
      outtok->type = NUMBER;
    }
    else if(!strcmp(func,"ACOS")) {
      if(numargs != 1) {
        error = 2; break;
      }
      if(toklist[0].type != NUMBER) {
        error = 3; break;
      }
      number = acos(atof(toklist[0].data));
      sprintf(outtok->data,"%lf",number);
      outtok->type = NUMBER;
    }
    else if(!strcmp(func,"ATAN")) {
      if(numargs != 1) {
        error = 2; break;
      }
      if(toklist[0].type != NUMBER) {
        error = 3; break;
      }
      number = atan(atof(toklist[0].data));
      sprintf(outtok->data,"%lf",number);
      outtok->type = NUMBER;
    }
    else if(!strcmp(func,"LOGN")) {
      if(numargs != 1) {
        error = 2; break;
      }
      if(toklist[0].type != NUMBER) {
        error = 3; break;
      }
      number = log(atof(toklist[0].data));
      sprintf(outtok->data,"%lf",number);
      outtok->type = NUMBER;
    }
    else if(!strcmp(func,"LOG10")) {
      if(numargs != 1) {
        error = 2; break;
      }
      if(toklist[0].type != NUMBER) {
        error = 3; break;
      }
      number = log10(atof(toklist[0].data));
      sprintf(outtok->data,"%lf",number);
      outtok->type = NUMBER;
    }
    else if(!strcmp(func,"ABS")) {
      if(numargs != 1) {
        error = 2; break;
      }
      if(toklist[0].type != NUMBER) {
        error = 3; break;
      }
      number = fabs(atof(toklist[0].data));
      sprintf(outtok->data,"%lf",number);
      outtok->type = NUMBER;
    }
    else if(!strcmp(func,"FLOOR")) {
      if(numargs != 1) {
        error = 2; break;
      }
      if(toklist[0].type != NUMBER) {
        error = 3; break;
      }
      number = floor(atof(toklist[0].data));
      sprintf(outtok->data,"%lf",number);
      outtok->type = NUMBER;
    }
    else if(!strcmp(func,"CEILING")) {
      if(numargs != 1) {
        error = 2; break;
      }
      if(toklist[0].type != NUMBER) {
        error = 3; break;
      }
      number = ceil(atof(toklist[0].data));
      sprintf(outtok->data,"%lf",number);
      outtok->type = NUMBER;
    }
    else if(!strcmp(func,"INT")) {
      if(numargs != 1) {
        error = 2; break;
      }
      if(toklist[0].type != NUMBER) {
        error = 3; break;
      }
      number = atoi(toklist[0].data);
      sprintf(outtok->data,"%lf",number);
      outtok->type = NUMBER;
    }
    else if(!strcmp(func,"RANDOM")) {
      if(numargs != 1) {
        error = 2; break;
      }
      if(toklist[0].type != NUMBER) {
        error = 3; break;
      }
      number = random(atoi(toklist[0].data));
      sprintf(outtok->data,"%lf",number);
      outtok->type = NUMBER;
    }

                                            // Multi-argument math functions
    else if(!strcmp(func,"MOD")) {
      if(numargs != 2) {
        error = 2; break;
      }
      if(toklist[0].type != NUMBER) {
        error = 3; break;
      }
      inum1 = atoi(toklist[0].data);
      if(toklist[1].type != NUMBER) {
        error = 3; break;
      }
      inum2 = atoi(toklist[1].data);

      number = inum1 % inum2;
      sprintf(outtok->data,"%lf",number);
      outtok->type = NUMBER;
    }
                                            //  Miscellaneous functions

    else if(!strcmp(func,"NUMTAG")) {       //  Create a serial tag
      if(numargs != 2) {
        error = 2; break;
      }
                                            // Both args need to be numbers
      if(toklist[0].type != NUMBER || toklist[1].type != NUMBER) return(3);
                                            // create the tag
      sprintf(string,"%d",atoi(toklist[1].data));
      j = strlen(string);
      k = atoi(toklist[0].data);
      for(i = k-1; i >= 0; i--) {           // scoot string and fill in 0's
        if(i-(k-j) < 0) string[i] = '0';
        else string[i] = string[i-(k-j)];
      }
      string[k] = 0;
      strcpy(outtok->data,string);
      outtok->type = STRING;
    }

    else if(!strcmp(func,"SET")) {           //  Create/alter a variable
      if(numargs != 2) {
        error = 2; break;
      }
      if(toklist[0].type != WORD ||         // Check argument integrity
         (toklist[1].type != STRING && toklist[1].type != NUMBER)) {
        error = 3; break;
      }

      updatevariable(toklist[0].data,&(toklist[1])); // set the variable

      strcpy(outtok->data,"");               // return an empty string
      outtok->type = STRING;
    }
    else if(!strcmp(func,"IF")) {           // Logical Branch
      if(numargs != 3) {
        error = 2; break;
      }
      if(toklist[0].type != NUMBER ) {      // Must be a number for the check
        error = 3; break;
      }
      if(atoi(toklist[0].data)) {
        error = povparse(toklist[1].data,outtok);  // process first token
        if(error) {                         // problems?
          error = -error;
          break;
        }
      }
      else {
        error = povparse(toklist[2].data,outtok);  // process second token
        if(error) {                         // problems?
          error = -error;
          break;
        }
      }
    }
    else if(!strcmp(func,"FROMFILE")) {     //  Grab a value from an input file
      if(numargs != 3) {
        error = 2; break;
      }

      input = fopen(toklist[0].data,"r");   //  Open the file;
      if(!input) {                          //  File access error
        error = 4; break;
      }

      if(toklist[1].type != NUMBER) {
        error = 3; break;
      }
      inum1 = atoi(toklist[1].data);        //  Line number where the token is.

      for(i = 1; i < inum1; i++) fgets(string,255,input);
      if(!fgets(string,255,input)) {
        error = 5; break;
      }
      *(string+strlen(string)-1) = 0;       // strip newline char
      fclose(input);

      if(!strcmp(toklist[2].data,"TEXT")) {
        strcpy(outtok->data,string);
        outtok->type = STRING;
      }
      else if(!strcmp(toklist[2].data,"NUMBER")) {
        strcpy(outtok->data,string);
        outtok->type = NUMBER;
      }
      else if(!strcmp(toklist[2].data,"TOKEN")) {
        error = povparse(string,outtok);
        if(error) error = 7;
      }
      else error = 3;
    }

    else {
      outtok->type = UNKNOWN;
      *(outtok->data) = 0;
      error = 1;
      break;
    }
    break;
  }

  for(j = 0; j < numargs; j++) free(toklist[j].data);

  return(error);
}


