/*********************************************************************
 *  ASM4QC, Assembly routines for QuickC                             *
 *  Version 2.02, Copyright (c) 1989-90 SoftCircuits (tm)            *
 *                                                                   *
 *  DEMO4QC.C, Demonstration C program for ASM4QC library            *
 *  C source code                                                    *
 *********************************************************************/

#include <ctype.h>                  /* for toupper() prototype */
#include <string.h>                 /* for strlen() prototype */
#include "asm4qc.h"                 /* include ASM4QC header file */

/* attribute variables (default to monochrome values) */
int menu_text = foreback(WHITE,BLACK);
int menu_frame = foreback(WHITE,BLACK);
int menu_title = foreback(BLACK,WHITE);
int menu_hilite = foreback(BLACK,WHITE);
int info_text = foreback(WHITE,BLACK);
int info_frame = foreback(WHITE,BLACK);
int input_text = foreback(WHITE,BLACK);
int input_frame = foreback(WHITE,BLACK);
int input_title = foreback(BLACK,WHITE);
int status_bar = foreback(BLACK,WHITE);

#define  TOP      7           /* define text window coordinates */
#define  BOTTOM   17
#define  LEFT     7
#define  RIGHT    73
#define  WINDOWHEIGHTH  ((BOTTOM - TOP) - 2)
#define  NUMOFLINES  22       /* number of lines of text to be displayed */

#define  HEADER_BAR  "ASM4QC, assembly routines for QuickC"
#define  FOOTER_BAR  "Version 2.02, Copyright (c) 1989-90 SoftCircuits (tm)"

/* declare screen window buffers */
int windowbuffer[((BOTTOM - TOP) + 1) * ((RIGHT - LEFT) + 1)];
int menubuffer[299], menubuffer2[299], headerbuffer[264];
int footerbuffer[264], footerbuffer2[264], editor_buffer[366];

/* declare miscellaneous global varaibles */
char input_buffer[60];           /* kbded() input buffer */
int  row,column;                 /* initial cursor location */
int  scroll_position = 0;        /* position within demoinfo text */
int  selection = 1;              /* current menu selection */
int  mtop = 6, mbottom = 18;     /* menu coordinates */
int  mleft = 29, mright = 51;

char *menu_data[] = {            /* define menu text */
	"Read About ASM4QC",
	"Line Input Editor",
	"Move This Menu",
	"Beep",
	"Exit to DOS"
};

char *window_data[] = {          /* define demo info text */
	"",
	"            DEMO4QC -- Demo program for ASM4QC",
	"   Version 2.02, Copyright (c) 1989-90 SoftCircuits (tm)",
	"",
	"",
	"   Welcome to ASM4QC (Assembly routines for QuickC). These",
	"routines give your C programs high performance video",
	"functions plus much more. Note: If you're using a CGA",
	"monitor that doesn't snow, restart this program with the /q",
	"option for even faster video performance.",
	"",
	"   Because ASM4QC was written in assembly language, it was",
	"possible to optimize these routines for both speed and",
	"power. And it naturally follows that they are compact and",
	"will add hardly anything to overall program size.",
	"",
	"   With ASM4QC, you'll be able to cut the time it takes to",
	"code colorful video routines down to seconds. And you'll be",
	"able to code professional windowing routines in minutes.",
	"And the resulting routines will run at the fastest speed",
	"possible on your computer. Period!",
	"",
	""
};

struct vconfig vdata;            /* declare structure for getvconfig() */

/*
** removes menu and restores original screen
*/
void restore_screen(void)
{
	/* restore text under menu */
	restorewin(menubuffer,mtop,mbottom,mleft,mright);

	/* restore text under header bar */
	restorewin(headerbuffer,1,1,1,vdata.textcols);

	/* restore text under footer bar */
	restorewin(footerbuffer,vdata.textrows,vdata.textrows,1,vdata.textcols);

	setcurs(row,column);     /* set cursor to original position */
	restorecurs();           /* and un-hide it */

} /* restore_screen */

/*
** moves the menu around the screen
*/
void move_menu(void)
{
	int done = FALSE;

	/* change footer bar information */
	_color(status_bar);
	savewin(footerbuffer2,vdata.textrows,vdata.textrows,1,vdata.textcols);
	setcurs(vdata.textrows,1);
	repch(' ',vdata.textcols);
	setvoff(vdata.textrows,10);
	prints("Move menu: \x18  \x19  \x1B  \x1A");
	setvoff(vdata.textrows,45);
	prints("Return to normal mode: Esc");

	/* fill a buffer with an image of the menu */
	savewin(menubuffer2,mtop,mbottom,mleft,mright);

	while(!done) {
		switch(getkey()) {
			case UP_KEY :
				if(mtop > 2) {
					restorewin(menubuffer,mtop,mbottom,mleft,mright);
					mtop--; mbottom--;
					savewin(menubuffer,mtop,mbottom,mleft,mright);
					restorewin(menubuffer2,mtop,mbottom,mleft,mright);
				}
				break;
			case DOWN_KEY :
				if(mbottom < (int)(vdata.textrows - 1)) {
					restorewin(menubuffer,mtop,mbottom,mleft,mright);
					mtop++; mbottom++;
					savewin(menubuffer,mtop,mbottom,mleft,mright);
					restorewin(menubuffer2,mtop,mbottom,mleft,mright);
				}
				break;
			case LEFT_KEY :
				if(mleft > 1) {
					restorewin(menubuffer,mtop,mbottom,mleft,mright);
					mleft--; mright--;
					savewin(menubuffer,mtop,mbottom,mleft,mright);
					restorewin(menubuffer2,mtop,mbottom,mleft,mright);
				}
				break;
			case RIGHT_KEY :
				if(mright < (int)vdata.textcols) {
					restorewin(menubuffer,mtop,mbottom,mleft,mright);
					mleft++; mright++;
					savewin(menubuffer,mtop,mbottom,mleft,mright);
					restorewin(menubuffer2,mtop,mbottom,mleft,mright);
				}
				break;
			case ESCAPE_KEY :
				done = TRUE;
				break;
		} /* switch */
		kbdflush();      /* clear any keystrokes waiting in keyboard buffer */
	}

	/* restore footer bar */
	restorewin(footerbuffer2,vdata.textrows,vdata.textrows,1,vdata.textcols);

} /* move_menu */

/*
** sounds a series of tones
*/
void test_beep(void)
{
	beep(55,8);
	beep(110,2);
	beep(220,2);
	beep(440,2);
	beep(880,2);
	kbdflush();              /* flush keyboard buffer */

} /* test_beep */

/*
** test input line-editor
*/
void input_editor(void)
{
	/* save information under input window */
	savewin(editor_buffer,16,21,10,70);

	/* clear window area */
	_color(input_text);
	scroll(0,16,21,10,70);

	/* create window frame */
	_color(input_frame);
	frame(16,21,10,70);

	/* create help bar */
	setvoff(21,26); _color(input_title);
	printsa(" Press Escape to close window ");

	_color(input_text); setcurs(17,11); biosprintsa(">");
	kbdflush();              /* clear keyboard buffer */

	/* allow user input until escape is pressed */
	while(kbded(input_buffer,57) != -1) {
		/* advance cursor to next line */
		if (cursrow() > 19) {
			scroll(1,17,20,11,69);
			setcurs(cursrow(),11);
		}
		else
			setcurs(cursrow() + 1,11);
		biosprintsa(">");
	}

	/* close input window */
	restorewin(editor_buffer,16,21,10,70);

} /* input_editor */

/*
** fills info window
*/
void fill_window(void)
{
	int element, line;
	/* clear the window */
	scroll(0,TOP + 1, BOTTOM - 1, LEFT + 1, RIGHT - 1);

	/* write text to window */
	for(element = scroll_position, line = 0;line <= WINDOWHEIGHTH;line++) {
		setvoff(line + TOP + 1,11);
		prints(window_data[element++]);
	}
} /* fill_window */

/*
** moves to bottom of demo info text
*/
void goto_end(void)
{
	if(scroll_position != NUMOFLINES - WINDOWHEIGHTH) {
		scroll_position = NUMOFLINES - WINDOWHEIGHTH;
		fill_window();
	}
} /* goto_end */

/*
** moves to top of demo info text
*/
void goto_home(void)
{
	if(scroll_position != 0) {
		scroll_position = 0;
		fill_window();
	}
} /* goto_home */

/*
** moves demo info text one page down
*/
void page_down(void)
{
	if(scroll_position < (NUMOFLINES - WINDOWHEIGHTH)) {
		if((scroll_position+= (WINDOWHEIGHTH + 1)) >= NUMOFLINES - WINDOWHEIGHTH) {
			scroll_position = NUMOFLINES - WINDOWHEIGHTH;
		}
		fill_window();
	}
} /* page_down */

/*
** moves demo video text one page up
*/
void page_up(void)
{
	if(scroll_position > 0) {
		if((scroll_position-= (WINDOWHEIGHTH + 1)) < 0) {
			scroll_position = 0;
			}
		fill_window();
	}
} /* page_up */

/*
** moves demo video text one line down
*/
void scroll_down(void)
{
	if(scroll_position > 0) {
		scroll(-1, TOP + 1, BOTTOM - 1, LEFT + 1, RIGHT -1);
		setvoff(TOP + 1,11);
		prints(window_data[--scroll_position]);
	}
} /* scroll_down */

/*
** moves demo video text one line up
*/
void scroll_up(void)
{
	if(scroll_position < (NUMOFLINES - WINDOWHEIGHTH)) {
		scroll(1, TOP + 1, BOTTOM - 1, LEFT + 1, RIGHT - 1);
		setvoff(BOTTOM - 1,11);
		prints(window_data[++scroll_position + WINDOWHEIGHTH]);
	}
} /* scroll_up */

/*
** displays scrollable information about asm4qc
*/
void demoinfo(void)
{
	int done = FALSE;

	/* save information under window */
	savewin(windowbuffer,TOP,BOTTOM,LEFT,RIGHT);

	/* create window border */
	_color(info_frame);
	frame(TOP,BOTTOM,LEFT,RIGHT);

	/* write text to window */
	_color(info_text);
	fill_window();

	/* save old footer bar */
	savewin(footerbuffer2,vdata.textrows,vdata.textrows,1,vdata.textcols);

	/* create new footer bar */
	_color(status_bar);
	setcurs(vdata.textrows,1);
	repch(' ',vdata.textcols);
	setvoff(vdata.textrows,6);
	prints("Move: \30  \31  PgUp  PgDn  Home  End");
	setvoff(vdata.textrows,59);
	prints("Close window: Esc");

	_color(info_text);
	while(!done) {
		switch(getkey()) {                 /* get a key from user */
			case HOME_KEY:                  /* home key */
				goto_home();
				break;
			case UP_KEY:                    /* cursor up */
				scroll_down();
				break;
			case PGUP_KEY:                  /* page up */
				page_up();
				break;
			case END_KEY:                   /* end key */
				goto_end();
				break;
			case DOWN_KEY:                  /* cursor down */
				scroll_up();
				break;
			case PGDN_KEY:                  /* page down */
				page_down();
				break;
			case ESCAPE_KEY:                /* escape key */
				done = TRUE;
				break;
		}
	}

	/* close window and restore footer bar */
	restorewin(windowbuffer,TOP,BOTTOM,LEFT,RIGHT);
	restorewin(footerbuffer2,vdata.textrows,vdata.textrows,1,vdata.textcols);

} /* demoinfo */

/*
** changes the current menu selection and moves the hilight bar
*/
void move_selection(int change)
{
	setvoff(selection+mtop+3,mleft+2);  /* unhighilight current selection */
	printa(menu_text,19);
	selection+=change;                  /* calculate new selection */
	if(selection > 5)
		selection = 1;
	else if(selection < 1)
		selection = 5;
	setvoff(selection+mtop+3,mleft+2);  /* hilight new selection */
	printa(menu_hilite,19);

} /* move_selection */

/*
** this function makes up the main program loop
*/
void run(void)
{
	int done = FALSE;                 /* initialize boolean variable */

	while(!done) {
		switch(getkey()) {               /* get key from user */
			case RETURN_KEY :            /* if it's a carriage return */
				switch(selection) {        /*  carry out command based on */
					case 1:                 /*  current selection */
						demoinfo();
						break;
					case 2:
						input_editor();
						break;
					case 3:
						move_menu();
						break;
					case 4:
						test_beep();
						break;
					case 5:
						done = TRUE;
						break;
				}
				break;
			case UP_KEY:                 /* cursor up */
				move_selection(-1);
				break;
			case DOWN_KEY:               /* cursor down */
				move_selection(1);
				break;
			case ESCAPE_KEY:             /* escape key */
				done = TRUE;
				break;
		}
	}
} /* run */

/*
** creates the main menu
*/
void draw_menu(void)
{
	int line;

	/* save text under the menu */
	savewin(menubuffer,mtop,mbottom,mleft,mright);

	/* clear menu area */
	_color(menu_text); scroll(0,mtop,mbottom,mleft,mright);

	/* create menu border */
	_color(menu_frame); frame(mtop,mbottom,mleft,mright);

	/* create menu title */
	_color(menu_title); setvoff(6,33); printsa("   Main Menu   ");

	/* create help bar at the bottom of the menu */
	_color(status_bar);
	setvoff(17,30); printsa("  \x18-\x19-\x11\xC4\xD9 to select  ");
	setvoff(17,32); printa(foreback(BLACK,WHITE),1);
	setvoff(17,34); printa(foreback(BLACK,WHITE),1);
	setvoff(17,36); printa(foreback(BLACK,WHITE),3);

	/* write menu text */
	for (line = 10; line < 15; line++) {
		setvoff(line,32);
		prints(menu_data[line - 10]);
	}

	/* create menu hilite bar */
	setvoff(10,31); printa(menu_hilite,19);

} /* draw_menu */

/*
** creates the initial screen
*/
void draw_screen(void)
{
	row = cursrow(); column = curscol(); /* save cursor position */
	hidecurs();                          /* hide the text cursor */

	/* save text under header bar */
	savewin(headerbuffer,1,1,1,vdata.textcols);

	/* save text under footer bar */
	savewin(footerbuffer,vdata.textrows,vdata.textrows,1,vdata.textcols);

	/* create header bar */
	_color(status_bar);
	setcurs(1,1);            /* clear header area */
	repch(' ',vdata.textcols);
	setvoff(1,(vdata.textcols - strlen(HEADER_BAR)) / 2);
	prints(HEADER_BAR);      /* print header */

	/* create footer bar */
	setcurs(vdata.textrows,1); /* clear header area */
	repch(' ',vdata.textcols);
	setvoff(vdata.textrows,(vdata.textcols - strlen(FOOTER_BAR)) / 2);
	prints(FOOTER_BAR);      /* print footer */

	/* cause frame() to use double-line frame characters */
	setdoubleframe();

	/* create the menu */
	draw_menu();
} /* draw_screen */

/*
** initializes video variables
*/
void initialize_video(int argc, char *argv[])
{
	int count, force_blackwhite = FALSE, force_nosnowcheck = FALSE;

	/* check command line options */
	for(count = 1; count < argc; count++) {
		switch(argv[count][0]) {
			case '/' :
			case '-' :
				switch(toupper(argv[count][1])) {
					case 'Q':
						force_nosnowcheck = TRUE;
						break;
					case 'B':
						force_blackwhite = TRUE;
						break;
					}
				break;
			}
		}

	/* if initvideo() returned a non-zero value and the /b option */
	/* was not specified, assign attribute variables with colors  */
	/* values that work well on color systems                     */
	if(initvideo() && !force_blackwhite) {
		menu_text = foreback(GREEN,BLACK);
		menu_frame = foreback(CYAN,BLACK);
		menu_title = foreback(BOLD+WHITE,RED);
		menu_hilite = foreback(BOLD+WHITE,YELLOW);
		info_text = foreback(BOLD+CYAN,BLUE);
		info_frame = foreback(BOLD+WHITE,BLUE);
		input_text = foreback(BOLD+WHITE,RED);
		input_frame = foreback(BOLD+YELLOW,RED);
		input_title = foreback(RED,WHITE);
		status_bar = foreback(BOLD+WHITE,BLUE);
	}

	/* turn off snow checking if specified */
	if(force_nosnowcheck)
		setsnowcheck(OFF);

	/* retrieve display information */
	getvconfig(&vdata);

} /* initialize_video */

/*
** here is main
*/
void main(int argc,char *argv[])
{
	initialize_video(argc,argv);  /* initialize video variables */
	draw_screen();                /* create initial screen */
	run();                        /* run main program loop */
	restore_screen();             /* restore the screen */

} /* main */
