//                                    Copyright John V. Rodley   1994
//
// NE_EXE.C - Module to deal with reading and dumping the contents of
// an NE executable.
//
// Theory of operation:
//
// The exe is made up of three interesting parts:
//    the header
//    the tables
//    the physical pages
//
// The header is read in one shot and printed out by the DumpLXExeHeader
// function
//
// The tables are read by a bunch of Load...Table functions into global
// arrays of structures (described in ne_exe.h).
// All the table entry types are described by a struct somewhere in 
// ne_exe.h.  If a table entry can have multiple "shapes" the 
// struct is a superset of all the possible shapes.  The Load... 
// functions read the tables and deconstruct those entries into the more
// abstract structs contained in the global arrays.  The Dump... 
// functions run through the global arrays printing out the members and
// incorporate most of the intelligence about the meaning of various
// structure members.
//
// The physical pages are read by the LoadPages() function into a 
// global array and printed out in hex and ascii by the DumpLXPages() 
// function.
//
// The Load... functions merely read the individual tables into 
// the global arrays.
// The Dump... functions contain any logic about linking the data
// in tables together (like pinning fixup records to imported functions).

#include <stddef.h>     /* Multi-threads live in mt directory.    */
#include <ctype.h>
#include <malloc.h>
#include <share.h>
#include <stdio.h>
#include <errno.h>
#include <io.h>
#include <fcntl.h>
#include <stdlib.h>
#include <sys\types.h>
#include <sys\stat.h>
#include <string.h>
#include "types.h"
#include "exe.h"
#include "utils.h"

static unsigned short magic;  // magic bytes == "NE"

// The global NE executable header.
NE_EXE ne_exe;

// The Segment Table
struct SEG *SegmentList[MAX_SEGMENTS+1];

// Another version of Segment Table, locating segment in file
struct SEG_DESC *SegmentDescList[MAX_SEGMENTS+1];

// The physical segment images
char *Segments[MAX_SEGMENTS+1];

// The single global Fixup Record Table a'la LX
struct NEW_REL *RelocationList[MAX_RELOCATIONS+1];

// The Resident Name Table
struct RES_NAME *ResNameList[MAX_RESNAMES+1];

// The Module Reference Table
struct MOD_REF *ModRefList[MAX_MODREFS+1];

// The Imported Name Table
struct IMP_NAME *ImpNameList[MAX_IMPNAMES+1];

// The Entry Table
struct ENTRY *EntryList[MAX_ENTRIES+1];

// The Non-Resident Name Table
struct NONRES_NAME *NonResNameList[MAX_NONRESNAMES+1];

// The Resource Table
struct RESOURCE *ResourceList[MAX_RESOURCES+1];

// The list of per-segment Fixup Record Tables
RELOCATION_SEGMENT *RelSegmentList[MAX_SEGMENTS+1];

// Contains indexes into RelocationList where the relocations
// for each segment begin.  RelIndex[SegmentNum-1] = the index
// in RelocationList where the relocations for SegmentNum begin.
int RelIndex[MAX_RELOCATIONS+1];   // terminated by -1

// The current index into RelIndex
unsigned RelocationSeg[MAX_RELOCATIONS+1];

// Possible values for new_rel.target, a Fixup Record field
typedef struct {
   char *name;
   USHORT target;
   } TARGETTYPE;

TARGETTYPE TargetTypes[] = {
"16 bit segment   ", NE_TARG_16SEG,
"16 bit seg:offset", NE_TARG_16SEGOFS,
"16 bit offset", NE_TARG_16OFS,
"16 bit seg, 32 bit offset", NE_TARG_16SEG32OFS,
"32 bit offset    ", NE_TARG_32OFS
};

// Count of the Fixup Record Tables, always <= count of segments
// can be 0 or 1 for each segment.
unsigned RelSegCnt = 0;
static int RelCnt = 0;

// The current count of fixup records for the entire exe.
unsigned uRelIndex = 0;

// A temporary buffer to load stuff of indeterminate size into.
char *b;
#define CLEARB() memset(b,'\0',BSIZE)

// used by FollowChain to name the called function
char callname[256];  

// a cheap way to look at a two byte word as individual bytes.
union { short w; char b[2]; } word;

// All the intelligence about the structures of the exe tables is
// located in the Load... functions below.
static void LoadSegmentTable(void);
static void LoadEntryTable(void);
static void LoadImportedNameTable(void);
static void LoadResidentNameTable(void);
static void LoadNonResidentNameTable(void);
static void LoadModRefTable(void);
static void LoadAllFixups(void);

static int AddToImpTable( char *szString, unsigned uIndex, unsigned short offset );
static int FollowChainAt( char *buffer, unsigned uSize, unsigned offset );
static int NewModRefIndex(void);
static void LoadFixupsForSegment( int i );
static void LoadSegmentDescription( int segnum );

// AddToImpTable - Add the specified name, with index and offset to the
// Imported Name Table.  return 0 of OK, 1 if error.
static int AddToImpTable( char *szString, unsigned uIndex, unsigned short offset )
{
struct IMP_NAME *s;

if(( s = (struct IMP_NAME *)malloc( sizeof( struct IMP_NAME ))) == NULL )
   goto ErrorReturn;

if( strlen( szString ) >= 255 )
{
	printf( "BAD NAME!\n" );
	exit( 0 );
}
memmove( s->name, szString, strlen( szString ) + 1);
s->index = uIndex;
s->offset = offset;
if( AddStructToList((char *)s, (char **)ImpNameList, sizeof( struct IMP_NAME ), MAX_IMPNAMES ))
   goto ErrorReturn;
return( 0 );
ErrorReturn:
printf( "AddToImpTable ERROR" );
return( 1 );

}  /* AddToImpTable */


// FreeNETables - Free and clear all the dynamically allocated NE tables.
void FreeNETables()
{
   ClearList((char **)&SegmentList[0], sizeof( struct SEG *), MAX_SEGMENTS);
   ClearList((char **)&SegmentDescList[0], sizeof( struct SEG_DESC *), MAX_SEGMENTS);
   ClearList((char **)&RelocationList[0], sizeof( struct NEW_REL *), MAX_RELOCATIONS );
   ClearList((char **)&ResNameList[0], sizeof( struct RES_NAME *), MAX_RESNAMES );
   ClearList((char **)&ImpNameList[0], sizeof( struct IMP_NAME *), MAX_IMPNAMES );
   ClearList((char **)&NonResNameList[0], sizeof( struct NONRES_NAME *), MAX_NONRESNAMES );
   ClearList((char **)&ModRefList[0], sizeof( struct MOD_REF *), MAX_MODREFS );
   ClearList((char **)&EntryList[0], sizeof( struct ENTRY *), MAX_ENTRIES );
   ClearList((char **)&RelSegmentList[0], sizeof( RELOCATION_SEGMENT *), MAX_SEGMENTS );
}

// ClearAllLists - Zero out all the NE global arrays.
void ClearAllLists()
{
   // dynamically alloced structure arrays
    memset( &SegmentList[0], '\0', sizeof( struct SEG *)* MAX_SEGMENTS);
    memset( &SegmentDescList[0], '\0', sizeof( struct SEG_DESC *)* MAX_SEGMENTS);
    memset( &RelocationList[0], '\0', sizeof( struct NEW_REL *) * MAX_RELOCATIONS );
    memset( &ResNameList[0], '\0', sizeof( struct RES_NAME *) * MAX_RESNAMES );
    memset( &ImpNameList[0], '\0', sizeof( struct IMP_NAME *) * MAX_IMPNAMES );
    memset( &NonResNameList[0], '\0', sizeof( struct NONRES_NAME *) * MAX_NONRESNAMES );
    memset( &ModRefList[0], '\0', sizeof( struct MOD_REF *) * MAX_MODREFS );
    memset( &EntryList[0], '\0', sizeof( struct ENTRY *) * MAX_ENTRIES );
    memset( &RelSegmentList[0], '\0', sizeof( RELOCATION_SEGMENT *) * MAX_SEGMENTS );

   // non-alloced arrays
    memset( &RelIndex[0], '\0', sizeof( USHORT ) * MAX_RELOCATIONS );
    memset( &RelocationSeg[0], '\0', sizeof( USHORT ) * MAX_RELOCATIONS );
   RelIndex[0] = -1;
   RelSegCnt = 0;
}

// DumpNEExeHdr - Dump the NE executable header.
int DumpNEExeHdr()
{
printf( "\n\nxxx   SEGMENTED EXE header \n" );

    printf( "%40s  %8.8u  0x%4.4x\n", "magic", magic,  magic );
    printf( "%40s: %08lXH\n", "Checksum", ne_exe.crc );
    printf( "%40s: %u\n", "Version number", ne_exe.ver );
    printf( "%40s: %u\n", "Revision number", ne_exe.rev );
    printf( "%40s: %u\n", "Number of segments", ne_exe.cseg );
    printf( "%40s: %04X:%04X\n", "Stack address", ne_exe.ss, ne_exe.sp );
    printf( "%40s: %u\n", "Automatic data segment number", ne_exe.autodata );
    printf( "%40s: %04XH\n", "Initial heap allocation in auto data", ne_exe.heap );
    printf( "%40s: %04XH\n", "Initial stack allocation in auto data", ne_exe.stack );
    printf( "%40s: %04XH\n", "Flags", ne_exe.flags );
    printf( "%40s? %s\n", "Program or library",(ne_exe.flags & NENOTP )? "LIBRARY":"PROGRAM" );
    printf( "%40s? %s\n", "Errors in image", (ne_exe.flags &NEIERR )? "YES":"NO" );
    printf( "%40s? %s\n", "Floating point instructions", (ne_exe.flags & NEFLTP )? "YES": "NO" );
    printf( "%40s? %s\n", "80386 instructions", (ne_exe.flags & NEI386)?"YES":"NO" );
    printf( "%40s? %s\n", "80286 instructions", (ne_exe.flags & NEI286)?"YES":"NO" );
    printf( "%40s? %s\n", "8086 instructions", (ne_exe.flags & NEI086)?"YES":"NO" );
    printf( "%40s? %s\n", "Protected mode only", (ne_exe.flags & NEPROT)?"YES":"NO" );
    printf( "%40s? %s\n", "Per-process initialization", (ne_exe.flags & NEPPLI)?"YES":"NO" );
    printf( "%40s? %s\n", "Per-instance data", (ne_exe.flags & NEINST)?"YES":"NO" );
    printf( "%40s? %s\n", "Solo data", (ne_exe.flags & NESOLO)?"YES":"NO" );
    printf( "%40s? %s\n", "Window compatible", (ne_exe.flags & NEWCPT)?"YES":"NO" );
    printf( "%40s? %s\n", "Window api", (ne_exe.flags & NEWAPI)?"YES":"NO" );
    printf( "%40s? %s\n", "LFN support", (ne_exe.resv[1] & NELFNS)?"YES":"NO" );

    if( ne_exe.resv[0] == '\0' )
      {
      if(( ne_exe.resv[8] != '\0' ) || ( ne_exe.resv[9] != '\0' ))
         printf( "%40s? %s\n", "Operating system", "(probably) WINDOWS" );
      else
         printf( "%40s? %s\n", "Operating system", "(probably) OS/2" );
      }
    else
      printf( "%40s? %s\n", "Operating system", (ne_exe.resv[0] & NEWIND)?"(probably) WINDOWS":"(probably) OS/2" );

   printf( "%40s %4.4u 0x%4.4x\n", "Segment table", ne_exe.segtab, ne_exe.segtab );
   printf( "%40s %4.4u 0x%4.4x\n", "Resident name table", ne_exe.restab, ne_exe.restab );
   printf( "%40s %4.4u 0x%4.4x\n", "Module reference table" ,ne_exe.modtab, ne_exe.modtab );
   printf( "%40s  %8.8u  0x%4.4x\n", "# Module references", ne_exe.cmod,  ne_exe.cmod );
   printf( "%40s %4.4u 0x%4.4x\n", "Import name table" ,ne_exe.imptab, ne_exe.imptab );
   printf( "%40s %4.4u 0x%4.4x\n", "Entry table" ,ne_exe.enttab, ne_exe.enttab );
   printf( "%40s  %8.8u  0x%4.4x\n", "Entry table byte count" ,ne_exe.cbenttab,  ne_exe.cbenttab );
   printf( "%40s %luL 0x%8.8lx\n", "Non res name table" ,ne_exe.nrestab, ne_exe.nrestab );
   printf( "%40s  %8.8u  0x%4.4x\n", "Non-res name table byte count", ne_exe.cbnrestab,  ne_exe.cbnrestab );
   printf( "%40s %4.4u 0x%4.4x\n", "Resource table" ,ne_exe.rsrctab, ne_exe.rsrctab );
   printf( "%40s  %8.8u  0x%4.4x\n", "# Resources", ne_exe.cres,  ne_exe.cres );

   printf( "%40s  %4.4X:%4.4X\n", "Initial CS:IP", ne_exe.cs,  ne_exe.ip );
   printf( "%40s  %8.8u  0x%4.4x\n", "# Moveable entries", ne_exe.cmovent,  ne_exe.cmovent );
   printf( "%40s  %8.8u  0x%4.4x\n", "Sector alignment", ne_exe.align,  ne_exe.align );
   ftdump( ne_exe.resv, 10 );
   printf( "\n" );

   return( 0 );

}     /* DumpExeHdr */

// DumpNETables - Dump all the NE exe tables, link together any data
// that can be linked, such as the Fixup Record Table and the Imported
// Name Table ...
int DumpNETables()
{
unsigned loop;
struct  NEW_REL *pnew_rel;
char *ptr;
char *modname;
char *funcname;
int i, j, k;
char *lump;

if( !bVerbose )
    return( 0 );

printf( "\nRELOCATION LIST\n" );
printf( "Segment Number   Offset  Target          Module.Function/Ordinal \n" );
for( loop = 0; RelocationList[loop] != NULL; loop++ )
   {
   ptr = "Unknown";
   for( i = 0; i < sizeof( TargetTypes )/sizeof( TARGETTYPE ); i++ )
      {
      if( TargetTypes[i].target == RelocationList[loop]->target )
         {
         ptr = TargetTypes[i].name;
         break;
         }
      }
   modname = "UNKNOWN";
   pnew_rel =  RelocationList[loop];
   if( RelocationList[loop]->source == NE_DEST_THISEXE )
      {
      modname = "THIS_EXE";
       printf( "     %u           %4.4x    %s  %s.%4.4u\n",
       pnew_rel->module_num, pnew_rel->offset, ptr, modname, pnew_rel->ordinal );
      }
   else
      {
      for( j = 0; ModRefList[j] != NULL && j < MAX_MODREFS; j++ )
         if( pnew_rel->module_num == ModRefList[j]->uModNum )
            {
            for( k = 0; ImpNameList[k] != NULL && k < MAX_IMPNAMES; k++ )
               if( ImpNameList[k]->index == ModRefList[j]->index )
                  {
                  modname = ImpNameList[k]->name;
                  break;
                  }
            }
      if( RelocationList[loop]->source == NE_DEST_DLLBYORDINAL )
         {
          printf( "     %u           %4.4x    %s  %s.%4.4u\n",
          pnew_rel->module_num, pnew_rel->offset, ptr, modname, pnew_rel->ordinal );
         }
      else
         {
         funcname = "Unknown";
         for( k = 0; ImpNameList[k] != NULL && k < MAX_IMPNAMES; k++ )
            {
            // The offset into Importnametable is stored as ordinal for byname
            if( ImpNameList[k]->offset == RelocationList[loop]->ordinal )
               {
               funcname = ImpNameList[k]->name;
               break;
               }
            }
          printf( "     %u           %4.4x    %s  %s.%s\n",
          pnew_rel->module_num, pnew_rel->offset, ptr, modname, funcname );
         }
      }
   }

printf( "\nENTRY LIST\n" );
printf( "Segment  Offset  Flags   Ordinal\n" );
for( loop = 0; EntryList[loop] != NULL; loop++ )
   printf( "   %4.4u 0x%4.4x  0x%4.4x   %u\n", EntryList[loop]->seg, EntryList[loop]->offset, EntryList[loop]->flags, EntryList[loop]->ordinal );

printf( "\nRESIDENT NAME LIST\n" );
printf( "Ordinal  Name\n" );
for( loop = 0; ResNameList[loop] != NULL; loop++ )
   printf( "   %4.4u  %s\n", ResNameList[loop]->ordinal,  ResNameList[loop]->name );

printf( "\nNON-RESIDENT NAME LIST\n" );
printf( "Ordinal  Name\n" );
for( loop = 0; NonResNameList[loop] != NULL; loop++ )
   printf( "%   4.4u      %s\n", NonResNameList[loop]->ordinal, NonResNameList[loop]->name );

printf( "\nSEGMENT LIST\n" );
printf( " #  Sector  Frame Flags   MinAlloc  Type  PL\n" );
for( loop = 0; SegmentList[loop] != NULL; loop++ )
{
   printf( "%2.2u   %4.4x  %4.4x   %4.4x   %4.4x      %s  ", loop+1, SegmentList[loop]->ns_sector, SegmentList[loop]->ns_cbseg, 
     SegmentList[loop]->ns_flags, SegmentList[loop]->ns_minalloc,
           (SegmentList[loop]->ns_flags & NSTYPE)?"Data":"Code" );
   printf( "%2.2x  ", (SegmentList[loop]->ns_flags >> 10) & 0x03 );
   if( NSITER & SegmentList[loop]->ns_flags )
      printf( "(Iterated data) " );
   if( NSMOVE & SegmentList[loop]->ns_flags )
      printf( "(Moveable) " );
   if( NSPURE & SegmentList[loop]->ns_flags )
      printf( "(Pure) " );
   else
      printf( "(Impure) " );
   if( NSPRELOAD & SegmentList[loop]->ns_flags )
      printf( "(Preload) " );
   else
      printf( "(Load on call) " );
   if( NSTYPE & SegmentList[loop]->ns_flags )
      {
      if( NSEXRD & SegmentList[loop]->ns_flags )
         printf( "(Read only) " );
      }
   else
      {
      if( NSEXRD & SegmentList[loop]->ns_flags )
         printf( "(Execute only) " );
      }
   if( NSCONFORM & SegmentList[loop]->ns_flags )
      printf( "(Debugging info) " );
   if( NSDISCARD & SegmentList[loop]->ns_flags )
      printf( "(Discardable)" );
   if( NS32BIT & SegmentList[loop]->ns_flags )
      printf( "(32 bits) " );
   if( NSHUGE & SegmentList[loop]->ns_flags )
      printf( "(Huge chunk) " );
   printf( "\n" );
}

printf( "\nIMPORTED NAME LIST\n" );
printf( "Index  Name\n" );
for( loop = 0; ImpNameList[loop] != NULL; loop++ )
   printf( "%4.4u   %s\n", ImpNameList[loop]->index, ImpNameList[loop]->name );

printf( "\nMODULE REFERENCE LIST\n" );
printf( "Module Number    Index\n" );
for( loop = 0; ModRefList[loop] != NULL; loop++ )
   printf( "  %4.4u           %4.4u\n", ModRefList[loop]->uModNum, ModRefList[loop]->index );

printf( "\nRELOCATION SEGMENT LIST\n" );
printf( "Seg  File start      File end       Size      Number of records\n" );
for( loop = 0; RelSegmentList[loop] != NULL; loop++ )
   {
   printf( "%d    ", loop );
   if( !RelSegmentList[loop]->uSize )
      printf( "NO RELOCATIONS\n" );
   else
      printf( "%8.8ldL       %8.8ldL      %5.5u     %5.5u\n",
         RelSegmentList[loop]->lFileStart,
         RelSegmentList[loop]->lFileEnd,
         RelSegmentList[loop]->uSize,
         RelSegmentList[loop]->uNumRecs );
/* printf( "0x%8.8lxL   0x%8.8lxL    0x%4.4x     0x%4.4x\n",
      RelSegmentList[loop]->lFileStart,
      RelSegmentList[loop]->lFileEnd,
      RelSegmentList[loop]->uSize,
      RelSegmentList[loop]->uNumRecs ); */
   }

printf( "\nFOLLOWING RELOCATION CHAINS\n" );
for( loop = 0; SegmentList[loop] != NULL; loop++ )
   {
   if( !SegmentList[loop]->ns_flags & RELOCATION_AVAIL )
      continue;
   if( !SegmentList[loop]->ns_cbseg )
      continue;
   printf( "\tSegment %u\n", loop+1 );
   printf( "      Start Links   End              #\n" );
   if(( lump = malloc( SegmentList[loop]->ns_cbseg )) == NULL )
      {
      printf( "malloc(%d) error\n", SegmentList[loop]->ns_cbseg );
      continue;
      }                        
   fseek( fp, (long )SegmentList[loop]->ns_sector*512L, SEEK_SET );
   if( fread( (char *)lump, SegmentList[loop]->ns_cbseg, 1, fp ) != 1 )
      {  
      printf( "read failed\n" );
      free( lump );
      continue;
      }

   for( i = 0; RelocationList[i] != NULL && i < MAX_RELOCATIONS; i++ )
      {
      if( RelocationSeg[i] == loop+1 )
         {
         pnew_rel =  RelocationList[i];
         if( RelocationList[i]->source == NE_DEST_THISEXE )
            {
            modname = "THIS_EXE";
            sprintf( callname, "%s.%d.%4.4u", modname, pnew_rel->module_num, pnew_rel->ordinal );
            }
         else
            {
            for( j = 0; ModRefList[j] != NULL && j < MAX_MODREFS; j++ )
               if( pnew_rel->module_num == ModRefList[j]->uModNum )
                  {
                  for( k = 0; ImpNameList[k] != NULL && k < MAX_IMPNAMES; k++ )
                     if( ImpNameList[k]->index == ModRefList[j]->index )
                        {
                        modname = ImpNameList[k]->name;
                        break;
                        }
                  }

               if( RelocationList[i]->source == NE_DEST_DLLBYORDINAL )
                  {
                   sprintf( callname, "%s.%4.4u", modname, pnew_rel->ordinal );
                  }
               else
                  {
                  funcname = "Unknown";
                  for( k = 0; ImpNameList[k] != NULL && k < MAX_IMPNAMES; k++ )
                     {
                     // The offset into Importnametable is stored as ordinal for byname
                     if( ImpNameList[k]->offset == RelocationList[i]->ordinal )
                        {
                        funcname = ImpNameList[k]->name;
                        break;
                        }
                     }
                   sprintf( callname, "%s.%4.4u", modname, pnew_rel->ordinal );
                  }
               }
         FollowChainAt( lump, SegmentList[loop]->ns_cbseg, RelocationList[i]->offset );
         }
      }
   free( lump );
   }
   return( 0 );
}                 /* ExPrintLists */

// FollowChainAt - Follow the Fixup target chain at the specified offset
// in the specified (loaded) segment of the specified size.  The end of
// the target chain is denoted by 0xFFFF (CHAIN_END).  Print out the
// head of the chain, the links in between, and the end of the chain.
// Return the offset of the end of the chain.
static int FollowChainAt( char *buffer, unsigned uSize, unsigned offset )
{
   int loop;
   unsigned NewOffset, *pU, LastOffset;

   NewOffset = offset;
   LastOffset = offset;

   for( loop = 0; NewOffset < uSize; loop++ )
      {
      pU = (unsigned *)&buffer[NewOffset];
      if(( *pU == CHAIN_END ) || ( *pU == 0 ))
         {
         printf( "      0x%4.4x        0x%4.4x %4.4u      %3.3d -> %s\n", offset, NewOffset, NewOffset, loop+1, callname );
         if( *pU == 0 )
            printf( "      2-byte relocation not supported\n" );
         return( NewOffset );
         }
      else
         {
         if( NewOffset != offset )
            printf( "              0x%4.4x\n", NewOffset );
         }
      NewOffset = *pU;
      LastOffset = NewOffset;
      }

   printf( "FollowChainAt 0x%x %u OVERFLOW ERROR\n", offset, offset );
   return( offset );
}               /* FollowChainAt */
// NewModRefIndex - Return the byte offset within the Imported Name Table 
// of the current Imported Name.  Imported Names are referred to not by
// an item index, but by the byte offset, so we have to keep this value
// to link impnames to fixups.
static int NewModRefIndex()
{
   int loop, index;

   for( loop = 0, index = 1; ModRefList[loop] != NULL && (loop < MAX_MODREFS); loop++ )
      {
      index += 1;
      index += strlen( ImpNameList[loop]->name );
      }

return( index );
}              /* NewModRefIndex */

// Read_NEExe - Read the NE executable, loading all Tables
int Read_NEExe()
{
    RelCnt = 0;

	 // Zero out all the tables.
	 ClearAllLists();
		
    if(( b = malloc( BSIZE )) == NULL )
       goto notanexe;

    if( bVerbose )
       printf( "xxx   Reading NE executable header.\n" );

    if( fseek( fp, (long)mz_exe.lfanew, 0 ))
      {
      printf(  "Can't seek to %04xh\n", mz_exe.lfarlc );
      goto notanexe;
      }

   // read past the two magic bytes
    if( fread( (char *)&magic, sizeof( magic ), 1, fp ) != 1 )
      goto notanexe;

   // read in the NE header
    if( fread( (char *)&ne_exe, sizeof( NE_EXE ), 1, fp ) != 1 )
      goto notanexe;

	 LoadSegmentTable();
	 LoadAllFixups();

    if( ne_exe.imptab )
		LoadImportedNameTable();

    if( ne_exe.cmod )
		LoadModRefTable();

    if( ne_exe.restab )
		LoadResidentNameTable();

    if( ne_exe.nrestab )
		LoadNonResidentNameTable();

    if( ne_exe.cbenttab )
		LoadEntryTable();

	 if( b )
		 free( b );
	 b = NULL;
    return( VALID_NEWEXE );


notanexe:
	 if( b )
	  free( b );
	 b = NULL;
    return( INVALID_EXE );
}    /* ReadExe */


// LoadNonResidentNameTable - Load the non-resident name table, loading
// names until we hit a zero-length string (first byte of entry is 0).
// Name format is 1-byte length, then length bytes of ASCII string
// non-null terminated.
static void LoadNonResidentNameTable()
{
   struct NONRES_NAME NonResName;
	int i, j;

      if( fseek( fp, ne_exe.nrestab, 0 ))
         {
         printf(  "Can't seek\n" );
         goto notanexe;
         }

      while( i=fgetc( fp ))
         {
			CLEARB();
         if(( i == EOF ) || (fread( b, i, 1, fp ) != 1 ))
            {
	         printf(  "Can't read\n" );
   	      goto notanexe;
            }
        b[i] = '\0';

	      if( fread( (char *)&j, 2, 1, fp ) != 1 )
	         {
   	      printf(  "Can't getc\n" );
  	   	   goto notanexe;
         	}
         memset( &NonResName, '\0', sizeof( struct  NONRES_NAME ));
         NonResName.ordinal = j;
         memmove( NonResName.name,  b, strlen( b )+1);
         if( AddStructToList( (char *)&NonResName, (char **)NonResNameList, sizeof( struct NONRES_NAME ), MAX_NONRESNAMES ))
            goto notanexe;
         }
		return;
notanexe:
		printf( "LoadNonResidentNameTable error!\n" );
		return;
}

// LoadResidentNameTable - Load the resident name table, reading 
// names until we hit a zero-length string (first byte of entry is 0).
// Name format is 1-byte length, then length bytes of ASCII string
// non-null terminated.
static void LoadResidentNameTable()
{
   struct RES_NAME ResName;
	int i, j;

      if( fseek( fp, (long )(mz_exe.lfanew + ne_exe.restab), 0 ))
         {
         printf(  "Can't seek\n" );
         goto notanexe;
         }
      while( i=fgetc( fp ))
         {
			CLEARB();
         if(( i == EOF ) || (fread( b, i, 1, fp ) != 1 ))
            {
	         printf(  "Can't read\n" );
   	      goto notanexe;
            }
        b[i] = '\0';

	      if( fread( (char *)&j, 2, 1, fp ) != 1 )
	         {
   	      printf(  "Can't getc\n" );
  	   	   goto notanexe;
         	}
         memset( &ResName, '\0', sizeof( struct RES_NAME ));
         ResName.ordinal = j;
         memmove( ResName.name,  b, strlen( b )+1);
         if( AddStructToList( (char *)&ResName, (char **)ResNameList, sizeof( struct RES_NAME ), MAX_RESNAMES ))
            goto notanexe;
         }

		return;
notanexe:
		printf( "LoadResidentNameTable error!\n" );
		return;
}

// LoadModRefTable - Load the module reference table reading "cmod" 
// Module References.
static void LoadModRefTable()
{
   struct MOD_REF ModRef;
	unsigned i, j;

      if( fseek( fp, (long )(mz_exe.lfanew + ne_exe.modtab), 0 ))
         {
         printf(  "Can't seek\n" );
         goto notanexe;
         }

      for( i=1; i <= ne_exe.cmod; i++ )
         {
	      if( fread( (char *)&j, 2, 1, fp ) != 1 )
	         {
   	      printf(  "Can't getc\n" );
  	   	   goto notanexe;
         	}
         ModRef.index  = j;
         ModRef.uModNum = i;
         if( AddStructToList((char *)&ModRef, (char **)ModRefList,  sizeof( struct MOD_REF ), MAX_MODREFS ))
            goto notanexe;
         }
		return;
notanexe:
		printf( "LoadModRefTable error!\n" );
		return;
}


// LoadImportedNameTable - Load the imported name table, reading 
// names until we hit a zero-length string (first byte of entry is 0) or
// until we hit the entry table.
// Name format is 1-byte length, then length bytes of ASCII string, ]
// non-null terminated.
static void LoadImportedNameTable()
{
	unsigned i, j;
   unsigned short offset;

      if( fseek( fp, (long )(mz_exe.lfanew + ne_exe.imptab + 1), 0 ))
         {
         printf(  "Can't seek\n" );
         goto notanexe;
         }
      offset = 0;
      for( i=ne_exe.imptab+1; i < ne_exe.enttab; i += j+1 )
         {
         j = fgetc( fp );
         if( j==EOF )
            {
	         printf(  "Can't getc\n" );
   	      goto notanexe;
            }
         offset++;
         if( j == 0 )
      	   break;
			CLEARB();
	      if( fread( b, j, 1, fp ) != 1 )
	         {
   	      printf(  "Can't getc\n" );
  	   	   goto notanexe;
         	}
         
      	b[j] = '\0';
         AddToImpTable( b,  (i-ne_exe.imptab), offset );
         offset += j;
	      }
		return;
notanexe:
		printf( "LoadImportedNameTable error!\n" );
		return;
}

// LoadEntryTable - Read through the entry table until we've read
// cbenttab bytes.  Termination of the table is a problem.  Certain
// flavors of NE seem to terminate it different ways, hence the 
// hack "fences" below.  Could be that I've missed one of the entry
// types and that's causing us to get out of sync.  Decent doc might
// help here.
static void LoadEntryTable()
{
	int i, index;
   char *ptr;
   struct ENTRY_BUNDLE *peb; 
   struct ENTRY_FIXED *pf; 
   struct ENTRY_MOVEABLE *pm;
   struct ENTRY Entry;
   unsigned short ordinal = 0;
   unsigned char *p;

      if( fseek( fp, (long )(mz_exe.lfanew + ne_exe.enttab), 0 ))
         {
         printf(  "Can't seek\n" );
         goto notanexe;
         }

      if(( p = malloc( ne_exe.cbenttab )) == NULL )
         {
         printf(  "Can't malloc\n" );
         goto notanexe;
         }
     if( fread( p, ne_exe.cbenttab, 1, fp ) != 1 )
        {
      	printf(  "Can't read\n" );
  		   goto notanexe;
	     	}

   peb = (struct ENTRY_BUNDLE *)p;
   while( 1 )
      {
      // We missed the two documented NULL bytes, but I suppose this will do.
      if( peb == (struct ENTRY_BUNDLE *)&p[ne_exe.cbenttab] )
         break;

      // This is a hack, we should not need this fence.
      if( peb > (struct ENTRY_BUNDLE *)&p[ne_exe.cbenttab-2] )
         {
         printf( "0 overflowed entry table\n" );
         break;
         }

      // The two documented null bytes that end the entry table.
      if( peb->num == 0 && peb->type == 0 )
         break;

      // This is a hack, we should not need this fence.
      if( peb > (struct ENTRY_BUNDLE *)&p[ne_exe.cbenttab-sizeof( struct ENTRY_BUNDLE )] )
         {
         printf( "1 overflowed entry table\n" );
         break;
         }

      ptr = (char *)&peb->placeholder;
      index = 0;
      switch( peb->type )
         {
         // if null bundle, increment pointer and continue
         case EB_NULL_BUNDLE:
            index += 2;
            ordinal = (unsigned short )peb->num;
            break;
         case EB_MOVEABLE_SEGMENT:
            for( i = 0, index = 0; i < (int )peb->num; i++ )
               {
               ordinal++;
               pm = (struct ENTRY_MOVEABLE *)&ptr[index];
               // This is a hack, we should not need this fence.
               if( pm >= (struct ENTRY_MOVEABLE *)&p[ne_exe.cbenttab-sizeof( struct ENTRY_MOVEABLE )] )
                  {
                  printf( "tried to read past end, moveable\n" );
                  break;
                  }
               Entry.seg = pm->seg;
               Entry.offset = pm->offset;
               Entry.flags = pm->flags;
               Entry.ordinal = ordinal;
               if( AddStructToList((char *)&Entry, (char **)EntryList, sizeof( struct ENTRY ), MAX_ENTRIES ))  
                  goto outoftheloop;
               index += sizeof( struct ENTRY_MOVEABLE );
               }                        
            break;
         default:
            for( i = 0, index = 0; i < (int )peb->num; i++ )
               {
               ordinal++;
               pf = (struct ENTRY_FIXED *)&ptr[index];
               // This is a hack, we should not need this fence.
               if( pf >= (struct ENTRY_FIXED *)&p[ne_exe.cbenttab-sizeof(struct ENTRY_FIXED)] )
                  {
                  printf( "tried to read past end, fixed\n" );
                  break;
                  }
               Entry.seg = peb->type;
               Entry.offset = pf->offset;
               Entry.flags = pf->flags;
               Entry.ordinal = ordinal;
               if( AddStructToList((char *)&Entry, (char **)EntryList, sizeof( struct ENTRY ), MAX_ENTRIES ))  
                  goto outoftheloop;
               index += sizeof( struct ENTRY_FIXED );
               }                        
            break;
         }
      peb = (struct ENTRY_BUNDLE *)&ptr[index];
      }
outoftheloop:
   free( p );
		return;
notanexe:
   if( p )
      free( p );
	printf( "LoadEntryTable error!\n" );
	return;
}

// LoadSegmentTable - Load the segment table, reading until the segment
// count cbseg is exhausted.
static void LoadSegmentTable()
{
    unsigned i;
    for( i = 1; i <= ne_exe.cseg; i++ )
		LoadSegmentDescription( i );
   return;
}

// LoadSegmentDescription - Load the segment description of the 
// specified segment from the Segment Table.
static void LoadSegmentDescription( int segnum )
{
struct SEG seg;
struct SEG_DESC s;

if( fseek( fp, (long )(mz_exe.lfanew + ne_exe.segtab)
                   + (sizeof( struct SEG ) * (segnum-1)), 0 ))
  {
  printf(  "Can't seek\n" );
  goto notanexe;
  }

if( fread( (char *)&seg, sizeof( struct SEG ), 1, fp ) != 1 )
   {
   printf(  "Can't seek\n" );
   goto notanexe;
   }

 if( AddStructToList((char *)&seg, (char **)SegmentList, sizeof(struct SEG), MAX_SEGMENTS ))
      {
      printf( "LoadSegmentDescription error\n" );
      goto notanexe;
      }

// This struct just makes loading the physical seg a little easier later on.
   s.number = (unsigned )segnum;
   s.file_start = ((long )seg.ns_sector * 512L);
   s.file_end   = ((long )seg.ns_sector * 512L) + (long )seg.ns_cbseg;
   s.size       = seg.ns_cbseg;
 if( AddStructToList((char *)&s, (char **)SegmentDescList, sizeof( struct SEG_DESC), MAX_SEGMENTS ))
      {
      printf( "LoadSegmentDescription error 1\n" );
      goto notanexe;
      }
		return;
notanexe:
		printf( "LoadSegmentDescription error!\n" );
		return;
}

// LoadAllFixups - Go through the segment list and load all the fixups
// for every segment into RelocationList[].
static void LoadAllFixups()
{
	unsigned i;

    for( i = 1; i <= ne_exe.cseg; i++ )
      {
      if( bVerbose )
         printf( "xxx   Reading segment %d\n", i );
		LoadFixupsForSegment( i );
		} /* for cseg */
		return;
}

// Load all the fixup records from the specified segment into the 
// RelocationList array, incrementing uRelIndex.
static void LoadFixupsForSegment( int segnum )
{
struct NEW_REL new_rel;
RELOCATION_SEGMENT rel_seg;
unsigned j;
long lStart;   // The file offset of the Fixup Record Table for this seg.
struct SEG seg;
unsigned num_relocs;

rel_seg.lFileStart = 0L;
rel_seg.uNumRecs = 0;
rel_seg.uSize = 0;
rel_seg.lFileEnd = 0L;

if( fseek( fp, (long )(mz_exe.lfanew + ne_exe.segtab)
                  + (sizeof( struct SEG ) * (segnum-1)), 0 ))
 {
  printf(  "Can't seek\n" );
  goto notanexe;
 }

if( fread( (char *)&seg, sizeof( struct SEG ), 1, fp ) != 1 )
  {
  printf(  "Can't seek\n" );
  goto notanexe;
  }

  lStart = (long )seg.ns_sector;

 for( j = 0; j < ne_exe.align; j++ )
  lStart *= 2L ;
 
 if( bVerbose )
    printf( "     %ld  %u  0x%lx  %x  RELOCATION DATA? %s\n\n", lStart, seg.ns_cbseg, lStart, seg.ns_cbseg,
                   (seg.ns_flags & RELOCATION_AVAIL)?"YES":"NO" ); 

 if( !(seg.ns_flags & 0x0100))
    {
    if( AddStructToList((char *)&rel_seg, (char **)RelSegmentList, sizeof( RELOCATION_SEGMENT ), MAX_SEGMENTS ))
         printf( "Relocation segment error!\n" );
	 return;
    }

 if( fseek( fp, lStart + seg.ns_cbseg, 0 ))
   {
    printf(  "Can't seek\n" );
    goto notanexe;
   }

 rel_seg.lFileStart = ftell( fp );
 if( fread( (char *)&num_relocs, sizeof( unsigned ), 1, fp ) != 1 )
    {
    printf(  "Can't seek\n" );
    goto notanexe;
    }
 rel_seg.uNumRecs = num_relocs;
 rel_seg.uSize = ( num_relocs * sizeof( struct NEW_REL )) + sizeof( unsigned ); 
 rel_seg.lFileEnd = rel_seg.lFileStart + (long )(num_relocs * sizeof( struct NEW_REL ));
 if( bVerbose )
    printf( "adding rel_seg[%d] size %u\n", segnum, rel_seg.uSize );
 if( AddStructToList((char *)&rel_seg, (char **)RelSegmentList, sizeof( RELOCATION_SEGMENT ), MAX_SEGMENTS ))
       goto notanexe;

RelIndex[segnum-1] = RelCnt;
 for( j = 0; j < num_relocs; j++ )
  {
   if( fread( (char *)&new_rel, sizeof( struct NEW_REL ), 1, fp ) != 1 )
      {
      printf(  "Can't seek\n" );
      goto notanexe;
      }
     if( AddStructToList((char *)&new_rel, (char **)RelocationList, sizeof( struct NEW_REL ), MAX_RELOCATIONS ))
         goto notanexe;
     RelocationSeg[RelSegCnt++] = segnum;
      RelCnt++;
     uRelIndex++;
   }	/* for num_relocs */
		return;
notanexe:
		printf( "LoadFixupsForSegment error!\n" );
		return;
}
