{ This file creates the zoom-path file ZOOMPATH.INC }

program MakePath;
uses Crt;

const
  MaxPics   = 4;        { CONFIG: max num of picture gaps }
  MaxFrames = 500;      { CONFIG: max num of frames }
  ZOOM      = 0.98;     { CONFIG: zoom-in of 98% steps }

const
  Fudge = 1 shl 24;
const
  MaxWidth  = 320;      { screen dimensions }
  MaxHeight = 200;
  WinPosX   = 110;      { zoom-window coords }
  WinPosY   = 69;
  WinSizeX  = 100;
  WinSizeY  = 62;
const
  CenterX   = -1.0165;  { zoom-in center complex point }
  CenterY   = -0.3245;
  Magnif    = +2.00;    { initial zoom-in region }

type
  TFrame = record
    XStart,XStop,YStart,YStop,PicNo: integer;
  end;

  TPic = record
    X,Y,DX,DY: longint;
  end;

function Fix(R:real):longint;
begin
  Fix := Trunc(FUDGE*R);
end;

var
  PX,PY,PDX,PDY: real;                     { current pic complex region }
  X,Y,DX,DY: real;                         { mandelbrot complex region }
  Frames: Array [1..MaxFrames] of TFrame;  { frame structure array }
  Pics: Array [1..MaxPics] of TPic;
  NumPics,NumFrames: word;
  R,G,B,IncR,IncG,IncB,I: integer;
  NeedNewPic: boolean;
  PathFile: text;

begin
  {**Compute data}

  { set frame#1 mandelbrot set complex region }
  DX := Magnif;
  DY := 0.75*Magnif;
  X  := CenterX - 0.5*DX;
  Y  := CenterY - 0.5*DY;

  { reset parameters }
  NeedNewPic := true;
  NumPics := 0;
  NumFrames := 0;

  while true do begin
    { check if new picture gap is needed }
    if NeedNewPic then begin
       NeedNewPic := false;
       if NumPics >= MaxPics then Break;
       Inc(NumPics);
       Pics[NumPics].X := FIX(X);
       Pics[NumPics].Y := FIX(Y);
       Pics[NumPics].DX := FIX(DX);
       Pics[NumPics].DY := FIX(DY);
       PX  := X;
       PY  := Y;
       PDX := DX;
       PDY := DY;
    end;

    { store new frame data }
    if NumFrames >= MaxFrames then begin
      Sound(1000); Delay(100); NoSound; Break;
    end;
    Inc(NumFrames);
    with Frames[NumFrames] do begin
      XStart := Trunc((MaxWidth-1)*(X-PX)/PDX);
      XStop  := Trunc((MaxWidth-1)*(X+DX-PX)/PDX);
      YStart := Trunc((MaxHeight-1)*(Y-PY)/PDY);
      YStop  := Trunc((MaxHeight-1)*(Y+DY-PY)/PDY);
      PicNo  := NumPics;
      { dont'a allow coarse resolution }
      if (Succ(XStop - XStart) < WinSizeX) or (Succ(YStop - YStart) < WinSizeY) then begin
        NeedNewPic := true;
        Dec(NumFrames);
      end;
    end;

    { zoom-in complex region (X,Y,X+DX,Y+DY) by ZOOM factor. }
    X  := X + ((1-ZOOM)/2)*DX;
    Y  := Y + ((1-ZOOM)/2)*DY;
    DX := ZOOM*DX;
    DY := ZOOM*DY;
  end;


  {**Write PATH.INC File }
  Assign(PathFile,'ZOOMPATH.INC');
  ReWrite(PathFile);
  Writeln(PathFile,'; MandelZoom Path Data Include File.');
  Writeln(PathFile);
  Writeln(PathFile,'WINPOSX         equ  ',WinPosX);
  Writeln(PathFile,'WINPOSY         equ  ',WinPosY);
  Writeln(PathFile,'WINSIZEX        equ  ',WinSizeX);
  Writeln(PathFile,'WINSIZEY        equ  ',WinSizeY);
  Writeln(PathFile,'NUMPICS         equ  ',NumPics);
  Writeln(PathFile,'NUMFRAMES       equ  ',NumFrames);
  Writeln(PathFile);
  Writeln(PathFile,'PicTable label dword');
  for I := 1 to NumPics do
    with Pics[I] do
      Writeln(PathFile,'  dd  ',X,',',Y,',',DX,',',DY);
  Writeln(PathFile);
  Writeln(PathFile,'FrameTable label word');
  for I := 1 to NumFrames do begin
    if (I mod 3 = 1) then Write(PathFile,'  dw  ');
    with Frames[I] do
      Write(PathFile,XStart:3,',',YStart:3,',',XStop:3,',',YStop:3,',',Pred(PicNo));
    if (I mod 3 = 0) or (I = NumFrames) then Writeln(PathFile) else Write(PathFile,',  ');
  end;
  Writeln(PathFile);

  Writeln(PathFile,'; Default Color Palette.');
  Writeln(PathFile);
  Writeln(PathFile,'Palette label byte');
  r:= 0; g:=0; b:=0;
  incR := $0053;
  incG := $017A;
  incB := $0097;
  for i := 255 downto 0 do begin
      Writeln(PathFile,'  db ',hi(r):3,',',hi(g):3,',',hi(b):3);
      inc(R,IncR); inc(G,IncG); inc(B,IncB);
      if not (Hi(R) in [0..63]) then begin IncR:=-IncR;Inc(R,IncR); end;
      if not (Hi(G) in [0..63]) then begin IncG:=-IncG;Inc(G,IncG); end;
      if not (Hi(B) in [0..63]) then begin IncB:=-IncB;Inc(B,IncB); end;
  end;
  Writeln(PathFile);
  Writeln(PathFile,'; End Of File.');
  Close(PathFile);
end.
