/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       sdm.c
**     SYSTEM   NAME:       BEHOLDER
**     ORIGINAL AUTHOR(S):  Dirk Wisse
**     VERSION  NUMBER:     
**     CREATION DATE:       1991/3/20
**
** DESCRIPTION: Source Destination Matrix for The Beholder.
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision$
** WORKFILE:    $Workfile$
** LOGINFO:     $Log$
*************************************************************************/
#if ! defined(PRD)
static char _pvcs_hdr[] =
"$Header$";
#endif

#include <stdlib.h>
#include <stdio.h>
#include <bufm.h>
#include <string.h>
#include <dnpap.h>
#include <error.h>
#include "sdm.h"

#define SDM_ALLOC(Mem)                  IPBufGet(Mem)
#define SDM_FREE(Mem)                   IPBufFree(Mem)


/**************************************************************
** NAME:        SdmCreate   
** SYNOPSIS:    SDM_MATRIX *SdmCreate
**              (
**                  WORD AddrSize,  sizeof MAC address
**                  WORD HostSize,  sizeof Host data
**                  WORD ConnSize,  sizeof Connection data
**                  WORD HostCount, number of Hosts
**                  WORD ConnCount, number of Connections
**                  WORD HashCount, sizeof Hash table
**                  WORD HashOver   overflow Hash table
**              )
** DESCRIPTION: Creates a source destination matrix.
**              You can specify your own data structures for
**              a host and a connection. Also the maximum
**              number of hosts and connections can be
**              specified. It is recommended that size of
**              the hash table is 1.5 times the number of
**              hosts and overflow is set to 0.5 times
**              the hash table. A reasonable example is:
**                  HostCount = 1500
**                  ConnCount = 4000
**                  HashCount = 2000
**                  HashOver  = 1000
**              The size of HostCount*HostSize and ConnCount*
**              ConnSize may not exceed 64Kbyte.
**              Data is allocated in five fixed size tables:
**                  Hash table
**                  Host link table, sorted on address
**                  Connection link table, unsorted
**                  Host data table, data for hosts
**                  Connection data table, data for connection
**              The function returns a pointer that is used
**              to identify the matrix in functions that
**              manipulate it.
** RETURNS:     NULL --> error
**              else     Matrix
**************************************************************/
SDM_MATRIX *SdmCreate
            (
                WORD AddrSize,
                WORD HostSize,
                WORD ConnSize,
                WORD HostCount,
                WORD ConnCount,
                WORD HashCount,
                WORD HashOver
            )
{
    SDM_MATRIX   *Matrix;

    Matrix=SDM_ALLOC(sizeof(SDM_MATRIX));
    if (Matrix!=NULL)
    {
        Matrix->AddrSize=AddrSize;
        Matrix->HostSize=HostSize;
        Matrix->ConnSize=ConnSize;
        Matrix->HostCount=HostCount;
        Matrix->ConnCount=ConnCount;
        Matrix->HashCount=HashCount;
        Matrix->HashOver=HashOver;
        Matrix->HashLink=SDM_ALLOC(sizeof(SDM_HASH)*HashCount);
        Matrix->HostLink=SDM_ALLOC(sizeof(SDM_HOST)*HostCount);
        Matrix->HostData=SDM_ALLOC(HostSize*HostCount);
        Matrix->ConnLink=SDM_ALLOC(sizeof(SDM_CONN)*ConnCount);
        Matrix->ConnData=SDM_ALLOC(ConnSize*ConnCount);
    }
    if (Matrix->HashLink!=NULL &&
        Matrix->HostLink!=NULL &&
        Matrix->HostData!=NULL &&
        Matrix->ConnLink!=NULL &&
        Matrix->ConnData!=NULL)
    {
        SdmClear(Matrix);
    }
    else
    {
        SdmRemove(Matrix);
        Matrix=NULL;
    }
    return Matrix;
}

/**************************************************************
** NAME:        SdmRemove   
** SYNOPSIS:    void SdmRemove(SDM_MATRIX *Matrix)
** DESCRIPTION: Removes a matrix created with SdmCreate.
**              Frees all the allocated data.
** RETURNS:     void
**************************************************************/
void SdmRemove(SDM_MATRIX *Matrix)
{
    if (Matrix->HashLink!=NULL)
        SDM_FREE(Matrix->HashLink);
    if (Matrix->HostLink!=NULL)
        SDM_FREE(Matrix->HostLink);
    if (Matrix->HostData!=NULL)
        SDM_FREE(Matrix->HostData);
    if (Matrix->ConnLink!=NULL)
        SDM_FREE(Matrix->ConnLink);
    if (Matrix->ConnData!=NULL)
        SDM_FREE(Matrix->ConnData);
    SDM_FREE(Matrix);
}

/**************************************************************
** NAME:        SdmClear
** SYNOPSIS:    void SdmClear(SDM_MATRIX *Matrix)
** DESCRIPTION: Flushes all hosts and connections form matrix.
**              Matrix is logically reset but space is not
**              freed.
**              This function can be used to forget everything
**              and start all over.
** RETURNS:     void
**************************************************************/
void SdmClear(SDM_MATRIX *Matrix)
{
    int Ind;
    
    Matrix->HashFree=Matrix->HashOver;
    Matrix->HostFree=0;
    Matrix->HostUsed=-1;
    Matrix->ConnFree=0;
    if (Matrix->HashLink!=NULL)
    {
        for (Ind=0;Ind<(Matrix->HashCount-1);Ind++)
            (Matrix->HashLink)[Ind].Host=-1;
    }
    if (Matrix->HostLink!=NULL)
    {
        for (Ind=0;Ind<(Matrix->HostCount-1);Ind++)
            (Matrix->HostLink)[Ind].Conn=-1;
    }
    if (Matrix->ConnLink!=NULL)
    {
        for (Ind=0;Ind<(Matrix->ConnCount-1);Ind++)
            (Matrix->ConnLink)[Ind].Host=-1;
    }
}



/**************************************************************
** NAME:        SdmHash      
** SYNOPSIS:    int SdmHash(SDM_MATRIX *Matrix, BYTE *Addr)
** DESCRIPTION: Calculates hash value form MAC address.
** RETURNS:     hash-value
**************************************************************/
int SdmHash(SDM_MATRIX *Matrix, BYTE *Addr)
{
    int Ind;

    Ind = (((unsigned)Addr[3] << 8) | (unsigned)Addr[4]) % Matrix->HashOver;
    return(Ind);
}


/**************************************************************
** NAME:        SdmGetHostLink       
** SYNOPSIS:    int SdmGetHostLink
**                      (SDM_MATRIX *Matrix, BYTE *Addr)  
** DESCRIPTION: Finds index of host link for specified address.
**              Uses hash table to find host link structure.
** RETURNS:     -1 --> host not found
**              else   index of host link structure
**************************************************************/
int SdmGetHostLink(SDM_MATRIX *Matrix, BYTE *Addr)
{
    int HashInd, HostInd;

    HashInd=SdmHash(Matrix,Addr);
    while (HashInd>=0)
    {
        if ((HostInd=Matrix->HashLink[HashInd].Host)<0)
            return(-1);
        if (memcmp(Addr,Matrix->HostData+Matrix->HostSize*HostInd,Matrix->AddrSize)==0)
            return(HostInd);
        HashInd=Matrix->HashLink[HashInd].Next;
    }
    return(-1);
}

/**************************************************************
** NAME:        SdmNextHostLink
** SYNOPSIS:    int SdmNextHostLink
**                      (SDM_MATRIX *Matrix, BYTE *Addr)
** DESCRIPTION: Finds index of host following the specified. 
**              Finds index of host link structure with address
**              lexicographically following the one specified.
** RETURNS:     -1 --> host not found
**              else   index of host link structure
**************************************************************/
int SdmNextHostLink(SDM_MATRIX *Matrix, BYTE *Addr)
{
    int HostInd;

    HostInd=Matrix->HostUsed;
    while (HostInd>=0)
    {
        if (memcmp(Addr,Matrix->HostData+Matrix->HostSize*HostInd,Matrix->AddrSize)<0)
            return(HostInd);
        HostInd=Matrix->HostLink[HostInd].Next;
    }
    return(-1);
}

/**************************************************************
** NAME:        SdmCreateHostLink
** SYNOPSIS:    int SdmCreateHostLink(SDM_MATRIX *Matrix,
**                                              BYTE *Addr)    
** DESCRIPTION: Creates host link for specified address.
**              No check is made if a host link with the
**              specified address already exists.
** RETURNS:     -1  -->  host table full
**              -2  -->  hash table full
**              else     host index of created host link
**************************************************************/
int SdmCreateHostLink(SDM_MATRIX *Matrix, BYTE *Addr)
{
    int HostInd, NewInd, HashInd, *PtrInd;
    
    if (Matrix->HostFree>=Matrix->HostCount)
    {
        ERR_DEB(ERR_SDM,1,"Host table full");
        return(-1);
    }
    HostInd=Matrix->HostFree++;
    ERR_DEB(ERR_SDM,1,"New host index %u",HostInd);
    Matrix->HostLink[HostInd].Conn=-1;
    memcpy(Matrix->HostData+Matrix->HostSize*HostInd,Addr,Matrix->AddrSize);
    PtrInd=&(Matrix->HostUsed);
    while (*PtrInd>=0)
    {
        if (memcmp(Addr,Matrix->HostData+Matrix->HostSize*(*PtrInd),Matrix->AddrSize)<0)
            break;
        PtrInd=&(Matrix->HostLink[*PtrInd].Next);
    }
    Matrix->HostLink[HostInd].Next=*PtrInd;
    *PtrInd=HostInd;
    HashInd=SdmHash(Matrix,Addr);
    if (Matrix->HashLink[HashInd].Host==-1)
    {
        ERR_DEB(ERR_SDM,1,"New hash index %u",HashInd);
        Matrix->HashLink[HashInd].Host=HostInd;
        Matrix->HashLink[HashInd].Next=-1;
    }
    else
    {
        if (Matrix->HashFree>=Matrix->HashCount)
        {
            ERR_DEB(ERR_SDM,1,"Hash table full");
            return(-2);
        }
        NewInd=Matrix->HashFree++;
        ERR_DEB(ERR_SDM,1,"New hash index %u",NewInd);
        Matrix->HashLink[NewInd].Host=HostInd;
        Matrix->HashLink[NewInd].Next=Matrix->HashLink[HashInd].Next;
        Matrix->HashLink[HashInd].Next=NewInd;
    }
    return(HostInd);
}

/**************************************************************
** NAME:        SdmGetConnLink
** SYNOPSIS:    int SdmGetConnLink(SDM_MATRIX *Matrix,
**                                  int HostInd, BYTE *Addr)  
** DESCRIPTION: Finds index of connection link.
**              The connection is specified by giving
**              the index of the source host and the address
**              of the destination host.
**              Uses SdmGetHostLink to find index of
**              destination host.
** RETURNS:     -1 --> no connections in matrix
**              -2 --> index of destination host not found
**              -3 --> connection doesn't exist
**              else   connection index
**************************************************************/
int SdmGetConnLink(SDM_MATRIX *Matrix, int HostInd, BYTE *Addr)
{
    int ConnInd;
    
    ConnInd=Matrix->HostLink[HostInd].Conn;
    if (ConnInd<0)
        return(-1);
    HostInd=SdmGetHostLink(Matrix, Addr);
    if (HostInd<0)
        return(-2);
    while (ConnInd>=0)
    {
        if (Matrix->ConnLink[ConnInd].Host==HostInd)
            return(ConnInd);
        ConnInd=Matrix->ConnLink[ConnInd].Next;
    }
    return(-3);
}

/**************************************************************
** NAME:        SdmNextConnLink    
** SYNOPSIS:    int SdmNextConnLink(SDM_MATRIX *Matrix,
**                                    int HostInd, BYTE *Addr)
** DESCRIPTION: Finds index of next connection link.
**              Finds index of connection link structure
**              lexicographically following the one specified
**              by the index of the source host and the address
**              of the destination host.
** RETURNS:     -1  --> connection not found
**              else    connection index
**************************************************************/
int SdmNextConnLink(SDM_MATRIX *Matrix, int HostInd, BYTE *Addr)
{
    int ConnInd;
    
    ConnInd=Matrix->HostLink[HostInd].Conn;
    while (ConnInd>=0)
    {
        HostInd=Matrix->ConnLink[ConnInd].Host;
        if (memcmp(Addr,Matrix->HostData+Matrix->HostSize*HostInd,Matrix->AddrSize)<0)
            return(ConnInd);
        ConnInd=Matrix->ConnLink[ConnInd].Next;
    }
    return(-1);
}


/**************************************************************
** NAME:        SdmGetConnLinkByIndex    
** SYNOPSIS:    int SdmGetConnLinkByIndex(SDM_MATRIX *Matrix,
**                                      int SrcInd, int DstInd)
** DESCRIPTION: Finds index of connection link.
**              The connection link is specified by the indices
**              of the source and destination host.
** RETURNS:     -1 --> connection not found
**              else   connection index
**************************************************************/
int SdmGetConnLinkByIndex(SDM_MATRIX *Matrix, int SrcInd, int DstInd)
{
    int ConnInd;
    
    ConnInd=Matrix->HostLink[SrcInd].Conn;
    while (ConnInd>=0)
    {
        if (Matrix->ConnLink[ConnInd].Host==DstInd)
            break;
        ConnInd=Matrix->ConnLink[ConnInd].Next;
    }
    return(ConnInd);
}

/***************************************************************
** NAME:        SdmCreateConnLinkByIndex
** SYNOPSIS:    int SdmCreateConnLinkByIndex(SDM_MATRIX *Matrix,
**                                      int SrcInd, int DstInd)  
** DESCRIPTION: Creates connection link.
**              No check is made if a connection link with the
**              specified indices already exists.
** RETURNS:     -1 --> connection table full
**              else   connection index
***************************************************************/
int SdmCreateConnLinkByIndex(SDM_MATRIX *Matrix, int SrcInd, int DstInd)
{
    int *PtrInd, NewInd;
    BYTE *DstAddr;

    if (Matrix->ConnFree>=Matrix->ConnCount)
    {
        ERR_DEB(ERR_SDM,1,"Connection table full");
        return(-1);
    }
    NewInd=Matrix->ConnFree++;
    ERR_DEB(ERR_SDM,1,"New connection index %u",NewInd);
    Matrix->ConnLink[NewInd].Host=DstInd;
    DstAddr=Matrix->HostData+Matrix->HostSize*DstInd;
    PtrInd=&(Matrix->HostLink[SrcInd].Conn);
    while (*PtrInd>=0)
    {
        DstInd=Matrix->ConnLink[*PtrInd].Host;
        if (memcmp(DstAddr,Matrix->HostData+Matrix->HostSize*DstInd,Matrix->AddrSize)<0)
            break;
        PtrInd=&(Matrix->ConnLink[*PtrInd].Next);
    }
    Matrix->ConnLink[NewInd].Next=*PtrInd;
    *PtrInd=NewInd;
    return(NewInd);
}


/**************************************************************
** NAME:        SdmGetHost
** SYNOPSIS:    void *SdmGetHost(SDM_MATRIX *Matrix,BYTE *Addr)   
** DESCRIPTION: Finds host with specified address.
**              Use this function to get a pointer to the
**              host data structure.
**              If no host with specified address exists a null
**              pointer is returned.
** RETURNS:     NULL --> error
**              else     pointer to host
**************************************************************/
void *SdmGetHost(SDM_MATRIX *Matrix, BYTE *Addr)
{
    int HostInd;
    
    HostInd=SdmGetHostLink(Matrix,Addr);
    if (HostInd>=0)
        return(Matrix->HostData + Matrix->HostSize*HostInd);
    else
        return(NULL);
}

/**************************************************************
** NAME:        SdmFirstHost
** SYNOPSIS:    void *SdmFirstHost(SDM_MATRIX *Matrix)   
** DESCRIPTION: Finds lexicographicaly first host.
**              Finds the first host in the specified matrix.
**              The hosts are sorted lexicographicaly on
**              address.
** RETURNS:     NULL --> error
**              else     pointer to host
**************************************************************/
void *SdmFirstHost(SDM_MATRIX *Matrix)
{
    int HostInd;
    
    HostInd=Matrix->HostUsed;
    if (HostInd>=0)
        return(Matrix->HostData + Matrix->HostSize*HostInd);
    else
        return(NULL);
}

/**************************************************************
** NAME:        SdmNextHost
** SYNOPSIS:    void *SdmNextHost(SDM_MATRIX *Matrix,BYTE *Addr)   
** DESCRIPTION: Finds lexicographicaly next host.
**              Finds the host following the one specified.
**              The hosts are sorted lexicographicaly on
**              address.
** RETURNS:     NULL --> error
**              else     pointer to host
**************************************************************/
void *SdmNextHost(SDM_MATRIX *Matrix, BYTE *Addr)
{
    int HostInd;
   
    HostInd=SdmGetHostLink(Matrix,Addr);
    if (HostInd>=0)
        HostInd=Matrix->HostLink[HostInd].Next;
    else
        HostInd=SdmNextHostLink(Matrix,Addr);
    if (HostInd>=0)
        return(Matrix->HostData + Matrix->HostSize*HostInd);
    else
        return(NULL);
}

/**************************************************************
** NAME:        SdmCreateHost
** SYNOPSIS:    void *SdmCreateHost(SDM_MATRIX *Matrix,
**                          BYTE *Addr,int *New)   
** DESCRIPTION: Creates host with specified address.
**              If the host already exists no host is created
**              but the existing host is returned.
**                  New = 1 : new host created
**                  New = 0 : existing host returned
**              Use this function to get a pointer to a
**              host if you are not sure wether the host
**              already exists in the specified matrix.
** RETURNS:     NULL --> error
**              else     pointer to host
**************************************************************/
void *SdmCreateHost(SDM_MATRIX *Matrix, BYTE *Addr, int *New)
{
    int Ind;
         
    *New=0;
    if ((Ind=SdmGetHostLink(Matrix,Addr))<0)
    {
        *New=1;
        if ((Ind=SdmCreateHostLink(Matrix,Addr))<0)
            return(NULL);
    }        
    return(Matrix->HostData+Matrix->HostSize*Ind);
}
    


/**************************************************************
** NAME:        SdmGetConn
** SYNOPSIS:    SDM_CONN_DATA *SdmGetConn(SDM_MATRIX *Matrix,
**                          BYTE *SrcAddr, BYTE *DstAddr)  
** DESCRIPTION: Finds connection with specified addresses.
**              Use this function to get a pointer to the
**              connection data structure with specified
**              source and destination hosts.
**              If no connection with specified address exists
**              a null pointer is returned.
** RETURNS:     NULL --> error
**              else     pointer to connection
**************************************************************/
SDM_CONN_DATA *SdmGetConn(SDM_MATRIX *Matrix, BYTE *SrcAddr, BYTE *DstAddr)
{
    static SDM_CONN_DATA Data;
    int HostInd, ConnInd;
    
    HostInd=SdmGetHostLink(Matrix,SrcAddr);
    if (HostInd>=0)
        ConnInd=SdmGetConnLink(Matrix,HostInd,DstAddr);
    else
        ConnInd=-1;
    if (HostInd>=0 && ConnInd>=0)
    {
        Data.Conn=Matrix->ConnData + Matrix->ConnSize*ConnInd;
        Data.Src=Matrix->HostData + Matrix->HostSize*HostInd;
        HostInd=Matrix->ConnLink[ConnInd].Host;
        Data.Dst=Matrix->HostData + Matrix->HostSize*HostInd;
        return(&Data);
    }
    else
        return(NULL);
}

/**************************************************************
** NAME:        SdmFirstConn
** SYNOPSIS:    SDM_CONN_DATA *SdmFirstConn(SDM_MATRIX *Matrix)  
** DESCRIPTION: Finds first connection in the specified matrix.
**              The connections in a matrix are sorted
**              lexicographicaly first on source address then
**              on destination address.
** RETURNS:     NULL --> error
**              else     pointer to connection
**************************************************************/
SDM_CONN_DATA *SdmFirstConn(SDM_MATRIX *Matrix)
{
    static SDM_CONN_DATA Data;
    int HostInd, ConnInd;
    
    HostInd=Matrix->HostUsed;
    ConnInd=Matrix->HostLink[HostInd].Conn;
    while (HostInd>=0 && ConnInd<0)
    {
        HostInd=Matrix->HostLink[HostInd].Next;
        ConnInd=Matrix->HostLink[HostInd].Conn;
    }
    if (HostInd>=0 && ConnInd>=0)
    {
        Data.Conn=Matrix->ConnData + Matrix->ConnSize*ConnInd;
        Data.Src=Matrix->HostData + Matrix->HostSize*HostInd;
        HostInd=Matrix->ConnLink[ConnInd].Host;
        Data.Dst=Matrix->HostData + Matrix->HostSize*HostInd;
        return(&Data);
    }
    else
        return(NULL);
}

/**************************************************************
** NAME:        SdmNextConn
** SYNOPSIS:    SDM_CONN_DATA *SdmNextConn(SDM_MATRIX *Matrix,
**                          BYTE *SrcAddr, BYTE *DstAddr)
** DESCRIPTION: Finds next connection in the specified matrix.
**              The connections in a matrix are sorted
**              lexicographicaly first on source address then
**              on destination address.
** RETURNS:     NULL --> error
**              else     pointer to connection
**************************************************************/
SDM_CONN_DATA *SdmNextConn(SDM_MATRIX *Matrix, BYTE *SrcAddr, BYTE *DstAddr)
{
    static SDM_CONN_DATA Data;
    int HostInd, ConnInd;
    
    HostInd=SdmGetHostLink(Matrix,SrcAddr);
    if (HostInd>=0)
    {
        ConnInd=SdmGetConnLink(Matrix,HostInd,DstAddr);
        if (ConnInd>=0)
            ConnInd=Matrix->ConnLink[ConnInd].Next;
        else
            ConnInd=SdmNextConnLink(Matrix,HostInd,DstAddr);
    }
    else
    {
        HostInd=SdmNextHostLink(Matrix,SrcAddr);
        ConnInd=Matrix->HostLink[HostInd].Conn;
    }
    while (HostInd>=0 && ConnInd<0)
    {
        HostInd=Matrix->HostLink[HostInd].Next;
        ConnInd=Matrix->HostLink[HostInd].Conn;
    }
    if (HostInd>=0 && ConnInd>=0)
    {
        Data.Conn=Matrix->ConnData + Matrix->ConnSize*ConnInd;
        Data.Src=Matrix->HostData + Matrix->HostSize*HostInd;
        HostInd=Matrix->ConnLink[ConnInd].Host;
        Data.Dst=Matrix->HostData + Matrix->HostSize*HostInd;
        return(&Data);
    }
    else
        return(NULL);
}

/***************************************************************
** NAME:        SdmCreateConn
** SYNOPSIS:    SDM_CONN_DATA *SdmCreateConn(SDM_MATRIX *Matrix,
**                      BYTE *SrcAddr, BYTE *DstAddr, int *New)   
** DESCRIPTION: Creates connection with specified addresses.
**              First it is checked wether the source and
**              destination hosts exist. If not they are
**              created.
**              Then it is checked wether the connection itself
**              exists. If not it is also created.
**                  New bit 0 : new source host
**                  New bit 1 : new destination host
**                  New bit 2 : new connection
**              For example 6 = 2^2 + 2^1 means that a new
**              destination and a new connection is created.
**              Use this function to get a pointer to a
**              connection if you are not sure wether the
**              source host, the destination host and the
**              connection already exist in the specified
**              matrix.
** RETURNS:     NULL --> error
**              else     pointer to connection
***************************************************************/
SDM_CONN_DATA *SdmCreateConn(SDM_MATRIX *Matrix, BYTE *SrcAddr, BYTE *DstAddr, int *New)
{
    static SDM_CONN_DATA Data;
    int SrcInd, DstInd, ConnInd;
    
    *New=0;
    if ((SrcInd=SdmGetHostLink(Matrix,SrcAddr))<0)
    {
        *New|=1;
        if ((SrcInd=SdmCreateHostLink(Matrix,SrcAddr))<0)
            return(NULL);
    }
    if ((DstInd=SdmGetHostLink(Matrix,DstAddr))<0)
    {
        *New|=2;
        if ((DstInd=SdmCreateHostLink(Matrix,DstAddr))<0)
            return(NULL);
    }
    if ((ConnInd=SdmGetConnLinkByIndex(Matrix,SrcInd,DstInd))<0)
    {
        *New|=4;
        if ((ConnInd=SdmCreateConnLinkByIndex(Matrix,SrcInd,DstInd))<0)
            return(NULL);
    }
    Data.Src=Matrix->HostData + Matrix->HostSize*SrcInd;
    Data.Dst=Matrix->HostData + Matrix->HostSize*DstInd;
    Data.Conn=Matrix->ConnData + Matrix->ConnSize*ConnInd;
    return(&Data);
}
