/**************************************************************************************************
*
*		Title:	NETWORK.H
*		Copyright (c) December 1991, Ryu Consulting, 916/722-1939
*		Written by Rahner James
*
*		This file contains all the structure and header information for the Novell network
*		programming library.
*
**************************************************************************************************/

#define	_NETWORK_H_
#pragma	pack(1)

#ifndef HUGE
#define	FAR		far
#define	HUGE	huge
#endif

#ifdef __WATCOMC__
	#ifdef __386__
		#pragma library ("netw386.lib");
	#else
		#ifdef __SMALL__
			#pragma library ("netw286s.lib");
		#else
			#pragma library ("netw286l.lib");
		#endif
	#endif
	#pragma aux MS_C	"_*";
#endif


/**************************************************************************************************
*
*										Definitions
*
**************************************************************************************************/

/*
** In use flags for ECBs
*/

#define	IU_DONE			0x00	// Indicates that the ECB has been processed and completion code is valid
#define	IU_HOLDING		0xFB	// Send/receive has occurred, but ECB is in holding queue waiting to be processed
#define	IU_SCHEDULED	0xFD	// Event has been scheduled and is waiting for time-out
#define	IU_LISTENING	0xFE	// IPX is listening for incoming packets at a socket
#define	IU_SENDING		0xFF	// ECB is being sent or in the process of being sent
#define	IU_PROCESSING	0xFA	// ECB is being processed by IPX
#define	IU_WAITING		0xF8	// ECB has been put in a waiting queue because IPX was busy


/*
** Completion codes for transmission ECBs
** Note: completion codes are not valid until in use flag contains IU_DONE value
*/

#define	CC_SUCCESS		0x00	// The packet has been sent successfully
#define	CC_CANCELLED	0xFC	// Send request for this ECB has been cancelled, just like Twin Peaks
#define	CC_CHALLEDGED	0xFD	// Packet was malformed (eg. IPX packet <30 bytes, SPX<42 bytes, packet > 576 bytes)
#define	CC_UNDELIVERED	0xFE	// Packet couldn't be sent, no listener available
#define	CC_FAILURE		0xFF	// Network was physically unable to deliver the packet


/*
** Additional completion codes for reception ECBs
** Note: completion codes are not valid until in use flag contains IU_DONE value
*/

#define	CC_OVERFLOW		0xFD	// The size of the fragment received is either 0 or too large (>576 bytes)
#define	CC_CLOSED		0xFF	// The reception socket was not open


/*
** Additional completion codes for cancellation of an ECB
** Note: completion codes are not valid until in use flag contains IU_DONE value
*/

#define	CC_CANCEL_FAIL	0xF9	// The ECB could not be cancelled


/*
** xPX_OPEN flags
*/

#define	XPX_READ		1		// Sets up listening packets for the stream
#define	XPX_WRITE		2		// Sets up talking packets for the stream
#define	XPX_RW			3		// Sets up talking and listening packets for the stream
#define	XPX_WAIT		4		// xPX_OPEN() waits for a response before returning
#define	XPX_NOWAIT		0		// xPX_OPEN() returns immediately after stream is opened
#define	XPX_SPX			8		// xPX_OPEN() creates an SPX stream
#define	XPX_IPX			0		// xPX_OPEN() creates an IPX stream


/*
** Error return codes used by the xPX library
*/

#define	ERR_SOCKET_OPEN	-1		// Socket was already open
#define	ERR_NO_SOCKET	-1		// Socket does not exist
#define	ERR_SOCKET_FULL	-2		// Socket table is full
#define	ERR_NO_IPX_PATH	-6		// No path to destination node found
#define	ERR_MEMORY		-8		// Not enough memory to allocated for function
#define	ERR_IPX_TIMEOUT	-9		// Timeout occurred during a waiting period
#define	ERR_POINTER		-10		// Pointer contents are not supported by the function
#define	ERR_PARAMETER	-11		// One of the passed parameters is not kosher
#define	ERR_OPEN		-12		// Stream is already open
#define	ERR_NOT_OPEN	-13		// Stream is not open or invalid stream pointer
#define	ERR_NO_SUPPORT	-14		// Tried to read a WRITE ONLY or write a READ ONLY stream
#define	ERR_NO_NULLS	-15		// Tried to write with a NULL flag while _Asynch_Flag == !0


/*
** Control parameters
*/

#define	DEFAULT_TALKERS		5	// Number of talkers normally created with an open
#define	DEFAULT_LISTENERS	5	// Number of listeners normally created with an open
#define	NODE_ADDRESS_SIZE	6	// Number of bytes in a node address, either local or global
#define	MAX_PACKET_SIZE		576	// Maximum size of a packet including IPX/SPX header
#define	IPX_TYPE			4	// Packet type number for IPX packets
#define	SPX_TYPE			5	// Packet type number for SPX packets


/**************************************************************************************************
*
*										Structures
*
**************************************************************************************************/

typedef struct NET_ADDRESS_S
{
	unsigned long	network;					// Network number
	unsigned char	node[NODE_ADDRESS_SIZE];	// Physical node address
	unsigned short	socket;						// Socket number
} NET_ADDRESS_T;


typedef struct IPX_S
{
	unsigned short	checksum;		// Dummy checksum of the 30-byte packet header, filled by IPX
	unsigned short	length;			// Length of the complete IPX packet, filled by IPX
	unsigned char	control;		// Transport control byte for internetwork bridges, filled by IPX
	unsigned char	type;			// Packet type: IPX=4, SPX=5, defined by Xerox, set by application	*
	NET_ADDRESS_T	dest;			// Destination node address, set by application						*
	NET_ADDRESS_T	src;			// Source node address, set by IPX
} IPX_T;


typedef struct SPX_S
{
	unsigned short	checksum;		// Dummy checksum of the 30-byte packet header, filled by SPX
	unsigned short	length;			// Length of the complete IPX packet, filled by SPX
	unsigned char	control;		// Transport control byte for internetwork bridges, filled by SPX
	unsigned char	type;			// Packet type: IPX=4, SPX=5, defined by Xerox, set by application	*
	NET_ADDRESS_T	dest;			// Destination node address, set by application						*
	NET_ADDRESS_T	src;			// Source node address, set by SPX

	struct
	{
		unsigned char reserved1:4;	// Reserved for future use
		unsigned char end_of_msg:1;	// End of message
		unsigned char reserved2:1;	// Reserved for future use
		unsigned char ACK_req:1;	// Acknowledge required
		unsigned char sys_packet:1;	// System packet
	} con_control;					// Connection control structure, set by SPX
	unsigned char	datastream;		// Datastream type, 0x00-0xFD are ignored, set by application		*
	unsigned short	src_ID;			// Source connection ID, used by SPX, filled by SPX
	unsigned short	dest_ID;		// Destination connection ID, used by SPX, filled by SPX
	unsigned short	sequence;		// Sequence number to keep track of the packet sequence, filled by SPX
	unsigned short	ACK_number;		// Acknowledge number, used by SPX, filled by SPX
	unsigned short	allocation;		// Allocation number, used by SPX, filled by SPX
} SPX_T;


typedef struct ECB_HEADER_S
{
	struct ECB_S FAR *next;			// Used by IPX/SPX when the ECB is active, can be used when ECB is inactive
	void (FAR *function)(void);		// Called when packet sent/recd, called Event Service Routine					* s,r
	unsigned char	in_use;			// Flag used by IPX/SPX while it processes the ECB, !0 while in use
	unsigned char	completion_code;// Completion code that is valid only when in_use is set to 0
	unsigned short	socket;			// Transmission/reception socket to use for this ECB							* s,r
	unsigned char	IPX_work[4];	// Workspace used internally by IPX
	unsigned char	driver_work[12];// Workspace used internally by the IPX driver
	unsigned char	dest_address[NODE_ADDRESS_SIZE];	// Address of node to which packet is sent/received			* s,r
	unsigned short	fragment_count;						// Number of address/size IPX/SPX fragments that follow		* s,r
} ECB_HEADER_T;


typedef struct FRAGMENT_S
{
	void FAR *		ptr;			// -> buffer that data is sent from or received into
	unsigned short	size;			// Size of that buffer
} FRAGMENT_T;


typedef struct SPX_PACKET_S
{
	struct SPX_PACKET_T FAR *next;	// Used by IPX/SPX when the ECB is active
	void (FAR *function)(void);		// Called after packet sent/recd, called ESR, 			* s,r
	unsigned char	in_use;			// Set to !0 by IPX/SPX when packet is in use
	unsigned char	completion_code;// Set by IPX/SPX after packet task is complete
	unsigned short	socket;			// Socket to use for this ECB							* s,r
	unsigned char	IPX_work[4];	// Workspace used internally by IPX
	unsigned char	driver_work[12];// Workspace used internally by the IPX driver
	unsigned char	dest_address[NODE_ADDRESS_SIZE];	// Destination address for packet	* s
	unsigned short	fragment_count;	// Fragments descriptors that follow					* s,r
	SPX_T FAR *		hdr;			// -> IPX/SPX packet descriptor to use					* s,r
	unsigned short	size_hdr;		// Size of the IPX(30) or SPX(42) descriptor			* s,r
	void FAR *		buffer_ptr;		// -> data buffer to use for transmission/reception		* s,r
	unsigned short	buffer_size;	// Number of bytes in that buffer						* s,r

	struct SPX_PACKET_S FAR *next_allocated;	// -> next allocated packet structure		* set by SPX_OPEN
	struct SPX_PACKET_S FAR *next_sibling;		// -> next packet for stream and condition	* various functions
	struct XPX_STREAM_S FAR *parent;			// -> parent stream definition packet		* set by SPX_OPEN
	void FAR *		default_buffer;	// -> default buffer to use for IPX or SPX				* set by SPX_OPEN
	unsigned short	default_size;	// Size of the default buffer							* set by SPX_OPEN
	char FAR *		done_flag;		// Set by the ESR with the completion code				* set by SPX_READ,SPX_WRITE

	unsigned short	checksum;		// Dummy checksum of the 30-byte packet header
	unsigned short	length;			// Length of the complete IPX packet, filled by IPX/SPX
	unsigned char	control;		// Transport control byte for internetwork bridges
	unsigned char	type;			// Packet type: IPX(4)/SPX(5), defined by Xerox			* s
	NET_ADDRESS_T	dest;			// Destination node address								* s
	NET_ADDRESS_T	src;			// Source node address, filled by IPX/SPX

	struct
	{
		unsigned char reserved1:4;	// Reserved for future use
		unsigned char end_of_msg:1;	// End of message
		unsigned char reserved2:1;	// Reserved for future use
		unsigned char ACK_req:1;	// Acknowledge required
		unsigned char sys_packet:1;	// System packet
	} con_control;					// Connection control structure
	unsigned char	datastream;		// Datastream type, 0x00-0xFD can be used by application	* s
	unsigned short	src_ID;			// Source connection ID, used by SPX, filled by SPX
	unsigned short	dest_ID;		// Destination connection ID, used by SPX, filled by SPX
	unsigned short	sequence;		// Sequence number to keep track of the packet sequence
	unsigned short	ACK_number;		// Acknowledge number
	unsigned short	allocation;		// Allocation number
} SPX_PACKET_T;


typedef struct IPX_PACKET_S
{
	struct IPX_PACKET_T FAR *next;	// Used by IPX/SPX when the ECB is active
	void (FAR *function)(void);		// Called after packet sent/recd, called ESR, 			* s,r
	unsigned char	in_use;			// Set to !0 by IPX/SPX when packet is in use
	unsigned char	completion_code;// Set by IPX/SPX after packet task is complete
	unsigned short	socket;			// Socket to use for this ECB							* s,r
	unsigned char	IPX_work[4];	// Workspace used internally by IPX
	unsigned char	driver_work[12];// Workspace used internally by the IPX driver
	unsigned char	dest_address[NODE_ADDRESS_SIZE];	// Destination address for packet	* s
	unsigned short	fragment_count;	// Fragments descriptors that follow					* s,r
	IPX_T FAR *		hdr;			// -> IPX/SPX packet descriptor to use					* s,r
	unsigned short	size_hdr;		// Size of the IPX(30) or SPX(42) descriptor			* s,r
	void FAR *		buffer_ptr;		// -> data buffer to use for transmission/reception		* s,r
	unsigned short	buffer_size;	// Number of bytes in that buffer						* s,r

	struct IPX_PACKET_S FAR *next_allocated;	// -> next allocated packet structure		* set by IPX_OPEN
	struct IPX_PACKET_S FAR *next_sibling;		// -> next packet for stream and condition	* various functions
	struct XPX_STREAM_S FAR *parent;			// -> parent stream definition packet		* set by IPX_OPEN
	void FAR *		default_buffer;	// -> default buffer to use for IPX or SPX				* set by IPX_OPEN
	unsigned short	default_size;	// Size of the default buffer							* set by IPX_OPEN
	char FAR *		done_flag;		// Set by the ESR with the completion code				* set by IPX_READ,IPX_WRITE

	unsigned short	checksum;		// Dummy checksum of the 30-byte packet header
	unsigned short	length;			// Length of the complete IPX packet, filled by IPX/SPX
	unsigned char	control;		// Transport control byte for internetwork bridges
	unsigned char	type;			// Packet type: IPX(4)/SPX(5), defined by Xerox			* s
	NET_ADDRESS_T	dest;			// Destination node address								* s
	NET_ADDRESS_T	src;			// Source node address, filled by IPX/SPX
} IPX_PACKET_T;


typedef struct XPX_STREAM_S
{
	struct XPX_STREAM_S FAR *next;		// -> next stream structure that has been opened
	SPX_PACKET_T FAR *first_allocated;	// -> first allocated packet for this handle
	SPX_PACKET_T FAR *last_allocated;	// -> last allocated packet for this handle
	SPX_PACKET_T FAR *first_unread;		// -> first unread packet
	SPX_PACKET_T FAR *last_unread;		// -> last unread packet in the list
	SPX_PACKET_T FAR *first_free;		// -> first packet available for talking
	SPX_PACKET_T FAR *first_error;		// -> first packet that encountered an error
	SPX_PACKET_T FAR *last_error;		// -> last packet that encountered an error

	NET_ADDRESS_T dest;						// -> destination network, node, and socket
	char local_target[NODE_ADDRESS_SIZE];	// Node address of the local target for the destination
	unsigned short connection_ID;			// Connection ID used for SPX
	int	total_talkers;						// Number of talkers allocated to this stream
	int	total_listeners;					// Number of listeners allocated to this stream
	int	unread_count;						// Number of packets unread by application
	int free_count;							// Number of free packets ready for tranmitting
	int	maximum_unread;						// Maximum number of unread packets that have occurred so far
	int	error_count;						// Number of unprocessed error packets

	unsigned long	total_transmissions;	// Number of transmissions performed by this stream
	unsigned long	total_receptions;		// Number of receptions performed by this stream
	unsigned long	total_errors;			// Number of errors encountered by this stream

	int				open_flags;				// Flags used when this stream was opened
} XPX_STREAM_T;


typedef struct ORPHAN_S
{
	NET_ADDRESS_T	src;					// Source address of the orphaned packet
	unsigned short	length;					// Number of bytes in the buffer that follows
	unsigned char	buffer[1];				// Start of the data buffer
} ORPHAN_T;


/**************************************************************************************************
*
*									Global Data
*
**************************************************************************************************/

extern NET_ADDRESS_T	_Our_Address;		// Full address into which the application will receive transmissions
extern char	_Socket_Life;					// 0 if all sockets are closed at application exit, 0xFF if only closed explicitly

extern unsigned short	_SPX_Version;
extern unsigned short	_SPX_Max_Connections;
extern unsigned short	_SPX_Available_Connections;
extern unsigned char	_SPX_Retry_Count;
extern unsigned char	_SPX_Bowser_Flag;

#ifndef _IPX_OPEN_C_
extern int _Default_Talkers;				// Default number of talkers created when a stream is opened
extern int _Default_Listeners;				// Default number of listeners created when a stream is opened
extern void (FAR *_Default_Talk_ESR)();		// -> default talking ESR
extern void (FAR *_Default_Listen_ESR)();	// -> default listening ESR
extern XPX_STREAM_T FAR *_First_Stream;		// -> first stream structure allocated by this application
extern SPX_PACKET_T FAR *_First_Nomatch;	// -> first received packet that does not match a stream address
extern SPX_PACKET_T FAR *_Last_Nomatch;		// -> last received packet that does not match a stream address
extern int _Total_Nomatchs;					// Count of the packets of nonmatched packets
extern int _Total_Open_Streams;				// Number of open streams
extern int _Wait_Timeout;					// Number of system ticks to wait for a response
extern char	_Asynch_Flag;					// Set to !0 if all writes will be asynchronous
extern char	_Ignore_Nomatch;				// Set to 0 if non matching packets should be kept
#endif


/**************************************************************************************************
*
*									Function Prototypes
*
**************************************************************************************************/

extern void atoaddr( char FAR *, char FAR * );
extern void addrtoa( char FAR *, char FAR * );
extern int broadcast_to_console( char * );
extern int close_semaphore( long );
extern int end_of_job( int );
extern int farcmpw( void FAR *, void FAR *, unsigned int );
extern void FAR *farmove( void FAR *, void FAR *, unsigned int );
extern void FAR *farset( void FAR *, char, unsigned int );
extern int get_semaphore_count( long );
extern int get_semaphore_value( long );
extern int isbroadcast( char FAR * );
extern int ipx_add_listener( ECB_HEADER_T FAR * );
extern int ipx_cancel_ecb( ECB_HEADER_T FAR * );
extern int ipx_close_socket( unsigned short );
extern int xpx_init( unsigned short );
extern int ipx_open_socket( unsigned short FAR * );
extern int ipx_send_packet( ECB_HEADER_T FAR * );
extern int ipx_target( char FAR *, NET_ADDRESS_T FAR * );
extern int open_semaphore( char *, char, char *, long * );
extern int signal_semaphore( long );
extern int spx_add_listener( ECB_HEADER_T FAR * );
extern int spx_establish_connection( ECB_HEADER_T FAR *, unsigned short FAR * );
extern int spx_send_packet( ECB_HEADER_T FAR * );
extern int wait_semaphore( long, unsigned short );

extern void FAR talk_esr( void );
extern void FAR listen_esr( void );

#define	xpx_error_status(ptr)	ptr->error_count
#define	xpx_orphan_status()		_Total_Nomatchs
#define	xpx_read_status(ptr)	ptr->unread_count
#define	xpx_write_status(ptr)	ptr->free_count


#ifndef _IPX_OPEN_C_
extern int xpx_close( XPX_STREAM_T FAR * );
extern int xpx_open( NET_ADDRESS_T *, int, XPX_STREAM_T FAR ** );
#endif

#ifndef _IPX_ORPH_C_
extern ORPHAN_T FAR *ipx_read_orphan( ORPHAN_T FAR * );
#endif

#ifndef _IPX_READ_C_
extern int xpx_read( XPX_STREAM_T FAR *, void FAR *, int );
#endif

#ifndef _IPX_WRIT_C_
extern int xpx_write( XPX_STREAM_T FAR *, void FAR *, int, char FAR * );
#endif

#ifdef __WATCOMC__
	#ifdef __386__
	#else
		#if __SMALL__
			#pragma aux broadcast_to_console "_*" parm [si];
		#else
			#pragma aux broadcast_to_console "_*" parm [cx si];
		#endif

		#pragma aux end_of_job "_*" parm [bx];
		#pragma aux farset "_*" parm [es di] [al] [cx] value [es bx];

	#endif
#endif

