{************************************************}
{                                                }
{   Turbo Pascal for Windows                     }
{   Communications Demo Program                  }
{   Copyright (c) 1991 by Borland International  }
{                                                }
{   by David Wilhelm                             }
{************************************************}

Program ModemX;

{$R ModemX}

uses WinTypes, WinProcs, Objects, ODialogs, OWindows, Strings, WinDos, StdDlgs;

type
  TEditLine = array[0..50] of Char;
const
  cmDial      = 201;
  cmConfigure = 202;
  cmCapture   = 203;
  cmXModem    = 204;
  idEdit      = 100;
  idDialStart = 101;
  idPhoneNum  = 102;
  idXModBlock = 100;
  idXModError = 101;
  id1200      = 101;
  id2400      = 102;
  id4800      = 103;
  id9600      = 104;
  idOdd       = 105;
  idEven      = 106;
  idNone      = 107;
  idComm1     = 108;
  idComm2     = 109;
  id1Stop     = 110;
  id2Stop     = 111;
  id7Data     = 112;
  id8Data     = 113;

  LineWidth   = 80;  { Width of each line displayed.                 }
  LineHeight  = 60;  { Number of line that are held in memory.       }

  Comm  : Char = '1';
  Baud  : Word = 24;
  Parity: Char = 'n';
  Stop  : Char = '1';
  Data  : Char = '8';

  DialStart: TEditLine = 'ATDT';
  PhoneNumber: TEditLine = '';

  SOH = #1;  { Start Of Header }
  EOT = #4;  { End Of Transmission }
  ACK = #6;  { Acknowledge (positive) }
  NAK = #21; { Negative Acknowledge }
  CAN = #24; { Cancel }
  XModemBlockSize = 132;

type
  TApp = object(TApplication)
    procedure Idle; virtual;
    procedure InitMainWindow; virtual;
    procedure MessageLoop; virtual;
  end;

  PCommWindow = ^TCommWindow;

  PDownLoad = ^TDownLoad;
  TDownLoad = object(TObject)
    F: File of Char;
    Count: Integer;
    Parent: PCommWindow;
    constructor Init(AParent: PCommWindow);
    destructor Done; virtual;
    function Echo: Boolean; virtual;
    procedure ReceiveChar(C: Char); virtual;
  end;

  PAscii = ^TAscii;
  TAscii = object(TDownLoad)
    function Echo: Boolean; virtual;
    procedure ReceiveChar(C: Char); virtual;
  end;

  PXModem = ^TXModem;
  TXModem = object(TDownLoad)
    NumberBlocks, NumberErrors: Integer;
    Block: array[1..XModemBlockSize] of Char;
    Status: PDialog;
    constructor Init(AParent: PCommWindow);
    destructor Done; virtual;
    procedure ReceiveChar(C: Char); virtual;
    procedure ShowStats; virtual;
  end;

  PBuffer = ^TBuffer;
  TBuffer = object(TCollection)
    Pos: Integer;
    constructor Init(AParent: PWindow);
    procedure FreeItem(Item: Pointer); virtual;
    function PutChar(C: Char): Boolean;
  end;

  TCommWindow = object(TWindow)
    Cid: Integer;
    Buffer: PBuffer;
    DownLoad: PDownLoad;
    FontRec: TLogFont;
    CharHeight: Integer;
    constructor Init(AParent: PWindowsObject; ATitle: PChar);
    destructor Done; virtual;
    procedure Capture(var Message: TMessage);
      virtual cm_First + cmCapture;
    procedure Configure(var Message: TMessage);
      virtual cm_First + cmConfigure;
    procedure Dial(var Message: TMessage);
      virtual cm_First + cmDial;
    procedure Error(E: Integer; C: PChar);
    procedure GetCommPort;
    procedure Paint(PaintDC: HDC; var PaintInfo: TPaintStruct); virtual;
    procedure ReadChar; virtual;
    procedure SendChar(C: Char);
    procedure SetConfigure;
    procedure SetHeight;
    procedure SetUpWindow; virtual;
    procedure wmChar(var Message: TMessage);
      virtual wm_Char;
    procedure wmSize(var Message: TMessage);
      virtual wm_Size;
    procedure WriteChar;
    procedure XModem(var Message: TMessage);
      virtual cm_First + cmXModem;
  end;

var
  App: TApp;


{ TDownLoad }
constructor TDownLoad.Init(AParent: PCommWindow);
var
  FileName: array[0..200] of Char;
begin
  TObject.Init;
  Parent := AParent;
  Count := 0;
  if Application^.ExecDialog(New(PFileDialog,
    Init(Parent, PChar(sd_FileOpen),
    StrCopy(FileName, '*.*')))) = id_Ok then
    begin
      Assign(F, FileName);
      {$I-}
      ReWrite(F);
      {$I+}
      if IOResult <> 0 then
      begin
        MessageBox(GetFocus, 'Error in Opening File', 'Error', mb_Ok);
        Fail;
      end;
    end
  else
    Fail;
end;

destructor TDownLoad.Done;
begin
  if TFileRec(F).Mode <> fmClosed then
    Close(F);
  TObject.Done;
end;

function TDownLoad.Echo: Boolean;
begin
  Echo := False;
end;

procedure TDownLoad.ReceiveChar(C: Char);
begin
end;

{ TAscii }
function TAscii.Echo: Boolean;
begin
  Echo := True;
end;

procedure TAscii.ReceiveChar(C: Char);
begin
  Write(F, C);
end;

{ TXModem }
constructor TXModem.Init(AParent: PCommWindow);
begin
  TDownLoad.Init(AParent);
  Parent^.SendChar(Nak);
  NumberErrors := 0;
  NumberBlocks := 0;
  Status := New(PDialog, Init(Parent, 'XModem'));
  Application^.MakeWindow(Status);
  Status^.Show(sw_ShowNormal);
end;

destructor TXModem.Done;
begin
  Dispose(Status, Done);
  TDownLoad.Done;
end;

procedure TXModem.ReceiveChar(C: Char);
var
  Sum: Byte;
  I: Integer;
  C1: Char;
  S: String;
begin
  Inc(Count);
  if (Count = 1) then
  begin
    if C = EOT then
      Parent^.SendChar(Ack);
    if (C = CAN) or (C = EOT) then
    begin
      Parent^.DownLoad := Nil;
      Dispose(PXModem(@Self), Done);
      exit;
    end;
  end;

  if Count < XModemBlockSize then
    Block[Count] := C
  else
  begin
    Block[Count] := C;
    Count := 0;
    Sum := 0;
    for I := 4 to 131 do
      Inc(Sum, Ord(Block[I]));
    if Sum <> Ord(Block[XModemBlockSize]) then
    begin
      Inc(NumberErrors);
      ShowStats;
      Parent^.SendChar(Nak)
    end
    else
    begin
      for I := 4 to 131 do
        Write(F, Block[I]);
      Inc(NumberBlocks);
      ShowStats;
      Parent^.SendChar(Ack);
    end;
  end;
end;

procedure TXModem.ShowStats;
var
  S: array[0..10] of Char;
begin
  Str(NumberBlocks, S);
  SetDlgItemText(Status^.HWindow, idXModBlock, S);
  Str(Error, S);
  SetDlgItemText(Status^.HWindow, idXModError, S);
end;


{ TBuffer }
constructor TBuffer.Init(AParent: PWindow);
var
  P: PChar;
  I: Integer;
begin
  TCollection.Init(LineHeight + 1, 10);
  GetMem(P, LineWidth + 1);
  P[0] := #0;
  Pos := 0;
  Insert(P);
  for I := 1 to LineHeight do
  begin
    GetMem(P, LineWidth + 1);
    P[0] := #0;
    Insert(P);
  end;
end;

procedure TBuffer.FreeItem(Item: Pointer);
begin
  FreeMem(Item, LineWidth + 1);
end;

function TBuffer.PutChar(C: Char): Boolean;
var
  Width: Integer;
  P: PChar;
begin
  PutChar := False;
  Case C of
    #13: Pos := 0;                          { if a Carriage Return.  }
    #10:                                    { if a Line Feed.        }
      begin
        GetMem(P, LineWidth + 1);
        FillChar(P^, LineWidth + 1, ' ');
        P[Pos] := #0;
        Insert(P);
      end;
    #8:
      if Pos > 0 then                       { if a Delete.           }
      begin
        Dec(Pos);
        P := At(Count - 1);
        P[Pos] := ' ';
      end;
   #32..#128:
    begin
      P := At(Count - 1);
      Width := StrLen(P);
      if Width > LineWidth then             { if line is to wide     }
      begin                                 { create a new line.     }
        Pos := 1;
        GetMem(P, LineWidth + 1);
        P[0] := C;
        P[1] := #0;
        Insert(P);
      end
      else                                   { else add character    }
      begin                                  { to current line.      }
        P[Pos] := C;
        Inc(Pos);
        P[Pos] := #0;
      end;
    end;
  end;
  if Count > LineHeight then
  begin
    AtFree(0);
    PutChar := True;
  end;
end;

{ TCommWindow }
constructor TCommWindow.Init(AParent: PWindowsObject; ATitle: PChar);
begin
  TWindow.Init(AParent, ATitle);
  Attr.Style := Attr.Style or ws_VScroll;
  Attr.Menu := LoadMenu(HInstance, 'Menu_1');
  Scroller := New(PScroller, Init(@Self, 1, 1, 100, 100));
  Buffer := New(PBuffer, Init(@Self));
  DownLoad := Nil;
end;

destructor TCommWindow.Done;
begin
  Error(CloseComm(Cid), 'Close');
  Dispose(Buffer, Done);
  if DownLoad <> Nil then
    Dispose(DownLoad, Done);
  TWindow.Done;
end;

procedure TCommWindow.Capture(var Message: TMessage);
begin
  if DownLoad = Nil then
  begin
    DownLoad := New(PAscii, Init(@Self));
    CheckMenuItem(GetMenu(HWindow), cmCapture, mf_ByCommand or mf_Checked);
  end
  else
  begin
    CheckMenuItem(GetMenu(HWindow), cmCapture, mf_ByCommand or mf_UnChecked);
    Dispose(Download, Done);
    DownLoad := Nil;
  end;
end;

procedure TCommWindow.Configure(var Message: TMessage);
var
  Trans: record
    R1200,
    R2400,
    R4800,
    R9600,
    ROdd,
    REven,
    RNone,
    RComm1,
    RComm2,
    R1Stop,
    R2Stop,
    R7Data,
    R8Data: Word;
  end;
  D: TDialog;
  P: PWindowsObject;
  I: Integer;
begin
  D.Init(@Self, 'Configure');
  For I := id1200 to id8Data do
    P := New(PRadioButton, InitResource(@D, I));
  With Trans do
  begin
    R1200 := Byte(Baud = 12);
    R2400 := Byte(Baud = 24);
    R4800 := Byte(Baud = 48);
    R9600 := Byte(Baud = 96);

    ROdd  := Byte(Parity = 'o');
    REven := Byte(Parity = 'e');
    RNone := Byte(Parity = 'n');

    RComm1 := Byte(Comm = '1');
    RComm2 := Byte(Comm = '2');

    R1Stop := Byte(Stop = '1');
    R2Stop := Byte(Stop = '2');

    R7Data := Byte(Data = '7');
    R8Data := Byte(Data = '8');
  end;
  D.TransferBuffer := @Trans;
  if D.Execute = id_Ok then
  begin
    with Trans do
    begin
      Baud := (R1200 * 12) + (R2400 * 24) + (R4800 * 48) + (R9600 * 96);
      if ROdd = bf_Checked then
        Parity := 'o';
      if REven = bf_Checked then
        Parity := 'e';
      if RNone = bf_Checked then
        Parity := 'n';
      if R1Stop = bf_Checked then
        Stop := '1'
      else
        Stop := '2';
      if RComm1 = bf_Checked then
        Comm := '1'
      else
        Comm := '2';
      if R7Data = bf_Checked then
        Data := '7'
      else
        Data := '8';
      SetConfigure;
    end;
  end;
  D.Done;
end;

procedure TCommWindow.Dial(var Message: TMessage);
var
  Trans: record
    Start: TEditLine;
    Phone: TEditLine;
  end;
  D: TDialog;
  P: PWindowsObject;
begin
  D.Init(@Self, 'Dial');
  P := New(PEdit, InitResource(@D, idDialStart, SizeOf(TEditLine)));
  P := New(PEdit, InitResource(@D, idPhoneNum, SizeOf(TEditLine)));
  StrCopy(Trans.Start, DialStart);
  StrCopy(Trans.Phone, PhoneNumber);
  D.TransferBuffer := @Trans;
  if D.Execute = id_Ok then
  begin
    StrCopy(DialStart, Trans.Start);
    StrCopy(PhoneNumber, Trans.Phone);
    StrCat(PhoneNumber, #13);
    StrCat(PhoneNumber, #10);
    Error(WriteComm(CId, DialStart, StrLen(DialStart)), 'Writing');
    Error(WriteComm(CId, PhoneNumber, StrLen(PhoneNumber)), 'Writing');
    PhoneNumber[StrLen(PhoneNumber) - 2] := #0;
  end;
  D.Done;
end;

procedure TCommWindow.Error(E: Integer; C: PChar);
var
  S: array[0..100] of Char;
begin
  if E >= 0 then exit;
  case E of
    ie_BadID    : StrCopy(S, ' Invalid or unsupported id    ');
    ie_Open     : StrCopy(S, ' Device Already Open          ');
    ie_NoPen    : StrCopy(S, ' Device Not Open              ');
    ie_Memory   : StrCopy(S, ' Unable to allocate queues    ');
    ie_Default  : StrCopy(S, ' Error in default parameters  ');
    ie_Hardware : StrCopy(S, ' Hardware Not Present         ');
    ie_ByteSize : StrCopy(S, ' Illegal Byte Size            ');
    ie_BaudRate : StrCopy(S, ' Unsupported BaudRate         ');
  end;
  MessageBox(GetFocus, S, C, mb_Ok);
  Halt;
end;

procedure TCommWindow.GetCommPort;
var
  S: array[0..20] of Char;
  D: TDialog;
  C: PControl;
  CommRec: Record
    Comm1: Bool;
    Comm2: Bool;
  end;
begin
  if GetProfileString('Terminal', 'Port', '', S, SizeOf(S)) = 0 then
  begin
    D.Init(@Self, 'GetComm');
    C := New(PRadioButton, InitResource(@D, idComm1));
    C := New(PRadioButton, InitResource(@D, idComm2));
    D.TransferBuffer := @CommRec;
    CommRec.Comm1 := True;
    CommRec.Comm2 := False;
    if D.Execute = id_Ok then
    begin
      if CommRec.Comm1 then
        Comm := '1'
      else
        Comm := '2';
      StrCopy(S, 'COMM ');
      S[3] := Comm;
      WriteProfileString('Terminal', 'Port', S);
    end;
    D.Done;
  end
  else
    Comm := S[3];
end;

procedure TCommWindow.Paint(PaintDC: HDC; var PaintInfo: TPaintStruct);
var
  I: Integer;
  Font: HFont;

  procedure WriteOut(Item: PChar); far;
  begin
    TextOut(PaintDC, 0, CharHeight * I, Item, StrLen(Item));
    inc(I);
  end;

begin
  I := 0;
  Font := SelectObject(PaintDC, CreateFontIndirect(FontRec));
  Buffer^.ForEach(@WriteOut);
  DeleteObject(SelectObject(PaintDC, Font));
end;

procedure TCommWindow.ReadChar;
var
  Stat: TComStat;
  I, Size: Integer;
  C: Char;
begin
  GetCommError(CID, Stat);
  for I := 1 to Stat.cbInQue do
  begin
    Size := ReadComm(CId, @C, 1);
    Error(Size, 'Read Comm');
    if Size = 0 then Exit;
    if Download <> Nil then
    begin
      DownLoad^.ReceiveChar(C);
      if (DownLoad <> Nil) and (Not DownLoad^.Echo) then
        Exit;
    end;
    if C <> #0 then
    begin
      if Buffer^.PutChar(C) then
      begin
        ScrollWindow(HWindow, 0, -CharHeight, Nil, Nil);
        UpDateWindow(HWindow);
      end;
      WriteChar;
    end;
  end;
end;

procedure TCommWindow.SendChar(C: Char);
begin
  Error(WriteComm(CId, @C, 1), 'Writing');
end;

procedure TCommWindow.SetConfigure;
var
  Config: array[0..20] of Char;
  S: array[0..5] of Char;
  DCB: TDCB;
begin
  StrCopy(Config, 'com?:??,?,?,?');
  Config[3] := Comm;
  Config[8] := Parity;
  Config[10] := Data;
  Config[12] := Stop;
  Str(Baud, S);
  Config[5] := S[0];
  Config[6] := S[1];
  BuildCommDCB(Config, DCB);
  DCB.ID := CID;
  Error(SetCommState(DCB), 'Set Comm State');
end;

procedure TCommWindow.SetUpWindow;
var
  DCB: TDCB;
  ComPort: Array[0..10] of Char;
begin
  TWindow.SetUpWindow;
  SetHeight;
  GetCommPort;
  StrCopy(ComPort, 'COM ');
  ComPort[3] := Comm;
  Cid := OpenComm(ComPort, 1024, 1024);
  Error(Cid, 'Open');
  SetConfigure;
  WriteComm(Cid, 'ATZ'#13#10, 5);  { Send a reset to Modem. }
end;

function GetFont(LogFont: PLogFont; TM: PTextMetric; FontType: Word;
  P: PCommWindow): Integer; export;
begin
  if P^.CharHeight = 0 then
  begin
    P^.FontRec := LogFont^;
    P^.CharHeight := P^.FontRec.lfHeight;
  end;
end;

procedure TCommWindow.SetHeight;
var
  DC: HDC;
  ProcInst: Pointer;
begin
  DC := GetDC(HWindow);
  CharHeight := 0;
  ProcInst := MakeProcInstance(@GetFont, HInstance);
  EnumFonts(DC, 'Courier', ProcInst, @Self);
  FreeProcInstance(ProcInst);
  ReleaseDC(HWindow, DC);

  Scroller^.SetUnits(CharHeight, CharHeight);
  Scroller^.SetRange(LineWidth, LineHeight);
  Scroller^.ScrollTo(0, LineHeight);
end;

procedure TCommWindow.wmChar(var Message: TMessage);
begin
  Error(WriteComm(CId, @Message.wParam, 1), 'Writing');
end;

procedure TCommWindow.wmSize(var Message: TMessage);
begin
  TWindow.wmSize(Message);
  Scroller^.SetRange(LineWidth, LineHeight - (Message.lParamhi div CharHeight));
end;

procedure TCommWindow.WriteChar;
var
  DC: HDC;
  Font: HFont;
  S: PChar;
  APos: Integer;
begin
  APos := Buffer^.Count - 1;
  S := Buffer^.AT(APos);
  APos := (APos - Scroller^.YPos) * CharHeight;
  if APos < 0 then exit;
  if Hwindow <> 0 then
  begin
    DC := GetDC(HWindow);
    Font := SelectObject(DC, CreateFontIndirect(FontRec));
    TextOut(DC, 0, APos, S, StrLen(S));
    DeleteObject(SelectObject(DC, Font));
    ReleaseDC(HWindow, DC);
  end;
end;

procedure TCommWindow.XModem(var Message: TMessage);
begin
  if DownLoad = Nil then
    DownLoad := New(PXmodem, Init(@Self))
  else
    begin
      dispose(DownLoad, Done);
      DownLoad := Nil;
    end;
end;


{ TApp }
procedure TApp.Idle;
var
  Stat: TComStat; { If you use Win31 then qualify with WinTypes.TComStat }
  I, Size: Integer;
  C: Char;
begin
  if MainWindow <> Nil then
    if MainWindow^.HWindow <> 0 then
      PCommWindow(MainWindow)^.ReadChar;
end;

procedure TApp.InitMainWindow;
begin
  MainWindow := New(PCommWindow, Init(Nil, 'ModemX'));
end;

procedure TApp.MessageLoop;
var
  Message: TMsg;
begin
  while True do
  begin
    if PeekMessage(Message, 0, 0, 0, pm_Remove) then
    begin
      if Message.Message = wm_Quit then
      begin
        Status := Message.WParam;
        Exit;
      end;
      if not ProcessAppMsg(Message) then
      begin
        TranslateMessage(Message);
        DispatchMessage(Message);
      end;
    end
    else
      Idle;
  end;
end;

begin
  App.Init('ModemX');
  App.Run;
  App.Done;
end.
