{ Dicts.pas         --Defines the StaticDict and DynamicDict types
                    --These types allow encapsulated access to a dictionary
                      with key->value pairs.  Keys are strings, values
                      are integers.
                    --They have the same protocol but are implemented
                      differently.  StaticDict uses fixed size arrays
                      whereas DynamicDict maintains a linked list of items.
                    --Two other classes, Assoc and Node provide support
                      for the DynamicDict type.
                    --Written by Zack Urlocker
                      Copyright 1990, The Whitewater Group.
                      All rights reserved.
                    --mzu 06/01/90
}

unit Dicts;

interface

const
   MaxDict = 10;

type

StaticDict = object           { Fixed size dictionary of key->value pairs }
{ Private Object fields }
   Keys : Array[1..MaxDict] of String;
   Values : Array[1..MaxDict] of Integer;
   Count : Integer;                         { How many items? }
   Current : Integer;                       { Index to current item }
{ Functions and procedures }
   constructor Init;
   destructor Done;
   function Size : Integer;
   function MaxValue : Integer;
   procedure Add(key : String; value : Integer);
   procedure Find(key : String);            { Sets Current        }
   procedure Remove(key : String);
   procedure Update(key : String; value : Integer);
   procedure Clear;                         { Empty the dictionary }
   procedure Reset;                         { Set Current to first }
   procedure Next;                          { Move Current to next }
   function AtEnd : Boolean;                { Is Current at end?   }
   function CurrentKey: String;
   function CurrentValue: Integer;
end; { StaticDict }


AssocPtr = ^Association;

Association = object          { A key->value pair }
{ Private Object fields }
  Key : String;
  Value : Integer;
{ Functions and procedures }
  constructor Init;
  destructor Done; virtual;
  function GetKey : String;
  function GetValue : Integer;
  procedure SetKey(S : String);
  procedure SetValue(I : Integer);
end;  { Association }

NodePtr = ^Node;

Node = Object(Association)   { A key->value pair with a link pointer }
  Link : NodePtr;
  constructor Init;
  destructor Done; virtual;
  function Next : NodePtr;
  procedure SetNext(P : NodePtr);
end;

DynamicDict = object      { A dynamic dictionary of key->value pairs }
{ Private Object fields }
   First : NodePtr;
   Prev : NodePtr;
   Current : NodePtr;
   Count : Integer;
{ Functions and procedures }
   constructor Init;
   destructor Done;
   function Size : Integer;
   function MaxValue : Integer;
   procedure Add(key : String; value : Integer);
   procedure Find(key : String);
   procedure Remove(key : String);
   procedure Update(key : String; value : Integer);
   procedure Clear;
   procedure Reset;
   procedure Next;
   function AtEnd : Boolean;
   function CurrentKey: String;
   function CurrentValue: Integer;
end; { DynamicDict }


implementation


{ ********** StaticDict ********** }

constructor StaticDict.Init;
begin
  Clear;
  Current := 0;
end;

destructor StaticDict.Done;
begin
  { not much happens here!}
end;

procedure StaticDict.Clear;
begin
  Count := 0;
end;

function StaticDict.Size : Integer;
begin
  Size := Count;
end;

procedure StaticDict.Reset;
begin
  Current := 1;
end;

function StaticDict.AtEnd : Boolean;
var flag : Boolean;
begin
  if Current > Count then
    flag := True
  else
    flag := False;
  AtEnd := flag;
end;

procedure StaticDict.Next;
begin
  Current := Current + 1;
  if Current > Size then
    Current := Size;
end;

function StaticDict.CurrentKey : String;
begin
  CurrentKey := Keys[Current];
end;

function StaticDict.CurrentValue : Integer;
begin
  CurrentValue := Values[Current];
end;

function StaticDict.MaxValue : Integer;
var I : Integer;
    max : Integer;
begin
  max := 0;
  for I := 1 to Size do
    if Values[I] > max then
       max := Values[i];
  MaxValue := max;
end;

procedure StaticDict.Add(key : String; value : Integer);
begin
  if Count >= MaxDict then
     WriteLn('** Dictionary Error: Dictionary full')
  else
    begin
      Count := Count + 1;
      Keys[Count] := key;
      Values[Count] := value;
    end;
end;

procedure StaticDict.Find(key : String);
{ Find an item in the dictionary.  Adjust Current pointer.
  If it can't find an entry, then AtEnd will be true. }
var found : Boolean;
begin
  Current := 1;
  found := False;
  while (Current <= count) and not found do   { locate key }
     if Keys[Current] = key then
       found := true
     else
       Current := Current + 1;
  if not found then
     writeLn('** Dictionary Error : key ', key, ' not found');
end;

procedure StaticDict.Update(key : String; value : Integer);
begin
  Find(key);
  if Current <> 0 then
  begin
     Keys[Current] := key;
     Values[Current] := value;
  end;
end;

procedure StaticDict.Remove(key : String);
{ Remove an item from the dictionary.  Reorganize the arrays. }
var I : Integer;
begin
  Find(key);
  if Current <> 0 then
  begin
    for I := Current to count-1 do  { overwrite the found item }
      begin
        Keys[I] := Keys[I+1];       { move up subsequent items }
        Values[I] := Values[I+1];
      end;
    Count := Count - 1;             { adjust count }
 end;
end;


{ ********** Association ********** }


constructor Association.Init;
begin
  Key := '';
  Value := 0;
end;

destructor Association.Done;
begin
  { not much happens here! }
end;

function Association.GetKey : String;
begin
  GetKey := Key;
end;

function Association.GetValue : Integer;
begin
  GetValue := Value;
end;

procedure Association.SetKey(S : String);
begin
  Key := S;
end;

procedure Association.SetValue(I : Integer);
begin
  Value := I;
end;

{ ********** Node ********** }

constructor Node.Init;
begin
  Link := nil;        { make sure point is nil }
  Association.Init;   { do the ancestor's init }
end;

destructor Node.Done;
begin
  if link <> nil then
    dispose(link, done);
  Association.Done;     { ancestor done! }
end;


function Node.Next : NodePtr;
begin
  Next := Link;
end;

procedure Node.SetNext(P : NodePtr);
begin
  Link := P;
end;


{ ********** DynamicDict ********** }


constructor DynamicDict.Init;
begin
  First:=nil;
  Current:=nil;
  Count := 0;

  {Fail; }  {  Test! }

end;

destructor DynamicDict.Done;
begin
  Clear;
end;

procedure DynamicDict.Clear;
{ Clear out the pointers and count. }
var P : NodePtr;
begin
  Current := First;
  while Current <> nil do
  begin
    P := Current;                   { save temp pointer }
    Current := Current^.Next;       { move along link }
    P^.SetNext(nil);                { remove old link }
    Dispose(P, Done);               { dispose of it }
  end;
  First := nil;
  Prev := nil;
  Count := 0;
end;

function DynamicDict.Size : Integer;
begin
  Size := Count;
end;

procedure DynamicDict.Reset;
begin
  Current := First;
end;

function DynamicDict.AtEnd : Boolean;
var flag : Boolean;
begin
  if Current = nil then
     flag := True
  else
     flag := False;
  AtEnd := flag;
end;

procedure DynamicDict.Next;
begin
  Current := Current^.Next;
end;

function DynamicDict.CurrentKey : String;
begin
  CurrentKey := Current^.GetKey;
end;

function DynamicDict.CurrentValue : Integer;
begin
  CurrentValue := Current^.GetValue;
end;

function DynamicDict.MaxValue : Integer;
{ Traverse the linked list of items and find the max. }
var P : NodePtr;
    max : Integer;
begin
  max := 0;                       { guess }
  P := First;                     { start at beginning }
  while P <> nil do               { until end of list  }
  begin
    if P^.GetValue > max then     { is it larger?      }
       max := P^.value;
    P := P^.Next;                 { go to next node    }
  end;
  MaxValue := max;
end;

procedure DynamicDict.Add(key : String; value : Integer);
{ Add a new item dynamically to the list }
var P : NodePtr;
begin
  New(P, Init);                   { Create a new Node   }
  P^.SetKey(key);                 { Set the key, value  }
  P^.SetValue(value);
  If First = nil then             { This is the first!  }
    First := P                    { Keep it separate    }
  else                            { For subsequent adds }
    Current^.SetNext(P);          { Add it to the list  }
  Current := P;                   { Make it current     }
  Count := Count + 1;             { Count the nodes     }
end;

procedure DynamicDict.Find(key : String);
{ Find an item in the dictionary. Set the current, prev pointers. }
var P : NodePtr;
    found : Boolean;
begin
  Current := First;
  Prev := nil;
  found := False;
  while (Current<>nil) and not found do  { search for key }
     if Current^.GetKey = key then
       found := true                     { found it! }
     else
       begin
         Prev := Current;                { keep track of prev  }
         Current := Current^.Next;       { move along in list  }
       end;
  if not found then
     writeLn('** Dictionary Error : key ', key, ' not found');
end;

procedure DynamicDict.Update(key : String; value : Integer);
begin
  Find(key);
  if Current <> nil then
  begin
    Current^.SetKey(key);
    Current^.SetValue(value);
  end;
end;

procedure DynamicDict.Remove(key : String);
{ Remove an item from the dictionary. Dispose of the object.  }
var P : NodePtr;
begin
  Find(key);                             { sets prev, current }
  if Current <> nil then                 { it was found }
    begin
      if Current = First then            { Adjust head of list }
        First := Current^.Next           { bypass found item   }
      else                               { its not the first   }
        Prev^.SetNext(Current^.Next);    { bypass found item   }
        P := Current;                    { hold on to temp     }
        Current := Current^.Next;        { adjust current item }
      dispose(P, Done);                  { dispose of item     }
      Count := Count - 1;                { adjust count        }
    end;
end;


{ no initialization }

end.
