/*  load.c - Digital Sound Interface Kit V1.01a loading routines.

    Copyright 1993,94 Carlos Hasan
*/

#include <io.h>
#include <fcntl.h>
#include <malloc.h>
#include <string.h>
#include <sys\stat.h>
#include "sound.h"

int DSMStatus;

static DSMInst *DSMLoadInst(int Handle)
{
    DSMInst *Inst;
    void far *Samples;

    if ((Inst = farmalloc(sizeof(DSMInst))) == NULL) {
        DSMStatus = ERR_NORAM;
        return NULL;
    }
    if (read(Handle,Inst,sizeof(DSMInst)) != sizeof(DSMInst)) {
        DSMStatus = ERR_ACCESS;
        farfree(Inst);
        return NULL;
    }
    if (!Inst->Length) {
        Inst->Address = NULL;
        return Inst;
    }
    if ((Inst->Address = Samples = farmalloc(Inst->Length)) == NULL) {
        DSMStatus = ERR_NORAM;
        farfree(Inst);
        return NULL;
    }
    if ((word)read(Handle,Inst->Address,Inst->Length) != Inst->Length) {
        DSMStatus = ERR_ACCESS;
        farfree(Samples);
        farfree(Inst);
        return NULL;
    }
    if (DSMAllocSampleData(Inst)) {
        DSMStatus = ERR_NODRAM;
        farfree(Samples);
        farfree(Inst);
        return NULL;
    }
    if (DSMTypeOfRAM() == RAM_CARD) {
        farfree(Samples);
    }
    return Inst;
}

static void DSMFreeInst(DSMInst *Inst)
{
    if (Inst) {
        if (Inst->Address) {
            DSMFreeSampleData(Inst);
            if (DSMTypeOfRAM() != RAM_CARD)
                farfree(Inst->Address);
        }
        farfree(Inst);
    }
}

static DSMPatt *DSMLoadPatt(int Handle)
{
    DSMPatt *Patt;
    word Length;

    if (read(Handle,&Length,sizeof(Length)) != sizeof(Length)) {
        DSMStatus = ERR_ACCESS;
        return NULL;
    }
    if ((Patt = farmalloc(Length)) == NULL) {
        DSMStatus = ERR_NORAM;
        return NULL;
    }
    Patt->Length = Length;
    Length -= sizeof(Length);
    if (read(Handle,Patt->Data,Length) != Length) {
        DSMStatus = ERR_ACCESS;
        farfree(Patt);
        return NULL;
    }
    return Patt;
}

static void DSMFreePatt(DSMPatt *Patt)
{
    if (Patt) farfree(Patt);
}

DSM *DSMLoad(char *FileName, dword FileOffset)
{
    int Handle;
    DSMHeader Header;
    DSMBlock Block;
    DSM *Module;
    DSMInst **Inst;
    DSMPatt **Patt;

    if ((Module = farcalloc(1,sizeof(DSM))) == NULL) {
        DSMStatus = ERR_NORAM;
        return NULL;
    }
    Inst = Module->Inst;
    Patt = Module->Patt;

    if ((Handle = open(FileName,O_RDONLY|O_BINARY)) == -1) {
        DSMStatus = ERR_NOFILE;
        DSMFree(Module);
        return NULL;
    }
    if (lseek(Handle,FileOffset,SEEK_SET) == -1) {
        DSMStatus = ERR_ACCESS;
        DSMFree(Module);
        close(Handle);
        return NULL;
    }
    if (read(Handle,&Header,sizeof(Header)) != sizeof(Header)) {
        DSMStatus = ERR_ACCESS;
        DSMFree(Module);
        close(Handle);
        return NULL;
    }
    if (Header.ID != ID_RIFF || Header.FileType != ID_DSMF) {
        DSMStatus = ERR_FORMAT;
        DSMFree(Module);
        close(Handle);
        return NULL;
    }
    Header.Length -= sizeof(Header.FileType);
    while (Header.Length) {
        if (read(Handle,&Block,sizeof(Block)) != sizeof(Block)) {
            DSMStatus = ERR_ACCESS;
            DSMFree(Module);
            close(Handle);
            return NULL;
        }
        Header.Length -= sizeof(Block) + Block.Length;
        if (Block.ID == ID_SONG) {
            if (read(Handle,&Module->Song,Block.Length) != Block.Length) {
                DSMStatus = ERR_ACCESS;
                DSMFree(Module);
                close(Handle);
                return NULL;
            }
        }
        else if (Block.ID == ID_INST) {
            if ((*Inst++ = DSMLoadInst(Handle)) == NULL) {
                DSMFree(Module);
                close(Handle);
                return NULL;
            }
        }
        else if (Block.ID == ID_PATT) {
            if ((*Patt++ = DSMLoadPatt(Handle)) == NULL) {
                DSMFree(Module);
                close(Handle);
                return NULL;
            }
        }
        else {
            if (lseek(Handle,Block.Length,SEEK_CUR) == -1) {
                DSMStatus = ERR_ACCESS;
                DSMFree(Module);
                close(Handle);
                return NULL;
            }
        }
    }
    close(Handle);
    return Module;
}

void DSMFree(DSM *Module)
{
    int i;
    DSMInst **Inst;
    DSMPatt **Patt;

    if (Module) {
        for (Inst = Module->Inst, i = 0; i < MAXSAMPLES; Inst++,i++)
            if (*Inst) DSMFreeInst(*Inst);
        for (Patt = Module->Patt, i = 0; i < MAXORDERS; Patt++,i++)
            if (*Patt) DSMFreePatt(*Patt);
        farfree(Module);
    }
}

DSMInst *DSMLoadSample(char *FileName, dword FileOffset)
{
    int Handle;
    DSMInst *Inst;
    void far *Samples;
    struct {
        DSMHeader Header;
        DSMBlock  Format;
        DSMWave   Fmt;
        DSMBlock  Data;
    } Wave;

    if ((Inst = farcalloc(1,sizeof(DSMInst))) == NULL) {
        DSMStatus = ERR_NORAM;
        return NULL;
    }
    if ((Handle = open(FileName,O_RDONLY|O_BINARY)) == -1) {
        DSMStatus = ERR_NOFILE;
        farfree(Inst);
        return NULL;
    }
    if (lseek(Handle,FileOffset,SEEK_SET) == -1) {
        DSMStatus = ERR_ACCESS;
        farfree(Inst);
        close(Handle);
        return NULL;
    }
    if (read(Handle,&Wave,sizeof(Wave)) != sizeof(Wave)) {
        DSMStatus = ERR_ACCESS;
        farfree(Inst);
        close(Handle);
        return NULL;
    }
    if (Wave.Header.ID != ID_RIFF || Wave.Header.FileType != ID_WAVE ||
        Wave.Format.ID != ID_FMT || Wave.Data.ID != ID_DATA ||
        Wave.Fmt.SampleFormat != 1 || Wave.Fmt.NumChannels != 1 ||
        Wave.Fmt.BitsPerSample != 8) {
        DSMStatus = ERR_FORMAT;
        farfree(Inst);
        close(Handle);
        return NULL;
    }
    Inst->Period = (MIDCFREQ*MIDCPERIOD)/Wave.Fmt.PlayRate;
    Inst->Length = Wave.Data.Length;
    Inst->MidCRate = Wave.Fmt.PlayRate;
    Inst->Volume = 64;
    if ((Inst->Address = Samples = farmalloc(Inst->Length)) == NULL) {
        DSMStatus = ERR_NORAM;
        farfree(Inst);
        close(Handle);
        return NULL;
    }
    if (read(Handle,Samples,Inst->Length) != Inst->Length) {
        DSMStatus = ERR_ACCESS;
        farfree(Samples);
        farfree(Inst);
        close(Handle);
        return NULL;
    }
    if (DSMAllocSampleData(Inst)) {
        DSMStatus = ERR_NODRAM;
        farfree(Samples);
        farfree(Inst);
        close(Handle);
        return NULL;
    }
    if (DSMTypeOfRAM() == RAM_CARD) {
        farfree(Samples);
    }
    close(Handle);
    return Inst;
}

void DSMFreeSample(DSMInst *Inst)
{
    if (Inst) {
        if (Inst->Address) {
            DSMFreeSampleData(Inst);
            if (DSMTypeOfRAM() != RAM_CARD)
                farfree(Inst->Address);
        }
        farfree(Inst);
    }
}

int DSMLoadSetup(DSMCard *Card)
{
    int Handle;
    if ((Handle = open("SOUND.CFG",O_RDONLY|O_BINARY)) == -1)
        return 1;
    if (read(Handle,Card,sizeof(DSMCard)) != sizeof(DSMCard)) {
        close(Handle);
        return 1;
    }
    close(Handle);
    return 0;
}

int DSMSaveSetup(DSMCard *Card)
{
    int Handle;
    if ((Handle = open("SOUND.CFG",O_CREAT|O_WRONLY|O_BINARY,S_IREAD|S_IWRITE)) == -1)
        return 1;
    if (write(Handle,Card,sizeof(DSMCard)) != sizeof(DSMCard)) {
        close(Handle);
        return 1;
    }
    close(Handle);
    return 0;
}

