EXTPROC CEnvi2
//******************************************
//*** ClipBrd - Command-line access to   ***
//*** ver.1     clipboard text functions ***
//******************************************

#include <ClipBrd.lib>

main(argc,argv)
{
   Cmd = argv[1];
   Text = argv[2];
   success = False;  // assume failure
   if ( argc == 2 ) {
      if      ( !stricmp(Cmd,"DELETE") )     success = DeleteClipBrd();
      else if ( !stricmp(Cmd,"GET") )        success = GetClipBrd();
      else if ( !stricmp(Cmd,"ISORT") )      success = SortClipBrd(False);
      else if ( !stricmp(Cmd,"LOWER") )      success = LowercaseClipBrd();
      else if ( !stricmp(Cmd,"QUERY") )      success = QueryClipBrd();
      else if ( !stricmp(Cmd,"SORT") )       success = SortClipBrd(True);
      else if ( !stricmp(Cmd,"TEXT") )       success = TextOnlyInClipBrd();
      else if ( !stricmp(Cmd,"UPPER") )      success = UppercaseClipBrd();
      else Instructions();
   } else if ( argc == 3 ) {
      if ( !stricmp(Cmd,"GET") && Text[0] == '@' )
         success = GetClipBrd(Text+1);
      else {
         if ( Text[0] == '@' ) {
            // Get text from FileSpec
            Text = ReadTextFromFile(Text+1);
         }
         if      ( !stricmp(Cmd,"APPEND") )     success = AppendToClipBrd(Text);
         else if ( !stricmp(Cmd,"FIND") )       success = FindInClipboard(Text,True);
         else if ( !stricmp(Cmd,"IFIND") )      success = FindInClipboard(Text,False);
         else if ( !stricmp(Cmd,"PREPEND") )    success = PrependToClipBrd(Text);
         else if ( !stricmp(Cmd,"PUT") )        success = PutInClipBrd(Text);
         else Instructions();
      }
   } else
      Instructions();
   return ( success ? EXIT_SUCCESS : EXIT_FAILURE );
}

Instructions()
{
   puts("\a")
   puts("SYNTAX: ClipBrd [ DELETE | GET | ISORT | LOWER | QUERY | SORT | TEXT | UPPER ]")
   puts("    or  ClipBrd [ APPEND | FIND | IFIND | PREPEND | PUT ] <text | @<file>>")
   puts("    or  ClipBrd GET @<file>")
   puts("")
   puts("Where: APPEND - Append new text to end of existing clipboard")
   puts("       DELETE - Delete contents of the clipboard")
   puts("       FIND - Find case-sensitive string in clipboard")
   puts("       GET - Display contents of clipboard; or write to @<file>")
   puts("       IFIND - Find case-insensitive string in clipboard")
   puts("       ISORT - Case-insenstive, alphabetic sort of clipboard contents")
   puts("       PREPEND - Prepend new text to beginning of existing clipboard")
   puts("       PUT - Put text in clipboard")
   puts("       QUERY - Does clipboard contain text?")
   puts("       SORT - Case-sensitive, alphabetic sort of clipboard contents")
   puts("       TEXT - Remove all clipboard types except for text")
   puts("       @<FILE> - file specification to get text from/copy-to")
   puts("")
   printf("Return: Return ERRORLEVEL %d if success, and %d if error. QUERY, FIND, and\n",EXIT_SUCCESS,EXIT_FAILURE)
   printf("        IFIND return %d if found, and %d if not found\n",EXIT_SUCCESS,EXIT_FAILURE)
   puts("");
   puts(`Examples: ClipBrd DELETE                   Delete clipboard contents`)
   puts(`          ClipBrd GET @CLIP.TXT            Save to file CLIP.TXT`)
   puts(`          ClipBrd Append "Little bunny."   Add text to clipboard`)
   puts(`          ClipBrd GET @\\DEV\\PRN          Send clipboard to printer`)
}

GetClipboardText(PrintError)   // return clipboard text; message and NULL if none
{
   lText = GetClipboardData(CF_TEXT);
   if ( !lText && PrintError )
      puts("No text in the clipboard.\n");
   return lText;
}

PutClipboardText(pText) // put text in clipboard; message and False if none
{
   success = PutClipboardData(pText,1+strlen(pText),CF_TEXT);
   if ( !success )
      puts("Erroring copying text to the clipboard.\n");
   return success;
}

ReadTextFromFile(pFileSpec)   // return string; else exit
{
   if ( !(fp = fopen(pFileSpec,"rb")) ) {
      printf("Unable to open file \%s\" for reading.\n",pFileSpec);
      exit(EXIT_FAILURE);
   }
   // read in file CHUNK_SIZE bytes at a time
   #define CHUNK_SIZE 1000
   lText = "", lTextLen = 0;
   do {
      lRead = fread(lChunk,CHUNK_SIZE,fp);
      memcpy(lText+lTextLen,lChunk,lRead);
      lTextLen += lRead;
   } while (CHUNK_SIZE == lRead);
   fclose(fp);
   lText[lTextLen] = '\0';
   return lText;
}

AppendToClipBrd(pText)
{
   if ( !(data = GetClipboardText(False)) )
      data = "";
   strcat(data,pText);
   return PutClipboardText(data);
}

DeleteClipBrd()
{
   return ( PutClipboardData(NULL,0,CF_TEXT) );
}

FindInClipboard(pText,pCaseSensitive)
{
   lTextLen = strlen(pText);
   if ( pCaseSensitive ) {
      lFindChars[0] = pText[0], lFindChars[1] = '\0';
      lFindFunction = "memcmp";
   } else {
      lFindChars[2] = '\0';
      lFindChars[1] = tolower(pText[0]);
      lFindChars[0] = toupper(pText[0]);
      lFindFunction = "memicmp";
   }
   if ( data = GetClipboardText() ) {
      while ( data = strpbrk(data,lFindChars) ) {
         if ( !function(lFindFunction,data,pText,lTextLen) ) {
            printf("Text found.\n");
            return True;
         }
         data++;
      }
   }
   printf("Text not found.\n");
   return False;
}

GetClipBrd(OptionalFileSpec)
{
   if ( data = GetClipboardText(True) ) {
      if ( va_arg() ) {
         if ( !(fp = fopen(OptionalFileSpec,"wb")) ) {
            printf("Error opening file \"%s\" for writing.\n",OptionalFileSpec);
            data = NULL;
         } else {
            fwrite(data,strlen(data),fp);
            fclose(fp);
         }
      } else
         fwrite(data,strlen(data),stdout);
   }
   return(data != NULL);
}

LowercaseClipBrd()
{
   if ( data = GetClipboardText(True) )
      strlwr(data);
   return( data != NULL && PutClipboardText(data));
}

PrependToClipBrd(pText)
{
   if ( !(data = GetClipboardText(False)) )
      data = "";
   strcat(pText,data);
   return PutClipboardText(pText);
}

PutInClipBrd(pText)
{
   return PutClipboardText(pText);
}

QueryClipBrd()
{
   return ( NULL != GetClipboardText(True) );
}

SortClipBrd(CaseSensitive)
{
   if ( data = GetClipboardText(True) ) {

      // determine if the last character is line-feed, so we'll
      // know whether to add line-feed in the sorted list
      AddFinalLineFeed = ( NULL != strchr("\r\n",data[strlen(data)-1]) );

      // build array of each line of text in the clipboard
      LineCount = 0;
      Line = strtok(data,"\r\n");
      while ( Line ) {
         strcpy(List[LineCount++],Line);
         Line = strtok(NULL,"\r\n");
      }

      // sort this array of text alphabetically, case-(in)sensitive
      qsort(List,LineCount,CaseSensitive ? "strcmp" : "stricmp" );

      // build one long string to put data back into the clipboard
      data[0] = '\0';
      for ( i = 0; i < LineCount; i++ ) {
         strcat(data,List[i]);
         strcat(data,"\r\n");
      }

      // if there wasn't supposed to be a line-feed at the end, then remove it
      if ( !AddFinalLineFeed )
         data[strlen(data)-2] = '\0';

      // put this new data string in clipboard, replacing old
      if ( !PutClipboardText(data) )
         data = NULL;
   }
   return(data != NULL);
}

TextOnlyInClipBrd()
{
   return ( NULL != (data=GetClipboardText(True))
         && PutClipboardText(data) );
}

UppercaseClipBrd()
{
   if ( data = GetClipboardText(True) )
      strupr(data);
   return( data != NULL && PutClipboardText(data));
}

