//////////////////////////////////////
//  Program Name: HARMONY.CPP
//  Programmer: Charlie Calvert
//  Description: Harmony windows program
//  Date: 08/07/93
//////////////////////////////////////

#define STRICT
#include <windows.h>
#include <windowsx.h>
#include <mmsystem.h>
#include <stdio.h>
#include <string.h>
#include "harmony.h"
#include "playinfo.h"
#include "cdinfo.h"
#include "cdutil.h"
#include "wavemidi.h"
#pragma warning (disable: 4068)
// --------------------------------------
// Interface
// --------------------------------------

static char szAppName[] = "Harmony";
static HWND MainWindow;
static HINSTANCE hInstance;

BYTE State;
BYTE PlayType;
char DirStr[DIRSTRSIZE];
char DeviceString[DEVICESTRSIZE];
static char NoCDOnSystem;

// -------------------------------------------------------
// Initialization
// -------------------------------------------------------

//////////////////////////////////////
// Program entry point
//////////////////////////////////////
#pragma argsused
int PASCAL WinMain(HINSTANCE hInst, HINSTANCE hPrevInstance,
                   LPSTR lpszCmdParam, int nCmdShow)
{
  MSG  Msg;

  if (!hPrevInstance)
    if (!Register(hInst))
      return FALSE;

  MainWindow = Create(hInst, nCmdShow);
  if (!MainWindow)
    return FALSE;

  while (GetMessage(&Msg, NULL, 0, 0))
  {
     TranslateMessage(&Msg);
     DispatchMessage(&Msg);
  }

  return Msg.wParam;
}

//////////////////////////////////////
// Register the window
//////////////////////////////////////
BOOL Register(HINSTANCE hInst)
{
  WNDCLASS WndClass;

  WndClass.style         = CS_HREDRAW | CS_VREDRAW;
  WndClass.lpfnWndProc   = WndProc;
  WndClass.cbClsExtra    = 0;
  WndClass.cbWndExtra    = DLGWINDOWEXTRA;
  WndClass.hInstance     = hInst;
  WndClass.hIcon         = LoadIcon(NULL, IDI_APPLICATION);
  WndClass.hCursor       = LoadCursor(NULL, IDC_ARROW);
  WndClass.hbrBackground = GetStockBrush(WHITE_BRUSH);
  WndClass.lpszMenuName  = NULL;
  WndClass.lpszClassName = szAppName;

  return RegisterClass (&WndClass);
}

//////////////////////////////////////
// Create the window
//////////////////////////////////////
HWND Create(HINSTANCE hInst, int nCmdShow)
{

  hInstance = hInst;

  HWND hwnd = CreateDialog(hInst, szAppName, 0, NULL);

  if (hwnd == NULL)
    return hwnd;

  if (NoCDOnSystem)
    EnableWindow(GetDlgItem(hwnd, CM_COMPACT), FALSE);

  ShowWindow(hwnd, nCmdShow);

  return hwnd;
}

// ------------------------------------------------------
// WndProc and Implementation
// ------------------------------------------------------

//////////////////////////////////////
// The Window Procedure
//////////////////////////////////////
LRESULT CALLBACK _export WndProc(HWND hwnd, UINT Message, WPARAM 
wParam,
                                 LPARAM lParam)
{
  switch(Message)
  {
    HANDLE_MSG(hwnd, WM_CREATE, Harmony_OnCreate);
    HANDLE_MSG(hwnd, WM_DESTROY, Harmony_OnDestroy);
    HANDLE_MSG(hwnd, WM_FILLLISTBOX, Harmony_OnFillListBox);
    HANDLE_MSG(hwnd, WM_COMMAND, Harmony_OnCommand);
    HANDLE_MSG(hwnd, MM_MCINOTIFY, Harmony_OnMCINotify);
    HANDLE_MSG(hwnd, WM_TIMER, Harmony_OnTimer);
    default: return Harmony_DefProc(hwnd, Message, wParam, lParam);
  }
}

#pragma argsused
BOOL Harmony_OnCreate(HWND hwnd, CREATESTRUCT FAR* lpCreateStruct)
{
  PlayType = WAVE;
  State = CLOSED;
  strcpy(DeviceString, "waveaudio");
  GetWindowsDirectory(DirStr, 200);
  FORWARD_WM_FILLLISTBOX(hwnd, PostMessage);
  if (!DoesDeviceExistOnSystem("cdaudio"))
    NoCDOnSystem = TRUE;
  else
    NoCDOnSystem = FALSE;
  return TRUE;
}

//////////////////////////////////////
// Handle WM_DESTROY
//////////////////////////////////////
#pragma argsused
void Harmony_OnDestroy(HWND hwnd)
{
  PostQuitMessage(0);
}

//////////////////////////////////////
// Handle WM_COMMAND
//////////////////////////////////////
#pragma argsused
void Harmony_OnCommand(HWND hwnd, int id, HWND hwndCtl, UINT 
codeNotify)
{
  char S[100];

  switch (id)
  {
    case CM_MIDI:
      CheckForClose();
      strcpy(DeviceString, "sequencer");
      PlayType = MIDI;
      FORWARD_WM_FILLLISTBOX(hwnd, PostMessage);
      break;

    case CM_WAVE:
      CheckForClose();
      strcpy(DeviceString, "waveaudio");
      PlayType = WAVE;
      FORWARD_WM_FILLLISTBOX(hwnd, PostMessage);
      break;

    case CM_COMPACT:
      if (PlayType != COMPACT)
      {
        CheckForClose();
        PlayType = COMPACT;
        strcpy(DeviceString, "cdaudio");
        FORWARD_WM_FILLLISTBOX(hwnd, PostMessage);
      }
      break;

    case CM_DIRECTORY:
      LoadFile(hwnd);
      FORWARD_WM_FILLLISTBOX(hwnd, PostMessage);
      break;

    case IDOK:
      CheckForClose();
      FORWARD_WM_CLOSE(hwnd, Harmony_DefProc);
      break;

    case CM_PAUSE:
    case CM_STOP:
      switch (PlayType)
      {
        case MIDI: HandleMIDIPAndS(hwnd, id); break;
        case WAVE: HandleWAVEPAndS(hwnd, id); break;
        case COMPACT: HandleCDPAndS(hwnd, id); break;
      }
      break;

    case CM_WAVEPLAY:
      if (State == PLAYING)
        return;

      if (PlayType == COMPACT)
      {
        PlayCD(hwnd);
        return;
      }

      if (FileOpen(hwnd))
      {
        State = PLAYING;
        SetWindowText(GetDlgItem(hwnd, ID_MODE), "Playing");
        SetTimeFormatMs();
        DWORD Result = GetLen();
        sprintf(S, "%ld ms", Result);
        SetWindowText(GetDlgItem(hwnd, ID_LENGTH), S);
        SetWindowText(GetDlgItem(hwnd, ID_NUMTRACKS),
                      GetInfo(S));
        if (PlayMCI())
          SetTimer(hwnd, HARMONY_TIMER, 10, NULL);
        else
        {
          CloseMCI();
          State = CLOSED;
          SetWindowText(GetDlgItem(hwnd, ID_MODE), "CLOSED");
        }
      }
      break;

    case ID_FILELIST:
      if (codeNotify == LBN_DBLCLK)
        MessageBox(hwnd, "They're talking to me!", NULL, MB_OK);
      break;

    case CM_RECORD:
      if ((PlayType == MIDI) || (PlayType == COMPACT))
        return;
      if (OpenMCI(hwnd, "", DeviceString))
      {
        DoRecord(10000);
        State = RECORDING;
        SetTimer(hwnd, HARMONY_TIMER, 10, NULL);
      }
      break;
  }
}

//////////////////////////////////////
// Handle WM_FILLLISTBOX
//////////////////////////////////////
#pragma argsused
void Harmony_OnFillListBox(HWND hwnd)
{
  if (PlayType == COMPACT)
  {
    FillCDTrackBox(hwnd);
    return;
  }

  char S[200];

  strcpy(S, DirStr);
  switch (PlayType)
  {
    case MIDI: strcat(S, "\\*.mid"); break;
    case WAVE: strcat(S, "\\*.wav"); break;
  }

  if(!DlgDirList(hwnd, S, ID_FILELIST, 0, DDL_ARCHIVE))
    MessageBox(hwnd, "No way", NULL, MB_OK);

  switch (PlayType)
  {
    case COMPACT:
      Button_SetCheck(GetDlgItem(hwnd, CM_COMPACT), TRUE); break;
    case MIDI:
      Button_SetCheck(GetDlgItem(hwnd, CM_MIDI), TRUE); break;
    case WAVE:
      Button_SetCheck(GetDlgItem(hwnd, CM_WAVE), TRUE); break;
  }
  SetWindowText(GetDlgItem(hwnd, ID_DIREDIT), DirStr);

  ListBox_SetCurSel(GetDlgItem(hwnd, ID_FILELIST), 0);
}

//////////////////////////////////////
// Handle MM_ONMCINOTIFY
//////////////////////////////////////
#pragma argsused
void Harmony_OnMCINotify(HWND hwnd, UINT status, int DeviceID)
{
  char S[100];

  if (State == ERROR_OCCURED)
    return;

  switch (status)
  {
    case MCI_NOTIFY_ABORTED:
      strcpy(S, "Aborted");
      break;

    case MCI_NOTIFY_SUCCESSFUL:
      if (State == RECORDING)
      {
        SaveFile("Albert.wav");
        DlgDirList(hwnd, S, ID_FILELIST, 0, DDL_ARCHIVE);
      }
      State = CLOSED;
      if (PlayType == COMPACT)
        CloseCDMCI();
      else
        CloseMCI();
      strcpy(S, "Success");
      break;

    case MCI_NOTIFY_SUPERSEDED:
      strcpy(S, "Superseded");
      break;

    case MCI_NOTIFY_FAILURE:
      State = CLOSED;
      if (PlayType == COMPACT)
      CloseCDMCI();
      else
        CloseMCI();
      strcpy(S, "Failure");
      break;
  }
  SetWindowText(GetDlgItem(hwnd, ID_MODE), S);
  SetWindowText(GetDlgItem(hwnd, ID_LENGTH), "...");
  SetWindowText(GetDlgItem(hwnd, ID_POSITION), "...");
  SetWindowText(GetDlgItem(hwnd, ID_NUMTRACKS), "...");
//  MessageBox(hwnd, S, "MCI_NOTIFY", MB_OK);
}

//////////////////////////////////////
// Handle WM_TIMER
//////////////////////////////////////
#pragma argsused
void Harmony_OnTimer(HWND hwnd, UINT id)
{
  char S[100];

  switch (State)
  {
    case PAUSED:
      KillTimer(hwnd, HARMONY_TIMER);
      break;

    case STOPPED:
    case CLOSED:
      KillTimer(hwnd, HARMONY_TIMER);
      break;

    case RECORDING:
    case PLAYING:
      if (PlayType == COMPACT)
      {
        HandleCDTimer(hwnd);
        return;
      }

      DWORD Result = GetLocation();
      
      if (Result == -1)
      {
        KillTimer(hwnd, HARMONY_TIMER);
        State = ERROR_OCCURED;
        SetWindowText(GetDlgItem(hwnd, ID_MODE), "ERROR");
        return;
      }
      
      sprintf(S, "%ld ms", Result);
      SetWindowText(GetDlgItem(hwnd, ID_POSITION), S);

      break;
  }
}
