{* Copyright (c) 1992-1995, Microsoft Corporation, all rights reserved
**
** ras.h
** Remote Access external API
** Public header for external API clients
*}

{ Delphi conversion by Davide Moretti <dmoretti@iper.net> }
{ Note: All functions and structures defaults to Ansi. If you want to use
  Unicode structs and funcs, use the names ending with 'W' }

unit Ras;

interface

uses Windows, Messages;

const

{ These are from lmcons.h }

  DNLEN = 15;  // Maximum domain name length
  UNLEN = 256; // Maximum user name length
  PWLEN = 256; // Maximum password length
  NETBIOS_NAME_LEN = 16; // NetBIOS net name (bytes)

  RAS_MaxDeviceType  = 16;
  RAS_MaxPhoneNumber = 128;
  RAS_MaxIpAddress   = 15;
  RAS_MaxIpxAddress  = 21;

{ Note: you must define WINVER31 before compiling this unit to get the sizes
  for the 3.x version of RAS. The default is for Windows 95 and NT 4.0 }

{$IFDEF WINVER31}
{Version 3.x sizes }
  RAS_MaxEntryName      = 20;
  RAS_MaxDeviceName     = 32;
  RAS_MaxCallbackNumber = 48;
{$ELSE}
{Version 4.x sizes }
  RAS_MaxEntryName      = 256;
  RAS_MaxDeviceName     = 128;
  RAS_MaxCallbackNumber = RAS_MaxPhoneNumber;
{$ENDIF}

type

  LPHRasConn = ^THRasConn;
  THRasConn = Longint;


{* Identifies an active RAS connection.  (See RasEnumConnections)
*}
  LPRasConnW = ^TRasConnW;
  TRasConnW = record
    dwSize: Longint;
    hrasconn: THRasConn;
    szEntryName: Array[0..RAS_MaxEntryName] of WideChar;
{$IFNDEF WINVER31}
   szDeviceType: Array[0..RAS_MaxDeviceType] of WideChar;
   szDeviceName: Array[0..RAS_MaxDeviceName] of WideChar;
{$ENDIF}
  end;

  LPRasConnA = ^TRasConnA;
  TRasConnA = record
    dwSize: Longint;
    hrasconn: THRasConn;
    szEntryName: Array[0..RAS_MaxEntryName] of AnsiChar;
{$IFNDEF WINVER31}
   szDeviceType: Array[0..RAS_MaxDeviceType] of AnsiChar;
   szDeviceName: Array[0..RAS_MaxDeviceName] of AnsiChar;
{$ENDIF}
  end;

  LPRasConn = ^TRasConn;
  TRasConn = TRasConnA;


const

{* Enumerates intermediate states to a connection.  (See RasDial)
*}
  RASCS_PAUSED = $1000;
  RASCS_DONE   = $2000;

type
  LPRasConnState = ^TRasConnState;
  TRasConnState = Integer;

const
  RASCS_OpenPort = 0;
  RASCS_PortOpened = 1;
  RASCS_ConnectDevice = 2;
  RASCS_DeviceConnected = 3;
  RASCS_AllDevicesConnected = 4;
  RASCS_Authenticate = 5;
  RASCS_AuthNotify = 6;
  RASCS_AuthRetry = 7;
  RASCS_AuthCallback = 8;
  RASCS_AuthChangePassword = 9;
  RASCS_AuthProject = 10;
  RASCS_AuthLinkSpeed = 11;
  RASCS_AuthAck = 12;
  RASCS_ReAuthenticate = 13;
  RASCS_Authenticated = 14;
  RASCS_PrepareForCallback = 15;
  RASCS_WaitForModemReset = 16;
  RASCS_WaitForCallback = 17;
  RASCS_Projected = 18;
{$IFNDEF WINVER31}
  RASCS_StartAuthentication = 19;
  RASCS_CallbackComplete = 20;
  RASCS_LogonNetwork = 21;
{$ENDIF}

  RASCS_Interactive = RASCS_PAUSED;
  RASCS_RetryAuthentication = RASCS_PAUSED + 1;
  RASCS_CallbackSetByCaller = RASCS_PAUSED + 2;
  RASCS_PasswordExpired = RASCS_PAUSED + 3;

  RASCS_Connected = RASCS_DONE;
  RASCS_Disconnected = RASCS_DONE + 1;

type

{* Describes the status of a RAS connection.  (See RasConnectionStatus)
*}
  LPRasConnStatusW = ^TRasConnStatusW;
  TRasConnStatusW = record
    dwSize: Longint;
    rasconnstate: TRasConnState;
    dwError: LongInt;
    szDeviceType: Array[0..RAS_MaxDeviceType] of WideChar;
    szDeviceName: Array[0..RAS_MaxDeviceName] of WideChar;
  end;

  LPRasConnStatusA = ^TRasConnStatusA;
  TRasConnStatusA = record
    dwSize: Longint;
    rasconnstate: TRasConnState;
    dwError: LongInt;
    szDeviceType: Array[0..RAS_MaxDeviceType] of AnsiChar;
    szDeviceName: Array[0..RAS_MaxDeviceName] of AnsiChar;
  end;

  LPRasConnStatus = ^TRasConnStatus;
  TRasConnStatus = TRasConnStatusA;


{* Describes connection establishment parameters.  (See RasDial)
*}
  LPRasDialParamsW = ^TRasDialParamsW;
  TRasDialParamsW = record
    dwSize: LongInt;
    szEntryName: Array[0..RAS_MaxEntryName] of WideChar;
    szPhoneNumber: Array[0..RAS_MaxPhoneNumber] of WideChar;
    szCallbackNumber: Array[0..RAS_MaxCallbackNumber] of WideChar;
    szUserName: Array[0..UNLEN] of WideChar;
    szPassword: Array[0..PWLEN] of WideChar;
    szDomain: Array[0..DNLEN] of WideChar;
  end;

  LPRasDialParamsA = ^TRasDialParamsA;
  TRasDialParamsA = record
    dwSize: LongInt;
    szEntryName: Array[0..RAS_MaxEntryName] of AnsiChar;
    szPhoneNumber: Array[0..RAS_MaxPhoneNumber] of AnsiChar;
    szCallbackNumber: Array[0..RAS_MaxCallbackNumber] of AnsiChar;
    szUserName: Array[0..UNLEN] of AnsiChar;
    szPassword: Array[0..PWLEN] of AnsiChar;
    szDomain: Array[0..DNLEN] of AnsiChar;
  end;

  LPRasDialParams = ^TRasDialParams;
  TRasDialParams = TRasDialParamsA;


{* Describes extended connection establishment options.  (See RasDial)
*}
  LPRasDialExtensions = ^TRasDialExtensions;
  TRasDialExtensions = record
    dwSize: LongInt;
    dwfOptions: LongInt;
    hwndParent: HWND;
    reserved: LongInt;
  end;

{* 'dwfOptions' bit flags.
*}
const
  RDEOPT_UsePrefixSuffix           = $00000001;
  RDEOPT_PausedStates              = $00000002;
  RDEOPT_IgnoreModemSpeaker        = $00000004;
  RDEOPT_SetModemSpeaker           = $00000008;
  RDEOPT_IgnoreSoftwareCompression = $00000010;
  RDEOPT_SetSoftwareCompression    = $00000020;


type

{* Describes an enumerated RAS phone book entry name.  (See RasEntryEnum)
*}
  LPRasEntryNameW = ^TRasEntryNameW;
  TRasEntryNameW = record
    dwSize: Longint;
    szEntryName: Array[0..RAS_MaxEntryName] of WideChar;
  end;

  LPRasEntryNameA = ^TRasEntryNameA;
  TRasEntryNameA = record
    dwSize: Longint;
    szEntryName: Array[0..RAS_MaxEntryName] of AnsiChar;
  end;

  LPRasEntryName = ^TRasEntryName;
  TRasEntryName = TRasEntryNameA;


{* Protocol code to projection data structure mapping.
*}
  LPRasProjection = ^TRasProjection;
  TRasProjection = Integer;

const
  RASP_Amb = $10000;
  RASP_PppNbf = $803F;
  RASP_PppIpx = $802B;
  RASP_PppIp = $8021;


type

{* Describes the result of a RAS AMB (Authentication Message Block)
** projection.  This protocol is used with NT 3.1 and OS/2 1.3 downlevel
** RAS servers.
*}
  LPRasAmbW = ^TRasAmbW;
  TRasAmbW = record
    dwSize: Longint;
    dwError: Longint;
    szNetBiosError: Array[0..NETBIOS_NAME_LEN] of WideChar;
    bLana: Byte;
  end;

  LPRasAmbA = ^TRasAmbA;
  TRasAmbA = record
    dwSize: Longint;
    dwError: Longint;
    szNetBiosError: Array[0..NETBIOS_NAME_LEN] of AnsiChar;
    bLana: Byte;
  end;

  LPRasAmb = ^TRasAmb;
  TRasAmb = TRasAmbA;


{* Describes the result of a PPP NBF (NetBEUI) projection.
*}
  LPRasPppNbfW = ^TRasPppNbfW;
  TRasPppNbfW = record
    dwSize: Longint;
    dwError: Longint;
    dwNetBiosError: Longint;
    szNetBiosError: Array[0..NETBIOS_NAME_LEN] of WideChar;
    szWorkstationName: Array[0..NETBIOS_NAME_LEN] of WideChar;
    bLana: Byte;
  end;

  LPRasPppNbfA = ^TRasPppNbfA;
  TRasPppNbfA = record
    dwSize: Longint;
    dwError: Longint;
    dwNetBiosError: Longint;
    szNetBiosError: Array[0..NETBIOS_NAME_LEN] of AnsiChar;
    szWorkstationName: Array[0..NETBIOS_NAME_LEN] of AnsiChar;
    bLana: Byte;
  end;

  LpRaspppNbf = ^TRasPppNbf;
  TRasPppNbf = TRasPppNbfA;


{* Describes the results of a PPP IPX (Internetwork Packet Exchange)
** projection.
*}
  LPRasPppIpxW = ^TRasPppIpxW;
  TRasPppIpxW = record
    dwSize: Longint;
    dwError: Longint;
    szIpxAddress: Array[0..RAS_MaxIpxAddress] of WideChar;
 end;

  LPRasPppIpxA = ^TRasPppIpxA;
  TRasPppIpxA = record
    dwSize: Longint;
    dwError: Longint;
    szIpxAddress: Array[0..RAS_MaxIpxAddress] of AnsiChar;
 end;

  LPRasPppIpx = ^TRasPppIpx;
  TRasPppIpx = TRasPppIpxA;


{* Describes the results of a PPP IP (Internet) projection.
*}
  LPRasPppIpW = ^TRasPppIpW;
  TRasPppIpW = record
    dwSize: Longint;
    dwError: Longint;
    szIpAddress: Array[0..RAS_MaxIpAddress] of WideChar;

{$IFNDEF WINNT35COMPATIBLE}

    {* This field was added between Windows NT 3.51 beta and Windows NT 3.51
    ** final, and between Windows 95 M8 beta and Windows 95 final.  If you do
    ** not require the server address and wish to retrieve PPP IP information
    ** from Windows NT 3.5 or early Windows NT 3.51 betas, or on early Windows
    ** 95 betas, define WINNT35COMPATIBLE.
    **
    ** The server IP address is not provided by all PPP implementations,
    ** though Windows NT server's do provide it.
    *}
    szServerIpAddress: Array[0..RAS_MaxIpAddress] of WideChar;

{$ENDIF}
  end;

  LPRasPppIpA = ^TRasPppIpA;
  TRasPppIpA = record
    dwSize: Longint;
    dwError: Longint;
    szIpAddress: Array[0..RAS_MaxIpAddress] of AnsiChar;

{$IFNDEF WINNT35COMPATIBLE}

    {* See RASPPPIPW comment.
    *}
    szServerIpAddress: Array[0..RAS_MaxIpAddress] of AnsiChar;

{$ENDIF}
  end;

  LPRasPppIp = ^TRasPppIp;
  TRasPppIp = TRasPppIpA;


const

{* If using RasDial message notifications, get the notification message code
** by passing this string to the RegisterWindowMessageA() API.
** WM_RASDIALEVENT is used only if a unique message cannot be registered.
*}
  RASDIALEVENT    = 'RasDialEvent';
  WM_RASDIALEVENT = $CCCD;


{* Prototypes for caller's RasDial callback handler.  Arguments are the
** message ID (currently always WM_RASDIALEVENT), the current RASCONNSTATE and
** the error that has occurred (or 0 if none).  Extended arguments are the
** handle of the RAS connection and an extended error code.
*}
{
typedef VOID (WINAPI *RASDIALFUNC)( UINT, RASCONNSTATE, DWORD );
typedef VOID (WINAPI *RASDIALFUNC1)( HRASCONN, UINT, RASCONNSTATE, DWORD, DWORD );

For Delphi: Just define the callback as
procedure RASCallback(msg: Integer; state: TRasConnState;
    dwError: Longint); stdcall;
 or
procedure RASCallback1(hConn: THRasConn; msg: Integer;
    state: TRasConnState; dwError: Longint; dwEexterror: Longint); stdcall;
}

{* External RAS API function prototypes.
*}
{Note: for Delphi the function without 'A' or 'W' is the Ansi one
  as on the other Delphi headers}

function RasDialA(
    lpRasDialExt: LPRasDialExtensions;
    lpszPhoneBook: PAnsiChar;
    var params: TRasDialParamsA;
    dwNotifierType: Longint;
    lpNotifier: Pointer;
    var rasconn: THRasConn
    ): Longint; stdcall;
function RasDialW(
    lpRasDialExt: LPRasDialExtensions;
    lpszPhoneBook: PWideChar;
    var params: TRasDialParamsW;
    dwNotifierType: Longint;
    lpNotifier: Pointer;
    var rasconn: THRasConn
    ): Longint; stdcall;
function RasDial(
    lpRasDialExt: LPRasDialExtensions;
    lpszPhoneBook: PAnsiChar;
    var params: TRasDialParams;
    dwNotifierType: Longint;
    lpNotifier: Pointer;
    var rasconn: THRasConn
    ): Longint; stdcall;

function RasEnumConnectionsA(
    rasconnArray: LPRasConnA;
    var lpcb: Longint;
    var lpcConnections: Longint
    ): Longint; stdcall;
function RasEnumConnectionsW(
    rasconnArray: LPRasConnW;
    var lpcb: Longint;
    var lpcConnections: Longint
    ): Longint; stdcall;
function RasEnumConnections(
    rasconnArray: LPRasConn;
    var lpcb: Longint;
    var lpcConnections: Longint
    ): Longint; stdcall;

function RasEnumEntriesA(
    reserved: PAnsiChar;
    lpszPhoneBook: PAnsiChar;
    entrynamesArray: LPRasEntryNameA;
    var lpcb: Longint;
    var lpcEntries: Longint
    ): Longint; stdcall;
function RasEnumEntriesW(
    reserved: PWideChar;
    lpszPhoneBook: PWideChar;
    entrynamesArray: LPRasEntryNameW;
    var lpcb: Longint;
    var lpcEntries: Longint
    ): Longint; stdcall;
function RasEnumEntries(
    reserved: PAnsiChar;
    lpszPhoneBook: PAnsiChar;
    entrynamesArray: LPRasEntryName;
    var lpcb: Longint;
    var lpcEntries: Longint
    ): Longint; stdcall;

function RasGetConnectStatusA(
    hConn: THRasConn;
    var lpStatus: TRasConnStatusA
    ): Longint; stdcall;
function RasGetConnectStatusW(
    hConn: THRasConn;
    var lpStatus: TRasConnStatusW
    ): Longint; stdcall;
function RasGetConnectStatus(
    hConn: THRasConn;
    var lpStatus: TRasConnStatus
    ): Longint; stdcall;

function RasGetErrorStringA(
    errorValue: Integer;
    erroString: PAnsiChar;
    cBufSize: Longint
    ): Longint; stdcall;
function RasGetErrorStringW(
    errorValue: Integer;
    erroString: PWideChar;
    cBufSize: Longint
    ): Longint; stdcall;
function RasGetErrorString(
    errorValue: Integer;
    erroString: PAnsiChar;
    cBufSize: Longint
    ): Longint; stdcall;

function RasHangUpA(
  hConn: THRasConn
  ): Longint; stdcall;
function RasHangUpW(
  hConn: THRasConn
  ): Longint; stdcall;
function RasHangUp(
  hConn: THRasConn
  ): Longint; stdcall;

function RasGetProjectionInfoA(
  hConn: THRasConn;
  rasproj: TRasProjection;
  lpProjection: Pointer;
  var lpcb: Longint
  ): Longint; stdcall;
function RasGetProjectionInfoW(
  hConn: THRasConn;
  rasproj: TRasProjection;
  lpProjection: Pointer;
  var lpcb: Longint
  ): Longint; stdcall;
function RasGetProjectionInfo(
  hConn: THRasConn;
  rasproj: TRasProjection;
  lpProjection: Pointer;
  var lpcb: Longint
  ): Longint; stdcall;

function RasCreatePhonebookEntryA(
  hwndParentWindow: HWND;
  lpszPhoneBook: PAnsiChar
  ): Longint; stdcall;
function RasCreatePhonebookEntryW(
  hwndParentWindow: HWND;
  lpszPhoneBook: PWideChar
  ): Longint; stdcall;
function RasCreatePhonebookEntry(
  hwndParentWindow: HWND;
  lpszPhoneBook: PAnsiChar
  ): Longint; stdcall;

function RasEditPhonebookEntryA(
  hwndParentWindow: HWND;
  lpszPhoneBook: PAnsiChar;
  lpszEntryName: PAnsiChar
  ): Longint; stdcall;
function RasEditPhonebookEntryW(
  hwndParentWindow: HWND;
  lpszPhoneBook: PWideChar;
  lpszEntryName: PWideChar
  ): Longint; stdcall;
function RasEditPhonebookEntry(
  hwndParentWindow: HWND;
  lpszPhoneBook: PAnsiChar;
  lpszEntryName: PAnsiChar
  ): Longint; stdcall;

function RasSetEntryDialParamsA(
  lpszPhoneBook: PAnsiChar;
  var lpDialParams: TRasDialParamsA;
  fRemovePassword: LongBool
  ): Longint; stdcall;
function RasSetEntryDialParamsW(
  lpszPhoneBook: PWideChar;
  var lpDialParams: TRasDialParamsW;
  fRemovePassword: LongBool
  ): Longint; stdcall;
function RasSetEntryDialParams(
  lpszPhoneBook: PAnsiChar;
  var lpDialParams: TRasDialParams;
  fRemovePassword: LongBool
  ): Longint; stdcall;

function RasGetEntryDialParamsA(
  lpszPhoneBook: PAnsiChar;
  var lpDialParams: TRasDialParamsA;
  var lpfPassword: LongBool
  ): Longint; stdcall;
function RasGetEntryDialParamsW(
  lpszPhoneBook: PWideChar;
  var lpDialParams: TRasDialParamsW;
  var lpfPassword: LongBool
  ): Longint; stdcall;
function RasGetEntryDialParams(
  lpszPhoneBook: PAnsiChar;
  var lpDialParams: TRasDialParams;
  var lpfPassword: LongBool
  ): Longint; stdcall;


{**
** raserror.h
** Remote Access external API
** RAS specific error codes
*}

const
  RASBASE = 600;
  SUCCESS = 0;

  PENDING                              = (RASBASE+0);
  ERROR_INVALID_PORT_HANDLE            = (RASBASE+1);
  ERROR_PORT_ALREADY_OPEN              = (RASBASE+2);
  ERROR_BUFFER_TOO_SMALL               = (RASBASE+3);
  ERROR_WRONG_INFO_SPECIFIED           = (RASBASE+4);
  ERROR_CANNOT_SET_PORT_INFO           = (RASBASE+5);
  ERROR_PORT_NOT_CONNECTED             = (RASBASE+6);
  ERROR_EVENT_INVALID                  = (RASBASE+7);
  ERROR_DEVICE_DOES_NOT_EXIST          = (RASBASE+8);
  ERROR_DEVICETYPE_DOES_NOT_EXIST      = (RASBASE+9);
  ERROR_BUFFER_INVALID                 = (RASBASE+10);
  ERROR_ROUTE_NOT_AVAILABLE            = (RASBASE+11);
  ERROR_ROUTE_NOT_ALLOCATED            = (RASBASE+12);
  ERROR_INVALID_COMPRESSION_SPECIFIED  = (RASBASE+13);
  ERROR_OUT_OF_BUFFERS                 = (RASBASE+14);
  ERROR_PORT_NOT_FOUND                 = (RASBASE+15);
  ERROR_ASYNC_REQUEST_PENDING          = (RASBASE+16);
  ERROR_ALREADY_DISCONNECTING          = (RASBASE+17);
  ERROR_PORT_NOT_OPEN                  = (RASBASE+18);
  ERROR_PORT_DISCONNECTED              = (RASBASE+19);
  ERROR_NO_ENDPOINTS                   = (RASBASE+20);
  ERROR_CANNOT_OPEN_PHONEBOOK          = (RASBASE+21);
  ERROR_CANNOT_LOAD_PHONEBOOK          = (RASBASE+22);
  ERROR_CANNOT_FIND_PHONEBOOK_ENTRY    = (RASBASE+23);
  ERROR_CANNOT_WRITE_PHONEBOOK         = (RASBASE+24);
  ERROR_CORRUPT_PHONEBOOK              = (RASBASE+25);
  ERROR_CANNOT_LOAD_STRING             = (RASBASE+26);
  ERROR_KEY_NOT_FOUND                  = (RASBASE+27);
  ERROR_DISCONNECTION                  = (RASBASE+28);
  ERROR_REMOTE_DISCONNECTION           = (RASBASE+29);
  ERROR_HARDWARE_FAILURE               = (RASBASE+30);
  ERROR_USER_DISCONNECTION             = (RASBASE+31);
  ERROR_INVALID_SIZE                   = (RASBASE+32);
  ERROR_PORT_NOT_AVAILABLE             = (RASBASE+33);
  ERROR_CANNOT_PROJECT_CLIENT          = (RASBASE+34);
  ERROR_UNKNOWN                        = (RASBASE+35);
  ERROR_WRONG_DEVICE_ATTACHED          = (RASBASE+36);
  ERROR_BAD_STRING                     = (RASBASE+37);
  ERROR_REQUEST_TIMEOUT                = (RASBASE+38);
  ERROR_CANNOT_GET_LANA                = (RASBASE+39);
  ERROR_NETBIOS_ERROR                  = (RASBASE+40);
  ERROR_SERVER_OUT_OF_RESOURCES        = (RASBASE+41);
  ERROR_NAME_EXISTS_ON_NET             = (RASBASE+42);
  ERROR_SERVER_GENERAL_NET_FAILURE     = (RASBASE+43);
  WARNING_MSG_ALIAS_NOT_ADDED          = (RASBASE+44);
  ERROR_AUTH_INTERNAL                  = (RASBASE+45);
  ERROR_RESTRICTED_LOGON_HOURS         = (RASBASE+46);
  ERROR_ACCT_DISABLED                  = (RASBASE+47);
  ERROR_PASSWD_EXPIRED                 = (RASBASE+48);
  ERROR_NO_DIALIN_PERMISSION           = (RASBASE+49);
  ERROR_SERVER_NOT_RESPONDING          = (RASBASE+50);
  ERROR_FROM_DEVICE                    = (RASBASE+51);
  ERROR_UNRECOGNIZED_RESPONSE          = (RASBASE+52);
  ERROR_MACRO_NOT_FOUND                = (RASBASE+53);
  ERROR_MACRO_NOT_DEFINED              = (RASBASE+54);
  ERROR_MESSAGE_MACRO_NOT_FOUND        = (RASBASE+55);
  ERROR_DEFAULTOFF_MACRO_NOT_FOUND     = (RASBASE+56);
  ERROR_FILE_COULD_NOT_BE_OPENED       = (RASBASE+57);
  ERROR_DEVICENAME_TOO_LONG            = (RASBASE+58);
  ERROR_DEVICENAME_NOT_FOUND           = (RASBASE+59);
  ERROR_NO_RESPONSES                   = (RASBASE+60);
  ERROR_NO_COMMAND_FOUND               = (RASBASE+61);
  ERROR_WRONG_KEY_SPECIFIED            = (RASBASE+62);
  ERROR_UNKNOWN_DEVICE_TYPE            = (RASBASE+63);
  ERROR_ALLOCATING_MEMORY              = (RASBASE+64);
  ERROR_PORT_NOT_CONFIGURED            = (RASBASE+65);
  ERROR_DEVICE_NOT_READY               = (RASBASE+66);
  ERROR_READING_INI_FILE               = (RASBASE+67);
  ERROR_NO_CONNECTION                  = (RASBASE+68);
  ERROR_BAD_USAGE_IN_INI_FILE          = (RASBASE+69);
  ERROR_READING_SECTIONNAME            = (RASBASE+70);
  ERROR_READING_DEVICETYPE             = (RASBASE+71);
  ERROR_READING_DEVICENAME             = (RASBASE+72);
  ERROR_READING_USAGE                  = (RASBASE+73);
  ERROR_READING_MAXCONNECTBPS          = (RASBASE+74);
  ERROR_READING_MAXCARRIERBPS          = (RASBASE+75);
  ERROR_LINE_BUSY                      = (RASBASE+76);
  ERROR_VOICE_ANSWER                   = (RASBASE+77);
  ERROR_NO_ANSWER                      = (RASBASE+78);
  ERROR_NO_CARRIER                     = (RASBASE+79);
  ERROR_NO_DIALTONE                    = (RASBASE+80);
  ERROR_IN_COMMAND                     = (RASBASE+81);
  ERROR_WRITING_SECTIONNAME            = (RASBASE+82);
  ERROR_WRITING_DEVICETYPE             = (RASBASE+83);
  ERROR_WRITING_DEVICENAME             = (RASBASE+84);
  ERROR_WRITING_MAXCONNECTBPS          = (RASBASE+85);
  ERROR_WRITING_MAXCARRIERBPS          = (RASBASE+86);
  ERROR_WRITING_USAGE                  = (RASBASE+87);
  ERROR_WRITING_DEFAULTOFF             = (RASBASE+88);
  ERROR_READING_DEFAULTOFF             = (RASBASE+89);
  ERROR_EMPTY_INI_FILE                 = (RASBASE+90);
  ERROR_AUTHENTICATION_FAILURE         = (RASBASE+91);
  ERROR_PORT_OR_DEVICE                 = (RASBASE+92);
  ERROR_NOT_BINARY_MACRO               = (RASBASE+93);
  ERROR_DCB_NOT_FOUND                  = (RASBASE+94);
  ERROR_STATE_MACHINES_NOT_STARTED     = (RASBASE+95);
  ERROR_STATE_MACHINES_ALREADY_STARTED = (RASBASE+96);
  ERROR_PARTIAL_RESPONSE_LOOPING       = (RASBASE+97);
  ERROR_UNKNOWN_RESPONSE_KEY           = (RASBASE+98);
  ERROR_RECV_BUF_FULL                  = (RASBASE+99);
  ERROR_CMD_TOO_LONG                   = (RASBASE+100);
  ERROR_UNSUPPORTED_BPS                = (RASBASE+101);
  ERROR_UNEXPECTED_RESPONSE            = (RASBASE+102);
  ERROR_INTERACTIVE_MODE               = (RASBASE+103);
  ERROR_BAD_CALLBACK_NUMBER            = (RASBASE+104);
  ERROR_INVALID_AUTH_STATE             = (RASBASE+105);
  ERROR_WRITING_INITBPS                = (RASBASE+106);
  ERROR_X25_DIAGNOSTIC                 = (RASBASE+107);
  ERROR_ACCT_EXPIRED                   = (RASBASE+108);
  ERROR_CHANGING_PASSWORD              = (RASBASE+109);
  ERROR_OVERRUN                        = (RASBASE+110);
  ERROR_RASMAN_CANNOT_INITIALIZE	     = (RASBASE+111);
  ERROR_BIPLEX_PORT_NOT_AVAILABLE      = (RASBASE+112);
  ERROR_NO_ACTIVE_ISDN_LINES           = (RASBASE+113);
  ERROR_NO_ISDN_CHANNELS_AVAILABLE     = (RASBASE+114);
  ERROR_TOO_MANY_LINE_ERRORS           = (RASBASE+115);
  ERROR_IP_CONFIGURATION               = (RASBASE+116);
  ERROR_NO_IP_ADDRESSES                = (RASBASE+117);
  ERROR_PPP_TIMEOUT                    = (RASBASE+118);
  ERROR_PPP_REMOTE_TERMINATED          = (RASBASE+119);
  ERROR_PPP_NO_PROTOCOLS_CONFIGURED    = (RASBASE+120);
  ERROR_PPP_NO_RESPONSE                = (RASBASE+121);
  ERROR_PPP_INVALID_PACKET             = (RASBASE+122);
  ERROR_PHONE_NUMBER_TOO_LONG          = (RASBASE+123);
  ERROR_IPXCP_NO_DIALOUT_CONFIGURED    = (RASBASE+124);
  ERROR_IPXCP_NO_DIALIN_CONFIGURED     = (RASBASE+125);
  ERROR_IPXCP_DIALOUT_ALREADY_ACTIVE   = (RASBASE+126);
  ERROR_ACCESSING_TCPCFGDLL            = (RASBASE+127);
  ERROR_NO_IP_RAS_ADAPTER              = (RASBASE+128);
  ERROR_SLIP_REQUIRES_IP               = (RASBASE+129);
  ERROR_PROJECTION_NOT_COMPLETE        = (RASBASE+130);
  ERROR_PROTOCOL_NOT_CONFIGURED        = (RASBASE+131);
  ERROR_PPP_NOT_CONVERGING             = (RASBASE+132);
  ERROR_PPP_CP_REJECTED                = (RASBASE+133);
  ERROR_PPP_LCP_TERMINATED             = (RASBASE+134);
  ERROR_PPP_REQUIRED_ADDRESS_REJECTED  = (RASBASE+135);
  ERROR_PPP_NCP_TERMINATED             = (RASBASE+136);
  ERROR_PPP_LOOPBACK_DETECTED          = (RASBASE+137);
  ERROR_PPP_NO_ADDRESS_ASSIGNED        = (RASBASE+138);
  ERROR_CANNOT_USE_LOGON_CREDENTIALS   = (RASBASE+139);
  ERROR_TAPI_CONFIGURATION             = (RASBASE+140);
  ERROR_NO_LOCAL_ENCRYPTION            = (RASBASE+141);
  ERROR_NO_REMOTE_ENCRYPTION           = (RASBASE+142);
  ERROR_REMOTE_REQUIRES_ENCRYPTION     = (RASBASE+143);
  ERROR_IPXCP_NET_NUMBER_CONFLICT      = (RASBASE+144);
  ERROR_INVALID_SMM                    = (RASBASE+145);
  ERROR_SMM_UNINITIALIZED              = (RASBASE+146);
  ERROR_NO_MAC_FOR_PORT                = (RASBASE+147);
  ERROR_SMM_TIMEOUT                    = (RASBASE+148);
  ERROR_BAD_PHONE_NUMBER               = (RASBASE+149);
  ERROR_WRONG_MODULE                   = (RASBASE+150);


  RASBASEEND                           = (RASBASE+150);

implementation

function RasCreatePhonebookEntryA; external 'rasapi32.dll' name 'RasCreatePhonebookEntryA';
function RasCreatePhonebookEntryW; external 'rasapi32.dll' name 'RasCreatePhonebookEntryW';
function RasCreatePhonebookEntry;  external 'rasapi32.dll' name 'RasCreatePhonebookEntryA';
function RasDialA;                 external 'rasapi32.dll' name 'RasDialA';
function RasDialW;                 external 'rasapi32.dll' name 'RasDialW';
function RasDial;                  external 'rasapi32.dll' name 'RasDialA';
function RasEditPhonebookEntryA;   external 'rasapi32.dll' name 'RasEditPhonebookEntryA';
function RasEditPhonebookEntryW;   external 'rasapi32.dll' name 'RasEditPhonebookEntryW';
function RasEditPhonebookEntry;    external 'rasapi32.dll' name 'RasEditPhonebookEntryA';
function RasEnumConnectionsA;      external 'rasapi32.dll' name 'RasEnumConnectionsA';
function RasEnumConnectionsW;      external 'rasapi32.dll' name 'RasEnumConnectionsW';
function RasEnumConnections;       external 'rasapi32.dll' name 'RasEnumConnectionsA';
function RasEnumEntriesA;          external 'rasapi32.dll' name 'RasEnumEntriesA';
function RasEnumEntriesW;          external 'rasapi32.dll' name 'RasEnumEntriesW';
function RasEnumEntries;           external 'rasapi32.dll' name 'RasEnumEntriesA';
function RasGetConnectStatusA;     external 'rasapi32.dll' name 'RasGetConnectStatusA';
function RasGetConnectStatusW;     external 'rasapi32.dll' name 'RasGetConnectStatusW';
function RasGetConnectStatus;      external 'rasapi32.dll' name 'RasGetConnectStatusA';
function RasGetEntryDialParamsA;   external 'rasapi32.dll' name 'RasGetEntryDialParamsA';
function RasGetEntryDialParamsW;   external 'rasapi32.dll' name 'RasGetEntryDialParamsW';
function RasGetEntryDialParams;    external 'rasapi32.dll' name 'RasGetEntryDialParamsA';
function RasGetErrorStringA;       external 'rasapi32.dll' name 'RasGetErrorStringA';
function RasGetErrorStringW;       external 'rasapi32.dll' name 'RasGetErrorStringW';
function RasGetErrorString;        external 'rasapi32.dll' name 'RasGetErrorStringA';
function RasGetProjectionInfoA;    external 'rasapi32.dll' name 'RasGetProjectionInfoA';
function RasGetProjectionInfoW;    external 'rasapi32.dll' name 'RasGetProjectionInfoW';
function RasGetProjectionInfo;     external 'rasapi32.dll' name 'RasGetProjectionInfoA';
function RasHangUpA;               external 'rasapi32.dll' name 'RasHangUpA';
function RasHangUpW;               external 'rasapi32.dll' name 'RasHangUpW';
function RasHangUp;                external 'rasapi32.dll' name 'RasHangUpA';
function RasSetEntryDialParamsA;   external 'rasapi32.dll' name 'RasSetEntryDialParamsA';
function RasSetEntryDialParamsW;   external 'rasapi32.dll' name 'RasSetEntryDialParamsW';
function RasSetEntryDialParams;    external 'rasapi32.dll' name 'RasSetEntryDialParamsA';

end.
