unit Raycast;

interface

uses
  Windows, Classes, Controls, Graphics;

type
  pPoint2D = ^TPoint2D;
  TPoint2D = record
    X,Y: Single;
  end;

  pVector2D = ^TVector2D;
  TVector2D = TPoint2D;

  pRect2D = ^TRect2D;
  TRect2D = record
    case Integer of
      0:  (Left, Top, Right, Bottom: Single);
      1:  (TopLeft, BottomRight: TPoint2D);
  end;

  { TCustomFilm
      This is a non-visual component that encapsulates a DIB section. A
      Canvas is provided for GDI calls to paint the DIB surface, and you
      can use the Pixels pointer to directly read/write the surface. A
      DC is created to support the canvas/colors for the DIB, so don't
      make a gazillion TCustomFilms in your app - that's not what it is
      designed for. It is designed to give you an off-screen buffer
      (for double-buffering techniques) which you can directly read/write.

      All TCustomFilm DIBs are top-down; that is, pChar(Pixels)[0] is
      the top-left pixel. If you want bottom-up, then rewrite
      RecreateBitmap so that FHeight (instead of -FHeight) is used in
      pInfo (used in a call to CreateDibSection).

      Use ScanWidth to find a scanline in the DIB:
        Line 0 = pChar(Pixels)
        Line 1 = pChar(Pixels) + ScanWidth
        Line 2 = pChar(Pixels) + 2*ScanWidth
        Line 3 = pChar(Pixels) + 3*ScanWidth
        ...

      By default TCustomFilm instantiates as a 320 x 200 DIB that has the
      same color depth as the system (ie an 8-bit DIB if you are
      running in 256-color mode, etc).

      Be carefull changing ColorDepth; on my I cannot blit 8-bit DIBs when
      in 16-bit or 24-bit graphics mode. I CAN blit 16-bit DIBs on a
      24-bit screen tho...
      I'd suggest using one of the following algorithms:
      
        "Grinch" approach: support only one mode (say, 8-bit). At startup
            test if the application is in this mode (you can check
            TCustomFilm.Create() for how I do this) and if it isn't, tell
            the user to change color modes and try again. Terminate the
            application.

        "Liberal" approach: support whichever mode the system comes up
            in. This probably means different code sets for different
            color depth support (ie code for 8-bit, 16-bit, 24-bit).
            
        "Hybrid" approach: a mixture of the two above. Allow a limited
            subset of modes (ie 8-bit..32-bit, but not 1-bit or 4-bit).

    procedure RecreateBitmap    (protected)
      Recreates the bitmap (due to size or colordepth change).

    procedure SetBounds(AWidth, AHeight: Integer)
      Resize the DIB (more efficient than changing Width, then Height).

    property Canvas: TCanvas    (read-only)
      Canvas that uses the DIB section.

    property ColorDepth: Integer    (read-write)
      Bits-per-pixel of the DIB section. By default is the same color
      depth as the display.

    property Height: Integer    (read-write)
      Height of the DIB section in pixels.

    property Palette: HPALETTE    (read-write)
      Canvas palette (only used when ColorDepth = 8).

    property Pixels: Pointer      (read-only)
      Pointer to the DIB section pixel data.

    property ScanWidth: Integer   (read-only)
      Width of a scan line in bytes.

    property Width: Integer     (read-write)
      Width of the DIB section in pixels.
  }

  TCustomFilm = class(TComponent)
  private
    FWidth: Integer;
    FScanWidth: Integer;
    FHeight: Integer;
    FColors: Integer;
    FCanvas: TCanvas;
    FOldBmp: HBITMAP;
    FOldPal: HPALETTE;
    FCurPal: HPALETTE;
    FCurBmp: HBITMAP;
    FPixels: Pointer;
    procedure SetColors(Value: Integer);
    procedure SetHeight(Value: Integer);
    procedure SetWidth(Value: Integer);
    procedure SetPalette(Pal: HPALETTE);
  protected
    procedure RecreateBitmap;
    property Canvas: TCanvas read FCanvas;
    property ColorDepth: Integer read FColors write SetColors;
    property Width: Integer read FWidth write SetWidth default 320;
    property ScanWidth: Integer read FScanWidth;
    property Height: Integer read FHeight write SetHeight default 200;
    property Palette: HPALETTE read FCurPal write SetPalette;
    property Pixels: Pointer read FPixels;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure SetBounds(AWidth, AHeight: Integer);
  end;

  TFilm = class(TCustomFilm)
  public
    property Canvas;
    property ScanWidth;
  published
    property ColorDepth;
    property Height;
    property Palette;
    property Pixels;
    property Width;
  end;

function CopyPalette(OldPal: HPALETTE): HPALETTE;

implementation

uses
  SysUtils, Dialogs;

type
  TPalEntryArray = array [0..255] of TPaletteEntry;
  pPalEntryArray = ^TPalEntryArray;
  TRGBQuadArray = array [0..255] of TRGBQuad;
  pRGBQuadArray = ^TRGBQuadArray;

procedure WindowsError;
begin
  raise Exception.Create('Windows error');
end;

function CopyPalette(OldPal: HPALETTE): HPALETTE;
var
  pLogPal: pLogPalette;
begin
  Result := 0;
  if OldPal = 0 then Exit;
  pLogPal := AllocMem(SizeOf(TLogPalette) + 255 * SizeOf(TPaletteEntry));
  try
    FillChar(pLogPal^, SizeOf(TLogPalette) + 255*SizeOf(TPaletteEntry), 0);
    pLogPal^.palVersion := $0300;
    pLogPal^.palNumEntries := GetPaletteEntries(OldPal, 0, 256, pLogPal^.palPalEntry);
    if pLogPal^.palNumEntries > 0 then begin
      Result := CreatePalette(pLogPal^);
      if Result = 0 then WindowsError;
    end;        
  finally
    FreeMem(pLogPal);
  end;
end;

{==== TFilm ========================================================}

constructor TCustomFilm.Create(AOwner: TComponent);
var
  DC: HDC;
  pInfo: pBitmapInfo;
  I, N: Integer;
  pEntry: pPalEntryArray;
begin
  inherited Create(AOwner);
  FCanvas := TCanvas.Create;
  DC := CreateCompatibleDC(0);
  if DC = 0 then WindowsError;
  try
    FCanvas.Handle := DC;
  except
    DeleteDC(DC);
    raise;
  end;
  FColors := GetDeviceCaps(FCanvas.Handle, BitsPixel) *
             GetDeviceCaps(FCanvas.Handle, Planes);
  { create a 1x1 (actually 4x1) DIB section }
  FWidth := 1;
  FScanWidth := 4;
  FHeight := 1;
  if FColors <= 8 then N := 1 shl FColors
  else N := 0;
  pInfo := AllocMem(SizeOf(TBitmapInfoHeader) + N*SizeOf(TRGBQuad));
  try
    FillChar(pInfo^.bmiHeader, SizeOf(TBitmapInfoHeader), 0);
    with pInfo^.bmiHeader do begin
      biSize := SizeOf(TBitmapInfoHeader);
      biWidth := FScanWidth;
      biHeight := -FHeight;
      biPlanes := 1;
      biBitCount := FColors;
      biCompression := BI_RGB;
    end;
    if N > 0 then begin
      pEntry := AllocMem(N*SizeOf(TPaletteEntry));
      try
        GetSystemPaletteEntries(FCanvas.Handle, 0, N, pEntry^);
        for I := 0 to N-1 do
          with pInfo^.bmiColors[I], pEntry^[I] do begin
            rgbRed := peRed;
            rgbGreen := peGreen;
            rgbBlue := peBlue;
            rgbReserved := 0;
          end;
      finally
        FreeMem(pEntry);
      end;
    end;
    FCurBmp := CreateDibSection(FCanvas.Handle, pInfo^, DIB_RGB_COLORS,
                FPixels, nil, 0);
    if FCurBmp = 0 then WindowsError;
    FOldBmp := SelectObject(FCanvas.Handle, FCurBmp);
  finally
    FreeMem(pInfo);
  end;
  SetBounds(320, 200);
end;

destructor TCustomFilm.Destroy;
var
  DC: HDC;
begin
  if FCanvas <> nil then begin
    DC := FCanvas.Handle;
    FCanvas.Handle := 0;
    if DC <> 0 then begin
      if FOldBmp <> 0 then begin
        DeleteObject(SelectObject(DC, FOldBmp));
        FOldBmp := 0;
      end;
      if FOldPal <> 0 then begin
        DeleteObject(SelectPalette(DC, FOldPal, False));
        FOldPal := 0;
      end;
      DeleteDC(DC);
    end;
  end;
  FCanvas.Free;
  inherited Destroy;
end;

procedure TCustomFilm.RecreateBitmap;
var
  Bmp: HBITMAP;
  pInfo: pBitmapInfo;
  N: Integer;
  P: Pointer;
begin
  FScanWidth := (((FWidth * FColors) div 8) + 3) and (not 3);
  if FColors <= 8 then N := 1 shl FColors
  else N := 0;
  pInfo := AllocMem(Sizeof(TBitmapInfoHeader) + N*SizeOf(TRGBQuad));
  try
    FillChar(pInfo^.bmiHeader, SizeOf(TBitmapInfoHeader), 0);
    with pInfo^.bmiHeader do begin
      biSize := SizeOf(TBitmapInfoHeader);
      biWidth := FWidth;
      biHeight := -FHeight;
      biPlanes := 1;
      biBitCount := FColors;
      biCompression := BI_RGB;
    end;
    if N > 0 then
      GetDIBColorTable(FCanvas.Handle, 0, N, pInfo^.bmiColors);
    Bmp := CreateDibSection(FCanvas.Handle, pInfo^, DIB_RGB_COLORS,
                P, nil, 0);
    if Bmp = 0 then WindowsError;
    DeleteObject(SelectObject(FCanvas.Handle, Bmp));
    FPixels := P;
  finally
    FreeMem(pInfo);
  end;
end;

procedure TCustomFilm.SetBounds(AWidth, AHeight: Integer);
begin
  if (AWidth <= 0) or (AHeight <= 0) then Exit;
{  AWidth := (AWidth + 3) and (not 3);}
  if (AWidth <> FWidth) or (AHeight <> FHeight) then begin
    FWidth := AWidth;
    FHeight := AHeight;
    RecreateBitmap;
  end;
end;

procedure TCustomFilm.SetColors(Value: Integer);
begin
  if (Value <> 1) and (Value <> 4) and (Value <> 8) and (Value <> 16) and
      (Value <> 24) and (Value <> 32) then Exit;
  if Value <> FColors then begin
    FColors := Value;
    RecreateBitmap;
  end;
end;

procedure TCustomFilm.SetHeight(Value: Integer);
begin
  SetBounds(FWidth, Value);
end;

procedure TCustomFilm.SetPalette(Pal: HPALETTE);
var
  H: HPALETTE;
  I, N: Integer;
  pPal: pPalEntryArray;
  pRGB: pRGBQuadArray;
begin
  if (Pal = 0) or (ColorDepth <> 8) then Exit;
  H := CopyPalette(Pal);
  if H = 0 then Exit;
  if FOldPal = 0 then FOldPal := SelectPalette(FCanvas.Handle, H, False)
  else DeleteObject(SelectPalette(FCanvas.Handle, H, False));
  pPal := AllocMem(256*SizeOf(TPaletteEntry));
  try
    pRGB := AllocMem(256*SizeOf(TRGBQuad));
    try
      N := GetPaletteEntries(H, 0, 256, pPal^);
      if N = 0 then WindowsError;
      for I := 0 to 255 do
        with pPal^[I], pRGB^[I] do begin
          rgbRed := peRed;
          rgbGreen := peGreen;
          rgbBlue := peBlue;
          rgbReserved := 0;
        end;
      SetDibColorTable(FCanvas.Handle, 0, 256, pRGB^);
    finally
      FreeMem(pRGB);
    end;
  finally
    FreeMem(pPal);
  end;
end;

procedure TCustomFilm.SetWidth(Value: Integer);
begin
  SetBounds(Value, FHeight);
end;

end.
