unit MsgSock;

{******************************************************************************
 Nome:			MsgSock
 Scopo:     MsgSock permette lo scambio di messaggi tra applicazioni diverse.
 Generalit:MsgSock permette di scambiare messaggi tra diverse applicazioni
 						Windows che ospitino una sua istanza. Esso fornisce metodi per
            inviare un messaggio in broadcast o ad una sua istanza di cui si
            conosce l'handle. Inoltre dispone di un evento attivato alla
            ricezione di un messaggio.
 Autore:    Ing. M. Venturini
 Revisore:
 Data:      15 Jul. 96
 Revisioni:
******************************************************************************}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls, Forms;

const
	MsgSockReservedID = 1024;
	MsgSockIDMin = wm_User + MsgSockReservedID;
	MsgSockIDMax = MsgSockIDMin + MsgSockReservedID - 1;

type
  TMsgReceived = procedure(				Sender: TObject;
  												 const	MsgID: Word;
  												 const	SourceHandle: Hwnd;
    											 const	DataPtr: Pointer) of object;
  TMsgSock = class(TComponent)
  private
    WindowHandle: Hwnd;
    FOnMsgReceived: TMsgReceived;
    procedure WndProc(var Msg: TMessage);
	public
   	procedure MsgBroadcast(const	MsgID: Word;
    											 const 	DataPtr: Pointer);
   	procedure MsgSend(const	MsgID: Word;
    									const DestinationHandle: Hwnd;
    									const DataPtr: Pointer);
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
  	property Handle: Hwnd read WindowHandle;
  	property OnMsgReceived: TMsgReceived read FOnMsgReceived write FOnMsgReceived;
  end;
  EMsgSock = class(Exception);
  EIllegalMsgID = class(EMsgSock);

procedure Register;

{*****************************************************************************}

implementation

const EIllegalMsgIDMsg = ' - Illegal message ID: ';

procedure Register;
begin

  RegisterComponents('Additional', [TMsgSock]);

end;

{******************************************************************************
 --	procedure WndProc [method]
    Window procedure per il componente. Si occupa della ridirezione dei
 		messaggi non destinati al componente.
******************************************************************************}

procedure TMsgSock.WndProc(var Msg: TMessage);
begin

  with Msg do
    if (Msg >= MsgSockIDMin) and (Msg <= MsgSockIDMax) then begin
      try
				if Assigned(FOnMsgReceived) then FOnMsgReceived(Self,Msg - MsgSockIDMin,Hwnd(WParam),Pointer(LParam));
      except
        Application.HandleException(Self);
      end
    end
    else
      Result := DefWindowProc(WindowHandle, Msg, WParam, LParam);

end;

{******************************************************************************
 --	procedure MsgBroadcast [method]
    Invia un messaggio in broadcast.
 -- Parametri
		MsgID:					in Word
    								Identificatore che individua il tipo di messaggio.
    DataPtr:				in Pointer
    								Puntatore ai dati che si vogliono trasmettere con il
                    messaggio.
******************************************************************************}

procedure TMsgSock.MsgBroadcast(const	MsgID: Word;
																const DataPtr: Pointer);
begin

	if MsgId < MsgSockReservedID then
	  SendMessage(hwnd_Broadcast,MsgSockIDMin + MsgID,WindowHandle,Longint(DataPtr))
  else
  	raise EIllegalMsgID.Create(TMsgSock.ClassName + EIllegalMsgIDMsg + IntToStr(MsgID));

end;

{******************************************************************************
 --	procedure MsgSend [method]
    Invia un messaggio ad un altro componente TMsgSock.
 -- Parametri
		MsgID:					in Word
    								Identificatore che individua il tipo di messaggio.
    DestinationHandle: in Hwnd
    								Handle del un componente TMsgSock a cui si vuole inviare
                    il messaggio.
    DataPtr:				in Pointer
    								Puntatore ai dati che si vogliono trasmettere con il
                    messaggio.
******************************************************************************}

procedure TMsgSock.MsgSend(const MsgID: Word;
													 const DestinationHandle: Hwnd;
    											 const DataPtr: Pointer);
begin

	if MsgId < MsgSockReservedID then
	  SendMessage(DestinationHandle,MsgSockIDMin + MsgID,WindowHandle,Longint(DataPtr))
  else
  	raise EIllegalMsgID.Create(TMsgSock.ClassName + EIllegalMsgIDMsg+ IntToStr(MsgID));

end;

{******************************************************************************
 --	procedure Create [constructor]
 Constructor per la classe.
******************************************************************************}

constructor TMsgSock.Create(AOwner: TComponent);
begin

	inherited Create(AOwner);
  WindowHandle := AllocateHWnd(WndProc);

end;

{******************************************************************************
 --	procedure Destroy [destructor]
 Destructor per la classe.
******************************************************************************}

destructor TMsgSock.Destroy;
begin

  DeallocateHWnd(WindowHandle);
  inherited Destroy;

end;


end.
