(*
   TdkwSplitPanel

   The SplitPanel is designed to automatically maintain a ratio between
   the size of two panels that divide a rectangular area horizontally or
   vertically

   (c) 1995 by DKW Systems Corporation, All Rights Reserved
   Originally written by Blake Stone, May 29, 1995

   Change History:

     May 31, 1995
       BWS - Fixed most archiving problems

     Sept 17, 1995
       BWS - Fixed cut and paste of entire SplitPanel in design mode
       BWS - Added black split bar

     ?
       BWS - << Fixed cut and paste INTO SplitPanels ... >>
*)

unit DKWSpPnl;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, ExtCtrls;

const
  TdkwSplitPanelBarWidth: Integer = 4 ;

type
  TdkwSplitPanelOrientation = ( dkwspHorizontal, dkwspVertical ) ;

  TdkwSplitPanel = class(TCustomPanel)
  private
    { Private declarations }
    FRatio: Array [ 0 .. 1 ] of Integer ;
    FDragOffset: Integer ;
    FAdjustable: Boolean ;
    FOrientation: TdkwSplitPanelOrientation ;
    procedure Resplit ;
    procedure SetRatio ( Index: Integer; Value: Integer ) ;
    function GetBevel ( Index: Integer ): TPanelBevel ;
    procedure SetBevel ( Index: Integer; Value: TPanelBevel ) ;
    procedure SetOrientation ( NewState: TdkwSplitPanelOrientation ) ;
    procedure SetAdjustable ( NewState: Boolean ) ;
  protected
    { Protected declarations }
  public
    { Public declarations }
    constructor Create ( AOwner : TComponent ) ; override ;
    procedure Resize ; override ;
    procedure Loaded ; override ;
    procedure GetChildren ( Proc: TGetChildProc ) ; override ;
    procedure ReadState ( Reader: TReader ) ; override ;
    procedure MouseDown ( Button: TMouseButton ; Shift: TShiftState; X, Y: Integer ) ; override ;
    procedure MouseUp ( Button: TMouseButton ; Shift: TShiftState; X, Y: Integer ) ; override ;
    procedure MouseMove ( Shift: TShiftState; X, Y: Integer ) ; override ;
  published
    { Published declarations }
    property Align ;
    property BevelOne: TPanelBevel index 0 read GetBevel write SetBevel stored False ;
    property BevelTwo: TPanelBevel index 1 read GetBevel write SetBevel stored False ;
    property RatioOne: Integer index 0 read FRatio [ 0 ] write SetRatio default 1 ;
    property RatioTwo: Integer index 1 read FRatio [ 1 ] write SetRatio default 1 ;
    property Orientation: TdkwSplitPanelOrientation read FOrientation write SetOrientation default dkwspVertical ;
    property Adjustable: Boolean read FAdjustable write SetAdjustable default False ;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('DKW', [TdkwSplitPanel]);
end;

constructor TdkwSplitPanel.Create ( AOwner : TComponent ) ;
begin
  inherited Create ( AOwner ) ;

  (* Set the defaults we declared earlier *)

  FRatio [ 0 ] := 1 ;
  FRatio [ 1 ] := 1 ;
  FOrientation := dkwspVertical ;
  FAdjustable := False ;

  Caption := '' ;
  Color := clBlack ;

  (*
     Only create the sub-panels in design mode.  They will be
     archived and restored automatically at run-time

     NOTE: The sub-panels are created with the TdkwSplitPanel
           as their owner to prevent developers from moving
           the panels around in design mode ... the owner
           becomes the form at run-time
  *)

  if ( csDesigning in ComponentState ) then
  begin
    InsertControl ( TPanel.Create ( self ) ) ;
    InsertControl ( TPanel.Create ( self ) ) ;
    Resplit ;
  end;
end;

(* Resize and reposition the sub-panels to the proper ratios *)

procedure TdkwSplitPanel.Resplit ;
var
  AdjustBarSize: Integer ;
begin
  if FAdjustable and ( Width > 8 ) and ( Height > 8 )
  then AdjustBarSize := TdkwSplitPanelBarWidth
  else AdjustBarSize := 0 ;

  if ControlCount >= 2 then
  begin
    if FOrientation = dkwspHorizontal then
    begin
      Cursor := crHSplit ;
      Controls [ 0 ].SetBounds ( 0, 0,
          ( LongInt ( FRatio [ 0 ] ) * ( Width - AdjustBarSize ) ) div ( FRatio [ 0 ] + FRatio [ 1 ] ), Height ) ;
      Controls [ 1 ].SetBounds ( Controls [ 0 ].Width + AdjustBarSize, 0,
          Width - Controls [ 0 ].Width - AdjustBarSize, Height ) ;
    end else begin
      Cursor := crVSplit ;
      Controls [ 0 ].SetBounds ( 0, 0,
          Width, ( LongInt ( FRatio [ 0 ] ) * ( Height - AdjustBarSize ) ) div ( FRatio [ 0 ] + FRatio [ 1 ] ) ) ;
      Controls [ 1 ].SetBounds ( 0, Controls [ 0 ].Height + AdjustBarSize,
          Width, Height - Controls [ 0 ].Height - AdjustBarSize ) ;
    end;
  end;
end;

(* Adjust the sub-panels during resize events *)

procedure TdkwSplitPanel.Resize ;
begin
  inherited Resize ;
  Resplit ;
end;

(* Deal with cut and paste problems in design mode *)

procedure TdkwSplitPanel.Loaded ;
begin
  inherited Loaded ;

  (*
     If in design mode, migrate the children from the cut 'n' pasted
     panels to our 'special' panels
  *)

  if ( csDesigning in ComponentState ) then
  begin
    while ( Controls [ 2 ] as TWinControl ).ControlCount > 0 do
      ( Controls [ 2 ] as TWinControl ).Controls [ 0 ].Parent :=
      Controls [ 0 ] as TWinControl ;
    while ( Controls [ 3 ] as TWinControl ).ControlCount > 0 do
      ( Controls [ 3 ] as TWinControl ).Controls [ 0 ].Parent :=
      Controls [ 1 ] as TWinControl ;
    Controls [ 3 ].Free ;
    Controls [ 2 ].Free ;
  end ;
end ;

(* << Write comment here >> *)

{procedure TdkwSplitPanel.InsertControl ( AControl: TControl ) ;
begin
  if ( csDesigning in ComponentState ) then
  begin
    if ControlCount > 2 then
    begin
      Controls [ 0 ].InsertControl ( AControl ) ;
    end;
  end ;
end;}

(* Allow indirect manipulation of the bevel type for sub-panels *)

function TdkwSplitPanel.GetBevel ( Index: Integer ): TPanelBevel ;
begin
  if ControlCount > Index
  then GetBevel := ( Controls [ Index ] as TPanel ).BevelOuter ;
end;

(* Allow indirect manipulation of the bevel type for sub-panels *)

procedure TdkwSplitPanel.SetBevel ( Index: Integer; Value: TPanelBevel ) ;
begin
  if ControlCount > Index
  then ( Controls [ Index ] as TPanel ).BevelOuter := Value ;
end;

(* Adjust the sub-panels when the ratio changes *)

procedure TdkwSplitPanel.SetRatio ( Index: Integer; Value: Integer ) ;
begin

  (*
     Permit the change for any positive value, or zero
     so long as BOTH ratios are not zero
  *)

  if ( Value > 0 ) or ( ( Value = 0 ) and
      ( FRatio [ 1 - Index ] <> 0 ) ) then
  begin
    FRatio [ Index ] := Value ;
    Resplit ;
  end;
end;

(* Adjust the sub-panels when the orientation changes *)

procedure TdkwSplitPanel.SetOrientation ( NewState: TdkwSplitPanelOrientation ) ;
begin
  if ( NewState <> FOrientation ) then
  begin
    FOrientation := NewState ;
    Resplit ;
  end;
end;

(* Adjust the sub-panels when the adjustability changes *)

procedure TdkwSplitPanel.SetAdjustable ( NewState: Boolean ) ;
begin
  if ( NewState <> FAdjustable ) then
  begin
    FAdjustable := NewState ;
    Resplit ;
  end;
end;

(*
   Since the TdkwSplitPanel is the owner, it needs to be responsible
   for archiving the panels and their contents
*)

procedure TdkwSplitPanel.GetChildren ( Proc: TGetChildProc ) ;
begin
  inherited ;
  Proc ( Controls [ 0 ] ) ;
  Proc ( Controls [ 1 ] ) ;
end ;

(*
   Since the TdkwSplitPanel class archives TPanel instances without
   Delphi's knowledge, we need to register the class just before
   reading instances from the archive
*)

procedure TdkwSplitPanel.ReadState ( Reader: TReader ) ;
begin
  RegisterClass ( TPanel ) ;
  inherited ReadState ( Reader ) ;
end;

(*
   When the user clicks in an exposed are of the SplitPanel
   it can be assumed to be a splitbar drag operation.
*)

procedure TdkwSplitPanel.MouseDown ( Button: TMouseButton ; Shift: TShiftState; X, Y: Integer ) ;
begin
  if Button = mbLeft then
  begin

    (*
       Preserve the relative distance from the bar to
       the cursor
    *)

    if FOrientation = dkwspHorizontal
    then FDragOffset := Controls [ 0 ].Width - X
    else FDragOffset := Controls [ 0 ].Height - Y ;

    (* Capture all mouse movements *)

    MouseCapture := True ;
  end
  else inherited MouseDown ( Button, Shift, X, Y ) ;
end;

(*
   Final cleanup is required when the splitbar is
   released
*)

procedure TdkwSplitPanel.MouseUp ( Button: TMouseButton ; Shift: TShiftState; X, Y: Integer ) ;
begin
  if Button = mbLeft then
  begin

    (* Make sure we're up to date on the split *)

    MouseMove ( Shift, X, Y ) ;

    (* Stop tracking the mouse *)

    MouseCapture := False ;
  end else inherited MouseUp ( Button, Shift, X, Y ) ;
end;

(*
   Perform real-time resplitting of the SplitPanel during
   drag operations
*)

procedure TdkwSplitPanel.MouseMove ( Shift: TShiftState; X, Y: Integer ) ;
begin
  if not MouseCapture then
  begin
    inherited MouseMove ( Shift, X, Y ) ;
    exit ;
  end;

  (* Adjust the split *)

  if FOrientation = dkwspHorizontal then
  begin
    FRatio [ 0 ] := X + FDragOffset ;
    FRatio [ 1 ] := Width - FRatio [ 0 ] - TdkwSplitPanelBarWidth ;
  end
  else
  begin
    FRatio [ 0 ] := Y + FDragOffset ;
    FRatio [ 1 ] := Height - FRatio [ 0 ] - TdkwSplitPanelBarWidth ;
  end;

  (* Don't allow negative split ratios! *)

  if FRatio [ 0 ] < 0 then
  begin
    FRatio [ 1 ] := FRatio [ 1 ] + FRatio [ 0 ] ;
    FRatio [ 0 ] := 0 ;
  end;
  if FRatio [ 1 ] < 0 then
  begin
    FRatio [ 0 ] := FRatio [ 0 ] + FRatio [ 1 ] ;
    FRatio [ 1 ] := 0 ;
  end;

  (* Redraw the results *)

  Resplit ;
end;

end.

