unit Graph;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls ;

type
  EdkwGraph = class ( Exception ) ;
  EdkwGraphData = class ( EdkwGraph ) ;

  TdkwGraph = class ;

  TdkwIntegerArrayData = Array [ 0..0 ] of Integer ;
  PdkwIntegerArrayData = ^TdkwIntegerArrayData ;
  TdkwIntegerArray = class(TObject)
  private
    Size: Integer ;
    Data: PdkwIntegerArrayData ;
    procedure SetValue ( Index: Integer ; Value: Integer ) ;
    function GetValue ( Index: Integer ): Integer ;
    constructor Create ;
    destructor Destroy ; override ;
  public
    property Value [ Index: Integer ]: Integer
      read GetValue write SetValue ; default ;
  end ;

  TdkwGraphAxis = class(TObject)
  private
    { Private declarations }
    FRect: TRect ;
    FColor: TColor ;
    FVisible: Boolean ;
    AxisRect: TRect ;
    Graph: TdkwGraph ;
  protected
    { Protected declarations }
    property Rect: TRect read FRect write FRect ;
    procedure Draw ( Name: String ; Canvas: TCanvas ) ; virtual ; abstract ;
  public
    { Public declarations }
    constructor Create ;
    destructor Destroy ; override ;
    property Color: TColor read FColor write FColor ;
    property Visible: Boolean read FVisible write FVisible ;
  end;

  TdkwGraphLabelEvent = procedure ( Sender: TObject ; Index: Integer;
      var AxisLabel: String ) of Object ;
  TdkwHorizontalGraphAxis = class(TdkwGraphAxis)
  private
    { Private declarations }
    FMinimum, FMaximum: Integer ;
    FMajorTick, FMinorTick: Integer ;
    FOnLabel: TdkwGraphLabelEvent ;
  protected
    { Protected declarations }
    procedure Draw ( Name: String ; Canvas: TCanvas ) ; override ;
  public
    { Public declarations }
    constructor Create ;
    function GetPoint ( Value: Integer ): Integer ;
    procedure GetRange ( Value: Integer ; var Low, High: Integer ) ;
    property Minimum: Integer read FMinimum write FMinimum ;
    property Maximum: Integer read FMaximum write FMaximum ;
    property MajorTick: Integer read FMajorTick write FMajorTick ;
    property MinorTick: Integer read FMinorTick write FMinorTick ;
    property OnLabel: TdkwGraphLabelEvent read FOnLabel write FOnLabel ;
  end ;

  TdkwVerticalGraphAxis = class(TdkwGraphAxis)
  private
    { Private declarations }
    FMinimum, FMaximum: Extended ;
  protected
    { Protected declarations }
    property Minimum: Extended read FMinimum write FMinimum ;
    property Maximum: Extended read FMaximum write FMaximum ;
    procedure Draw ( Name: String ; Canvas: TCanvas ) ; override ;
    procedure AdjustMinMax ;
  public
    { Public declarations }
    function GetPoint ( Value: Extended ): Integer ;
  end ;

  TdkwGraphColorEvent = procedure ( Sender: TObject ; Index: Integer ;
      var BarColor: TColor ) of Object ;
  TdkwGraphLayer = class(TObject)
  private
    { Private declarations }
    FXAxis: TdkwHorizontalGraphAxis ;
    FYAxis: TdkwVerticalGraphAxis ;
    FColor: TColor ;
    FOnColor: TdkwGraphColorEvent ;
    FVisible: Boolean ;
    Graph: TdkwGraph ;
  protected
    { Protected declarations }
    FData: TdkwIntegerArray ;
    procedure Draw ( Name: String ; Canvas: TCanvas ; Rect: TRect ) ; virtual ;
  public
    { Public declarations }
    constructor Create ;
    destructor Destroy ; override ;
    property Color: TColor read FColor write FColor ;
    property XAxis: TdkwHorizontalGraphAxis read FXAxis write FXAxis ;
    property YAxis: TdkwVerticalGraphAxis read FYAxis write FYAxis ;
    property OnColor: TdkwGraphColorEvent read FOnColor write FOnColor ;
    property Visible: Boolean read FVisible write FVisible ;
  end;

  TdkwLineGraphLayer = class(TdkwGraphLayer)
  private
    { Private declarations }
  protected
    { Protected declarations }
    procedure Draw ( Name: String ; Canvas: TCanvas ; Rect: TRect ) ; override ;
  public
    { Public declarations }
    property Data: TdkwIntegerArray read FData ;
  end;

  TdkwBarGraphLayer = class(TdkwGraphLayer)
  private
    { Private declarations }
  protected
    { Protected declarations }
    procedure Draw ( Name: String ; Canvas: TCanvas ; Rect: TRect ) ; override ;
  public
    { Public declarations }
    property Data: TdkwIntegerArray read FData ;
  end;

  TdkwHiLowGraphLayer = class(TdkwGraphLayer)
  private
    { Private declarations }
  protected
    { Protected declarations }
    FDataHigh: TdkwIntegerArray ;
    procedure Draw ( Name: String ; Canvas: TCanvas ; Rect: TRect ) ; override ;
  public
    { Public declarations }
    constructor Create ;
    destructor Destroy ; override ;
    property DataHigh: TdkwIntegerArray read FDataHigh ;
    property DataLow: TdkwIntegerArray read FData ;
  end;

  TdkwGraphClickEvent = procedure ( Sender: TObject ; Index: Integer ) of Object ;
  TdkwGraph = class(TCustomControl)
  private
    { Private declarations }
    FFixedWidth: Integer ;
    FOnClick: TdkwGraphClickEvent ;
    HorizontalAxes, VerticalAxes, GraphLayers: TStringList ;
    NeedsRestructure: Boolean ;
    GraphRect: TRect ;
    ScrollBar: TScrollBar ;
  protected
    { Protected declarations }
    procedure MouseDown ( Button: TMouseButton ; Shift: TShiftState ;
        X, Y: Integer ); override ;
    procedure Paint ; override ;
    procedure ScrollGraph ( Sender: TObject ; ScrollCode: TScrollCode ;
        var ScrollPos: Integer ) ;
  public
    { Public declarations }
    constructor Create ( AOwner: TComponent ) ; override ;
    destructor Destroy ; override ;
    procedure AddAxis ( Name: String ; Axis: TdkwGraphAxis ) ;
    procedure AddLayer ( Name: String ; Layer: TdkwGraphLayer ) ;
    procedure RemoveAxis ( Axis: TdkwGraphAxis ) ;
    procedure RemoveLayer ( Layer: TdkwGraphLayer ) ;
    procedure SetBounds ( ALeft, ATop, AWidth, AHeight: Integer ) ; override ;
    procedure Restructure ;
    property FixedWidth: Integer read FFixedWidth write FFixedWidth ;
    property OnClick: TdkwGraphClickEvent read FOnClick write FOnClick ;
  published
    { Published declarations }
  end;

procedure Register;

implementation

uses ExtCtrls ;

const
  VertLabelWidth = 30 ;
  VertLabelHeight = 40 ;
  HorizLabelWidth = 30 ;
  HorizLabelHeight = 30 ;

(**************************************************************
 *
 * Utility functions
 *
 **************************************************************)

(* Draw a label horizontally centered around a specific point *)

procedure dkwHorizLabel ( X, Y: Integer ; Canvas: TCanvas ; Value: String ) ;
begin
  Canvas.TextOut ( X - Canvas.TextWidth ( Value ) div 2, Y, Value ) ;
end ;

(* Draw a label vertically centered around a specific point *)

procedure dkwVertLabel ( X, Y: Integer ; Canvas: TCanvas ; Value: String ) ;
var
  Height, Index: Integer ;
begin
  Height := Canvas.TextHeight ( Value ) - 3 ;
  for Index := 1 to Length ( Value ) do
  begin
    Canvas.TextOut ( X - Canvas.TextWidth ( Value [ Index ] ) div 2,
        Y - ( Height * Length ( Value ) ) div 2 +
        Height * ( Index-1 ), Value [ Index ] ) ;
  end ;
end ;

(**************************************************************
 *
 * TdkwIntegerArray Implementation
 *
 * This class maintains a dynamic array which automatically
 * resizes to accomodate a large number of entries (up to the
 * 64k limit of the 16-bit Delphi memory allocation.)
 *
 **************************************************************)

(* Create a new dynamic array *)

constructor TdkwIntegerArray.Create ;
begin
  inherited Create ;

  { Start out with space for 128 entries }

  Size := 128 ;
  Data := AllocMem ( Size * sizeof ( Integer ) ) ;
end ;

(* Destroy a dynamic array *)

destructor TdkwIntegerArray.Destroy ;
begin

  { Free the array storage }

  FreeMem ( Data, Size ) ;
  inherited Destroy ;
end ;

(* Set a value in the array *)

procedure TdkwIntegerArray.SetValue ( Index: Integer ; Value: Integer ) ;
var
  OldData: PdkwIntegerArrayData ;
  NewSize: Integer ;
  LoopIndex: Integer ;
begin

  { Fail for negative indexes }

  if Index < 0 then
    raise EdkwGraphData.Create ( 'Negative array indexes are not supported' ) ;

  { Increase the allocated size if necessary, in increments of 128
    array elements }

  if Index >= Size then
  begin
    NewSize := Size + 128 ;
    while Index >= NewSize do
      NewSize := NewSize + 128 ;

    { Allocate new storage }

    OldData := Data ;
    Data := AllocMem ( NewSize * sizeof ( Integer ) ) ;

    { Copy the existing elements }

    for LoopIndex := 0 to Size-1 do
      Data^ [ LoopIndex ] := OldData^ [ LoopIndex ] ;
    FreeMem ( OldData, Size ) ;
    Size := NewSize ;
  end ;

  { Store the value }

  Data^ [ Index ] := Value ;
end ;

(* Read a value from the array *)

function TdkwIntegerArray.GetValue ( Index: Integer ): Integer ;
begin

  { Fail for out-of-range indexes }

  if ( Index >= Size ) or ( Index < 0 ) then
    raise EdkwGraphData.Create ( 'Integer array fetch out of bounds' ) ;

  { Fetch the value }

  Result := Data^ [ Index ] ;
end ;

(**************************************************************
 *
 * TdkwGraph Implementation
 *
 **************************************************************)

(* Create a multi-layer graph *)

constructor TdkwGraph.Create ( AOwner: TComponent ) ;
begin
  inherited Create ( AOwner ) ;

  { Allocate additional storage }

  HorizontalAxes := TStringList.Create ;
  VerticalAxes := TStringList.Create ;
  GraphLayers := TStringList.Create ;

  { Mark the graph as 'dirty' }

  NeedsRestructure := True ;
end ;

(* Destroy a multi-layer graph *)

destructor TdkwGraph.Destroy ;
begin

  { Clean up }

  { << Free things IN lists, too? >> }
  HorizontalAxes.Free ;
  VerticalAxes.Free ;
  GraphLayers.Free ;
  inherited Destroy ;
end ;

(* Add a new X or Y axis to the graph *)

procedure TdkwGraph.AddAxis ( Name: String ; Axis: TdkwGraphAxis ) ;
begin

  { Make sure the axis is unattached }

  if Axis.Graph <> nil then
    raise EdkwGraph.Create ( 'AddAxis: An axis cannot be added to more than one graph' ) ;

  { Mark the graph as 'dirty' }

  NeedsRestructure := True ;

  { Deal with horizontal axes }

  if Axis is TdkwHorizontalGraphAxis then
  begin
    Axis.Graph := self ;
    HorizontalAxes.AddObject ( Name, Axis ) ;
    Exit ;
  end ;

  { Deal with vertical axes }

  if Axis is TdkwVerticalGraphAxis then
  begin
    Axis.Graph := self ;
    VerticalAxes.AddObject ( Name, Axis ) ;
    Exit ;
  end ;

  { Any other axis type is unrecognized }

  raise EdkwGraph.Create ( 'AddAxis: Unknown axis type' ) ;
end ;

(* Add a new display layer to the graph *)

procedure TdkwGraph.AddLayer ( Name: String ; Layer: TdkwGraphLayer ) ;
begin

  { Make sure the layer is unattached }

  if Layer.Graph <> nil then
    raise EdkwGraph.Create ( 'AddLayer: A layer cannot be added to more than one graph' ) ;

  { Mark the graph as 'dirty' }

  NeedsRestructure := True ;

  Layer.Graph := self ;
  GraphLayers.AddObject ( Name, Layer ) ;
end ;

(* Remove an existing axis from the graph *)

procedure TdkwGraph.RemoveAxis ( Axis: TdkwGraphAxis ) ;
var
  Index: Integer ;
  AxisList: TStringList ;
begin

  { Make sure it is already attached }

  if Axis.Graph <> self then
    raise EdkwGraph.Create ( 'RemoveAxis: The graph and axis are not connected' ) ;

  { Determine the type of axis }

  AxisList := nil ;
  if Axis is TdkwHorizontalGraphAxis then
    AxisList := HorizontalAxes ;
  if Axis is TdkwVerticalGraphAxis then
    AxisList := VerticalAxes ;

  if AxisList = nil then
    raise EdkwGraph.Create ( 'RemoveAxis: Unknown axis type' ) ;

  Index := AxisList.IndexOfObject ( Axis ) ;
  if Index <> -1 then
  begin

    { Mark the graph as 'dirty' }

    NeedsRestructure := True ;
    Axis.Graph := nil ;
    AxisList.Delete ( Index ) ;
  end ;
end ;

(* Remove an existing display layer from the graph *)

procedure TdkwGraph.RemoveLayer ( Layer: TdkwGraphLayer ) ;
var
  Index: Integer ;
begin

  { Make sure it is already attached }

  if Layer.Graph <> self then
    raise EdkwGraph.Create ( 'RemoveLayer: The graph and layer are not connected' ) ;

  Index := GraphLayers.IndexOfObject ( Layer ) ;
  if Index <> -1 then
  begin

    { Mark the graph as 'dirty' }

    NeedsRestructure := True ;
    Layer.Graph := nil ;
    GraphLayers.Delete ( Index ) ;
  end ;
end ;

(* Prepare to draw the graph after changes *)

procedure TdkwGraph.Restructure ;
var
  Index, PointIndex, AxisCount: Integer ;
  TestRect: TRect ;
begin

  GraphRect := GetClientRect ;

  { Reduce the graph size for each visible axis present }

  for Index := 0 to VerticalAxes.Count-1 do
  with VerticalAxes.Objects [ Index ] as TdkwVerticalGraphAxis do
  begin
    if Visible then
      GraphRect.Left := GraphRect.Left + VertLabelWidth ;
  end ;

  for Index := 0 to HorizontalAxes.Count-1 do
  with HorizontalAxes.Objects [ Index ] as TdkwHorizontalGraphAxis do
  begin
    if Visible then
      GraphRect.Bottom := GraphRect.Bottom - HorizLabelHeight ;
  end ;

  { Test to see if we need a scroll bar }

  If FixedWidth > GraphRect.Right - GraphRect.Left then
  begin

    { Scrollbar needed, create one if not present }

    if ScrollBar = nil then
    begin
      ScrollBar := TScrollBar.Create ( self ) ;
      ScrollBar.Align := alBottom ;
      ScrollBar.Parent := self ;
      ScrollBar.OnScroll := ScrollGraph ;
    end ;

    { Adjust the existing scroll bar }

    ScrollBar.SmallChange := ( GraphRect.Right - GraphRect.Left ) div 8 ;
    ScrollBar.LargeChange := GraphRect.Right - GraphRect.Left ;
    ScrollBar.Max := FixedWidth - ( GraphRect.Right - GraphRect.Left ) ;
    GraphRect.Bottom := GraphRect.Bottom - ScrollBar.Height ;
  end else begin

    { Eliminate the scrollbar, if present }

    if ScrollBar <> nil then
    begin
      ScrollBar.Free ;
      ScrollBar := nil ;
    end ;

    { Reduce the graph size if needed }

    if FixedWidth <> 0 then
      GraphRect.Right := GraphRect.Left + FixedWidth ;
  end ;

  { Prepare vertical axes by setting their size,
    location, and default maxima / minima }

  AxisCount := 0 ;
  for Index := 0 to VerticalAxes.Count-1 do
  with VerticalAxes.Objects [ Index ] as TdkwVerticalGraphAxis do
  begin

    { Set the size and location }

    TestRect.Left := AxisCount * VertLabelWidth ;
    TestRect.Right := ( AxisCount + 1 ) * VertLabelWidth ;
    TestRect.Top := GraphRect.Top ;
    TestRect.Bottom := GraphRect.Bottom ;
    Rect := TestRect ;

    if Visible then AxisCount := AxisCount + 1 ;

    { Set default maxima / minima }

    Maximum := low ( Integer ) ;
    Minimum := high ( Integer ) ;
  end ;

  { Prepare horizontal axes by setting their
    size and location }

  AxisCount := 0 ;
  for Index := 0 to HorizontalAxes.Count-1 do
  with HorizontalAxes.Objects [ Index ] as TdkwHorizontalGraphAxis do
  begin

    { Set the size and location }

    TestRect.Left := GraphRect.Left ;
    TestRect.Right := GraphRect.Right ;
    TestRect.Top := GraphRect.Bottom + HorizLabelHeight * AxisCount ;
    TestRect.Bottom := GraphRect.Bottom + HorizLabelHeight * ( AxisCount + 1 ) ;
    Rect := TestRect ;

    if Visible then AxisCount := AxisCount + 1 ;
  end ;

  { Iterate through all appropriate layers,
    determining new maxima / minima for Y-axes }

  for Index := 0 to GraphLayers.Count-1 do
  begin

    { Walk through all normal data }

    with GraphLayers.Objects [ Index ] as TdkwGraphLayer do
    begin
      if Visible then for PointIndex := XAxis.Minimum to XAxis.Maximum do
      begin
        if FData [ PointIndex ] < YAxis.Minimum then
          YAxis.Minimum := FData [ PointIndex ] ;
        if FData [ PointIndex ] > YAxis.Maximum then
          YAxis.Maximum := FData [ PointIndex ] ;
      end ;
    end ;

    { Walk through hi / low data }

    if GraphLayers.Objects [ Index ] is TdkwHiLowGraphLayer then
    begin
      with TdkwHiLowGraphLayer ( GraphLayers.Objects [ Index ] ) do
      begin
        if Visible then for PointIndex := XAxis.Minimum to XAxis.Maximum do
        begin
          if FDataHigh [ PointIndex ] < YAxis.Minimum then
            YAxis.Minimum := FDataHigh [ PointIndex ] ;
          if FDataHigh [ PointIndex ] > YAxis.Maximum then
            YAxis.Maximum := FDataHigh [ PointIndex ] ;
        end ;
      end ;
    end ;

  end ;

  { Adjust minimum / maximum to reasonable round
    values }

  for Index := 0 to VerticalAxes.Count - 1 do
  with VerticalAxes.Objects [ Index ] as TdkwVerticalGraphAxis do
  begin
    AdjustMinMax ;
  end ;

  { Reset the 'dirty' flag and redraw }

  NeedsRestructure := False ;
  Refresh ;
end ;

(* Draw the graph *)

procedure TdkwGraph.Paint ;
var
  Index: Integer ;
  TempRect: TRect ;
begin

  { Restructure the graph if necessary }

  if NeedsRestructure then
  begin
    Restructure ;
    Exit ;
  end ;

  { Draw background }

  TempRect := GetClientRect ;
  if ScrollBar <> nil then
    TempRect.Bottom := TempRect.Bottom -
      ScrollBar.Height ;

  Canvas.Brush.Color := Color ;
  Canvas.FillRect ( TempRect ) ;

  { Draw axes }

  for Index := 0 to VerticalAxes.Count-1 do
    with VerticalAxes.Objects [ Index ] as TdkwGraphAxis do
      if Visible then
        Draw ( VerticalAxes.Strings [ Index ], Canvas ) ;

  { Change the clipping region to allow scrolling
    axes to draw arbitrarily }

  IntersectClipRect ( Canvas.Handle,
      GraphRect.Left,
      GraphRect.Top,
      GraphRect.Right,
      GraphRect.Top + Height ) ;

  for Index := 0 to HorizontalAxes.Count-1 do
    with HorizontalAxes.Objects [ Index ] as TdkwGraphAxis do
      if Visible then
        Draw ( HorizontalAxes.Strings [ Index ], Canvas ) ;

  { Change the clipping region to allow layers some
    freedom in drawing }

  IntersectClipRect ( Canvas.Handle,
      GraphRect.Left,
      GraphRect.Top,
      GraphRect.Right,
      GraphRect.Bottom ) ;

  { Draw graph layers }

  for Index := 0 to GraphLayers.Count-1 do
    with GraphLayers.Objects [ Index ] as TdkwGraphLayer do
      if Visible then
        Draw ( GraphLayers.Strings [ Index ], Canvas, GraphRect ) ;

  { Place graph border }

  TempRect := GraphRect ;
  Frame3D ( Canvas, TempRect, clBtnShadow, clBtnHighlight, 1 ) ;
end ;

(* Respond to scroll bar changes *)

procedure TdkwGraph.ScrollGraph ( Sender: TObject ; ScrollCode: TScrollCode ;
    var ScrollPos: Integer ) ;
begin
  if ScrollCode = scEndScroll then
    Refresh ;
end ;

(* Restructure when bounds change occurs *)

procedure TdkwGraph.SetBounds ( ALeft, ATop, AWidth, AHeight: Integer ) ;
begin
  inherited SetBounds ( ALeft, ATop, AWidth, AHeight ) ;
  Restructure ;
end ;

(* Allow clients to respond to mouse clicks on the graph *)

procedure TdkwGraph.MouseDown ( Button: TMouseButton ; Shift: TShiftState ;
    X, Y: Integer );
var
  Index, Low, High: Integer ;
begin
  if not Assigned ( FOnClick ) then
    Exit ;
  if HorizontalAxes.Count = 0 then
    Exit ;

  { << Inefficient and inflexible!  Fix this >> }

  with HorizontalAxes.Objects [ 0 ] as TdkwHorizontalGraphAxis do
  for Index := Minimum to Maximum do
  begin
    GetRange ( Index, Low, High ) ;
    if ( X >= Low ) and ( X <= High ) then
    begin
      OnClick ( self, Index ) ;
      Exit ;
    end ;
  end ;
end ;

(**************************************************************
 *
 * TdkwGraphAxis Implementation
 *
 **************************************************************)

constructor TdkwGraphAxis.Create ;
begin
  inherited Create ;
  FVisible := True ;
  FColor := clBlack ;
end ;

destructor TdkwGraphAxis.Destroy ;
begin

  { Remove ourselves from any existing graph }

  if Graph <> nil then
    Graph.RemoveAxis ( self ) ;

  inherited Destroy ;
end ;

(**************************************************************
 *
 * TdkwHorizontalGraphAxis Implementation
 *
 **************************************************************)

(* Initialize a horizontal axis *)

constructor TdkwHorizontalGraphAxis.Create ;
begin
  inherited Create ;
  MajorTick := 1 ;
  MinorTick := 1 ;
end ;

(* Draw the axis label and markers *)

procedure TdkwHorizontalGraphAxis.Draw ( Name: String ; Canvas: TCanvas ) ;
var
  Index: Integer ;
  TextLabel: String ;
  LastTick: Integer ;
begin
  with Canvas do
  begin

    { Set drawing characteristics }

    Pen.Color := FColor ;
    Font.Color := FColor ;
    Font.Name := 'Ariel' ;
    Font.Size := 8 ;
    Brush.Style := bsClear ;

    { Draw the axis line }

    MoveTo ( GetPoint ( FMinimum ), Rect.Bottom - HorizLabelHeight div 2 ) ;
    LineTo ( GetPoint ( FMaximum ), Rect.Bottom - HorizLabelHeight div 2 ) ;

    { Draw the 'tick' marks and labels }

    LastTick := -999 ;
    for Index := Minimum to Maximum do
    begin

      { Draw major ticks }

      if ( Index mod MajorTick = 0 ) or
          ( Index = FMaximum ) then
      begin
        MoveTo ( GetPoint ( Index ), Rect.Bottom - HorizLabelHeight div 2 + 2 ) ;
        LineTo ( GetPoint ( Index ), Rect.Bottom - HorizLabelHeight div 2 - 3 ) ;

        if ( GetPoint ( Index ) > LastTick + HorizLabelWidth ) and
            ( ( Index = FMaximum ) or
            ( GetPoint ( Index ) <
            GetPoint ( FMaximum ) - HorizLabelWidth ) ) then
        begin

          { Draw the axis labels }

          TextLabel := IntToStr ( Index ) ;
          if Assigned ( FOnLabel ) then
            OnLabel ( self, Index, TextLabel ) ;

          dkwHorizLabel ( GetPoint ( Index ), Rect.Top + 1, Canvas, TextLabel ) ;
          LastTick := GetPoint ( Index ) ;
        end ;
        Continue ;
      end ;

      { Draw minor ticks }

      if Index mod MinorTick = 0 then
      begin
        MoveTo ( GetPoint ( Index ), Rect.Bottom - HorizLabelHeight div 2 + 1 ) ;
        LineTo ( GetPoint ( Index ), Rect.Bottom - HorizLabelHeight div 2 - 2 ) ;
      end ;
    end ;

    { Draw the axis title }

    dkwHorizLabel ( Rect.Left + ( Rect.Right - Rect.Left ) div 2,
        Rect.Bottom - TextHeight ( Name ) - 1, Canvas, Name ) ;
  end ;
end ;

(* Find a point on a horizontal axis *)

function TdkwHorizontalGraphAxis.GetPoint ( Value: Integer ): Integer ;
begin
  if Graph.FixedWidth = 0 then
    Result := Round ( Rect.Left + HorizLabelWidth div 2 +
        ( Rect.Right - Rect.Left - HorizLabelWidth ) *
        ( ( Value - Minimum ) / ( Maximum - Minimum ) ) )
  else
    Result := Round ( Rect.Left + HorizLabelWidth div 2 +
        ( Graph.FixedWidth - HorizLabelWidth ) *
        ( ( Value - Minimum ) / ( Maximum - Minimum ) ) ) ;

  { Adjust for horizontal scrolling }

  if Graph.ScrollBar <> nil then
    Result := Result - Graph.ScrollBar.Position ;
end ;

(* Find a range on a horizontal axis *)

procedure TdkwHorizontalGraphAxis.GetRange ( Value: Integer ; var Low, High: Integer ) ;
var
  Mid: Integer ;
begin
  Mid := GetPoint ( Value ) ;
  if Value > Minimum then
  begin
    Low := GetPoint ( Value - 1 ) ;
  end else begin
    Low := GetPoint ( Value + 1 ) ;
  end ;
  Low := ( Mid - Low ) div 2 + Mid ;

  if Mid > Low then
    High := Mid + ( Mid - Low )
  else begin
    High := Low ;
    Low := Mid - ( High - Mid ) ;
  end ;
  High := High + 1 ;
end ;

(**************************************************************
 *
 * TdkwVerticalGraphAxis Implementation
 *
 **************************************************************)

(* Draw the axis label and markers *)

procedure TdkwVerticalGraphAxis.Draw ( Name: String ; Canvas: TCanvas ) ;
var
  Value, Spread: Extended ;
  AxisGaps, Index: Integer ;
begin
  with Canvas do
  begin

    { Set drawing characteristics }

    Pen.Color := FColor ;
    Font.Color := FColor ;
    Font.Name := 'Ariel' ;
    Font.Size := 8 ;
    Brush.Style := bsClear ;

    { Draw the axis line }

    MoveTo ( Rect.Left + VertLabelWidth div 2, GetPoint ( FMinimum ) ) ;
    LineTo ( Rect.Left + VertLabelWidth div 2, GetPoint ( FMaximum ) ) ;

    { Axis label }

    dkwVertLabel ( Rect.Left + VertLabelWidth div 4,
        Rect.Top + ( Rect.Bottom - Rect.Top ) div 2,
        Canvas, Name ) ;

    { Determine how many points will fit }

    Spread := Abs ( GetPoint ( Maximum ) - GetPoint ( Minimum ) ) ;
    AxisGaps := 1 ;
    if Spread / VertLabelHeight > 2 then
      AxisGaps := 2 ;
    if Spread / VertLabelHeight > 5 then
      AxisGaps := 5 ;
    if Spread / VertLabelHeight > 10 then
      AxisGaps := 10 ;

    { Draw axis ticks and labels }

    for Index := 0 to AxisGaps do
    begin

      { Don't draw labels for axes without visible
        layers }

      if Minimum < Maximum then
      begin
        Value := FMinimum + ( FMaximum - FMinimum ) * Index / AxisGaps ;
        dkwVertLabel ( Rect.Right - VertLabelWidth div 4,
            GetPoint ( Value ), Canvas, FloatToStr ( Value ) ) ;
      end ;
      MoveTo ( Rect.Left + HorizLabelWidth div 2 + 2, GetPoint ( Value ) ) ;
      LineTo ( Rect.Left + HorizLabelWidth div 2 - 3, GetPoint ( Value ) ) ;
    end ;
  end ;
end ;

(* Find a point on a vertical axis *)

function TdkwVerticalGraphAxis.GetPoint ( Value: Extended ): Integer ;
begin
  Result := Round ( Rect.Bottom - VertLabelHeight div 2 -
      ( Rect.Bottom - Rect.Top - VertLabelHeight ) *
      ( ( Value - Minimum ) / ( Maximum - Minimum ) ) ) ;
end ;

(* Adjust minimum and maximum values to nice round values *)

procedure TdkwVerticalGraphAxis.AdjustMinMax ;
var
  AxisRange, RoundLevel: Extended ;
begin
  AxisRange := Maximum - Minimum ;

  { Determine the appropriate rounding level,
    allowing room for approximately two significant
    digits on the axis labels }

  RoundLevel := 1 ;

  { Don't try to round invalid maxima / minima
    which will be produced for axes with no
    visible layers }

  if Maximum < Minimum then
    Exit ;

  { Get at least SOME range for equal values }

  if Maximum = Minimum then
  begin
    Maximum := Maximum + 1 ;
    Minimum := Minimum - 1 ;
    Exit ;
  end ;

  { For all other cases, do the default }

  if AxisRange > RoundLevel * 100 then
    while AxisRange > RoundLevel * 100 do
      RoundLevel := RoundLevel * 10
  else
    while AxisRange < RoundLevel * 10 do
      RoundLevel := RoundLevel / 10 ;

  { Round the minimum down and the maximum up }

  Minimum := Round ( Minimum / RoundLevel - 0.5 ) * RoundLevel ;
  Maximum := Round ( Maximum / RoundLevel + 0.49999999999 ) * RoundLevel ;
end ;

(**************************************************************
 *
 * TdkwGraphLayer Implementation
 *
 **************************************************************)

(* Create a new display layer *)

constructor TdkwGraphLayer.Create ;
begin
  inherited Create ;
  FVisible := True ;

  { Create a dynamic data array }

  FData := TdkwIntegerArray.Create ;
end ;

(* Destroy a display layer *)

destructor TdkwGraphLayer.Destroy ;
begin

  { Remove ourselves from any existing graph }

  if Graph <> nil then
    Graph.RemoveLayer ( self ) ;

  { Destroy the dynamic data array }

  FData.Free ;
  inherited Destroy ;
end ;

(* Basic drawing for all graph layers *)

procedure TdkwGraphLayer.Draw ( Name: String ; Canvas: TCanvas ;
    Rect: TRect ) ;
begin

  { Fail to draw if conditions are not correct }

  if XAxis = nil then
    raise EdkwGraph.Create ( 'Draw: X axis missing from graph layer' ) ;
  if YAxis = nil then
    raise EdkwGraph.Create ( 'Draw: Y axis missing from graph layer' ) ;
  if Graph = nil then
    raise EdkwGraph.Create ( 'Draw: Graph layer not associated with graph' ) ;
  if XAxis.Graph <> Graph then
    raise EdkwGraph.Create ( 'Draw: X axis not associated with the same graph' ) ;
  if YAxis.Graph <> Graph then
    raise EdkwGraph.Create ( 'Draw: Y axis not associated with the same graph' ) ;
  if XAxis.Minimum < 0 then
    raise EdkwGraph.Create ( 'Draw: X axis minimum should not be less than zero' ) ;
  if XAxis.Maximum > FData.Size then
    raise EdkwGraph.Create ( 'Draw: X axis maximum extends beyond available data' ) ;
end ;

(**************************************************************
 *
 * TdkwLineGraphLayer Implementation
 *
 **************************************************************)

(* Draw a line graph *)

procedure TdkwLineGraphLayer.Draw ( Name: String ; Canvas: TCanvas ;
    Rect: TRect ) ;
var
  Points: Array [ 0..127 ] of TPoint ;
  PointIndex: Integer ;
  Index: Integer ;
  TempColor: TColor ;
begin

  { Check drawing conditions }

  inherited Draw ( Name, Canvas, Rect ) ;

  { Set the correct color }

  Canvas.Pen.Color := FColor ;

  { Use an efficient mechanism for single-color
    line graphs }

  PointIndex := FXAxis.Minimum ;

  if not Assigned ( FOnColor ) then
  begin

    { Use PolyLine to draw line segments 128 at a time
      for efficiency until we don't have enough left }

    while PointIndex + 127 < FXAxis.Maximum do
    begin
      for Index := 0 to 127 do
      begin
        Points [ Index ].X := FXAxis.GetPoint ( PointIndex + Index ) ;
        Points [ Index ].Y := FYAxis.GetPoint ( Data [ PointIndex + Index ] ) ;
      end ;
      Canvas.PolyLine ( Points ) ;
      PointIndex := PointIndex + 127 ;
    end ;

    { Finish remaining line segments one at a time }

    Canvas.MoveTo ( FXAxis.GetPoint ( PointIndex ),
        FYAxis.GetPoint ( Data [ PointIndex ] ) ) ;
    PointIndex := PointIndex + 1 ;
    while PointIndex < FXAxis.Maximum do
    begin
      Canvas.LineTo ( FXAxis.GetPoint ( PointIndex ),
          FYAxis.GetPoint ( Data [ PointIndex ] ) ) ;
      PointIndex := PointIndex + 1 ;
    end ;
  end else begin

    { Draw multi-color line graph }

    Canvas.MoveTo ( FXAxis.GetPoint ( PointIndex ),
        FYAxis.GetPoint ( Data [ PointIndex ] ) ) ;
    PointIndex := PointIndex + 1 ;
    while PointIndex < FXAxis.Maximum do
    begin
      TempColor := FColor ;
      FOnColor ( self, PointIndex, TempColor ) ;
      Canvas.Pen.Color := TempColor ;

      Canvas.LineTo ( FXAxis.GetPoint ( PointIndex ),
          FYAxis.GetPoint ( Data [ PointIndex ] ) ) ;
      PointIndex := PointIndex + 1 ;
    end ;
  end ;
end ;

(**************************************************************
 *
 * TdkwHiLowGraphLayer Implementation
 *
 **************************************************************)

(* Create a high / low graph layer *)

constructor TdkwHiLowGraphLayer.Create ;
begin
  inherited Create ;

  { Create a second data array for 'high' points }

  FDataHigh := TdkwIntegerArray.Create ;
end ;

(* Destroy a high / low graph layer *)

destructor TdkwHiLowGraphLayer.Destroy ;
begin

  { Destroy the additional data array }

  FDataHigh.Free ;
  inherited Destroy ;
end ;

(* Draw a high / low graph layer *)

procedure TdkwHiLowGraphLayer.Draw ( Name: String ; Canvas: TCanvas ;
    Rect: TRect ) ;
var
  BarRect: TRect ;
  Adjust: Integer ;
  Index: Integer ;
  TempColor: TColor ;
begin

  { Check drawing conditions }

  inherited Draw ( Name, Canvas, Rect ) ;
  if XAxis.Maximum > FDataHigh.Size then
    raise EdkwGraph.Create ( 'Draw: X axis maximum extends beyond available data' ) ;

  { For each bar ... }

  for Index := FXAxis.Minimum to FXAxis.Maximum do
  begin

    { Create a representative bar }

    FXAxis.GetRange ( Index, BarRect.Left, BarRect.Right ) ;
    BarRect.Top := FYAxis.GetPoint ( DataHigh [ Index ] ) ;
    BarRect.Bottom := FYAxis.GetPoint ( DataLow [ Index ] ) ;

    { Deal with inverted high / low pairs }

    if BarRect.Top > BarRect.Bottom then
    begin
      Adjust := BarRect.Top ;
      BarRect.Top := BarRect.Bottom ;
      BarRect.Bottom := Adjust ;
    end ;

    { Enlarge 'invisible' bars }

    if BarRect.Bottom < BarRect.Top + 2 then
    begin
      Inc ( BarRect.Bottom ) ;
      Dec ( BarRect.Top ) ;
    end ;

    { Make all bars 1/2 width }

    Adjust := ( BarRect.Right - BarRect.Left ) div 4 ;
    BarRect.Left := BarRect.Left + Adjust ;
    BarRect.Right := BarRect.Right - Adjust ;

    { Draw it in the layer's color and outline it }

    TempColor := Color ;
    if Assigned ( FOnColor ) then
      FOnColor ( self, Index, TempColor ) ;
    Canvas.Brush.Color := TempColor ;

    { Force fillrect to use a larger bar }

    Dec ( BarRect.Top ) ;
    Canvas.FillRect ( BarRect ) ;
    Canvas.Brush.Color := clBlack ;
    Canvas.FrameRect ( BarRect ) ;
  end ;
end ;

(**************************************************************
 *
 * TdkwBarGraphLayer Implementation
 *
 **************************************************************)

(* Draw a bar graph layer *)

procedure TdkwBarGraphLayer.Draw ( Name: String ; Canvas: TCanvas ;
    Rect: TRect ) ;
var
  BarRect: TRect ;
  Index: Integer ;
  TempColor: TColor ;
begin

  { Check drawing conditions }

  inherited Draw ( Name, Canvas, Rect ) ;

  { For each bar ... }

  for Index := FXAxis.Minimum to FXAxis.Maximum do
  begin

    { Set the color }

    TempColor := Color ;
    if Assigned ( FOnColor ) then
      FOnColor ( self, Index, TempColor ) ;
    Canvas.Brush.Color := TempColor ;

    { Create and fill the appropriate area }

    FXAxis.GetRange ( Index, BarRect.Left, BarRect.Right ) ;
    BarRect.Top := FYAxis.GetPoint ( Data [ Index ] ) ;
    BarRect.Bottom := FYAxis.GetPoint ( 0 ) ;

    { Force fillrect to use a larger bar }

    Canvas.FillRect ( BarRect ) ;
  end ;
end ;

(**************************************************************
 *
 * Registration Procedure
 *
 **************************************************************)

procedure Register;
begin
  RegisterComponents('DKW', [TdkwGraph]);
end;

end.

