unit Copyfrm;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, StdCtrls, Buttons, ExtCtrls, Gauges,
  FileTool, CopyMsg, {$IFDEF WIN32} CpyObj32, {$ELSE} CpyObj, {$ENDIF} Timelabl;

type

  { TCopyForm }
  TCopyForm = class(TForm)
    Image: TImage;
    TargetLabel: TLabel;
    FileNameLabel: TLabel;
    CancelBtn: TBitBtn;
    Panel1: TPanel;
    Gauge1: TGauge;
    Image1: TImage;
    ImageLeft: TImage;
    ImageRight: TImage;
    procedure FormKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure FormCreate(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure CancelBtnClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
  private
    TimeLabel1: TTimeLabel;
    CopyObj1: TCopyObj;

    ActiveWindow: HWnd;
    WindowList: Pointer;
    OldPos, CurPos: Integer;
    procedure ComputeImagePos;
    procedure WMClose(var Msg: TMessage); message wm_Close;
    procedure WMCommand(var Msg: TMessage); message wm_Command;
    procedure WMFCReset(var Msg: TMessage); message wm_fcReset;
    procedure WMFCSearching(var Msg: TMessage); message wm_fcSearching;
    procedure WMFCName(var Msg: TMessage); message wm_fcName;
    procedure WMFCRead(var Msg: TMessage); message wm_fcRead;
    procedure WMFCWrite(var Msg: TMessage); message wm_fcWrite;
    procedure WMFCIdle(var Msg: TMessage); message wm_fcIdle;
    procedure AbortCopying(Aborted: Boolean);
  protected
    procedure CreateParams(var Params: TCreateParams); override;
  public
    Copying: Boolean;
    Sources, Targets: TStringList;
    procedure SetupFiles(TheSources, TheTargets: TStringList; TargetFree: LongInt);
    function  CopyFiles(OneByOne, MoveFile: Boolean): Boolean;
  end;

var
  CopyForm: TCopyForm;

implementation

{$R *.DFM}

{$DEFINE LIB}

uses
  LZExpand;

function AbortProc: Boolean;
var
  Msg: TMsg;
begin
  while not UserAbort and PeekMessage(Msg, 0, 0, 0, pm_Remove) do
  begin
    TranslateMessage(Msg);
    DispatchMessage(Msg);
  end;
  Result := not UserAbort;
end;

function CountBytes(C: TStringList): LongInt;
var
  SourceHandle, I: Integer;
  OpenBuf: TOFStruct;
  S: array [0..255] of Char;
begin
  Result := 0;
  for I := 0 to C.Count - 1 do
  begin
    StrPCopy(S, C[I]);
    SourceHandle := LZOpenFile(S, OpenBuf, of_Share_Deny_Write or of_Read);
    Inc(Result, LZSeek(SourceHandle, 0, 2));
    LZClose(SourceHandle);
  end;
end;

{ TCopyForm }
procedure TCopyForm.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  with Params do
  begin
    WndParent := TForm(Owner).Handle;
    Style := Style or ws_DlgFrame;
  end;
end;

procedure TCopyForm.FormCreate(Sender: TObject);
begin
  ActiveWindow := GetActiveWindow;
  WindowList := DisableTaskWindows(0);
  EnableMenuItem(GetSystemMenu(Handle, False), sc_Close, mf_Grayed);

  Copying := True;
  UserAbort := False;
  {$IFDEF SHOWFLY}
  CurPos := 45;
  ComputeImagePos;
  {$ELSE}
  ImageLeft.Visible := False;
  ImageRight.Visible := False;
  {$ENDIF}
  OldPos := 0;

  CopyObj1 := TCopyObj.Create(Self);
  {CopyObj1.Parent := Self;}
  { if you install TCopyObj in IDE, you can set next
    properties through Object Inspector, and of course
    you don't need the above two lines}
  with CopyObj1 do
  begin
    CreateBackup := False;
    MaxBufSize := MaxBlockSize;
    MoveFile := False;
    OneByOne := True;
    OverwriteMsg := False;
    Step := 1;
  end;

  TimeLabel1 := TTimeLabel.Create(Self);
  TimeLabel1.Parent := Self;
  { if you install TTimeLabel in IDE, you can set next
    properties through Object Inspector, and of course
    you don't need the above two lines}
  TimeLabel1.SetBounds(13, 121, 3, 13);
  TimeLabel1.Font.Assign(TargetLabel.Font);
end;

procedure TCopyForm.AbortCopying(Aborted: Boolean);
begin
  Copying := False;
  UserAbort := Aborted;
  EnableMenuItem(GetSystemMenu(Handle, False), sc_Close, mf_Enabled);
end;

procedure TCopyForm.FormKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
  if Key in [VK_RETURN, VK_ESCAPE] then
    AbortCopying(True);
end;

procedure TCopyForm.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
  CanClose := not Copying;
end;

procedure TCopyForm.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  CopyObj1.FlushBuffers;
  if not UserAbort then
    if Gauge1.Progress < Gauge1.MaxValue then
      Application.MessageBox('Some files could not be copied.', 'Copy', MB_OK);
  Action := caFree;
  EnableTaskWindows(WindowList);
  SetActiveWindow(ActiveWindow);
end;

procedure TCopyForm.CancelBtnClick(Sender: TObject);
begin
  AbortCopying(True);
end;

procedure TCopyForm.WMCommand(var Msg: TMessage);
begin
  AbortCopying(True);
end;

procedure TCopyForm.WMClose(var Msg: TMessage);
begin
  if Copying then
    AbortCopying(True)
  else
    inherited;
end;

procedure TCopyForm.WMFCReset(var Msg: TMessage);
begin
  FileNameLabel.Caption := '';
  TargetLabel.Caption := 'Change Disk';
end;

procedure TCopyForm.WMFCSearching(var Msg: TMessage);
begin
  FileNameLabel.Caption := '';
  TargetLabel.Caption := 'Searching ...';
end;

procedure TCopyForm.WMFCName(var Msg: TMessage);
begin
  with TNames(Pointer(Msg.LParam)^) do
  begin
    FileNameLabel.Caption := LowerCase(ExtractFilename(Dest));
    TargetLabel.Caption := Format('From ''%s'' to ''%s''',
      [UpperCase(ExtractFilePath(Source)), UpperCase(ExtractFilePath(Dest))]);
  end;
end;

procedure TCopyForm.WMFCRead(var Msg: TMessage);
begin
  with TNameAndPos(Pointer(Msg.LParam)^) do
  begin
    Gauge1.Progress := Gauge1.Progress + Pos;
    TimeLabel1.UpdateIt(Trunc(Pos*1.9));
    Caption := 'Reading...';
  end;
end;

procedure TCopyForm.WMFCWrite(var Msg: TMessage);
begin
  with TNameAndPos(Pointer(Msg.LParam)^) do
  begin
    Gauge1.Progress := Gauge1.Progress + Pos;
    TimeLabel1.UpdateIt(Trunc(Pos*0.1));
    Caption := 'Writing...';
  end;
end;

procedure TCopyForm.SetupFiles(TheSources, TheTargets: TStringList; TargetFree: LongInt);
begin
  Gauge1.Progress := 0;
  Sources := TheSources;
  Targets := TheTargets;
  CopyObj1.SetDiskFree(TargetFree);
end;

function TCopyForm.CopyFiles(OneByOne, MoveFile: Boolean): Boolean;
var
  I: Integer;
  TotalSize: Longint;
begin
  TotalSize := CountBytes(Sources);
  Show;
  Gauge1.Progress := 0;
  Gauge1.MaxValue := TotalSize*2;
  CopyObj1.OneByOne := OneByOne;
  CopyObj1.MoveFile := MoveFile;
  TimeLabel1.Reset;
  TimeLabel1.Start(TotalSize*2);
  Application.ProcessMessages;
  for I := 0 to Sources.Count - 1 do
    if not CopyObj1.Copy(Sources[I], Targets[I], coNormal) then
      Break;
  Copying := False;
  Result := not UserAbort;
end;

procedure TCopyForm.WMFCIdle(var Msg: TMessage);
begin
  AbortProc;
  {$IFDEF SHOWFLY}
  Inc(CurPos, Msg.lParam);
  if CurPos > 215 then CurPos := 45;
  ComputeImagePos;
  {$ENDIF}
end;

procedure TCopyForm.ComputeImagePos;

function SolveForX(X: Integer): Integer;
begin
  Result := 20-Trunc(-0.0025*X*X+0.687*X-25.68);
end;

begin
  if not Image1.Visible then
    Image1.Visible := True;
  Image1.SetBounds(Curpos, SolveForX(CurPos), Image1.Width, Image1.Height);
end;

end.

