{$I COPYOBJ.INT}

{$DEFINE LIB}
uses
  LZExpand, CopyMsg, FileTool;

const
  fmReadOnly = OF_SHARE_DENY_WRITE or OF_READ;
  fmReadWrite = OF_SHARE_EXCLUSIVE or OF_READWRITE;

const
  { I/O errors }
  HFILE_ERROR   =  -1;

  erWriteOpen   =  -1;     { error opening for write }
  erReadOpen    =  -2;     { error opening for read  }
  erNoFile      =  -3;     { file not found }
  erResetAFlag  =  -4;     { Unable to reset archive flag on original file }
  erNoRoom      =  -5;     { not enough room on target destination for file }

  { Internal error }
  erOutOfMemory = -99;     { could not allocate more memory }

type
  { TOneFileRec represents a single file that is being processed. }
  TOneFileRec = class(TObject)
  private
    FFileName: String;
    FOrigName: String;
    FFTime: Longint;
    FFSize: Longint;
    FOffset: Longint;
    FCreate: Boolean;
    FOptFlags: Word;
    FBuffers: TList;
  public
    constructor Create(OldName, NewName: String);
    destructor Destroy; override;
  end;

{ TOneFileRec }
constructor TOneFileRec.Create(OldName, NewName: String);
begin
  inherited Create;
  FOrigName := OldName;
  FFileName := NewName;
  FBuffers := nil;
end;

destructor TOneFileRec.Destroy;
begin
  if Assigned(FBuffers) then FBuffers.Free;
  inherited Destroy;
end;

{ TCopyObj }
constructor TCopyObj.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  {$IFNDEF WIN32}
  LZStart;
  {$ENDIF}

  FFileList := TList.Create;
  FOneByOne := False;
  FMaxBufSize := MaxBlockSize;
  FStep := 1;
  FState := sIdle;
  FBusy := False;
  FOverwriteMsg := False;
  FCreateBackup := True;
  FMoveFile := False;
end;

destructor TCopyObj.Destroy;
begin
  {$IFNDEF WIN32}
  LZDone;
  {$ENDIF}

  if Assigned(FFileList) then
    FFileList.Free;
  inherited Destroy;
end;

procedure TCopyObj.SetDiskFree(Value: LongInt);
begin
  FFreeDisk := Value;
end;

function TCopyObj.UpdateDiskFree(FName: String; Value: LongInt): Boolean;
begin
  if not FIncomplete then
    SysUtils.DeleteFile(FName);
  FreeDisk := DiskFree(Ord(UpCase(FName[1]))-64);
  Result := FFreeDisk - Value > 512;
  if not Result then Exit;
  Dec(FFreeDisk, Value);
end;

function TCopyObj.SetBlockSize(Value: LongInt): Word;
begin
  if Value > FMaxBufSize then
    Result := FMaxBufSize
  else Result := Value;
end;

procedure TCopyObj.SetMaxBufSize(Value: Integer);
begin
  if Value > MaxBlockSize then Value := MaxBlockSize;
  FMaxBufSize := Value;
end;

procedure TCopyObj.FlushBuffers;

procedure FlushFile(CurFile: TOneFileRec);

procedure ErrFree;
var
  RemainingBytes: Longint;
  BytesToXFer: Word;
  BufAddr: Pointer;
begin
  RemainingBytes := CurFile.FFSize;
  with CurFile do
  repeat
    BytesToXFer := SetBlockSize(RemainingBytes);
    BufAddr := FBuffers.Items[0];
    FreeMem(BufAddr, BytesToXFer);
    FBuffers.Delete(0);
    Dec(RemainingBytes, BytesToXFer);
  until (FBuffers.Count = 0) or (RemainingBytes = 0);
end;

var
  TargetFile: File;
  RemainingBytes: Longint;
  Attr: Word;
  BytesDone, BytesToXFer: Integer;
  BufAddr: Pointer;
begin
  ResetIOResult;
  FileMode := fmReadWrite;
  FAction := erRetry;
  FResult := 1;
  while (FAction <> erAbort) and (FResult <> 0) do
  begin
    AssignFile(TargetFile, CurFile.FFileName);
    {$I-}
    if CurFile.FCreate then Rewrite(TargetFile, 1)
    else Reset(TargetFile, 1);
    {$I+}
    FResult := IOResult;
    if FResult <> 0 then
    begin
      FAction := IOError(CurFile.FFileName, erWriteOpen);
      if FAction = erAbort then
      begin
        ErrFree;
        Exit;
      end;
    end;
  end;
  Seek(TargetFile, CurFile.FOffset);
  if (FFileList.IndexOf(CurFile) = FFileList.Count - 1) and FIncomplete
    then Inc(FOffset, CurFile.FFSize);
  RemainingBytes := CurFile.FFSize;
  with CurFile do
  repeat
    BytesToXFer := SetBlockSize(RemainingBytes);
    BufAddr := FBuffers.Items[0];
    if UserAbort then BytesDone := BytesToXFer
    else BlockWrite(TargetFile, BufAddr^, BytesToXFer, BytesDone);
    FreeMem(BufAddr, BytesToXFer);
    FBuffers.Delete(0);
    Dec(RemainingBytes, BytesDone);
    WriteMsg(FFileName, BytesDone);
  until (FBuffers.Count = 0) or (RemainingBytes = 0);
  if not FIncomplete then
    FileSetDate(TFileRec(TargetFile).Handle, CurFile.FFTime);
  CloseFile(TargetFile);
  if ((FFileList.IndexOf(CurFile) <> FFileList.Count-1) or (not FIncomplete)) then
    if CurFile.FOptFlags and coResetAFlag <> 0 then
    begin
      Attr := FileGetAttr(CurFile.FOrigName);
      Attr := Attr and (not faArchive);
      if FileSetAttr(CurFile.FOrigName, Attr) <> 0 then
        IOError(CurFile.FOrigName, erResetAFlag);
    end;
  if UserAbort then
  begin
    SysUtils.DeleteFile(CurFile.FFileName);
    FIncomplete := False;
  end;
end;

var
  I: Integer;
begin
  with FFileList do
  begin
    if Count = 0 then Exit;
    FState := sWriting;
    for I := 0 to Count - 1 do
      FlushFile(Items[I]);
    FState := sIdle;
    while Count <> Word(FIncomplete) do
    begin
      TOneFileRec(Items[0]).Free;
      Delete(0);
    end;
  end;
end;

function TCopyObj.Copy(ASource, ADest: OpenString; Options: Word): Boolean;

procedure FreeZero;
var
  Temp: TObject;
begin
  Temp := FFileList.Items[0];
  Temp.Free;
  FFileList.Delete(0);
end;

const
  Safety = 8192;
var
  Names: TNames;
  Flush: Boolean;
  CurFile: TOneFileRec;
  RemainingBytes, CurMaxAvail: Longint;
  SRec: TSearchRec;
  IORes: Integer;
  Msg, Dst, Dest: array [0..255] of char;
  BytesDone, BytesToXFer: Word;
  BufAddr: Pointer;
  SourceHandle: Integer;
  OpenBuf: TOFStruct;
  Overwrite: Boolean;
begin
  Result := False;
  ResetIOResult;
  IORes := FindFirst(ASource, faAnyFile, SRec);
  if Options and coCopyAOnly <> 0 then
    if (IORes = 0) and ((SRec.Attr and faArchive) = 0) then Exit
    else IOError(ASource, erNoFile);
  CurFile := TOneFileRec.Create(ASource, ADest);
  CurFile.FBuffers := TList.Create;
  if (CurFile = nil) or (CurFile.FBuffers = nil) then
  begin
    InternalError(erOutOfMemory);
    Exit;
  end;
  CurFile.FOptFlags := Options;
  CurFile.FOffset := 0;
  CurFile.FCreate := True;
  FFileList.Add(CurFile);
  FOffset := 0;
  FIncomplete := False;
  FIsNewFile := False;
  FBase := 0;
  repeat
    FState := sReading;
    Flush := OneByOne;
    repeat
      SourceHandle := LZOpenFile(StringAsPChar(ASource), OpenBuf, fmReadOnly);
      if SourceHandle = HFILE_ERROR then
        if IOError(ASource, erReadOpen) = erAbort then Break;
    until SourceHandle <> HFILE_ERROR;
    if SourceHandle = HFILE_ERROR then
    begin
      FreeZero;
      Exit;
    end;
    FillChar(Dst, SizeOf(Dst), #0);
    StrPCopy(Dst, CurFile.FOrigName);
    with CurFile do
      if (FOrigName[Length(FOrigName)] = '_') then
      begin
        FillChar(Dest, SizeOf(Dest), #0);
        GetExpandedName(Dst, Dest);
      end
      else StrPCopy(Dest, ADest);
    Names.Source := ASource;
    Names.Dest := System.Copy(ADest, 1, pos('.', ADest)-1)+ExtractFileExt(StrPas(Dest));
    SendMsg(wm_fcName, LongInt(@Names));
    CurFile.FFileName := Names.Dest;
    CurFile.FFTime := SRec.Time;
    CurFile.FFSize := LZSeek(SourceHandle, 0, 2) - FOffset + FBase;
    if not FIncomplete and FileExists(Names.Dest) then
    begin
      Overwrite := not FOverwriteMsg or
        (Application.MessageBox(StrPCopy(Msg, 'Overwrite '+Names.Dest+' ?'),
          'Copy',MB_YESNO) = mrYes);
      if not Overwrite then
      begin
        ReadMsg('Skipped', CurFile.FFSize);
        Result := True;
        FreeZero;
        Exit;
      end
      else
        if FCreateBackup then
        begin
          SysUtils.DeleteFile(ChangeFileExt(Names.Dest, '.BAK'));
          RenameFile(ADest, ChangeFileExt(Names.Dest, '.BAK'));
        end;
      SendMsg(wm_fcIdle, 0);
    end;
    while not UpdateDiskFree(ADest, CurFile.FFSize) do
    begin
      if IOError(ASource, erNoRoom) = erAbort then
      begin
        Result := True;
        FreeZero;
        Exit;
      end;
    end;
    if FIncomplete then
    begin
      LZSeek(SourceHandle, FOffset-FBase, 0);
      CurFile.FOffset := FOffset;
      CurFile.FCreate := False;
    end
    else LZSeek(SourceHandle, 0, 0);
    FIncomplete := False;
    {$IFDEF WIN32}
    CurMaxAvail := GetFreeSpace(0) - Safety;
    {$ELSE}
    CurMaxAvail := MaxAvail - Safety;
    {$ENDIF}
    if CurFile.FFSize > CurMaxAvail then
    begin
      CurFile.FFSize := CurMaxAvail;
      Flush := True;
      FIncomplete := True;
      CurFile.FOffset := FOffset;
    end;
    RemainingBytes := CurFile.FFSize;
    with CurFile do
    repeat
      BytesToXFer := SetBlockSize(RemainingBytes);
      GetMem(BufAddr, BytesToXFer);
      FBuffers.Add(BufAddr);
      BytesDone := LZRead(SourceHandle, PChar(BufAddr), BytesToXFer);
      Dec(RemainingBytes, BytesDone);
      ReadMsg(FOrigName, BytesDone);
    until UserAbort or (RemainingBytes = 0);
    LZClose(SourceHandle);
    if Flush and not UserAbort then
      FlushBuffers;
  until not FIncomplete or UserAbort;
  if not UserAbort and FMoveFile then
    SysUtils.DeleteFile(ASource);
  FState := sIdle;
  Result := not UserAbort;
end;

function TCopyObj.ErrorMsg(ECode: Integer): String;
begin
  case ECode of
    erWriteOpen   : Result := 'Unable to open for write access';
    erReadOpen    : Result := 'Unable to open for read access';
    erNoFile      : Result := 'File not found.';
    erNoRoom      : Result := #13#10'Not enough room for this file.'#13#10'Change disk and clik retry.';
    erOutOfMemory : Result := 'Unable to allocate memory.';
    else            Result := 'Unknown error.';
  end;
end;

function TCopyObj.IOError(FName: String; ECode: Integer): erAction;
var
  mr: Word;
  S: array [0..255] of char;
begin
  StrPCopy(S, FName+': '+ErrorMsg(ECode));
  mr := Application.MessageBox(S, 'Copy', MB_RETRYCANCEL);
  if mr = idCancel then Result := erAbort
  else Result := erRetry;
end;

function TCopyObj.InternalError(ECode: Integer): erAction;
var
  S: array [0..255] of char;
begin
  Application.MessageBox(StrPCopy(S, ErrorMsg(ECode)), 'Copy', MB_OK);
  Result := erAbort;
end;

function TCopyObj.ReadMsg(FName: String; Progress: Longint): LongInt;
begin
  Result := ShellMsg(FName, wm_fcRead, Progress);
end;

function TCopyObj.WriteMsg(FName: String; Progress: Longint): LongInt;
begin
  Result := ShellMsg(FName, wm_fcWrite, Progress);
end;

function TCopyObj.SendMsg(Msg: Word; lParam: LongInt): LongInt;
begin
  Result := SendMessage(TForm(Owner).Handle, Msg, 0, lParam);
end;

function TCopyObj.ShellMsg(FName: String; Msg: Word; Progress: Longint): LongInt;
var
  N: TNameAndPos;
begin
  if UserAbort then Exit;
  N.Name := FName;
  N.Pos := Progress;
  Result := SendMsg(Msg, LongInt(@N));
  SendMsg(wm_fcIdle, FStep);
end;

procedure Register;
begin
  RegisterComponents('Samples', [TCopyObj]);
end;

