unit Resizer;

interface

uses
(* If you have the Client/Server version of Delphi you can take off from
   comments the line below to take advantage of the support to the
   additional components *)
 {$DEFINE CLIENTSERVER}

{$IFDEF WIN32}
  Windows, ComCtrls,
{$ELSE}
  WinTypes, WinProcs,
{$ENDIF}
{$IFDEF VER100}
{$IFDEF CLIENTSERVER}
  MxGrid,
{$ENDIF}
{$ENDIF}
  Classes, Controls, DBGrids, Graphics, TypInfo, DBCtrls, DBLookup, CustomRP,
  NoBDERP, DsgnIntf;

type
  TMRControlHandlerBDE = class(TMRControlHandler)
  public
    function CalcNewPosition: TRect; override;
  end;


  TResizerPanel = class(TNoBDEResizerPanel)
  protected
    procedure AddHandlers( Control: TControl; List: TList ); override;
  public
    constructor Create( AOwner: TComponent ); override;
  end;


  TMRDBGrid = class(TMRPropertyHandler)
  private
    {$IFDEF WIN32}
    FColumns: TDBGridColumns;
    {$ENDIF}
    FData: TMRLValues;
    FFontHandler: TMRFont;
  public
    constructor Create( Control: TControl ); override;
    destructor Destroy; override;
    procedure BeforeResize; override;
    procedure AfterResize; override;
  end;


  procedure Register;

implementation



procedure Register;
begin
  RegisterComponents('Additional', [TResizerPanel]);
  RegisterPropertyEditor( TypeInfo(TStringList), TResizerPanel,'',
    TWarningListProperty );
end;


constructor TResizerPanel.Create( AOwner: TComponent );
begin
  inherited Create( AOwner );

  ControlHandlerClass := TMRControlHandlerBDE;
end;


procedure TResizerPanel.AddHandlers( Control: TControl; List: TList );
begin
  inherited AddHandlers( Control, List );

  if ((Control is TDBComboBox) or (Control is TDBListBox)) then
    List.Add(TMRItemHeight.Create( Control ));

  if (Control is TDBGrid) then
    List.Add(TMRDBGrid.Create( Control ));

  if (Control is TDBImage) then
    List.Add(TMRStretch.Create( Control ));

  if (Control is TDBText) then
    List.Add(TMRAutoSize.Create( Control ));

  {$IFDEF VER100}
  {$IFDEF CLIENTSERVER}
  if (Control is TDecisionGrid) then
  begin
    List.Add(TMRFont.CreateHandler( Control, 'CaptionFont' ));
    List.Add(TMRFont.CreateHandler( Control, 'DataFont' ));
  end;
  {$ENDIF}
  {$ENDIF}
end;


function TMRControlHandlerBDE.CalcNewPosition: TRect;
var
  pInfo: PPropInfo;
  CFont: TFont;
begin
  (* patch for the ComboBox bug, the vertical size is upper limited by the
   font height *)

  {$IFDEF WIN32}
  if (FControl is TDBLookupComboBox) then
  begin
    pInfo := GetPropInfo( Control.ClassInfo, 'Font' );
    if (pInfo <>  nil) then
    begin
      CFont := TFont(GetOrdProp(Control, pInfo));
      CFont.Size := 1000;
    end;
  end;
  {$ENDIF}

  Result := inherited CalcNewPosition;
end;


(**************************************
**************************************
**************************************
 ** List Columns property handler ***)

constructor TMRDBGrid.Create( Control: TControl );
var
  pInfo: PPropInfo;
begin
  inherited Create(Control);

  FData := TMRLValues.Create;
  {$IFDEF WIN32}
  pInfo := GetPropertyInfo( Control.ClassInfo, 'Columns' );
  FColumns := TDBGridColumns(GetOrdProp(Control, pInfo));
  {$ENDIF}

  FFontHandler := TMRFont.CreateHandler( Control, 'TitleFont' );
end;


destructor TMRDBGrid.Destroy;
begin
  FFontHandler.Free;

  FData.Free;

  inherited Destroy;
end;

procedure TMRDBGrid.BeforeResize;
var
  W, WC: Single;
  Cnt: Integer;
begin
  FFontHandler.BeforeResize;

  {$IFDEF WIN32}
  (* verifies if the panels changed *)
  if ((FColumns.Count <> FData.Count) or (Invalid = True)) then
    Invalid := True
  else
  begin
    for Cnt := 0 to FColumns.Count-1 do
      if (FColumns.Items[Cnt].Width <> FData.LastValues[Cnt]) then
      begin
        Invalid := True;
        break;
      end;
  end;

  if (Invalid = False) then
    exit;

  Change; // DEBUG

  // certifies that the width isn't 0
  WC := FControl.ClientWidth;
  if (WC = 0) then
    exit;

  FData.Clear;
  for Cnt := 0 to FColumns.Count-1 do
  begin
    W := FColumns.Items[Cnt].Width;
    FData.Add( 0, W / WC );
  end;

  Invalid := False;
  {$ENDIF}
end;

procedure TMRDBGrid.AfterResize;
var
  Cnt: Integer;
  W: Single;
begin
  FFontHandler.AfterResize;

  {$IFDEF WIN32}
  if (Invalid = True) then
    exit;

  W := FControl.ClientWidth;

  for Cnt := 0 to FData.Count-1 do
  begin
    FColumns.Items[Cnt].Width := Trunc(FData.CValues[Cnt] * W);

    FData.LastValues[Cnt] := FColumns.Items[Cnt].Width;
  end;
  {$ENDIF}
end;

end.
