
{*******************************************************}
{                                                       }
{          Delphi TLED & TLEDLabel Components           }
{           Copyright  1997, Gary Hardman.             }
{                   10th August 1997                    }
{                                                       }
{   No warranties whatsoever are implied or offered     }
{                    for this unit                      }
{                                                       }
{  TLED      V1.1                                       }
{  TLEDLABEL V1.0                                       }
{*******************************************************}

unit LED;

interface

uses

  WinTypes, WinProcs, Classes, Graphics, Controls, Messages, SysUtils, Menus;

type

  TLEDColor = (lcRed,lcGreen,lcYellow);   { LED colour - Red, Green or Yellow }
  TCaptionAlignment = (caLeft, caRight, caTop, caBottom); { LEDLabel Caption position }

  TLED = class(TGraphicControl)
    private
  { Private declarations }

      fLit         : Boolean;             { LED On or Off }
      fColor       : TLEDColor;           { LED colour }
      fChange      : TNotifyEvent;        { Custom Event }

      procedure SetState(const NewState: Boolean);
      procedure SetColor(const NewColor: TLEDColor);

      procedure CM_ENABLEDCHANGED(var Msg: TMessage);
                message cm_EnabledChanged;

      procedure CM_PARENTCOLORCHANGED(var Msg: TMessage);
                message cm_ParentColorChanged;

      procedure WMWINDOWPOSCHANGED(var Msg: TWMWindowPosChanged);
                message wm_WindowPosChanged;

    protected
  { Protected declarations }
      procedure Paint; override;
      procedure PaintLED; virtual;

    public
  { Public declarations }
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
      function ChangeState : Boolean;
      function ChangeColor: TLEDColor;

    published
  { Published declarations }

      property Lit: Boolean                { LED is On or Off }
               read fLit
               write SetState
               default False;

      property Color: TLEDColor            { LED is Red, Green or Yellow }
               read fColor
               write SetColor
               default lcRed;

      property Height default 13;
      property Width  default 12;

      property Enabled;
      property Visible;
      property Hint;
      property ParentShowHint;
      property ShowHint;
      property Tag;

      property OnClick;
      property OnDblClick;

      property OnChangeState: TNotifyEvent { User-defined Method }
               read fChange
               write fChange;

      property OnDragDrop;
      property OnDragOver;
      property OnEndDrag;
      property OnMouseDown;
      property OnMouseMove;
      property OnMouseUp;
  end;

{ TLEDLabel }

  TLEDLabel = class(TLED)
    private
  { Private declarations }

      fAlignment   : TCaptionAlignment;  { Caption to Left,Right,Top or Bottom }
      fLEDRect     : TRect;              { Rect in which to draw LED }
      fTextRect    : TRect;              { Rect in which to draw Caption }

      procedure SetAlignment(const Value: TCaptionAlignment);
      function  GetTransparent: Boolean; { Caption text transparent? }
      procedure SetTransparent(Value: Boolean);

      procedure CM_TEXTCHANGED(var Msg: TMessage);
                message cm_TextChanged;

      procedure WMWINDOWPOSCHANGED(var Msg: TWMWindowPosChanged);
                message wm_WindowPosChanged;

    protected
  { Protected declarations }
      procedure Paint; override;
      procedure PaintLED; override;
      procedure PaintCaption; virtual;

    public
  { Public declarations }
      constructor Create(AOwner: TComponent); override;

    published
  { Published declarations }

      property Caption;                      { Publish the Caption property }
      property Font;                         { Publish the Font property }
      property ParentFont;                   { Publish the ParentFont property }
      property PopupMenu;                    { Publish the PopupMenu property }

      property Alignment: TCaptionAlignment  { Caption Alignment }
               read fAlignment
               write SetAlignment
               default caRight;

      property Transparent: Boolean          { Transparent Caption area }
               read GetTransparent
               write SetTransparent
               default False;

      property Height default 17;            { New default values for height }
      property Width  default 97;            { and Width properties }

  end;

implementation

{$IFDEF WIN32}
{$R LED_32.RES}
{$ELSE}
{$R LED_16.RES}
{$ENDIF}

type
  TLEDBitmap = class
  public
    LED_Pics: TBitmap;    { LED Bitmap matrix - all colours, all states }
    LED_List: TList;      { List to keep track of LEDs created }
    Trans_Color : TColor; { Bottom left pixel of LED Bitmap matrix }
    constructor Create;
    destructor Destroy; override;
end;

const
  LEDBitmap : TLEDBitmap = nil;
  NumRows = 4;                 { LED_PICS bitmap has four 'state' rows }
  bmpHeight = 13;              { Height of individual LED bitmap }
  bmpWidth = 12;               { Width of an individual LED bitmap }
  CaptionSpacing = 3;          { Spacing of Caption from LED in LEDLabel}

constructor TLEDBitmap.Create;
begin
  inherited Create;
  LED_Pics := TBitmap.Create;
{ Load the LED_PICS Bitmap matrix from the resource file }
  LED_Pics.Handle:= LoadBitmap(hInstance, 'LED_PICS');
{ Determine Transparent Colour from LED bitmap matrix }
  Trans_Color := LED_PICS.TransparentColor;
{ Create list to keep track of LEDs created }
  LED_List := TList.Create;
end;

destructor TLEDBitmap.Destroy;
begin
{ Free both the LED bitmap matrix, and LED_List }
  LED_Pics.Free;
  LED_List.Free;
  inherited Destroy;
end;

constructor TLED.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
{ If this is the first LED then create the LED Bitmap and List object }
  if LEDBitmap = nil then LEDBitmap := TLEDBitmap.Create;
  LEDBitmap.LED_LIST.Add(Self); { Add this new LED instance to the list }
  fLit := False;                { Default = not Lit }
  fColor := lcRed;              { Default = clRed }
  ControlStyle := ControlStyle + [csOpaque]      + [csDoubleClicks]
                               + [csFixedHeight] + [csFixedWidth];
end;

destructor TLED.Destroy;
begin
{ Delete the current control from the LED_LIST }
  with LEDBitmap.LED_LIST do
    Delete (IndexOf(Self));
{ If there are no more LEDs, delete the Bitmap and List object }
    if LEDBitmap.LED_LIST.Count = 0 then
    begin
      LEDBitmap.Free; { Free the Bitmap and List object }
      LEDBitmap:=nil; { and point to NIL }
    end;
 inherited Destroy;
end;

{ The Enabled property has changed }
procedure TLED.CM_ENABLEDCHANGED(var Msg: TMessage);
begin
  if Assigned(fChange) then OnChangeState(Self);
  inherited;
end;

{ The Parent Colour has changed }
procedure TLED.CM_PARENTCOLORCHANGED(var Msg: TMessage);
begin
  Canvas.Brush.Color := TWincontrol(Parent).Brush.Color;
  inherited;
end;

{ Previous version used SetBounds - I think this is more elegant }
procedure TLED.WMWINDOWPOSCHANGED(var Msg: TWMWindowPosChanged);
begin
  if Height <> bmpHeight then Height:= bmpHeight;
  if Width <> bmpWidth then Width:= bmpWidth;
end;

{ Set LED Lit state }
procedure TLED.SetState(const NewState : Boolean);
begin
 if fLit <> NewState then begin
   fLit:= NewState;
   if csDesigning in ComponentState then Invalidate else
   begin
     PaintLED;
     if Assigned(fChange) then OnChangeState(Self);
   end; { if not designing }
 end; { if fLit <> NewState }
end;

{ Set LED Colour }
procedure TLED.SetColor(const NewColor : TLEDColor);
begin
 if fColor <> NewColor then
 begin
   fColor:= NewColor;
   if csDesigning in ComponentState then Invalidate else
   begin
     PaintLED;
     if Assigned(fChange) then OnChangeState(Self);
   end; {if not designing}
 end; {if fColor <> NewColor}
end;

{ Toggle LED Lit state }
function TLED.ChangeState : Boolean;
begin
 fLit := not fLit;
 PaintLED;
 if Assigned(fChange) then OnChangeState(Self);
 Result := fLit;
end;

{ Cycle LED Colour through Red...Green...Yellow...Red... etc. }
function TLED.ChangeColor : TLEDColor;
begin
 case fColor of
   lcRed: fColor := lcGreen;
   lcGreen: fColor := lcYellow;
   lcYellow: fColor := lcRed;
 end; { case }
 PaintLED;
 if Assigned(fChange) then OnChangeState(Self); { Call User Method }
 Result := fColor;
end;

procedure TLED.Paint;
begin
  PaintLED;                { Paint the LED Bitmap }
end;

procedure TLED.PaintLED;
var
  IndexX, IndexY: Integer; { Identifies LED position in LED_PIC grid }
begin
  if not Enabled then      { Use disabled picture }
  begin
    IndexX := 0;
    IndexY := NumRows-1
  end else
  begin
    IndexX := Ord(Lit);    { 0= Left, 1 = Right column in LED_PICS Bitmap }
    IndexY := Ord(fColor); { 0= Red, 1 = Green, 2 = Yellow };
  end;
  Canvas.Brush.Color := TWincontrol(Parent).Brush.Color;
  Canvas.Brush.Style := bsSolid;
  Canvas.BrushCopy(ClientRect, LEDBitmap.LED_Pics,
                   Rect(bmpWidth*IndexX, bmpHeight*IndexY,
                        bmpWidth*(IndexX+1), bmpHeight*(IndexY+1)),
                   LEDBitmap.Trans_Color);
end;

{ LEDLabel }

constructor TLEDLabel.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Height := 17;
  Width := 97;
  fAlignment := caRight;
end;

procedure TLEDLabel.Paint;
begin
  Canvas.Brush.Color := TWincontrol(Parent).Brush.Color;
  if not Transparent then { Opaque Background }
  begin
    Canvas.Brush.Style := bsSolid;
    Canvas.FillRect(ClientRect);
  end;
  PaintLED;               { Paint the LED Bitmap  }
  PaintCaption;           { Paint the Caption text }
end;

procedure TLEDLabel.PaintLED;
var
  IndexX, IndexY: Integer; { Identifies LED position in LED_PIC grid }
begin
  if not Enabled then      { Use Disabled picture }
  begin
    IndexX := 0;
    IndexY := NumRows-1
  end else
  begin
    IndexX := Ord(Lit);    { 0= Left, 1 = Right column in LED_PICS Bitmap }
    IndexY := Ord(fColor); { 0= Red, 1 = Green, 2 = Yellow };
  end;
  Canvas.Brush.Color := TWincontrol(Parent).Brush.Color;
  Canvas.Brush.Style := bsSolid;
  { Paint the LED in fLEDRect bounds }
  Canvas.BrushCopy(fLEDRect, LEDBitmap.LED_Pics,
                   Rect(bmpWidth*IndexX, bmpHeight*IndexY,
                        bmpWidth*(IndexX+1), bmpHeight*(IndexY+1)),
                   LEDBitmap.Trans_Color);
end;

procedure TLEDLabel.PaintCaption;
var
  CaptionStr: array [0..255] of char;
  TempRect: TRect;

  procedure DoDrawCaption(TextRect: TRect);
  begin
    case fAlignment of
    caTop:
      DrawText( Canvas.Handle, CaptionStr, StrLen(CaptionStr),
                TextRect, dt_Bottom or dt_SingleLine or dt_Center);
    caBottom:
      DrawText( Canvas.Handle, CaptionStr, StrLen(CaptionStr),
                TextRect, dt_Top or dt_SingleLine or dt_Center);
    else
      DrawText( Canvas.Handle, CaptionStr, StrLen(CaptionStr),
                TextRect, dt_VCenter or dt_SingleLine);
    end; {case}
  end; {DoDrawCaption}

begin
  Canvas.Font:=Font;
  Canvas.Brush.Style := bsClear;
  StrPCopy(CaptionStr, Caption);

  { if Enabled = False then draw text 'greyed' }
  if not Enabled then
  begin
    TempRect := fTextRect;
    OffsetRect(TempRect,1,1);
    Canvas.Font.Color:= clBtnHighLight;
    DoDrawCaption(TempRect);
    Canvas.Font.Color := clBtnShadow;
    Canvas.Brush.Style := bsClear;
  end; { if not Enabled}

  { Draw the Caption }
  DoDrawCaption(fTextRect);

end;

procedure TLEDLabel.SetAlignment(const Value: TCaptionAlignment);
begin
  if fAlignment <> Value then
  begin
    fAlignment := Value;
    Invalidate;
    { Force LED and Caption rectangle re-calculations }
    Perform(WM_WINDOWPOSCHANGED,0,0);
  end;
end;

function TLEDLabel.GetTransparent: Boolean;
begin
  Result := not (csOpaque in ControlStyle);
end;

procedure TLEDLabel.SetTransparent(Value: Boolean);
begin
  if Transparent <> Value then
  begin
    if Value then
      ControlStyle := ControlStyle - [csOpaque] else
      ControlStyle := ControlStyle + [csOpaque];
    Invalidate;
  end;
end;

{ Respond to changes in Caption text }
procedure TLEDLabel.CM_TEXTCHANGED(var Msg: TMessage);
begin
  Invalidate;
  inherited;
end;

{ Re-calculate LED and Caption rectangles when LEDLabel size changes }
procedure TLEDLabel.WMWINDOWPOSCHANGED(var Msg: TWMWindowPosChanged);
var
  HW: Integer;
begin
  case fAlignment of
    caLeft:
      begin
        HW := ((Height-bmpHeight)div 2);
        fLEDRect := Rect(Width-bmpWidth, HW, Width, HW + bmpHeight);
        fTextRect := Rect(0, 0, Width-bmpWidth-CaptionSpacing, Height);
      end;
    caRight:
      begin
        HW := ((Height-bmpHeight)div 2);
        fLEDRect := Rect(0, HW, bmpwidth, HW + bmpHeight);
        fTextRect := Rect(bmpWidth + CaptionSpacing, 0, Left+Width, Height);
      end;
    caTop:
      begin
        HW := ((Width-bmpWidth)div 2);
        fLEDRect := Rect(HW, Height - bmpHeight, HW+bmpWidth, Height);
        fTextRect := Rect(0, 0, Width, Height-bmpHeight-CaptionSpacing);
      end;
    caBottom:
      begin
        HW := ((Width-bmpWidth)div 2);
        fLEDRect := Rect(HW, 0, HW+bmpWidth, bmpHeight);
        fTextRect := Rect(0, bmpHeight+CaptionSpacing, Width, Height);
      end;
  end; {case}
end;

end.
