unit cmpEvaluator;

// Copyright (c) Colin Wilson 1997.  All rights reserved.
// E-mail to woozle@compuserve.com

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs;

type
  TExpressionEvaluator = class(TComponent)
  private
    fExpression : string;

    pos : Integer;
    ch : char;

    function GetChar : char;
    function GetNonWhitespace : char;
    procedure SkipWhitespace;

    function CalcAddition : extended;
    function CalcMultiplication : extended;
    function CalcPower : extended;
    function CalcSignedTerm : extended;
    function CalcTerm : extended;
    function CalcNumber : extended;
    function CalcExpressionInBrackets : extended;

  protected
  public
    function Calculate : extended;
    property Position : Integer read pos;
  published
    property Expression : string read fExpression write fExpression;
  end;

procedure Register;

implementation

function TExpressionEvaluator.GetChar : char;
begin
  if pos < Length (Expression) then
  begin
    ch := Expression [pos + 1];
    Inc (pos)
  end
  else ch := #0;
  result := ch;
end;

function TExpressionEvaluator.GetNonWhitespace : char;
begin
  GetChar;
  SkipWhitespace;
  result := ch;
end;

procedure TExpressionEvaluator.SkipWhitespace;
begin
  if ch in [' ', #9] then
  repeat
    GetChar
  until not (ch in [' ', #9])
end;

function TExpressionEvaluator.CalcAddition : extended;
begin
  result := CalcMultiplication;
  while True do
    case ch of
      '+' : result := result + CalcMultiplication;
      '-' : result := result - CalcMultiplication;
      else break
    end
end;

function TExpressionEvaluator.CalcMultiplication : extended;
begin
  result := CalcPower;
  while True do
    case ch of
      '*' : result := result * CalcPower;
      '/' : result := result / CalcPower;
      else break
    end
end;

function TExpressionEvaluator.CalcPower : extended;
begin
  result := CalcSignedTerm;
  while True do
    case ch of
      '^' : result := exp (ln (result) * CalcSignedTerm);
      else break
    end
end;

function TExpressionEvaluator.CalcSignedTerm : extended;
begin
  case GetNonWhitespace of
    '+' : result := CalcSignedTerm;
    '-' : result := -CalcSignedTerm;
    else result := CalcTerm
  end
end;

function TExpressionEvaluator.CalcTerm : extended;
begin
  case ch of
    '0'..'9' : result := CalcNumber;
    '(' : result := CalcExpressionInBrackets;
    else
      raise Exception.CreateFmt ('Syntax error in expression at position %d', [pos + 1])
  end
end;

function TExpressionEvaluator.CalcNumber : extended;
var
   s : string;
begin
  result := 0;
  s := '';
  while ch in ['0'..'9'] do
  begin
    s := s + ch;
    GetChar;
    if ch = '.' then
    begin
      repeat
        s := s + ch;
        GetChar
      until not (ch in ['0'..'9']);
      break
    end
  end;
  result := StrToFloat (s);
  SkipWhitespace
end;

function TExpressionEvaluator.CalcExpressionInBrackets : extended;
begin
  result := CalcAddition;
  if ch = ')' then
    GetNonWhitespace
  else
    raise Exception.CreateFmt ('Mismatched parentheses at position %d', [pos + 1])
end;

function TExpressionEvaluator.Calculate : extended;
begin
  pos := 0;
  result := CalcAddition;
  if ch <> #0 then
    raise Exception.CreateFmt ('Unexpected end of expression at position %d', [pos + 1]);
end;

procedure Register;
begin
  RegisterComponents('Samples', [TExpressionEvaluator]);
end;

end.
