unit Mainform;
(*-----
    File: MAINFORM.PAS for Project CODEAPP.DPR
   Description:
  Application to select a code file for viewing or copy and pasting
  code sections from the file, or printing the file.
  For text (ASCII) files only.

-----*)

(* uncomment this to enable DDE
 {$DEFINE Using DDE} {enables DDE}
*)

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, FileCtrl,
  IniFiles, Printers, Grids, Buttons, ExtCtrls, ShellAPI,
  Viewcode, FindWhat, PrintDlg, FileDlg, FileFunc, TextClip, DdeMan;

type
  TFileForm = class(TForm)
    BtnExit: TBitBtn;
    BtnHelp: TBitBtn;
    ViewBtn: TBitBtn;
    PrintBtn: TBitBtn;
    Panel1: TPanel;
    FilesName: TLabel;
    FileInfo: TLabel;
    UpdateBox: TCheckBox;
    SelectFileBtn: TBitBtn;
    DdeClientConv1: TDdeClientConv;
    DdeClientItem1: TDdeClientItem;
    DDELinkBtn: TBitBtn;
    procedure ViewBtnClick(Sender: TObject);
    procedure PrintBtnClick(Sender: TObject);
    procedure BtnExitClick(Sender: TObject);
    procedure BtnHelpClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormShow(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure SelectFileBtnClick(Sender: TObject);
    procedure FormActivate(Sender: TObject);
    procedure FormDragDrop(Sender, Source: TObject; X, Y: Integer);
    procedure FormDragOver(Sender, Source: TObject; X, Y: Integer;
      State: TDragState; var Accept: Boolean);
    procedure DdeClientItem1Change(Sender: TObject);
    procedure DDELinkBtnClick(Sender: TObject);
  private
    { Private declarations }
    printing : boolean;
    theFile: string; {name of file}
    FDateStr: string; {file's date string}
    procedure WndProc(var Msg: TMessage); override;
    procedure ShowSelection(const FN: string);
    procedure CheckCmdLine;
  public
    { Public declarations }
  end;

var
  FileForm: TFileForm;

implementation

{$R *.DFM}

procedure TFileForm.ShowSelection(const FN: string);
{-Show info on file selected or dropped on form}
var
  F: TSearchRec;
begin
  FilesName.Caption := ExtractFilePath(FN);
  if GetFileInfo(FN, F) then
  begin
    FDateStr := FormatDateTime(DateTimeFormat,
    FileDateToDateTime(F.Time));
    FileInfo.Caption := ShowFileStats(F);
    ViewBtn.Enabled := True;
    PrintBtn.Enabled := True;
  end
  else
  begin
    FDateStr := '';
    FileInfo.Caption := 'FILE INFO UNAVAILABLE';
    theFile := '';
  end;
end;

procedure TFileForm.BtnExitClick(Sender: TObject);
{-End this App}
begin
  Close
end;

procedure TFileForm.BtnHelpClick(Sender: TObject);
{-Tell user basic use}
begin
  MessageDlg('Select a File by dragging it from another App'+#13+
  'and dropping it on this form or '+#13+
  'use the Select File Dialog.',
  mtInformation, [mbCancel], 0);
end;

procedure TFileForm.FormCreate(Sender: TObject);
{-Set up main form}
begin
  DragAcceptFiles(handle, True);
  ViewBtn.Enabled := False;
  PrintBtn.Enabled := False;
  Top := 0;
  Left := 0;
  {$IFDEF Using DDE}
  DDELinkBtn.Visible := True;
  {$ELSE}
  DDELinkBtn.Visible := False;
  {$ENDIF}
end;

procedure TFileForm.SelectFileBtnClick(Sender: TObject);
begin
  SelectFileDlg.FileSpec := theFile;
  SelectFileDlg.Show
end;

procedure TFileForm.FormActivate(Sender: TObject);
{-Show selected file}
begin
  if SelectFileDlg.FileSpec <> '' then
  begin
    theFile := SelectFileDlg.FileSpec;
    ShowSelection(theFile);
  end
end;

procedure TFileForm.ViewBtnClick(Sender: TObject);
{-Exec the text viewer for file selected}
begin
  if theFile = '' then
    MessageBeep(0)
  else
    with ViewText do
    begin
      LoadFile(theFile, FDateStr);
      theTarget := ''; {let user choose new}
    end
end;

procedure TFileForm.PrintBtnClick(Sender: TObject);
{-Set up print options, then print file}
begin
  with PRNformatDlg do
  begin
    if printing then {quit}
    begin
      pcancel := True;
      exit;
    end;
    TextList := TStringList.Create;
    try
      TextList.LoadFromFile(theFile);
      SetPrintFactors;
      if ShowModal <> mrCancel then {do it}
      begin
        PrintBtn.Caption := 'Cancel';
        pcancel := False;
        printing := True;
        PrintTheFile(theFile, FilesName);
        PrintBtn.Caption := '&Print...';
        printing := False;
      end;
    finally
      TextList.Free;
    end
  end;
end;

{ .INI file stuff }

procedure TFileForm.FormClose(Sender: TObject;
  var Action: TCloseAction);
{-Write to .INI file}
begin
  if UpdateBox.Checked then
    with TIniFile.Create(ChangeFileExt
    (ExtractFilename(Application.ExeName),'.INI')) do
    try
      with SelectFileDlg do
      begin
        WriteString('DirectoryPath', 'FilePath', FilePath);
        WriteString('FileTypes', 'FileTypes', FileListBox1.Mask);
      end;
      with FindWhatDlg do
      begin
        WriteString('Search Options', 'SearchItem', ComboBox1.Text);
        WriteBool('Search Options', 'StartAtTop', StartTop.Checked);
        WriteBool('Search Options', 'AnyCaseChars', AnyCase.Checked);
        WriteBool('Search Options', 'WholeWordsOnly', WholeWords.Checked);
      end;
      with PRNformatDlg do
      begin
        WriteBool('Print Format', 'Formatted', HasTitle.Checked);
        WriteBool('Print Format', 'LineNumbering', LineNumbering.Checked);
        WriteBool('Print Format', 'LastPageFirst', LastPageFirst.Checked);
        WriteBool('Print Format', 'AutoSetCPI', AutoWidth.Checked);
        WriteInteger('Print Format', 'PageWidth', page_width);
      end;
    finally
      Free;
    end;
end;

procedure TFileForm.CheckCmdLine;
begin
  if ParamCount > 0 then
  begin
    theFile := ParamStr(1);
    try
      SelectFileDlg.DirectoryListBox1.Directory :=
        ExtractFilePath(theFile);
      ShowSelection(theFile);
      if ParamCount > 1 then
        with ViewText do
        begin
          theTarget := ParamStr(2);
          LoadFile(theFile, FDateStr);
        end
    except
      ShowMessage('Unable to set Directory List Box to '+theFile);
    end;
  end;
end;

procedure TFileForm.FormShow(Sender: TObject);
{-Read from .INI file}
var
  mp, ic, ix: integer;
begin
  with TIniFile.Create(ChangeFileExt(ExtractFilename
  (Application.ExeName), '.INI')) do
  try
    with SelectFileDlg do
    begin
      FilePath := ReadString('DirectoryPath', 'FilePath', '');
      FileTypes := ReadString('FileTypes', 'FileTypes', '*.*');
    end;
    with FindWhatDlg do
    begin
      ComboBox1.Text := ReadString('Search Options', 'SearchItem', '');
      StartTop.Checked := ReadBool('Search Options', 'StartAtTop', True);
      AnyCase.Checked := ReadBool('Search Options', 'AnyCaseChars', True);
      WholeWords.Checked := ReadBool('Search Options', 'WholeWordsOnly', False);
    end;
    with PRNformatDlg do
    begin
      HasTitle.Checked := ReadBool('Print Format', 'Formatted', True);
      LineNumbering.Checked := ReadBool('Print Format',
       'LineNumbering', True);
      LastPageFirst.Checked := ReadBool('Print Format',
       'LastPageFirst', False);
      AutoWidth.Checked := ReadBool('Print Format',
       'AutoSetCPI', False);
      page_width := ReadInteger('Print Format', 'PageWidth', 80);
    end;
  finally
    Free;
  end;

  with SelectFileDlg do {set up Select File Dialog}
  begin
    if FilePath = '' then {get current}
      GetDir(0, FilePath);
    DirectoryListBox1.Directory := FilePath;
    DriveComboBox1.Drive := DirectoryListBox1.Drive;
    {Set selected item in FilterComboBox to stored FileTypes }
    mp := pos('|'+FileTypes, FilterComboBox1.Filter);
    if mp > 0 then {it's in the combo's filter list, so use it}
    begin
      FileListBox1.Mask := FileTypes;
      {make combo selection match filelist at start}
      ic := 0;
      for ix := 1 to mp-1 do
        if FilterComboBox1.Filter[ix] = '|' then
          inc(ic);
      FilterComboBox1.ItemIndex := ic div 2;
    end;
  end;

  {-Cmd line will override .INI}
  CheckCmdLine;
end;

{ Drag and drop stuff }

procedure TFileForm.WndProc(var Msg: TMessage);
{-Respond to DropFile message}
var
  buff: Array[0..80] of Char;
begin
  if Msg.Msg = WM_DropFiles then
  begin
    Msg.Result := 0;
    DragQueryFile(Msg.wParam, 0, @buff, SizeOf(buff));
    if StrLen(buff) <> 0 then
    begin
      theFile := StrPas(buff);
      SetFocus;
      ShowSelection(theFile);
    end;
    DragFinish(Msg.wParam);
  end
  else
    inherited WndProc(Msg);
end;

procedure TFileForm.FormDragDrop(Sender, Source: TObject; X, Y: Integer);
{-Drop from file select dialog}
begin
  theFile := SelectFileDlg.FileSpec;
  SelectFileDlg.Close;
  SetFocus;
  ShowSelection(theFile);
end;

procedure TFileForm.FormDragOver(Sender, Source: TObject; X, Y: Integer;
  State: TDragState; var Accept: Boolean);
begin
  Accept := Source is TFileListBox;
end;

{ DDE Stuff }

procedure TFileForm.DdeClientItem1Change(Sender: TObject);
{-Get server data}
var
  F: string;
  ps: Integer;
begin
  F := DdeClientItem1.Text;
  { Data format is 'filename searchstr'}
  ps := pos(' ', F);
  if ps > 0 then {have valid data}
  begin
    theFile := copy(F, 1, ps-1);
    { Removing DDE added spaces -- ok, but then your search
      string can never have trailing spaces}
    while F[Length(F)] = ' ' do {remove extraneous spaces}
       System.Delete(F, Length(F), 1);
    ViewText.theTarget := copy(F, ps+1, Length(F)-ps);
    ShowSelection(theFile);
    ViewText.LoadFile(theFile, FDateStr);
  end;
end;

procedure TFileForm.DDELinkBtnClick(Sender: TObject);
const
  Service = 'DELGREP';
  Topic = 'Delphi Grep';
begin
  if DdeClientConv1.SetLink(Service, Topic) then
  begin
    DdeClientItem1.DdeItem := 'DdeServerItem1';
    if DdeClientConv1.ConnectMode = ddeManual then
      DdeClientConv1.OpenLink;
    DDELinkBtn.Enabled := False;
  end
  else
    ShowMessage('Unable to create DDE Link to '+Service);
end;

end.
