unit Printdlg;
(*-----
    File: PRINTDLG.PAS for Project CODEAPP.DPR
    Sends a text file to printer
-----*)

{.$DEFINE Testing} {enable for out to file}

interface

uses WinTypes, WinProcs, Classes, Graphics, Forms, Controls, Buttons,
  StdCtrls, ExtCtrls, SysUtils, Dialogs, Spin, Printers, FileFunc;

type
  TPRNformatDlg = class(TForm)
    OKBtn: TBitBtn;
    CancelBtn: TBitBtn;
    HelpBtn: TBitBtn;
    Bevel1: TBevel;
    LineNumbering: TCheckBox;
    PrintPitch: TRadioGroup;
    HasTitle: TCheckBox;
    LastPageFirst: TCheckBox;
    GroupBox1: TGroupBox;
    Label2: TLabel;
    Label3: TLabel;
    Panel2: TPanel;
    AutoWidth: TCheckBox;
    LinesLabel: TLabel;
    SpinEditLast: TSpinEdit;
    SpinEditFirst: TSpinEdit;
    procedure HelpBtnClick(Sender: TObject);
    procedure OKBtnClick(Sender: TObject);
    procedure AutoWidthClick(Sender: TObject);
    procedure SpinEditFirstChange(Sender: TObject);
    procedure SpinEditLastChange(Sender: TObject);
    procedure HasTitleClick(Sender: TObject);
  private
    { Private declarations }
    FirstPage, LastPage : integer;
    PrintLength: Integer;               {lines per page}
    Pages: Integer;
    procedure AutoSetCPI;
    procedure UpdateLineRange;
  public
    { Public declarations }
    TextList: TStringList;
    page_width : Integer;  {print width in # of columns}
    pcancel: boolean;
    procedure SetPrintFactors;
    procedure PrintTheFile(const FileSpec: String; FilesName: TLabel);
  end;


const
  LinesPerPage = 55;               {nominal lines per page}

var
  PRNformatDlg: TPRNformatDlg;

implementation

{$R *.DFM}

const
  Widths: array[0..3] of integer = (40, 80, 132, 160);

procedure TPRNformatDlg.PrintTheFile(const FileSpec: String;
   FilesName: TLabel);
{-Print the file to the printer}
const
  Esc = ^[;                       { ASCII Escape }
  BoldOff = Esc+'(s0B';           { Bold Print Off }
  BoldOn = Esc+'(s3B';            { Bold Print On }
  PRNDateTimeFormat = 'mmm d, yy  h:mm:ss am/pm';
var
  Page: Integer;
  PrintText: System.Text;
  PrinterMode: Integer; {print format mode}
  S, HeaderStr1, FooterStr: string;
  F: TSearchRec;
  oldMode: Word;

  procedure SelectPrintMode(const col : Integer);
  var
    M: string;
  begin
    case col of
       40 : M := '(s5H';     { 5 cpi }
       80 : M := '(s10H';    { 10 cpi }
      132 : M := '(s16.67H'; { 16.67 cpi }
      160 : M := '(s20H';    { 20 cpi }
    else
      exit; {.. nothing}
    end;                          { case }
    write(PrintText, Esc, M);
  end;

  procedure InitPrinter;
  {- laser printer setup: select PC-8 font; perf skip on, 66 lines }
  const
    PrnInitStr = Esc+'(10U'+Esc+'&l1L'+Esc+'&l66P';
  begin
    write(PrintText, PrnInitStr);
  end;

  procedure WriteHeader;
  begin
    if HasTitle.Checked then
    begin
      if page_width <> 80 then  { restore it }
        SelectPrintMode(80); { 10.0 cpi }
      writeln(PrintText, BoldOn, HeaderStr1, BoldOff);
      writeln(PrintText);
      if page_width <> 80 then
        SelectPrintMode(page_width);
    end;
  end;

  procedure WriteFooter;
  begin
    if HasTitle.Checked then
    begin
      if page_width <> 80 then  { restore it }
        SelectPrintMode(80); { 10.0 cpi }
      writeln(PrintText);
      writeln(PrintText, BoldOn, FooterStr,
      'Page ':65-Length(FooterStr),Page,' of ',Pages, BoldOff);
    end;
    write(PrintText, ^L); {form feed}
  end;

  procedure OutputPage;
  {-Output a print page}
  var
    Line, firstline, lastline : Integer;
    rlines: Integer;
    S: string;
  begin
    firstline := ((Page-1) * PrintLength) +1;
    if Page >= Pages then
      lastline := TextList.Count
    else
      lastline := Page * PrintLength;

    WriteHeader;
    for Line := firstline to lastline do
    begin
      if LineNumbering.Checked then
        write(PrintText, Line:5,': ');
      writeln(PrintText, TextList.Strings[Line-1]);
      Application.ProcessMessages;
    end;
    if Page >= Pages then {last page}
      if HasTitle.Checked then
      begin
        rlines := TextList.Count mod PrintLength;
        if rlines <> 0 then
        begin
          for Line := rlines+1 to PrintLength do {feed out last page}
            writeln(PrintText);
        end
      end;
    WriteFooter;
    S := 'File: '+ExtractFileName(FileSpec)+
      '  Page:'+IntToStr(Page);
    FilesName.Caption := S;
  end; {OutputPage}

begin {PrintTheFile}
  FooterStr := '  Listing date: '+ FormatDateTime(DateTimeFormat,
    Now);
  HeaderStr1 := '';
  if GetFileInfo(FileSpec, F) then
  try
    HeaderStr1 := Format('  %13s   File Size: %6s   File Date: %s',
      [F.Name, FormatFloat(',##########', F.Size),
      FormatDateTime(PRNDateTimeFormat,
      FileDateToDateTime(F.Time))]);
  except
    ShowMessage('Unable to get file data for print-out');
    exit;
  end;

  oldMode := SetErrorMode(SEM_FAILCRITICALERRORS);
  {$IFDEF Testing}
  AssignFile(PrintText, ChangeFileExt(FileSpec, '.lst'));
  {$ELSE}
  AssignFile(PrintText, 'PRN');
  {$ENDIF}
  try
    Rewrite (PrintText);
    try
      InitPrinter; { Init the printer }
      Screen.Cursor := crHourGlass;
      SelectPrintMode(page_width);

      { Get range of pages}
      FirstPage := SpinEditFirst.Value;
      LastPage := SpinEditLast.Value;

      { Print Pages}
      if LastPageFirst.Checked then {backwards}
        for Page := LastPage downto FirstPage do
        begin
          OutputPage;
          if pcancel then break;  {get out}
        end
      else
        for Page := FirstPage to LastPage do
        begin
          OutputPage;
          if pcancel then break;
        end;

      { Check result }
      if LastPage >= FirstPage then
      begin
        S := 'File: '+ExtractFileName(FileSpec);
        if pcancel then
          FilesName.Caption := 'Printing of '+S+' ABORTED'
        else
        begin
          S := S + '  Pages printed: '+IntToStr(LastPage-FirstPage+1);
          FilesName.Caption := S;
        end
      end;
      pcancel := False;
      {-Restore printer to default state}
      if page_width <> 80 then  { restore it }
        SelectPrintMode(80); { 10.0 cpi }
    finally
      CloseFile(PrintText);
      Screen.Cursor := crDefault;
      SetErrorMode(oldMode);
   end;
  except
    on EInOutError do
    begin
      S := Format('Unable to print text for file: %s'+
      #13+'Check Printer Status', [FileSpec]);
      MessageDlg(S, mtError, [mbOk], 0);
    end;
  end;
end;  {PrintTheFile}

procedure TPRNformatDlg.HelpBtnClick(Sender: TObject);
{-Tell user basic use}
begin
  MessageDlg('Select print format options,'+#13+
  'then click on OK to start printing.',
  mtInformation, [mbCancel], 0);
end;

procedure TPRNformatDlg.AutoSetCPI;
{-Set up print format}
var
  leadin, MaxLen, ix: integer;
begin
  {Get max line width}
  MaxLen := 0;
  leadin := 0;
  if LineNumbering.Checked then
    leadin := 7;
  for ix := 0 to TextList.Count-1 do
  begin
    if Length(TextList.Strings[ix]) > MaxLen then
      MaxLen := Length(TextList.Strings[ix]);
  end;
  page_width := Widths[0];
  {set page width}
  for ix := 0 to 2 do
    if MaxLen+leadin > Widths[ix] then
      page_width := Widths[ix+1];
  {change pitch to match}
  for ix := 0 to 3 do
    if page_width = Widths[ix] then
      PrintPitch.ItemIndex := ix
end;

procedure TPRNformatDlg.SetPrintFactors;
var
  ix: integer;
begin
  { set width}
  if AutoWidth.Checked then
    AutoSetCPI
  else
  begin
    PrintPitch.ItemIndex := 1; {default, 80 col.}
    for ix := 0 to 3 do
      if page_width = Widths[ix] then {get from .INI}
        PrintPitch.ItemIndex := ix;
  end;
  { set lines, pages }
  if HasTitle.Checked then
    PrintLength := LinesPerPage              {set lines per page}
  else
    PrintLength := LinesPerPage+4;           {unformtd lines per page}
  Pages := TextList.Count div PrintLength;
  if TextList.Count mod PrintLength <> 0 then
    inc(Pages);

  with SpinEditFirst do
  begin
    MinValue := 1;
    MaxValue := Pages;
    Value := 1;        {set last}
  end;
  with SpinEditLast do
  begin
    MinValue := 1;
    MaxValue := Pages;
    Value := Pages;    {set last}
  end;
end;

procedure TPRNformatDlg.OKBtnClick(Sender: TObject);
begin
  page_width := Widths[PrintPitch.ItemIndex];
end;

procedure TPRNformatDlg.UpdateLineRange;
{-Show how many lines}
var
  lastline: integer;
begin
  lastline:= SpinEditLast.Value * PrintLength;
  if lastline > TextList.Count then
    lastline := TextList.Count;
  LinesLabel.Caption := Format('Lines %4d to %4d',
  [((SpinEditFirst.Value - 1) * PrintLength)+1, lastline]);
end;

procedure TPRNformatDlg.SpinEditFirstChange(Sender: TObject);
begin
  if SpinEditFirst.Value > SpinEditLast.Value then
  begin
    MessageBeep(0);
    SpinEditFirst.Value := SpinEditLast.Value;
  end
  else
    UpdateLineRange;
end;

procedure TPRNformatDlg.SpinEditLastChange(Sender: TObject);
begin
  if SpinEditLast.Value < SpinEditFirst.Value then
  begin
    MessageBeep(0);
    SpinEditLast.Value := SpinEditFirst.Value;
  end
  else
    UpdateLineRange;
end;

procedure TPRNformatDlg.HasTitleClick(Sender: TObject);
begin
  if Visible then
    SetPrintFactors;
end;

procedure TPRNformatDlg.AutoWidthClick(Sender: TObject);
begin
  if Visible and AutoWidth.Checked then
    AutoSetCPI
end;

end.
