{===EZDSLPQU==========================================================

Part of the Delphi Structures Library--the priority queue.

EZDSLPQU is Copyright (c) 1993, 1996 by  Julian M. Bucknall

VERSION HISTORY
13Mar96 JMB 2.00 release for Delphi 2.0
18Jun95 JMB 1.00 conversion of EZStrucs to Delphi
======================================================================}
{ Copyright (c) 1993, 1996, Julian M. Bucknall. All Rights Reserved   }

unit EZDSLPQu;

{$I EZDSLDEF.INC}
{---Place any compiler options you require here-----------------------}


{---------------------------------------------------------------------}
{$I EZDSLOPT.INC}

interface

uses
  SysUtils,
  WinTypes,
  WinProcs,
  Classes,
  EZDSLCts,
  EZDSLSup,
  EZDSLBse;

type
  TPriorityQueue = class(TAbstractContainer)
    {-Priority queue object}
    private
      Rt : PNode;

    protected
      procedure pqBubbleUp(Node : PNode);
      function  pqGetNodeFromIndex(Inx : longint) : PNode;
      procedure pqTrickleDown(Node : PNode);

    public
      constructor Create(DataOwner : boolean); override;
      constructor Clone(Source : TAbstractContainer;
                        DataOwner : boolean; NewCompare : TCompareFunc); override;

      procedure Append(aData : pointer);
      procedure Empty; override;
      function  Examine : pointer;
      function  Pop : pointer;
      function  Replace(aData : pointer) : pointer;
  end;

implementation

{=TPriorityQueue======================================================
A priority queue

Much like an ordinary queue, expect that the smallest data object in
the queue is popped first. Another name for a priority queue is a
heap.

If the Compare method returns values in the 'normal' sense (ie -1 if
Data1 < Data2, etc), then data objects will be popped off smallest
first, ie in increasing order. If Compare returns values in the
'reverse' sense (ie -1 if Data1 > Data2, etc), then elements will be
popped off largest first, ie in decreasing order. Thus by carefully
selecting Compare, this object will provide a min-heap and a max-heap.
======================================================================}
constructor TPriorityQueue.Create(DataOwner : boolean);
  begin
    NodeSize := 16;
    inherited Create(DataOwner);
    Rt := acNewNode(nil);
    Rt^.TLink[CLeft] := Rt;
    Rt^.TLink[CRight] := nil;
    FCount := 0;
  end;
{--------}
constructor TPriorityQueue.Clone(Source : TAbstractContainer;
                                 DataOwner : boolean;
                                 NewCompare : TCompareFunc);
  var
    OldQueue : TPriorityQueue absolute Source;
    NodeInx  : longint;
    NewData  : pointer;
  begin
    Create(DataOwner);
    Compare := NewCompare;
    DupData := OldQueue.DupData;
    DisposeData := OldQueue.DisposeData;

    if not (Source is TPriorityQueue) then
      RaiseError(escBadSource);

    if OldQueue.IsEmpty then Exit;

    for NodeInx := 1 to OldQueue.Count do
      begin
        if DataOwner then
             NewData := DupData(OldQueue.pqGetNodeFromIndex(NodeInx)^.Data)
        else NewData := OldQueue.pqGetNodeFromIndex(NodeInx)^.Data;
        try
          Append(NewData);
        except
          DisposeData(NewData);
          raise;
        end;
      end;
  end;
{--------}
procedure TPriorityQueue.Append(aData : pointer);
  var
    Node,
    NewParent : PNode;
  begin
    Node := acNewNode(aData);
    if (Count = 1) then
         NewParent := Rt
    else NewParent := pqGetNodeFromIndex(Count shr 1);
    with NewParent^ do
      if Odd(Count) then
           TLink[CRight] := Node
      else TLink[CLeft] := Node;
    Node^.PLink := NewParent;
    pqBubbleUp(Node);
  end;
{--------}
procedure TPriorityQueue.Empty;
  begin
    if IsDataOwner then
      while not IsEmpty do DisposeData(Pop)
    else
      while not IsEmpty do Pop;
    if InDone then
      if Assigned(Rt) then
        acDisposeNode(Rt);
  end;
{--------}
function  TPriorityQueue.Examine : pointer;
  begin
    {$IFDEF DEBUG}
    Assert(not IsEmpty, ascEmptyExamine);
    {$ENDIF}
    Examine := Rt^.TLink[CRight]^.Data;
  end;
{--------}
function TPriorityQueue.Pop : pointer;
  var
    Root,
    LastNode : PNode;
  begin
    {$IFDEF DEBUG}
    Assert(not IsEmpty, ascEmptyPop);
    {$ENDIF}
    Root := Rt^.TLink[CRight];
    LastNode := pqGetNodeFromIndex(Count);
    Pop := Root^.Data;
    Root^.Data := LastNode^.Data;
    with LastNode^.PLink^ do
      if Odd(Count) then
           TLink[CRight] := nil
      else TLink[CLeft] := nil;
    acDisposeNode(LastNode);
    if not IsEmpty then
      pqTrickleDown(Root);
  end;
{--------}
function TPriorityQueue.Replace(aData : pointer) : pointer;
  begin
    Rt^.Data := aData;
    pqTrickleDown(Rt);
    Replace := Rt^.Data;
    Rt^.Data := nil;
  end;
{--------}
procedure TPriorityQueue.pqBubbleUp(Node : PNode);
  var
    AllDone : boolean;
    OurData : pointer;
  begin
    AllDone := false;
    OurData := Node^.Data;
    repeat
      with Node^ do
        {If our parent is 'larger' than we are, swap data and move up}
        if (PLink <> Rt) and (Compare(PLink^.Data, OurData) > 0) then
          begin
            Node^.Data := PLink^.Data;
            Node := PLink;
          end
        else
          AllDone := true;
    until AllDone;
    Node^.Data := OurData;
  end;
{--------}
function TPriorityQueue.pqGetNodeFromIndex(Inx : longint) : PNode;
  {$IFNDEF Win32}
  type
    LH = record L, H : word; end;
  {$ENDIF}
  var
    Temp : PNode;
    Mask : longint;
  begin
    {$IFDEF DEBUG}
    Assert((0 < Inx) and (Inx <= Count), ascOutOfRange);
    {$ENDIF}
    Temp := Rt;
    {$IFDEF Win32}
    {find first bit in Inx}
    Mask := $40000000;
    while ((Mask and Inx) = 0) do Mask := Mask shr 1;
    {walk the tree:
     if the next bit in Inx is zero go left, otherwise right }
    while (Mask <> 0) do
      begin
        if ((Mask and Inx) = 0) then
             Temp := Temp^.TLink[CLeft]
        else Temp := Temp^.TLink[CRight];
        Mask := Mask shr 1;
      end;
    {$ELSE}
    if (LH(Inx).H = 0) then
      asm
        mov ax, $8000
        mov bx, Inx.Word[0]
        les di, Temp
        jmp @@StartTest

      @@Again:
        shr ax, 1
      @@StartTest:
        test ax, bx
        jz @@Again

      {The first walk is always right}
      @@WalkRight:
        les di, es:[di].TNode.BLink  {BLink is equivalent to TLink[CRight]}

      @@TestForAnotherWalk:
        shr ax, 1
        jz @@AllDone
        test ax, bx
        jnz @@WalkRight

      @@WalkLeft:
        les di, es:[di].TNode.FLink  {FLink is equivalent to TLink[CLeft]}
        jmp @@TestForAnotherWalk

      @@AllDone:
        mov Temp.Word[0], di
        mov Temp.Word[2], es
      end
    else
      begin
        {find first bit in Inx}
        Mask := $40000000;
        while ((Mask and Inx) = 0) do Mask := Mask shr 1;
        {walk the tree:
         if the next bit in Inx is zero go left, otherwise right }
        while (Mask <> 0) do
          begin
            if ((Mask and Inx) = 0) then
                 Temp := Temp^.TLink[CLeft]
            else Temp := Temp^.TLink[CRight];
            Mask := Mask shr 1;
          end;
      end;
    {$ENDIF}
    pqGetNodeFromIndex := Temp;
  end;
{--------}
procedure TPriorityQueue.pqTrickleDown(Node : PNode);
  var
    Temp : PNode;
    AllDone : boolean;
    OurData : pointer;
  begin
    if not Assigned(Node^.TLink[CLeft]) then Exit;
    AllDone := false;
    OurData := Node^.Data;
    repeat
      with Node^ do
        begin
          {Find our 'smaller' child}
          if (not Assigned(TLink[CRight])) or
             (Compare(TLink[CLeft]^.Data, TLink[CRight]^.Data) < 0) then
               Temp := TLink[CLeft]
          else Temp := TLink[CRight];
          {If our 'smaller' child is smaller than we are, swap the data,
           and move down}
          if (Compare(Temp^.Data, OurData) < 0) then
            begin
              Node^.Data := Temp^.Data;
              Node := Temp;
            end
          else
            AllDone := true;
        end;
    until AllDone or (not Assigned(Node^.TLink[CLeft]));
    Node^.Data := OurData;
  end;
{---------------------------------------------------------------------}

end.
