{ Copyright  1996 by Ray Konopka.  Distributed with Component Create courtesy }
{ of Ray Konopka.                                                              }
{==============================================================================}
{= RzBizCmp Unit                                                              =}
{=                                                                            =}
{= This unit implements the TRzBusinessComponent class.  This class           =}
{= represents an abstract class which is used as the base class for building  =}
{= data-aware business components.                                            =}
{=                                                                            =}
{= Building Custom Delphi Components - Ray Konopka                            =}
{==============================================================================}


unit RzBizCmp;

interface

uses
  Classes, DB, DBTables;

type
  TRzBusinessComponent = class( TComponent )
  private
    FDataset : TDataset;
  protected
    procedure Notification( AComponent : TComponent;
                            Operation : TOperation ); override;
    procedure SetDataset( Value : TDataset ); virtual;

    procedure SetupFields; virtual;
    procedure CreateFields; virtual; abstract;
    function CreateField( const FieldName : string ) : TField;
    function CreateCalcField( const FieldName : string;
                              FieldClass : TFieldClass;
                              Size : Word ) : TField;
  published
    property Dataset : TDataset
      read FDataset
      write SetDataset;
  end;


implementation

uses
  DBConsts, Dialogs;


{==================================}
{== TRzBusinessComponent Methods ==}
{==================================}


{= TRzBusinessComponent.Notification                                          =}
{=   This method is overridden to ensure that FDataset is set to nil if the   =}
{=   corresponding Dataset is deleted from the form. This method should be    =}
{=   used whenever a contains a reference to another component.               =}

procedure TRzBusinessComponent.Notification( AComponent : TComponent;
                                             Operation : TOperation );
begin
  inherited Notification( AComponent, Operation );
  if ( Operation = opRemove ) and ( AComponent = FDataset ) then
    FDataset := nil;
end;


{= TRzBusinessComponent.SetDataset                                            =}
{=   After setting the Dataset, you must call SetupFields to ensure that the  =}
{=   field objects get created.                                               =}

procedure TRzBusinessComponent.SetDataset( Value : TDataset );
begin
  if Value <> FDataset then
  begin
    FDataset := Value;
    try
      SetupFields;
    except
      on EDBEngineError do
      begin
        MessageDlg( 'To connect to a Dataset, the Dataset must be able to ' +
                    'generate a result set.', mtError, [ mbOk ], 0 );
        FDataset := nil;
      end;

      on EDatabaseError do
      begin
        FDataset := nil;
        raise;
      end;
    end;
  end;
end;


{= TRzBusinessComponent.SetupFields                                           =}
{=   This method is responsible for making sure the appropriate functions are =}
{=   called when the field objects are created. More precisely, there is pre- =}
{=   and post- operations that must be performed.                             =}
{=                                                                            =}
{=   Pre-Operations   Dataset must be closed                                  =}
{=                    Dataset's FieldDefs must be updated                     =}
{=                                                                            =}
{=   Post-Operations  Re-establish original Dataset state                     =}
{=                    Send DataEvent message to Dataset.Designer              =}

procedure TRzBusinessComponent.SetupFields;
var
  ActiveState : Boolean;
begin
  if FDataset = nil then
    Exit;                                 { No need to setup if Dataset is nil }

  { Dataset must be closed in order to create new field objects }
  ActiveState := FDataset.Active;                         { Save current state }
  if ActiveState then
    FDataset.Active := False;

  { Must call Update to gain Access to All Field Defs }
  FDataset.FieldDefs.Update;

  CreateFields;                     { Call to Descendant's CreateFields method }

  FDataset.Active := ActiveState;

  { If the developer has not yet invoked the Field Editor, then the Dataset    }
  { Designer does not exist. If the designer doesn't exist, it is not          }
  { necessary to call DataEvent.  Besides if you do, it will cause a GPF       }

  if FDataset.Designer <> nil then
    FDataset.Designer.DataEvent( deFieldListChange, 0 );
end; {= TRzBusinessComponent.SetupFields =}


{= TRzBusinessComponent.CreateField                                           =}
{=   This method simplifies the process of creating fields.  First, the       =}
{=   Dataset is searched for a matching FieldName. If one exists, then that   =}
{=   one is used.  If not, then a  new field is created from the FieldDef     =}
{=   object corresponding to the desired field name.                          =}

function TRzBusinessComponent.CreateField( const FieldName : string ) : TField;
begin
  { First, try to find an existing field object.  FindField is the same as     }
  { FieldByName, but doesn't raise an exception if the field cannot be found.  }
  Result := FDataset.FindField( FieldName );

  if Result = nil then
  begin
    { If cannot find an existing field object... }
    { Instruct the FieldDefs object create a Field Object }

    Result := FDataset.FieldDefs.Find( FieldName ).CreateField( Owner );

    { We need to give the new field object a name so that it may appear in }
    { the Object Inspector. Use the default naming convention. }

    Result.Name := FDataset.Name + FieldName;
  end;
end; {= TRzBusinessComponent.CreateField =}


{= TRzBusinessComponent.CreateCalcField                                       =}
{=   This method simplifies the process of creating calculated fields. First, =}
{=   the Dataset is searched for a matching FieldName. If one exists, then    =}
{=   that one is used.  If not, then a new field is created from the passed   =}
{=   FieldClass parameter.                                                    =}

function TRzBusinessComponent.CreateCalcField( const FieldName : string;
                                               FieldClass : TFieldClass;
                                               Size : Word ) : TField;
begin
  Result := FDataset.FindField( FieldName );
  if Result = nil then
  begin
    if FieldClass = nil then
      DBErrorFmt( SUnknownFieldType, [ FieldName ] );

    { Create Desired Field Object using a Class Reference Variable }
    Result := FieldClass.Create( Owner );
    try
      Result.FieldName := FieldName;

      { Size is only necessary if new field is one of the following types }
      if ( Result is TStringField ) or
         ( Result is TBCDField ) or
         ( Result is TBlobField ) or
         ( Result is TBytesField ) or
         ( Result is TVarBytesField ) then
      begin
        Result.Size := Size;
      end;
      Result.Calculated := True;
      Result.Dataset := FDataset;
      Result.Name := FDataset.Name + FieldName;
    except
      Result.Free;                { If error occurs, be sure to release memory }
      raise;
    end;
  end;
end; {= TRzBusinessComponent.CreateCalcField =}


end.
