//***************************************************************************
// 2D_OBJS.CPP					    Copyright 1996 by MUTTLEY
// Routines for manipulating wireframe objects in Mode X
//***************************************************************************

#include <stdlib.h>
#include <conio.h>
#include <math.h>
#include "modex32w.h"			// Header file for Mode X routines
#include "2d_objs.h"			// Header file for 2D routines

//---------------------------------------------------------------------------
void Thrust_Object(ObjectType *Object)
// Routine to find the direction of an object so that it can be thrusted
{
 // Adjusts the acceleration of the object based on it's direction 
 // This is done by adding the Sin and Cos of it's current angle
    Object->ax = Cos_Look[Object->Angle] * SPEED;     // Find x accel
    Object->ay = Sin_Look[Object->Angle] * SPEED;     // Find y accel
}

//---------------------------------------------------------------------------
void Trans_Object(ObjectType *ObjToMove)
// Routine to rotate and translate an object
{
 int i, rx, ry; 
 float newx, newy;

 // Change angle of object by adding it's rotation to it's current angle
 ObjToMove->Angle += ObjToMove->Rotate;
 if (ObjToMove->Angle>359) ObjToMove->Angle -= 360;
 if (ObjToMove->Angle<1) ObjToMove->Angle += 360;

 ObjToMove->vx += ObjToMove->ax;	    // Add x accel to x velocity
 ObjToMove->vy += ObjToMove->ay;	    // Add y accel by y velocity

 if (ObjToMove->vx > ObjToMove->mx) 	    // Check to make sure
    ObjToMove->vx = ObjToMove->mx; else	    // the velocity of the object
 if (ObjToMove->vy > ObjToMove->my)	    // does not exceed it's maximum	
    ObjToMove->vy = ObjToMove->my; else	    // speed
 if (ObjToMove->vx < -ObjToMove->mx) 	    // Also check to make sure the 
    ObjToMove->vx = -ObjToMove->mx; else    // negative velocity of the 
 if (ObjToMove->vy < -ObjToMove->my)	    // object does not exceed it's 	
    ObjToMove->vy = -ObjToMove->my;	    // negative maximum speed

 if ((ObjToMove->Type == ENEMYTYPE) && (ObjToMove->Scale == LARGE)) {
    newx = Wave_3Leaf[ObjToMove->Track_Index].x;
    newy = Wave_3Leaf[ObjToMove->Track_Index].y;
    ObjToMove->Track_Index += 1;
    if (ObjToMove->Track_Index > 399) 
        ObjToMove->Track_Index = 0;
 }
 else {
    newx = ObjToMove->x + ObjToMove->vx;     // Find new x position of object
    newy = ObjToMove->y + ObjToMove->vy;     // Find new y position of object
 }

 // If object is a line fragment and is near edge of screen make it die
 if (ObjToMove->Type == FRAGTYPE) {
    if ((newx < WIN_XMIN) || (newx > WIN_XMAX) || (newy < WIN_YMIN) || (newy > WIN_YMAX)) {
        ObjToMove->State = DYING; 
    }
 } else 

 // Else object needs to be wrapped around the display window
 if (newx < WIN_XMIN) newx = WIN_XMAX; else 
 if (newx > WIN_XMAX) newx = WIN_XMIN; else 
 if (newy < WIN_YMIN) newy = WIN_YMAX; else         
 if (newy > WIN_YMAX) newy = WIN_YMIN;    

 ObjToMove->x = newx;			    // Set the new x,y position
 ObjToMove->y = newy;			    // of the object

 // Now save the actual screen co-ords for the OLD position of each vert   
 for (i=0; i<ObjToMove->NumVerts; i++) { 
     ObjToMove->VertPtr[i].ox = ObjToMove->VertPtr[i].sx;
     ObjToMove->VertPtr[i].oy = ObjToMove->VertPtr[i].sy;
 }

 // Now calculate the new screen co-ords for the new position of each vert
 for (i=0; i<ObjToMove->NumVerts; i++) { 

     // First , rotate the local co-ords of the object about it's centre point
     ObjToMove->VertPtr[i].rx = ObjToMove->VertPtr[i].lx * Cos_Look[ObjToMove->Angle] - 
	  ObjToMove->VertPtr[i].ly * Sin_Look[ObjToMove->Angle];

     ObjToMove->VertPtr[i].ry = ObjToMove->VertPtr[i].lx * Sin_Look[ObjToMove->Angle] +
	  ObjToMove->VertPtr[i].ly * Cos_Look[ObjToMove->Angle];	 

     // Screen co-ords for each vertex = obj's new pos + obj's rotated co-ords 
     ObjToMove->VertPtr[i].sx = ObjToMove->x + ObjToMove->VertPtr[i].rx;
     ObjToMove->VertPtr[i].sy = ObjToMove->y + ObjToMove->VertPtr[i].ry;
 }
} 


//---------------------------------------------------------------------------
void Draw_Object(ObjectType *ObjToDraw, int PageNum)
// Routine to draw an object in it's new position
{
 int i, v2;
 
 // If object is dying then don't draw in this page
 if (ObjToDraw->State == DYING) return;	    

 for (i=0; i<ObjToDraw->NumVerts; i++) {    // For every vert in the object..
     v2 = i + 1;			    // Get next vert after current one
     if (v2 >= ObjToDraw->NumVerts) {       // Wrap v2 to point to the first
        v2 = 0;				    // vertex if necessary
     }
     // Now draw/clip the new line between the current vertex and the next vertex
     Clip_Line(ObjToDraw->VertPtr[i].sx, ObjToDraw->VertPtr[i].sy,
           ObjToDraw->VertPtr[v2].sx, ObjToDraw->VertPtr[v2].sy,
           ObjToDraw->Color, PageNum);
 }
}

//---------------------------------------------------------------------------
void Erase_Object(ObjectType *ObjToDraw, int PageNum)
// Routine to erase an object's old position and draw it's new position
{
 int i, v2;

 // If object is dying then kill it off and erase it one last time
 if (ObjToDraw->State == DYING) ObjToDraw->State = INACTIVE;

 for (i=0; i<ObjToDraw->NumVerts; i++) {    // For every vert in the object..
     v2 = i + 1;			    // Get next vert after current one
     if (v2 >= ObjToDraw->NumVerts) {       // Wrap v2 to point to the first
        v2 = 0;				    // vertex if necessary
     }

     // Erase/clip the old line between the current vertex and the next vertex
     Clip_Line(ObjToDraw->VertPtr[i].ox, ObjToDraw->VertPtr[i].oy,
           ObjToDraw->VertPtr[v2].ox, ObjToDraw->VertPtr[v2].oy,
           0, PageNum);
 }
}

//---------------------------------------------------------------------------
static int Encode_Point(int x, int y)
// Encodes a line segment endpoint in terms of it's 'half space' position
// and returns a four-bit code (See book for details)
{
 return ((x<WIN_XMIN)<<3) | ((x>WIN_XMAX)<<2) | ((y<WIN_YMIN)<<1) | (y>WIN_YMAX);
}


//---------------------------------------------------------------------------
void Clip_Line(int x1, int y1, int x2, int y2, int Color, int PageNum)
// Clips a line to the display window and then calls the LineX draw routine
// Uses the Cohen / Sutherland algorithm
{
 int c1 = Encode_Point(x1, y1), c2 = Encode_Point(x2, y2);
 float dx, dy;

 while (c1 | c2) {	// While either point is not inside display window...

   if (c1 & c2) return; // If both points lie outside display window - don't draw

   dx = (x2 - x1); dy = (y2 - y1);

   if (c1) {		// If point c1 lies outside display window...
      // Find intersection of point c1 with relevant 1/2 space line	
      if (c1 & 8) y1 += (WIN_XMIN - x1) * dy/dx, x1 = WIN_XMIN; else
      if (c1 & 4) y1 += (WIN_XMAX - x1) * dy/dx, x1 = WIN_XMAX; else
      if (c1 & 2) x1 += (WIN_YMIN - y1) * dx/dy, y1 = WIN_YMIN; else
      if (c1 & 1) x1 += (WIN_YMAX - y1) * dx/dy, y1 = WIN_YMAX; 
      c1 = Encode_Point(x1, y1);
   }      
   else {		// If point c2 lies outside display window...
      // Find intersection of point c2 with relevant 1/2 space line
      if (c2 & 8) y2 += (WIN_XMIN - x2) * dy/dx, x2 = WIN_XMIN; else
      if (c2 & 4) y2 += (WIN_XMAX - x2) * dy/dx, x2 = WIN_XMAX; else
      if (c2 & 2) x2 += (WIN_YMIN - y2) * dx/dy, y2 = WIN_YMIN; else
      if (c2 & 1) x2 += (WIN_YMAX - y2) * dx/dy, y2 = WIN_YMAX; 
      c2 = Encode_Point(x2, y2);
   }
 }

 LineX(x1, y1, x2, y2, Color, PageOffsets[PageNum]);
}

//---------------------------------------------------------------------------
void Check_Object(ObjectType *Object1, ObjectType *Object2)
{
// Routine to check if an object has collided with another object
// Very similar to the Check_Bullet routine except that every point of 
// one object is checked with every line segment of the other object

 int i, j, v2;				    // Used as indices into vertices
 float x1, y1, x2, y2;			    // Holds co-ords of line segment	
 int x1flag, x2flag, y1flag, y2flag;	    // Used for checking co-ords
 float c, ix;				    // Holds intercepts
 float bx, by;				    // Holds bullet's x,y position
 int Intersects;			    // Intersect counter

 for (j=0; j<Object1->NumVerts; j++) {      // For every point in object1
     Intersects = 0;			    // Reset intersects counter
     bx = Object1->VertPtr[j].sx;	    // Get next x co-ord of object1
     by = Object1->VertPtr[j].sy;	    // get next y co-ord of object1

     // For every line segment in object2...
     for (i=0; i<Object2->NumVerts; i++) {  // For every vert in object2..
         v2 = i + 1;			    // Get next vert after current one
         if (v2 >= Object2->NumVerts)       // Wrap v2 to point to the first
            v2 = 0;			    // vertex if necessary
     
         // Get co-ords of current line segment
         x1 = Object2->VertPtr[i].sx;
         y1 = Object2->VertPtr[i].sy;
         x2 = Object2->VertPtr[v2].sx;
         y2 = Object2->VertPtr[v2].sy;
	
         // Get y co-ords of the line segment and set their flags accordingly
         y1flag = (y1 >= by);  		   // Set flag of y1
         y2flag = (y2 >= by); 		   // Set flag of y2

         // If y co-ords of line fall either side of bullet's y position then..
         if (y1flag != y2flag) {		    	 
      
            // Get x co-ords of line segment and set their flags accordingly
            x1flag = (x1 >= bx); 	   // Set flag of x1
            x2flag = (x2 >= bx);	   // Set flag of x2
       
            // If x co-ords of line BOTH fall on right side of bullets x position..
            if (x1flag == x2flag) Intersects +=1;
            else {				// Else x co-ords fall either side	
               c = y1 - (y2-y1)/(x2-x1) * x1;   // Find c of line, as in y=mx + c
	       // Find where line cuts a line drawn parallel with x-axis along by
	       ix = (by - c) / (y2-y1)/(x2-x1); 
               // If line cuts at x positon > bullet's x positon then...
               if (ix >= bx) Intersects +=1;    
            }	
         }     	
     }

     if ((Intersects % 2) > 0) {	   // If odd number of intersects...
        Object1->State = DYING;		   // then kill objects
        Object2->State = DYING;		   
        Explode_Object(Object1);	   // Explode objects into fragments
	Explode_Object(Object2);
	break;
     }
 }					   // Repeat for every object1 point
}

//---------------------------------------------------------------------------
void Fire_Thrust(ObjectType *Object, ParticleType *Thrust)
// Routine to initiate a thrust particle
{
 // Angle of thrust will be opposite direction of object (+-20 degrees)
 int ThrustAngle = Object->Angle + 160 + rand()%40;
 if (ThrustAngle>360) ThrustAngle -= 360;
 if (ThrustAngle<0) ThrustAngle += 360;
 
 Thrust->Color = rand()%256; 

 // Get the starting position and direction of the thrust 
 Thrust->x = Object->x + 4 * Cos_Look[ThrustAngle];
 Thrust->y = Object->y + 4 * Sin_Look[ThrustAngle];
 
 // Make sure particle falls within display window
 if (Thrust->x < WIN_XMIN) Thrust->x = WIN_XMIN; else
 if (Thrust->x > WIN_XMAX) Thrust->x = WIN_XMAX; else
 if (Thrust->y < WIN_YMIN) Thrust->y = WIN_YMIN; else
 if (Thrust->y > WIN_YMAX) Thrust->y = WIN_YMAX; else

 // Get the speed of the thrust (= opposite speed of object)
 Thrust->vx = 2 * Cos_Look[ThrustAngle];
 Thrust->vy = 2 * Sin_Look[ThrustAngle];
 
 // This is a brand new active thrust particle so...
 Thrust->State = ACTIVE;
}

//---------------------------------------------------------------------------
void Trans_Particle(ParticleType *Part)
// Routine to move a particle
{
 float newx, newy;
 
 newx = Part->x + Part->vx;		        // Set the new x,y position
 newy = Part->y + Part->vy;		        // of the particle

 // If a particle hits the edge of the screen then kill it
 if (newx < WIN_XMIN) Part->State = DYING; else                    
 if (newx > WIN_XMAX) Part->State = DYING; else               	    
 if (newy < WIN_YMIN) Part->State = DYING; else          
 if (newy > WIN_YMAX) Part->State = DYING; 

 Part->ox = Part->x; Part->oy = Part->y; 	// Save old particle position
 Part->x = newx; Part->y = newy;		// Set new particle position
}

//---------------------------------------------------------------------------
void Draw_Particle(ParticleType *Part, int PageNum)
// Routine draw a particle in it's new position
{
 // If particle is dying then don't draw in this page
 if (Part->State == DYING) return;	

 WrPixelX(Part->x, Part->y, Part->Color, PageOffsets[PageNum]);
}

//---------------------------------------------------------------------------
void Erase_Particle(ParticleType *Part, int PageNum)
// Routine to erase a particle in it's old position 
{
 // If particle is dying then kill it and erase it one last time 
 if (Part->State == DYING) {
     Part->State = INACTIVE;
 }
 WrPixelX(Part->ox, Part->oy, 0, PageOffsets[PageNum]);
}

//---------------------------------------------------------------------------
void Fire_Bullet(ObjectType *Object, ParticleType *Bullet)
// Routine to initiate a fired bullet
{
 // Checking for Angle here too will resolve the 'stray' bullet problem
 if (Object->Angle>360) Object->Angle -= 360;
 if (Object->Angle<0) Object->Angle += 360;
 
 Bullet->Color = Object->Color; 

 // Get the direction of the bullet
 Bullet->x = Object->x + 7 * Cos_Look[Object->Angle];
 Bullet->y = Object->y + 7 * Sin_Look[Object->Angle];

 // Make sure particle falls within display window
 if (Bullet->x < WIN_XMIN) Bullet->x = WIN_XMIN; else
 if (Bullet->x > WIN_XMAX) Bullet->x = WIN_XMAX; else
 if (Bullet->y < WIN_YMIN) Bullet->y = WIN_YMIN; else
 if (Bullet->y > WIN_YMAX) Bullet->y = WIN_YMAX; else

 // Get the speed of the bullet
 Bullet->vx = Object->vx + 4 * Cos_Look[Object->Angle];
 Bullet->vy = Object->vy + 4 * Sin_Look[Object->Angle];
 
 // This is a brand new active bullet so...
 Bullet->State = ACTIVE;
}


//---------------------------------------------------------------------------
void Check_Bullet(ParticleType *Bullet, ObjectType *Object)
// Routine to check if a bullet has collided with an enemy object
// For each line segment of object, check it's y co-ords to see if they fall 
// either side of the bullet's y co-ord. If they do then check to see if the 
// x co-ords of the line segment BOTH fall to the right of the bullet's 
// x co-ord - if they do then a definate intersect occurs.  
// If line segment's x co-ords fall either side of the point's x co-ord then
// find where the line segment intersects an imaginary x axis line passing 
// through the bullett. If the line segment intersects at an x positon > than
// the bullet's x co-ord then again a definate intersect exists  
// If none of these conditions are met then move on to next line segment
// If an odd number of intersects occured then bullet is inside the object
{
 int i, v2;				    // Used as indices into vertices
 float x1, y1, x2, y2;			    // Holds co-ords of line segment	
 int x1flag, x2flag, y1flag, y2flag;	    // Used for checking co-ords
 float c, ix;				    // Holds intercepts
 float bx, by;				    // Holds bullet's x,y position
 int Intersects = 0;			    // Intersect counter

 bx = Bullet->x;
 by = Bullet->y;

 // For every line segment in the object...
 for (i=0; i<Object->NumVerts; i++) {       // For every vert in the object..
     v2 = i + 1;			    // Get next vert after current one
     if (v2 >= Object->NumVerts) {          // Wrap v2 to point to the first
        v2 = 0;				    // vertex if necessary
     }
     
     // Get co-ords of current line segment
     x1 = Object->VertPtr[i].sx;
     y1 = Object->VertPtr[i].sy;
     x2 = Object->VertPtr[v2].sx;
     y2 = Object->VertPtr[v2].sy;
	
     // Get y co-ords of the line segment and set their flags accordingly
     y1flag = (y1 >= by);  		    // Set flag of y1
     y2flag = (y2 >= by); 		    // Set flag of y2

     // If y co-ords of line fall either side of bullet's y position then..
    if (y1flag != y2flag) {		    	 
      
       // Get x co-ords of line segment and set their flags accordingly
       x1flag = (x1 >= bx); 		    // Set flag of x1
       x2flag = (x2 >= bx);		    // Set flag of x2
       
       // If x co-ords of line BOTH fall on right side of bullets x position..
       if (x1flag == x2flag) Intersects +=1;
       else {				   // Else x co-ords fall either side	
          c = y1 - (y2-y1)/(x2-x1) * x1;   // Find c of line, as in y=mx + c
	  // Find where line cuts a line drawn parallel with x-axis along by
	  ix = (by - c) / (y2-y1)/(x2-x1); 
          // If line cuts at x positon > bullet's x positon then...
          if (ix >= bx) Intersects +=1;    
       }	
     }     	
 }

 if ((Intersects % 2) > 0) {		   // If odd number of intersects...
    Object->State = DYING;		   // then kill object
    Bullet->State = DYING;		   // and kill bullet and...
    if (Object->Scale != SMALL) 
       Spawn_Object(Object); else	   // If object is big split it up - 
       Explode_Object(Object);		   // else blow it up
 }
} 	
