//***************************************************************************
// PCXGRAF.CPP - PCX graphics and sprite routines
//***************************************************************************

#include <io.h>
#include <conio.h>
#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <bios.h>
#include <fcntl.h>
#include <memory.h>
#include <malloc.h>
#include <math.h>
#include <iostream.h>

#include "MODEX.H"
#include "VGA.H"
#include "PCXGRAF.H"

int Screen_Width, Screen_Height;

//---------------------------------------------------------------------------
int PCX_Init(PCXPicturePtr image)
{
 // Allocates the buffer that the PCX image data will be loaded into
 if (!(image->buffer = (unsigned char far *)farmalloc(Screen_Width * Screen_Height + 1)))
    {
    cout << "PCX SYSTEM - Couldn't allocate PCX image buffer";
    return(0);
    }
 return(1);                             // Success
}

//---------------------------------------------------------------------------
int PCX_Load(char *filename, PCXPicturePtr image, int load_palette)
{
 // Loads a PCX file into the image structure. First, the PCX header is
 // loaded, then the image data is decompressed and loaded, then the palette
 // data is used to update the VGA palette if we want the palette

 FILE *fp;                             // File pointer used to open PCX file
 int num_bytes;                        // Number of bytes in current RLE run
 int index;                            // Loop variable
 long bytes_processed;                 // Total number of bytes decompressed
 unsigned char data;                   // Holds current pixel data
 char far *temp_buffer;                // Working buffer

 if ((fp = fopen(filename,"rb"))==NULL) {
    cout << "PCX SYSTEM - Couldn't open file " << filename;
    return(0);
 }

 temp_buffer = (char far *)image;

 for (index=0; index<128; index++) {
      temp_buffer[index] = (char)getc(fp);  // Get the 128 byte PCX header
 }

 // Load and decompress data into 64,000 byte buffer
 bytes_processed=0;

 while(bytes_processed <= Screen_Width * Screen_Height) {
  	   data = (unsigned char)getc(fp);         // Get first piece of data

       if (data>=192 && data<=255) {           // Is this an RLE run of bytes?
           num_bytes = data-192;               // Compute size of RLE run
           data  = (unsigned char)getc(fp);    // Get actual data for the run

  		    while(num_bytes-->0) {                         // Put data in buffer
                image->buffer[bytes_processed++] = data; // num_bytes times
          }
       }
       else {
         // Else not an RLE run so just copy actual data into buffer
         image->buffer[bytes_processed++] = data;
	 	   }
 }

 // Load color palette by moving to end of file then back up 768 bytes to the
 // start of palette
 fseek(fp, -768L, SEEK_END);

 // Load the pallete into the VGA color registers
 for (index=0; index<256; index++) {
      // Get red component
      image->palette[index].red = (unsigned char)(getc(fp) >> 2);
      // Get green component
      image->palette[index].green = (unsigned char)(getc(fp) >> 2);
      // Get blue component
      image->palette[index].blue = (unsigned char)(getc(fp) >> 2);
 }
 fclose(fp);
 // change the palette to newly loaded palette if commanded to do so

 if (load_palette) {                   // Change palette to loaded one?
     // For each palette register, set to the new color values
     for (index=0; index<256; index++) {
          Write_Color_Reg(index,(RGBColorPtr)&image->palette[index]);
     }
 }
 return(1);                            // Success
}

//---------------------------------------------------------------------------
void PCX_Delete(PCXPicturePtr image)
{
 // De-allocates the buffer region used for the PCX file load
 farfree(image->buffer);
}

//---------------------------------------------------------------------------
void PCX_Show_Buffer(PCXPicturePtr image)
{
 // Copies the PCX buffer into the video buffer
 char far *data;                       // Temp variable used for aliasing
 data = image->buffer;                 // Alias the umage buffer
 _asm {
  	   push ds                         // Save the data segment
  	   les di, video_buffer            // Point es:di to video buffer
  	   lds si, data                    // Point ds:si to data area
  	   mov cx,320*200/2                // Move 32000 words
	     cld                             // Set direction to foward
	     rep movsw                       // Do the string operation
	     pop ds                          // Restore the data segment
	}
}

/*/---------------------------------------------------------------------------
void PCX_Copy_To_Buffer(PCXPicturePtr image,unsigned char far *buffer)
{
 // Copies data in the PCX buffer to another buffer (usually double buffer)
 fwordcpy((void far *)buffer,(void far *)image->buffer,double_buffer_size);

}*/

//---------------------------------------------------------------------------
void PCX_Get_Sprite(PCXPicturePtr image, SpritePtr sprite,
                    int sprite_frame, int loc_x, int loc_y)
{
 // Load the images for a sprite into the sprite frame array. Uses size of
 // sprite and the position of requested cell to compute the location in the
 // PCX image buffer to extract the data from.

 int x_off, y_off;            // Position of sprite cell in PCX image buffer
 int y;                       // Looping variable
 int width, height;           // Size of sprite

 unsigned char far *sprite_data;

 width  = sprite->width;      // Extract dimensions of sprite
 height = sprite->height;

 // First allocate the memory for the sprite in the sprite structure
 sprite->frames[sprite_frame] = (unsigned char far *)farmalloc(width * height + 1);

 // Create an alias to the sprite frame for ease of access
 sprite_data = sprite->frames[sprite_frame];

 // Now load the sprite data into the sprite frame array from the PCX picture
 x_off = (width+1)  * loc_x + 1;
 y_off = (height+1) * loc_y + 1;

 y_off = y_off * 320;                   // Compute starting y address

 for (y=0; y<height; y++, y_off+=320) { // Scan the data row by row
     // Copy a line of data
     _fmemcpy((void far *)&sprite_data[y*width],
             (void far *)&(image->buffer[y_off + x_off]), width);
 }
 sprite->num_frames++;                  // Increment number of sprite frames
}

//---------------------------------------------------------------------------
void Sprite_Init(SpritePtr sprite,int x,int y,int dx, int dy,
                 int width,int height,int c1,int c2,int c3,
                 int t1,int t2,int t3)
{
 int index;
 sprite->x = x;
 sprite->y = y;
 sprite->dx = dx;
 sprite->dy = dy;
 sprite->width = width;
 sprite->height = height;
 sprite->visible = 1;
 sprite->cntr1 = c1;
 sprite->cntr2 = c2;
 sprite->cntr3 = c3;
 sprite->thr1 = t1;
 sprite->thr2 = t2;
 sprite->thr3 = t3;
 sprite->curr_frame = 0;
 sprite->state = SPRITE_DEAD;
 sprite->num_frames = 0;
 sprite->background = (unsigned char far *)farmalloc(width * height+1);

 // Set all bitmap pointers to null
 for (index=0; index<MAX_SPR_FRAMES; index++)
     sprite->frames[index] = NULL;
}

//---------------------------------------------------------------------------
void Sprite_Delete(SpritePtr sprite)
{
 int index;
 farfree(sprite->background);
 // Now de-allocate all the animation frames
 for (index=0; index<MAX_SPR_FRAMES; index++)
     farfree(sprite->frames[index]);
}

//---------------------------------------------------------------------------
void Sprite_Under(SpritePtr sprite, unsigned char far *buffer)
{
 // Scans the background under a sprite so that when the sprite
 // is drawn the background isn't obliterated

 unsigned char far *back_buffer;       // Background buffer for sprite
 int y;                                // Current line being scanned
 int width, height;                    // Dimensions of sprite

 // Alias a pointer to sprite background for ease of access
 back_buffer = sprite->background;

 // Alias width and height
 width  = sprite->width;
 height = sprite->height;

 // Compute offset of background in source buffer
 buffer = buffer + (sprite->y << 8) + (sprite->y << 6) + sprite->x;

 for (y=0; y<height; y++) {
     // Copy next row out of image buffer into sprite background buffer
     _fmemcpy((void far *)back_buffer, (void far *)buffer, width);

    // Move to next line in source buffer and in sprite background buffer
    buffer      += Screen_Width;
    back_buffer += width;
 }
}

//---------------------------------------------------------------------------
void Sprite_Erase(SpritePtr sprite,unsigned char far *buffer)
{
 // Replace the background that was behind the sprite
 unsigned char far *back_buffer;       // Background buffer for sprite
 int y;                                // Current line being scanned
 int width, height;                    // Dimensions of sprite

 // Alias a pointer to sprite background for ease of access
 back_buffer = sprite->background;

 // Alias width and height
 width  = sprite->width;
 height = sprite->height;

 // Compute offset in destination buffer
 buffer = buffer + (sprite->y << 8) + (sprite->y << 6) + sprite->x;

 for (y=0; y<height; y++) {
     // Copy next line from sprite background buffer to destination buffer
     _fmemcpy((void far *)buffer, (void far *)back_buffer, width);

	  // Move to next line in destination buffer and sprite background buffer
    buffer      += Screen_Width;
    back_buffer += width;
 }
}

//---------------------------------------------------------------------------
void Sprite_Erase_Quick(SpritePtr sprite,unsigned char far *buffer)
{
 // Quick erase that replaces the sprite with color 0
 int y;                                // Current line being scanned

 // Alias width and height
 int width  = sprite->width;
 int height = sprite->height;

 // Compute offset in destination buffer
 buffer = buffer + (sprite->y << 8) + (sprite->y << 6) + sprite->x;

 for (y=0; y<height; y++) {
     // Set next line of sprite to color 0
     _fmemset((void far *)buffer, 0, width);

	  // Move to next line in buffer
    buffer      += Screen_Width;
 }
}

//---------------------------------------------------------------------------
void Sprite_Draw(SpritePtr sprite, int transparent)
{
 // Draws a sprite on the screen row by row. If transparent flag is true then
 // pixels will be drawn 1 by 1 else a memcpy will be used to draw each line

 unsigned char far *sprite_data;       // Pointer to sprite data
 int width, height;
 int xpos, ypos;


 // Alias a pointer to sprite for ease of access
 sprite_data = sprite->frames[sprite->curr_frame];

 width = sprite->width;
 height = sprite->height;
 xpos = sprite->x;
 ypos = sprite->y;

 Tdraw_Bitmap(sprite_data, xpos, ypos, width, height);

}

/*/---------------------------------------------------------------------------
void Sprite_Under_Clip(SpritePtr sprite, unsigned char far *buffer)
{
 // Scans the background under a sprite, but only portions that are visible

 unsigned char far *back_buffer;       // Pointer to sprite background buffer
 unsigned char far *source_buffer;     // Pointer to source buffer

 int x,y;                              // Looping variables
 int sx,sy;                            // Position of sprite
 int width;                            // Width of sprite
 int bitmap_width = 0;                 // Width and height of sub-bitmap
 int bitmap_height = 0;

 unsigned char pixel;                  // Holds current pixel being processed

 // Alias a variable to sprite size
 width = sprite->width;
 bitmap_width = width;
 bitmap_height = sprite->height;
 sx = sprite->x;
 sy = sprite->y;

 // Perform trivial rejection tests
 if (sx >= (int)Screen_Width || sy >= (int)double_buffer_height ||
   (sx+width) <= 0 || (sy+bitmap_height) <= 0) {
   // Sprite is totally invisible so don't scan

   // Set invisible flag in structure so that draw sub-function does nothing
   sprite->visible = 0;
   return;
 }

 // Sprite background region must be clipped before scanning
 // so therefore compute visible portion...

 // First compute upper left hand corner of clipped sprite background
 if (sx<0) {
     bitmap_width += sx;
	   sx = 0;
 }
 else
 if (sx+width>=(int)Screen_Width) {
     bitmap_width  = (int)Screen_Width-sx;
 }
 // Now process y
 if (sy<0) {
     bitmap_height += sy;
     sy = 0;
 }
 else
 if (sy+bitmap_height>=(int)double_buffer_height) {
     bitmap_height  = (int)double_buffer_height - sy;
 }

 // At this point we know were to start scanning the bitmap i.e. sx, sy
 // and we know the size of the bitmap to be scanned i.e. width, height so...

 // Compute number of bytes between adjacent video lines after a row
 // of pixels has been drawn

 // Compute offset of sprite background in source buffer
 source_buffer = buffer + (sy << 8) + (sy << 6) + sx;

 // Alias a pointer to sprite background
 back_buffer = sprite->background;

 for (y=0; y<bitmap_height; y++) {
      // Copy the next row into the destination buffer
      _fmemcpy((void far *)back_buffer,(void far *)source_buffer,bitmap_width);

      // Move to next line in desintation buffer and sprite image buffer
      source_buffer += Screen_Width;
      back_buffer += width;  // This is the actual width of whole bitmap
 }

 // Set variables in structure so that erase sub-function can operate faster
 sprite->x_clip = sx;
 sprite->y_clip = sy;
 sprite->width_clip = bitmap_width;
 sprite->height_clip = bitmap_height;
 sprite->visible = 1;
}

//---------------------------------------------------------------------------
void Sprite_Erase_Clip(SpritePtr sprite,unsigned char far *buffer)
{
 // Replace the background that was behind the sprite

 unsigned char far *back_buffer;       // Background buffer for sprite
 int y;                                // Current line being scanned
 int width;                            // Size of sprite background buffer
 int bitmap_height;                    // Size of clipped bitmap
 int bitmap_width;

 if (!sprite->visible)                 // Make sure sprite was visible
    return;

 // Alias a pointer to sprite background for ease of access
 back_buffer = sprite->background;

 // Alias width and height
 bitmap_width  = sprite->width_clip;
 bitmap_height = sprite->height_clip;
 width = sprite->width;

 // Compute offset in destination buffer
 buffer = buffer + (sprite->y_clip << 8) + (sprite->y_clip << 6) + sprite->x_clip;

 for (y=0; y<bitmap_height; y++) {
      // Copy next row from sprite background buffer to destination buffer
     _fmemcpy((void far *)buffer, (void far *)back_buffer, bitmap_width);

 	   // Move to next line in destination buffer and in sprite background buffer
     buffer += Screen_Width;
     back_buffer += width;
 }
}

//---------------------------------------------------------------------------
void Sprite_Draw_Clip(SpritePtr sprite, unsigned char far *buffer,int transparent)
{
 // Draws a sprite on the screen row by row. If the transparent flag is true
 // then pixels will be draen 1 by 1 else a memcpy will be used to draw each
 // line. Function will test if the sprite is totally visible/invisible and
 // will only draw the portions that are visible

 unsigned char far *sprite_data;       // Pointer to sprite data
 unsigned char far *dest_buffer;       // Pointer to destination buffer

 int x,y;                              // Looping variables
 int sx,sy;                            // Position of sprite
 int width;                            // Width of sprite
 int bitmap_x = 0;                     // Upper-left corner of sub-bitmap
 int bitmap_y = 0;                     // to be drawn after clipping
 int bitmap_width = 0;                 // Dimensions of sub-bitmap
 int bitmap_height = 0;

 unsigned char pixel;                  // Current pixel being processed

 // Alias a variable to sprite size
 width = sprite->width;
 bitmap_width = width;
 bitmap_height = sprite->height;
 sx = sprite->x;
 sy = sprite->y;

 // Perform trivial rejection tests
 if (sx >= (int)Screen_Width || sy >= (int)double_buffer_height ||
    (sx+width) <= 0  || (sy+bitmap_height) <= 0 || !sprite->visible) {
    // Sprite is totally invisible so don't draw
    // Set invisible flag in structure so that the erase function does nothing

    sprite->visible = 0;
    return;
 }

 // The sprite needs some clipping or no clipping at all
 // so compute visible portion of sprite rectangle

 // First compute upper left hand corner of clipped sprite

 if (sx<0) {
     bitmap_x = -sx;
     sx = 0;
     bitmap_width -= bitmap_x;
 }
 else
 if (sx+width>=(int)Screen_Width) {
     bitmap_x = 0;
     bitmap_width  = (int)Screen_Width-sx;
 }

 // Now process y
 if (sy<0) {
     bitmap_y = -sy;
     sy = 0;
	   bitmap_height -= bitmap_y;
 }
 else
 if (sy+bitmap_height>=(int)double_buffer_height) {
     bitmap_y = 0;
     bitmap_height  = (int)double_buffer_height - sy;
 }
 // This point we know were to start drawing the bitmap i.e. sx,sy
 // and we know were in data to extract the bitmap i.e. bitmap_x, bitmap_y
 // and finally we know size of the bitmap to be drawn i.e. width,height so..

 // Compute number of bytes between adjacent video lines after a
 // row of pixes has been drawn

 // Compute offset of sprite in destination buffer
 dest_buffer = buffer + (sy << 8) + (sy << 6) + sx;

 // Alias a pointer to sprite for ease of access and locate starting sub
 // bitmap that will be drawn
 sprite_data = sprite->frames[sprite->curr_frame] + (bitmap_y*width) + bitmap_x;

 // Copy each line of the sprite data into destination buffer
 if (transparent) {
     for (y=0; y<bitmap_height; y++) {
          // Copy the next row into the destination buffer
          for (x=0; x<bitmap_width; x++) {
              // Test for transparent pixel i.e. 0, if not transparent then draw
              if ((pixel=sprite_data[x]))
                   dest_buffer[x] = pixel;
          }
          // Move to next line in desintation buffer and sprite image buffer
          dest_buffer += Screen_Width;
          sprite_data += width;  // This width is the actual width of the
                                 // entire bitmap NOT the visible portion
     }
 }
 else
   {
   // Draw sprite with transparency off
 	 for (y=0; y<bitmap_height; y++) {
        // Copy the next row into the destination buffer
        _fmemcpy((void far *)dest_buffer,(void far *)sprite_data,bitmap_width);

       // move to next line in desintation buffer and sprite image buffer
       dest_buffer += Screen_Width;
       sprite_data += width;  // Note this width is the actual width of the
                              // entire bitmap NOT the visible portion
   }

 }

 // Set variables in structure so that erase sub-function can operate faster
 sprite->x_clip = sx;
 sprite->y_clip = sy;
 sprite->width_clip = bitmap_width;
 sprite->height_clip = bitmap_height;
 sprite->visible = 1;
}


//---------------------------------------------------------------------------
int Sprites_Collide(SpritePtr sprite_1, SpritePtr sprite_2)
{
 // Tests if two sprites have intersected by testing their bounding boxes
 return(0);
}


//---------------------------------------------------------------------------
void fwordcpy(void far *destination, void far *source,int num_words)
{
 // This function is similar to fmemcpy except that is moves data in words
 // it is about 25% faster than memcpy which uses bytes

 _asm {
	     push ds               // Need to save segment registers i.e. ds
	     les di,destination    // Point es:di to destination of memory move
	     lds si,source         // Point ds:si to source of memory move
	     mov cx,num_words      // Move into cx the number of words to be moved
	     rep movsw             // Let the processor do the memory move
	     pop ds                // Restore the ds segment register
 }
}

//---------------------------------------------------------------------------
void Bitmap_Put(BitmapPtr image, unsigned char far *destination,int transparent)
{
 // Draws a bitmap on the destination buffer which could
 // be a double buffer or the video buffer

 int x,y;                            // Looping variables
 int width,height;                   // Size of bitmap

 unsigned char far *bitmap_data;     // Pointer to bitmap buffer
 unsigned char far *dest_buffer;     // Pointer to destination buffer

 unsigned char pixel;                // Current pixel value being processed

 // Compute offset of bitmap in destination buffer
 // Note that all video/double buffers must be 320 bytes wide
 dest_buffer = destination + (image->y << 8) + (image->y << 6) + image->x;

 // Create aliases to variables
 height      = image->height;
 width       = image->width;
 bitmap_data = image->buffer;

 // Test if transparency is on or off
 if (transparent) {
    // Use version that will draw a transparent bitmap(slighlty slower)
    // Draw each line of the bitmap

    for (y=0; y<height; y++) {
        // Copy the next row into the destination buffer
        for (x=0; x<width; x++) {
			      // Test for transparent pixel i.e. 0, if not transparent then draw
            if ((pixel=bitmap_data[x]))
                 dest_buffer[x] = pixel;
        }
        // Move to next line in double buffer and in bitmap buffer
        dest_buffer += Screen_Width;
        bitmap_data += width;
    }
 }
 else	{
   // Draw each line of the bitmap, note how each pixel doesn't need
   // to be tested for transparency hence a memcpy can be used
   for (y=0; y<height; y++) {
       // copy the next row into the destination buffer using memcpy for speed
       _fmemcpy((void far *)dest_buffer, (void far *)bitmap_data,width);

       // Move to next line in destination buffer and in bitmap buffer
       dest_buffer += Screen_Width;
       bitmap_data += width;
	 }
 }
}

//---------------------------------------------------------------------------
void Bitmap_Get(BitmapPtr image, unsigned char far *source)
{
 // Scans a bitmap from the source buffer

 unsigned int source_off;         // Offset into source buffer
 unsigned int bitmap_off;         // Offset into destination buffer

 int y;                           // Looping variable
 int width,height;                // Size of bitmap

 unsigned char far *bitmap_data;  // pointer to bitmap buffer

 // Compute offset of bitmap in source buffer
 // Note that all buffers must be 320 bytes wide!

 source_off   = (image->y << 8) + (image->y << 6) + image->x;
 bitmap_off = 0;

 // Create aliases to variables
 height = image->height;
 width = image->width;
 bitmap_data = image->buffer;

 // Draw each line of the bitmap, note how each pixel doesn't need to be
 // tested for transparency hence a memcpy can be used

 for (y=0; y<height; y++) {
     // Copy the next row into the bitmap buffer using memcpy for speed
     _fmemcpy((void far *)&bitmap_data[bitmap_off],
             (void far *)&source[source_off],width);

	   // Move to next line in source buffer and in bitmap buffer
     source_off += Screen_Width;
     bitmap_off += width;
 }
}

//---------------------------------------------------------------------------
int Bitmap_Allocate(BitmapPtr image, int width, int height)
{
 // Allocates the memory needed for a bitmap
 if ((image->buffer = (unsigned char far *)farmalloc(width*height+1))==NULL)
	  return(0);
 else
    return(1);
}

//---------------------------------------------------------------------------
void Bitmap_Delete(BitmapPtr the_bitmap)
{
 // Deletes the memory used by a bitmap
 if (the_bitmap->buffer)
    farfree(the_bitmap->buffer);
}

*/