//GTILE.C
//Copyright (c) 1996 Kevin Hoffman
//All Rights Reserved

//Feel free to use this code in your own applications and programs :)
//If you're nice you can give me some credit :)

//This file contains some C api functions to draw a tiled world :)
//Enjoy! It should compile on the Microsoft,Turbo C++, and Watcom C++
//compilers. :)

#include <stdlib.h>
#include <memory.h>
#include "mydefs.h" //Include some standard defs :)
#include "gtile.h"  //Include some more stuff :)

#ifdef __DOS4GW__
  KDATA char KFAR *vidmemory = (char KFAR *)0xA0000;
#else
  KDATA char KFAR *vidmemory = (char KFAR *)0xA0000000;
#endif

//Initilizes a tile engine structure...
//You must free memory for the map yourself! We don't copy the map...
//We copy the pointer to the map.
//Returns 0 on error or 1 on success!
KCALL int TileEngine_Init(TileMapInfo &cTileEI,   //Tile Engine Info Struct
                     int iMapWidth,          //Map Width
                     int iMapHeight,         //Map Height
                     char KFAR *ptrTheMap,   //Pointer to map data
                     int iNumTiles,          //Number of bitmap tiles
                     int iTileWidth,         //Width of tile
                     int iTileHeight,        //Height of tile
                     char KFAR *aTiles[])    //Array of (char *) that point to the data for the different tiles!
{
    //Try to find the correct shift values for the tile width!
    int iFoundShift=0;
    for (int iTryShift=1; iTryShift<16; iTryShift++)
    {
        if ((1 << iTryShift) == iTileWidth)
        {
            //It matches!
            iFoundShift=1;
            cTileEI.tmi_iShiftX=iTryShift;
            break;
        }
    }
    cTileEI.tmi_iTileWidth=iTileWidth;
    //If we didn't find the correct shift value it's not a power of 2!
    if (iFoundShift == 0) return 0;

    //Try to find the correct shift values for the tile height!
    iFoundShift=0;
    for (iTryShift=1; iTryShift<16; iTryShift++)
    {
        if ((1 << iTryShift) == iTileHeight)
        {
            //It matches!
            iFoundShift=1;
            cTileEI.tmi_iShiftY=iTryShift;
            break;
        }
    }
    cTileEI.tmi_iTileHeight=iTileHeight;
    //If we didn't find the correct shift value it's not a power of 2!
    if (iFoundShift == 0) return 0;

    //Set the data...
    cTileEI.tmi_iMapWidth =iMapWidth;
    cTileEI.tmi_iMapHeight=iMapHeight;
    cTileEI.tmi_ptrTheMap =ptrTheMap;
    cTileEI.tmi_iNumTiles =iNumTiles;
    for (int i=0; i<iNumTiles; i++)
    {
        cTileEI.tmi_aTiles[i] = aTiles[i];
    }
    //We are initilized!
    cTileEI.tmi_bInit     =1;
    //Success!
    return 1;
}

//Uninitilizes a tile engine structure.
KCALL void TileEngine_UnInit(TileMapInfo &cTileEI)   //Tile Engine Info Struct
{
    //Reset all members :)
    cTileEI.tmi_iMapWidth   =0;
    cTileEI.tmi_iMapHeight  =0;
    cTileEI.tmi_ptrTheMap   =NULL;
    cTileEI.tmi_iTileWidth  =0;
    cTileEI.tmi_iTileHeight =0;
    cTileEI.tmi_iShiftX     =0;
    cTileEI.tmi_iShiftY     =0;
    for (int i=0; i<cTileEI.tmi_iNumTiles; i++)
    {
        cTileEI.tmi_aTiles[i] = NULL;
    }
    cTileEI.tmi_iNumTiles   =0;
    //We aren't initilialized!
    cTileEI.tmi_bInit       =0;
}

//Sets the drawing surface for the tile engine structure.
//Returns 0 on error or 1 on success!
KCALL int TileEngine_SetDrawSurface(TileMapInfo &cTileEI,     //Tile Engine Info Struct
                               char KFAR *ptrDraw,      //Pointer to drawing surface
                               int  iBytesPerScan,      //Number of bytes per scanline
                               int  iViewX,             //Start of view
                               int  iViewY,             //Start of view
                               int  iViewWidth,         //The viewport clip width
                               int  iViewHeight)        //The viewport clip height
{
    //Make sure we are initialized!
    if (cTileEI.tmi_bInit != 1) return 0;
    //Set info
    cTileEI.tmi_ptrDrawSurface   =ptrDraw;
    cTileEI.tmi_iBytesPerScanline=iBytesPerScan;
    cTileEI.tmi_iClipX           =iViewX;
    cTileEI.tmi_iClipY           =iViewY;
    cTileEI.tmi_iClipWidth       =iViewWidth;
    cTileEI.tmi_iClipHeight      =iViewHeight;
    //Set other info
    cTileEI.tmi_iMaxX            =((long)(cTileEI.tmi_iMapWidth)  << (long)(cTileEI.tmi_iShiftX) )-(long)(iViewWidth);
    cTileEI.tmi_iMaxY            =((long)(cTileEI.tmi_iMapHeight) << (long)(cTileEI.tmi_iShiftY) )-(long)(iViewHeight);
    cTileEI.tmi_iTileDrawX       =(iViewWidth / cTileEI.tmi_iTileWidth) + 2;
    cTileEI.tmi_iTileDrawY       =(iViewHeight/ cTileEI.tmi_iTileHeight) + 2;
    //Success!
    cTileEI.tmi_bSurfaceInit=1;
    return 1;
}

//Sets the drawing surface for the tile engine structure.
//Instead of width and height its coords...
//Returns 0 on error or 1 on success!
KCALL int TileEngine_SetDrawSurface2(TileMapInfo &cTileEI,     //Tile Engine Info Struct
                               char KFAR *ptrDraw,      //Pointer to drawing surface
                               int  iBytesPerScan,      //Number of bytes per scanline
                               int  iViewX,             //Start of view
                               int  iViewY,             //Start of view
                               int  iViewX2,            //End of view
                               int  iViewY2)            //End of view
{
    //Make sure we are initialized!
    if (cTileEI.tmi_bInit != 1) return 0;
    int iViewWidth = iViewX2-iViewX;
    int iViewHeight= iViewY2-iViewY;
    //Set info
    cTileEI.tmi_ptrDrawSurface   =ptrDraw;
    cTileEI.tmi_iBytesPerScanline=iBytesPerScan;
    cTileEI.tmi_iClipX           =iViewX;
    cTileEI.tmi_iClipY           =iViewY;
    cTileEI.tmi_iClipWidth       =iViewWidth;
    cTileEI.tmi_iClipHeight      =iViewHeight;
    //Set other info
    cTileEI.tmi_iMaxX            =((long)(cTileEI.tmi_iMapWidth)  << (long)(cTileEI.tmi_iShiftX) )-(long)(iViewWidth);
    cTileEI.tmi_iMaxX           -=cTileEI.tmi_iTileWidth;
    cTileEI.tmi_iMaxY            =((long)(cTileEI.tmi_iMapHeight) << (long)(cTileEI.tmi_iShiftY) )-(long)(iViewHeight);
    cTileEI.tmi_iMaxY           -=cTileEI.tmi_iTileHeight;
    cTileEI.tmi_iTileDrawX       =(iViewWidth / cTileEI.tmi_iTileWidth) + 2;
    cTileEI.tmi_iTileDrawY       =(iViewHeight/ cTileEI.tmi_iTileHeight) + 2;
    //Success!
    cTileEI.tmi_bSurfaceInit=1;
    return 1;
}


//Clears Drawing surface information!
KCALL void TileEngine_UnSetDrawSurface(TileMapInfo &cTileEI)  //Tile Engine Info Struct
{
    //Set info
    cTileEI.tmi_ptrDrawSurface   =NULL;
    cTileEI.tmi_iBytesPerScanline=0;
    cTileEI.tmi_iClipX           =0;
    cTileEI.tmi_iClipY           =0;
    cTileEI.tmi_iClipWidth       =0;
    cTileEI.tmi_iClipHeight      =0;
    cTileEI.tmi_iMaxX            =0;
    cTileEI.tmi_iMaxY            =0;
    cTileEI.tmi_iTileDrawX       =0;
    cTileEI.tmi_iTileDrawY       =0;
    //Success!
    cTileEI.tmi_bSurfaceInit=0;    
}

//----------------TILE ENGINE DRAW FUNCTIONS-------------------//

//Draws the tiled view on the current drawing surface
//at iDrawX,iDrawY. it does not let iDrawX and iDrawY
//go beyond our limits.
//Returns 0 on error or 1 on success!
KCALL int TileEngine_DrawAtPos(TileMapInfo &cTileEI,      //Tile Engine Info Structure
                         long &iDrawX,long &iDrawY) //Position to draw tiled map.
{
    //If the drawing surface isn't set or we aren't initilized then return error!
    if (1 != cTileEI.tmi_bInit || 1 != cTileEI.tmi_bSurfaceInit)
    {
        //Not initilized!
        return 0;
    }
    //Make sure iDrawX and iDrawY are in a valid range!
    if (iDrawX > cTileEI.tmi_iMaxX) iDrawX = cTileEI.tmi_iMaxX;
    else if (iDrawX < 0) iDrawX=0;

    if (iDrawY > cTileEI.tmi_iMaxY) iDrawY = cTileEI.tmi_iMaxY;
    else if (iDrawY < 0) iDrawY=0;
    
    //Get width and height of tiles
    int iTileW = cTileEI.tmi_iTileWidth;
    int iTileH = cTileEI.tmi_iTileHeight;

    //Get what tile the drawing position landed on!
    int iMapX=(int)(iDrawX >> cTileEI.tmi_iShiftX);
    int iMapY=(int)(iDrawY >> cTileEI.tmi_iShiftY);

    //Get what drawing positions to start on(should be negative!)
    //Note that they should also be >=(-tilewidth) and >=(-tileheight)
    //int iStartX=(int)(0) - ((int)(iTileW) - (int)(iDrawX % iTileW));
    //int iStartY=(int)(0) - ((int)(iTileH) - (int)(iDrawY % iTileH));
    int iStartX=(int)(0) - (int)(iDrawX % iTileW);
    int iStartY=(int)(0) - (int)(iDrawY % iTileH);
    
    //If we are exactly even on a tile boundary then make start drawing positions 0!
//    if (iStartX == -(iTileW)) /*Make start pos 0!*/ iStartX=0;
//    if (iStartY == -(iTileH)) /*Make start pos 0!*/ iStartY=0;

    ////////////TILE DRAWING VARS/////////////
    char cMapValue=0;
    int iOurStartX=iStartX;
    ////////////OTHER VARS////////////////

    //Get a pointer to the map at the starting drawing position
    char KFAR *ptrBaseMap=(char KFAR *)(cTileEI.tmi_ptrTheMap + (int)((iMapY * cTileEI.tmi_iMapWidth) + iMapX));
    //This pointer is a temp pointer we use to do fast map value lookup...
    char KFAR *ptrDrawMap=ptrBaseMap;
    //Loop through the Rows drawing columns...
    int iLoopY=0;
    int iLoopX=0;
    char KFAR *ptrTileBitmap=NULL;

    //Now that we have the number of tiles to draw on the viewport draw them.
    for (iLoopY=0; iLoopY<cTileEI.tmi_iTileDrawY; iLoopY++)
    {
        //Reset X starting drawing position
        iOurStartX=iStartX;
        //Reset draw map pointer
        ptrDrawMap=ptrBaseMap;
        //Loop through the columns for this row
        for (iLoopX=0; iLoopX<cTileEI.tmi_iTileDrawX; iLoopX++)
        {
            //Get the value for this tile.
            //This also points the pointer to the next tile after getting value.
            cMapValue=*ptrDrawMap++;
            //Get the pointer to the tile bitmap for this tile!
            ptrTileBitmap=cTileEI.tmi_aTiles[cMapValue];
            //Now draw this bitmap clipped to the viewport(if map value isn't 0!)
            if (cMapValue)
                DrawBitmapClipped(ptrTileBitmap,        //Pointer to bitmap
                                  iOurStartX,           //X Coord to draw bitmap
                                  iStartY,              //Y Coord to draw bitmap
                                  iTileW,               //bitmap width
                                  iTileH,               //bitmap height,
                                  cTileEI.tmi_ptrDrawSurface,//Where we draw to
                                  cTileEI.tmi_iBytesPerScanline,//Num of bytes per scanline
                                  cTileEI.tmi_iClipX,   //Start of X viewport
                                  cTileEI.tmi_iClipY,   //Start of Y viewport
                                  cTileEI.tmi_iClipWidth,   //Drawing Dest width
                                  cTileEI.tmi_iClipHeight); //Drawing Dest height
            //Next X tile position!
            iOurStartX += iTileW;
        }
        //Point to the next row on the map...
        ptrBaseMap += (int)(cTileEI.tmi_iMapWidth);
        //Next Y tile position!
        iStartY += iTileH;
    }
    //Success!
    return 1;
}

//Draws a bitmap onto another bitmap clipped!
KCALL void DrawBitmapClipped(char KFAR *ptrBitmap,    //Pointer to bitmap
                       int iDrawX,              //X Coord to draw bitmap
                       int iDrawY,              //Y Coord to draw bitmap
                       int iBWidth,             //Width of bitmap
                       int iBHeight,            //Height of bitmap
                       char KFAR *ptrDrawDest,  //Where we draw to
                       int iBytesPerScanline,   //Num of bytes per scanline
                       int iDrawDestClipX,      //Start of clip region
                       int iDrawDestClipY,      //End of clip region
                       int iDrawDestWidth,      //Clip region width
                       int iDrawDestHeight)     //Clip region height
{
    //The REAL spot where we draw the bitmap.
    unsigned int iRealX=iDrawX;
    unsigned int iRealY=iDrawY;

    //Number of bytes to add to bitmap pointer before drawing scanline
    int iSPrevAddBytes=0;

    //Number of bytes to actually draw!
    int iSDrawBytes=iBWidth;

    //Number of bytes to add to bitmap pointer after drawing scanline
    int iSAftAddBytes=0;

    //Number of scanlines to draw
    int iSDrawScanlines=iBHeight;

    //Get the end clip region coords...
    int iEndX=iDrawDestClipX+iDrawDestWidth;
    int iEndY=iDrawDestClipY+iDrawDestHeight;

    //Clip X

    //Do we need to clip before?
    if (iDrawX < iDrawDestClipX)
    {
        int iBytes=iDrawDestClipX-iDrawX;
        iSPrevAddBytes+=iBytes;

        //If we add more bytes to source before drawing than width of bitmap the bitmap can't be seen!
        if (iSPrevAddBytes >= iBWidth) return;

        iSDrawBytes-=iBytes;
        iRealX=iDrawDestClipX;
    }
    //Do we need to clip after?
    int iTemp=iDrawX+iBWidth;
    if (iTemp > iEndX)
    {
        int iBytes=iTemp-iEndX;
        iSAftAddBytes+=iBytes;

        //If we add more bytes to source after drawing than width of bitmap the bitmap can't be seen!
        if (iSAftAddBytes >= iBWidth) return;

        iSDrawBytes-=iBytes;
    }

    //Clip Y

    //Do we need to clip before?
    if (iDrawY < iDrawDestClipY)
    {
        int iBytes=iDrawDestClipY-iDrawY;
        ptrBitmap+=(iBytes * iBWidth);

        //If we add more bytes to source before drawing than width of bitmap the bitmap can't be seen!
        if (iSPrevAddBytes >= iBWidth) return;

        iSDrawScanlines-=iBytes;
        iRealY=iDrawDestClipY;
    }
    //Do we need to clip after?
    iTemp=iDrawY+iBHeight;
    if (iTemp > iEndY)
    {
        int iBytes=iTemp-iEndY;
        iSDrawScanlines-=iBytes;
        //If the bitmap can't be seen then discard it!
        if (iSDrawScanlines <=0) return;
    }

    //Calculate the offset into the destination bitmap
    ptrDrawDest += ((unsigned int)(iRealY) * (unsigned int)(iBytesPerScanline)) + (unsigned int)(iRealX);
    //Now actually draw the bitmap(finally!)

    //Draw 1 scanline at a time!
    for (int iDrawScan=0; iDrawScan <iSDrawScanlines; iDrawScan++)
    {
        //Add the bytes previous to drawing to source
        ptrBitmap += iSPrevAddBytes;
        //Draw the scanline
        KMEMCPY(ptrDrawDest,ptrBitmap,iSDrawBytes);
        ptrBitmap+=iSDrawBytes;
        //Add the bytes after to drawing to source
        ptrBitmap += iSAftAddBytes;
        //Next scanline!
        ptrDrawDest += iBytesPerScanline;
    }
}
