/****************************************************************************\
*                                                                            *
*  FISH-HC.C                                                                 *
*                                                                            *
*  High color version of the Fastgraph fish tank.                            *
*                                                                            *
*  This program requires a 32-bit compiler.                                  *
*                                                                            *
\****************************************************************************/

#include "fastgraf.h"
#include <stdio.h>
#include <stdlib.h>

#define RIGHT 0
#define LEFT  1

#define UP   -1
#define DOWN  1

/* generate a random integer between two numbers */
#define IRAND(min,max) ((rand()%(max-min+1))+min)

/* location of each fish frame as stored in a PCX file */
typedef struct _frame
{
   int x1; int x2; int y1; int y2;
} FRAME;

/* type FISH defines each species of fish */
typedef struct _fish
{
   char  *fname;
   int   direction;
   int   nframes;
   int   frame_delay;
   FRAME frame[4];
   int   width[4];
   int   height[4];
   char  *rgb;
   char  *bitmap[4];
} FISH;

FISH fish[6] =
{{ "BLUEDAMS.PCX", LEFT, 4, 4,
    79,232,218,281,
   270,423,219,281,
   457,610,219,281,
    73,226,334,396,
   0,0,0,0,
   0,0,0,0,
   NULL,
   NULL,NULL,NULL,NULL
},{
   "BLUETANG.PCX", RIGHT, 4, 4,
   492,635,200,275,
   325,468,200,275,
   165,308,200,275,
     1,144,199,274,
   0,0,0,0,
   0,0,0,0,
   NULL,
   NULL,NULL,NULL,NULL
},{
   "BUTTRFLY.PCX", RIGHT, 4, 4,
   244,397,191,286,
    41,194,191,286,
    41,194,337,433,
   244,397,338,433,
   0,0,0,0,
   0,0,0,0,
   NULL,NULL,NULL,NULL
},{
   "GUDGEON.PCX", RIGHT, 4, 4,
    23,203,198,280,
   231,411,195,280,
   438,618,195,280,
    23,203,340,425,
   0,0,0,0,
   0,0,0,0,
   NULL,
   NULL,NULL,NULL,NULL
},{
   "KILLIFSH.PCX", RIGHT, 4,  4,
   133,285,153,205,
   350,502,153,205,
   136,288,273,324,
   350,502,272,324,
   0,0,0,0,
   0,0,0,0,
   NULL,
   NULL,NULL,NULL,NULL
},{
   "SEAHORSE.PCX", LEFT, 3, 4,
   388,435,205,293,
   295,342,205,293,
   199,246,205,293,
   0,0,0,0,
   0,0,0,0,
   0,0,0,0,
   NULL,
   NULL,NULL,NULL,NULL
}};

/* forward declarations */
struct _fishlist;
typedef struct _fishlist FISHLIST;

/* Each fish is stored as a sprite in a linked list */
typedef struct _fishlist
{
   FISHLIST *next;
   int fishnum;
   int x;
   int y;
   int xmin;
   int xmax;
   int xinc;
   int ymin;
   int ymax;
   int yinc;
   int frame;
   int x_direction;
   int y_direction;
   int f_direction;
   int frame_counter;
} FISHLIST;

FISHLIST *head;
FISHLIST *tail;

char *background_buffer;  /* virtual buffer for background */
int  background;

char *wheel_buffer;       /* virtual buffer for water wheel */
int  wheel;

char *work_buffer;        /* virtual buffer for building frames */
int  work;

/* buffers required water wheel flic file */
char wheel_context[16];
char wheel_rgb[768];
char *image_buf;
FILE *stream;

/* buffer required for jpeg decoding */
char *jpeg_buf;
char jpeg_header[10];

/* function declarations */
void get_fish(int fishnum);
void init_fish(void);
void animate_fish(void);

void main(void)
{
   register int i;
   int size;

   /* initialize for protected mode */
   fg_initpm();

   /* initialize video card and check for high color mode */
   fg_setmode(19);
   fg_svgainit(0);
   if (fg_testmode(34,1) == 0)
   {
      fg_setmode(3);
      fg_reset();
      printf("Unable to set 640x480 high color video mode.\n");
      exit(1);
   }
   fg_setmode(34);

   /* initialize virtual buffers */
   fg_vbinit();
   wheel_buffer      = malloc(fg_vbsize(186,153));
   wheel             = fg_vbdefine(wheel_buffer,186,153);
   background_buffer = malloc(fg_vbsize(640,480));
   background        = fg_vbdefine(background_buffer,640,480);
   work_buffer       = malloc(fg_vbsize(640,480));
   work              = fg_vbdefine(work_buffer,640,480);

   /* load background into virtual buffer and paste to screen */
   fg_vbopen(background);
   fg_jpeghead("AQUARIUM.JPG",jpeg_header);
   size = fg_jpegmem(jpeg_header);
   jpeg_buf = malloc(size);
   fg_jpegbuf(jpeg_buf,size);
   fg_showjpeg("AQUARIUM.JPG",0);
   fg_vbpaste(0,639,0,479,0,479);
   free(jpeg_buf);
   fg_vbclose();

   /* get all the fish and put them into a linked list */
   for (i = 0; i < 6; i++)
      get_fish(i);
   init_list();

   /* initialize water wheel flic file and load into memory */
   fg_flicopen("WHEEL.FLC",wheel_context);
   fg_flicdone(wheel_context);
   stream = fopen("WHEEL.FLC","rb");
   size = filelength(fileno(stream));
   image_buf = malloc(size);
   fread(image_buf,sizeof(char),size,stream);
   fclose(stream);
   fg_imagebuf(image_buf,size);

   /* load water wheel into its virtual buffer and grab DACs */
   fg_vbopen(wheel);
   fg_flicplay(wheel_context,1,5);
   fg_getdacs(0,256,wheel_rgb);
   fg_vbclose();

   /* everything's loaded -- proceed to animate */
   animate_fish();

   /* reset the video mode and exit */
   fg_vbclose();
   fg_setmode(3);
   fg_reset();
}

void get_fish(int fishnum)
{
   register int i;

   /* display the fish in the work virtual buffer */
   fg_vbopen(work);
   fg_loadpcx(fish[fishnum].fname,0);

   /* grab the 256-color palette information */
   fish[fishnum].rgb = malloc(768);
   fg_getdacs(0,256,fish[fishnum].rgb);

   /* grab the bitmaps for all the frames */
   for (i = 0; i < fish[fishnum].nframes; i++)
   {
      fish[fishnum].width[i] =
         fish[fishnum].frame[i].x2 - fish[fishnum].frame[i].x1+1;
      fish[fishnum].height[i] =
         fish[fishnum].frame[i].y2 - fish[fishnum].frame[i].y1+1;

      fish[fishnum].bitmap[i] =
         malloc(fish[fishnum].width[i] * fish[fishnum].height[i]);

      fg_move(fish[fishnum].frame[i].x1,fish[fishnum].frame[i].y2);
      fg_getimage(fish[fishnum].bitmap[i],
         fish[fishnum].width[i],fish[fishnum].height[i]);
   }
}

void init_list(void)
{
   FISHLIST *node;
   int nodes;
   register int i;

   /* choose a random number of fish */
   srand(fg_getclock());
   nodes = IRAND(6,12);

   /* build a linked list */
   for (i = 0; i < nodes; i++)
   {
      node = (FISHLIST *) malloc(sizeof(FISHLIST));
      node->x_direction = IRAND(RIGHT,LEFT);
      node->y_direction = IRAND(UP,DOWN);
      node->f_direction = IRAND(0,1);
      node->fishnum     = IRAND(0,5);
      node->next        = (FISHLIST *)NULL;
      node->xmin        = IRAND(-700,-200);
      node->xmax        = IRAND(700,1200);
      if (node->x_direction == RIGHT)
         node->x        = node->xmin;
      else
         node->x        = node->xmax;
      node->xinc        = IRAND(1,4);
      node->ymin        = IRAND(60,400);
      node->ymax        = IRAND(node->ymin,479);
      node->y           = IRAND(node->ymin,node->ymax);
      node->yinc        = IRAND(1,4);
      node->frame       = IRAND(1,2);
      node->frame_counter   = IRAND(0,2);
      if (head == (FISHLIST *)NULL)
      {
         head = node;
         tail = head;
      }
      else
      {
         tail->next = node;
         tail = node;
      }
   }
}

void animate_fish(void)
{
   FISHLIST *node;
   int r, g, b;
   int counter;
   int flic_frames;
   int bubble_frame;
   register int i;
   unsigned char key, aux;

   short bubble[14*14];
   static char bubble_mask[14*14] = {
   0,0,0,0,0,1,1,1,1,0,0,0,0,0,
   0,0,0,1,1,1,1,1,1,1,1,0,0,0,
   0,0,1,1,1,1,1,1,1,1,1,1,0,0,
   0,0,1,1,1,1,1,1,1,1,1,1,0,0,
   0,1,1,1,1,1,1,1,1,1,1,1,1,0,
   0,1,1,1,1,1,1,1,1,1,1,1,1,0,
   1,1,1,1,1,1,1,1,1,1,1,1,1,1,
   1,1,1,1,1,1,1,1,1,1,1,1,1,1,
   0,1,1,1,1,1,1,1,1,1,1,1,1,0,
   0,1,1,1,1,1,1,1,1,1,1,1,1,0,
   0,0,1,1,1,1,1,1,1,1,1,1,0,0,
   0,0,1,1,1,1,1,1,1,1,1,1,0,0,
   0,0,0,1,1,1,1,1,1,1,1,0,0,0,
   0,0,0,0,0,1,1,1,1,0,0,0,0,0};

   counter = 0;
   bubble_frame = 0;
   do
   {
      /* display background in work virtual buffer */
      fg_vbcopy(0,639,0,479,0,479,background,work);

      /* display next frame of water wheel flic file */
      fg_vbopen(wheel);
      fg_setdacs(0,256,wheel_rgb);
      flic_frames = fg_flicplay(wheel_context,1,5);
      if (flic_frames == 0)
      {
         fg_flicskip(wheel_context,-1);
         fg_flicskip(wheel_context,1);
         flic_frames = fg_flicplay(wheel_context,1,5);
         bubble_frame = 0;
      }
      fg_vbcopy(0,185,0,152,454,479,wheel,work);

      /* create and display a translucent bubble moving up */
      fg_vbopen(work);
      bubble_frame++;
      fg_move(458,(33-bubble_frame)*11);
      fg_getdcb(bubble,14,14);
      for (i = 0; i < 14*14; i++)
      {
         if (bubble_mask[i])
         {
            fg_unmaprgb(bubble[i],&r,&g,&b);
            bubble[i] = fg_maprgb(r+30,g+30,b+30);
         }
      }
      fg_clipdcb(bubble,14,14);

      /* display all the fish in the linked list */
      for (node = head; node != (FISHLIST *)NULL; node = node->next)
      {
         /* set the colors for this 256-color fish */
         fg_setdacs(0,256,fish[node->fishnum].rgb);

         /* blit the fish, may reverse while blitting */
         fg_move(node->x,node->y);
         if (node->x_direction == fish[node->fishnum].direction)
         {
            fg_clpimage(fish[node->fishnum].bitmap[node->frame],
               fish[node->fishnum].width[node->frame],
               fish[node->fishnum].height[node->frame]);
         }
         else
         {
            fg_flpimage(fish[node->fishnum].bitmap[node->frame],
               fish[node->fishnum].width[node->frame],
               fish[node->fishnum].height[node->frame]);
         }

         /* increment the horizontal position */
         if (node->x_direction == RIGHT)
         {
            node->x += node->xinc;
            if (node->x > node->xmax)
               node->x_direction = LEFT;
         }
         else
         {
            node->x -= node->xinc;
            if (node->x < node->xmin)
               node->x_direction = RIGHT;
         }

         /* adjust vertical position */
         node->y += node->y_direction;

         /* keep vertical motion within tolerance area */
         if (node->y >= node->ymax)
            node->y_direction = UP;
         else if (node->y <= node->ymin)
            node->y_direction = DOWN;

         /* add a random element to speed and vertical motion */
         counter++;
         if (counter % node->y == 0)
         {
            node->y_direction = IRAND(-1,5);
            if (node->y_direction > 1)
               node->y_direction = 0;
            node->xinc = IRAND(1,4);
            node->ymin = IRAND(60,400);
            node->ymax = IRAND(node->ymin,479);
         }

         /* increment fish frame (swish tail slowly) */
         node->frame_counter++;
         if (node->frame_counter >= fish[node->fishnum].frame_delay)
         {
            if (node->f_direction == RIGHT)
            {
               node->frame++;
               if (node->frame >= fish[node->fishnum].nframes-1)
                  node->f_direction = LEFT;
            }
            else
            {
               node->frame--;
               if (node->frame <= 1)
                  node->f_direction = RIGHT;
            }
            node->frame_counter = 0;
         }
      }

      /* display the frame we just constructed */
      fg_vbpaste(0,639,0,479,0,479);

      /* intercept a keystroke, break if Escape key pressed */
      fg_intkey(&key,&aux);
   }
   while (key != 27);
}
