/*
	POB-File-Format by Thomas Baier April/97


	Description:
	Complete pob-parser implementation

	History:
	01.04.97	Version 0.8	First Version, all things implemented, 
					rarly tested

	Comments:
	Tis code opens a pob-file and prints out all contents to a povray file.
	This file is provided as template for own implementations and as
	test application for pob-files.

	To Do List:
	- optimized mesh input
*/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <ctype.h>

#include "pob_id.h"            // POV-Ray idendifier
#include "pob_id1.h"           // System Identifier

/* Platform stuff */
/*
	if your Platform does not support IEEE with DOUBLE == 8 Bytes
	you have to rewrite following procedures:
		read_double_ieee()
*/

#define __IEEE				// switch on IEEE floating point support

/* debug stuff */
#define __POB_DEBUG1			// only chunk header
#define __POB_DEBUG2			// plus evaluated values
#define __POB_DEBUG3			// plus chunk data

void start_debug(void);			// start debug session
void end_debug(void);			// end debug session
void print_debug_chunk(CHUNK *chunk, CHUNK *mainchunk);	// debug print chunk header + raw data
void set_tab( FILE *fp );		// print tabs
void inc_tab( void );			// add tab
void dec_tab( void );			// remove tab

static P_WORD tab_size = 0;		// store formating info
static FILE *dbg_fpt = NULL;		// file pointer debug file
char *pob_log_file = "pob.log";		// debug file name

/* binary stuff */
static FILE *fptr;			// POB file pointer
static P_DWORD file_size;		// end of file checking
					// very important, has to be set
					// before first chunk is parsed

/* read data types */
P_BYTE read_byte( void );
P_WORD read_word( void );
P_DWORD read_dword( void );
P_DOUBLE read_double( void );
char *read_string( void );
void read_vector (P_DVECTOR v);
void read_vector_uv (P_DVECTOR v);
void read_matrix(P_MATRIX m);

/* chunk stuff */
void start_chunk (CHUNK *chunk, CHUNK *mainchunk);	// read next chunk header
void end_chunk (CHUNK *chunk);				// skip to next chunk header
P_DWORD filesize(FILE *stream);				// calculate overall file size

/* POB file stuff */
void parse_pob_file (CHUNK *chunk);
void parse_povray_section (CHUNK *chunk);

void parse_camera_section (CHUNK *mainchunk);
void parse_camera (CHUNK *mainchunk);

void parse_global_section (CHUNK *mainchunk);
void parse_global_environment (CHUNK *mainchunk);
void parse_rainbow (CHUNK *mainchunk);
void parse_sky_sphere (CHUNK *mainchunk);
void parse_fog (CHUNK *mainchunk);
void parse_athmosphere (CHUNK *mainchunk);

void parse_material_section (CHUNK *mainchunk);
void parse_simple_texture (CHUNK *mainchunk);
void parse_layered_texture (CHUNK *mainchunk);
void parse_material_map_texture(CHUNK *mainchunk);
void parse_checker_texture(CHUNK *mainchunk);
void parse_brick_texture(CHUNK *mainchunk);
void parse_hexagon_texture(CHUNK *mainchunk);
void parse_pattern_texture(CHUNK *mainchunk);

void parse_simple_pigment (CHUNK *mainchunk);
void parse_checker_pigment(CHUNK *mainchunk);
void parse_brick_pigment(CHUNK *mainchunk);
void parse_hexagon_pigment(CHUNK *mainchunk);
void parse_bitmap_pigment(CHUNK *mainchunk);
void parse_pattern_pigment(CHUNK *mainchunk);

void parse_simple_normal (CHUNK *mainchunk);
void parse_checker_normal(CHUNK *mainchunk);
void parse_brick_normal(CHUNK *mainchunk);
void parse_hexagon_normal(CHUNK *mainchunk);
void parse_bitmap_normal(CHUNK *mainchunk);
void parse_pattern_normal(CHUNK *mainchunk);

void parse_simple_finish (CHUNK *mainchunk);

void parse_simple_halo (CHUNK *mainchunk);

void parse_pattern(void);
void parse_pattern_mod(CHUNK *mainchunk);
void parse_color(void);

void parse_object_section (CHUNK *mainchunk);
void parse_bicubic_patch (CHUNK *mainchunk);
void parse_blob (CHUNK *mainchunk);
void parse_box (CHUNK *mainchunk);
void parse_cylinder (CHUNK *mainchunk);
void parse_cone (CHUNK *mainchunk);
void parse_group (CHUNK *mainchunk);
void parse_disc (CHUNK *mainchunk);
void parse_fractal (CHUNK *mainchunk);
void parse_heightfield (CHUNK *mainchunk);
void parse_lathe (CHUNK *mainchunk);
void parse_mesh (CHUNK *mainchunk);
void parse_plane (CHUNK *mainchunk);
void parse_poly (CHUNK *mainchunk);
void parse_polygon (CHUNK *mainchunk);
void parse_prism (CHUNK *mainchunk);
void parse_quadric (CHUNK *mainchunk);
void parse_sor (CHUNK *mainchunk);
void parse_sphere (CHUNK *mainchunk);
void parse_superellipsoid (CHUNK *mainchunk);
void parse_torus (CHUNK *mainchunk);
void parse_triangle (CHUNK *mainchunk);
void parse_smoothtriangle (CHUNK *mainchunk);
void parse_text (CHUNK *mainchunk);
void parse_light_source (CHUNK *mainchunk);

void parse_obj_texture(CHUNK *mainchunk);
void parse_object_mod(CHUNK *mainchunk);

void parse_image(void);

/* POV printout stuff */
void print_double(P_DOUBLE t);
void print_word(P_WORD t);
void print_vector(P_DVECTOR t);
void print_uv_vector(P_DVECTOR t);
void print_matrix(P_MATRIX matrix);
void print_flags(P_WORD flags);
void print_obj_texture(void);
void print_comment(char *t);

void my_print(char *format, ...);		// main print routine, goes to stdout


/* Some helpfull print defines */

#define PRINT_COMMA             my_print( ",")
#define PRINT_BEGIN             my_print( "{\n")
#define PRINT_END               my_print( "}\n")
#define PRINT_NEWLINE   	my_print( "\n")

#define PRINT_NAME print_comment(read_string())

#define GET_TOKEN(a)    {my_print( "%s ", get_token_str(a));PRINT_NEWLINE;}
#define GET_VTOKEN(a)   {P_DVECTOR tmp;read_vector(tmp);my_print( "%s ", get_token_str(a));print_vector(tmp);PRINT_NEWLINE;}
#define GET_ITOKEN(a)   {P_WORD tmp;tmp = read_word();my_print( "%s ", get_token_str(a));print_word(tmp);PRINT_NEWLINE;}
#define GET_DTOKEN(a)   {P_DOUBLE tmp;tmp = read_double();my_print( "%s ", get_token_str(a));print_double(tmp);PRINT_NEWLINE;}
#define GET_CTOKEN(a)   {my_print( "%s ", get_token_str(a));parse_color();PRINT_NEWLINE;}


/* main loop */
void main ( int argc, char **argv )
{
	CHUNK chunk;

	fprintf(stderr, ">>Sample POB File reader by Thomas Baier<<\n");

	/* check if DOUBLE is 8 Bytes long */
	/* Floating Point stuff requires P_DOUBLE == 8 Bytes long */
	if(sizeof(P_DOUBLE) != 8)
	{
		fprintf(stderr, "POB - P_DOUBLE requires 8 Bytes\n");
		exit(0);
	}

	if(argv[1] == NULL)
	{
		fprintf(stderr, "No file specified\n");
		fprintf(stderr, "Usage: pob_rd pob_file\n");
		fprintf(stderr, "Example: pob_rd test.pob\n");
		exit(1);
	}

	fptr = fopen(argv[1], "rb");
	if(fptr == NULL)
	{
		fprintf(stderr, "Cannot open file %s\n",argv[1]);
		exit(1);
	}

	start_debug();						// start debug log

	file_size = filesize(fptr);				// init eof check

	start_chunk(&chunk, NULL);				// read chunk header

	if (chunk.tag == POB_FILESECTION_TAG)			// check chunk identifier
	{
		P_WORD file_version;

		if( filesize(fptr) == chunk.end) 		// Check for correct len
		{
			file_version = read_word();		// read POB version
			fprintf(stderr, "Pob File version: %04X\n", file_version);
			parse_pob_file (&chunk);		// read contents of POB-file
		}
		else
			fprintf(stderr, "!! No Pob file\n");
	}
	else
		fprintf(stderr, "!! No Pob file\n");
	end_chunk (&chunk);					// close first chunk

	end_debug();						// close debug log

	fclose(fptr);						// close POB-file
	exit(0);						// thats all
}

/* parse contents of POB-file*/

void parse_pob_file (CHUNK *mainchunk)
{
	CHUNK chunk;

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case POB_POVRAYSECTION_TAG:
				{
					P_WORD povray_version;
					povray_version = read_word();
					fprintf(stderr, "Povray Section version: %04X\n", povray_version);
					print_comment(read_string());
					parse_povray_section (&chunk);
				}
				break;
				case POB_MODELLERSECTION_TAG:
				{
					P_WORD modeller_version;
					modeller_version = read_word();
					fprintf(stderr, "Modeller Section version: %04X\n", modeller_version);
					fprintf(stderr, "Modeller Name: %s\n", read_string());
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);
}

/* Parse contents of POV-Ray Section */

void parse_povray_section (CHUNK *mainchunk)
{
	CHUNK chunk;

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{

				case POB_GLOBALSECTION_TAG:
				{

					print_comment("Global Section");
					my_print( "%s ", get_token_str(GLOBAL_SETTINGS_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_global_section(&chunk);
					PRINT_END;

				}
				break;

				case POB_ENVIRONMENTSECTION_TAG:
				{
					print_comment("Environment Section");
					parse_global_environment(&chunk);
				}
				break;

				case POB_CAMERASECTION_TAG:
				{
					print_comment("Camera Section");
					parse_camera_section(&chunk);
				}
				break;

				case POB_MATERIALSECTION_TAG:
				{
					print_comment("Material Section");
					parse_material_section(&chunk);
				}
				break;

				case POB_OBJECTSECTION_TAG:
				{
					print_comment("Object Section");
					parse_object_section(&chunk);
				}
				break;

			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);

}

/* Camera Section */

void parse_camera_section (CHUNK *mainchunk)
{
	CHUNK chunk;

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case CAMERA_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(CAMERA_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_camera(&chunk);
					PRINT_END;
				}
				break;
				case SIMPLENORMAL_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;
					parse_simple_normal(&chunk);
				}
				break;
				case CHECKERNORMAL_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;
					parse_checker_normal(&chunk);
				}
				break;
				case BRICKNORMAL_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;
					parse_brick_normal(&chunk);
				}
				break;
				case HEXAGONNORMAL_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;
					parse_hexagon_normal(&chunk);
				}
				break;
				case BITMAPNORMAL_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;
					parse_bitmap_normal(&chunk);
				}
				break;
				case PATTERNNORMAL_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;
					parse_pattern_normal(&chunk);
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);
}

/* Material (Texture) Section */

void parse_material_section (CHUNK *mainchunk)
{
	CHUNK chunk;

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case SIMPLETEXTURE_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;
					parse_simple_texture(&chunk);
				}
				break;
				case LAYEREDTEXTURE_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;

					parse_layered_texture(&chunk);
				}
				break;
				case MATERIAL_MAPTEXTURE_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;

					parse_material_map_texture(&chunk);
				}
				break;
				case CHECKERTEXTURE_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;

					parse_checker_texture(&chunk);
				}
				break;
				case BRICKTEXTURE_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;

					parse_brick_texture(&chunk);
				}
				break;
				case HEXAGONTEXTURE_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;

					parse_hexagon_texture(&chunk);
				}
				break;
				case PATTERNTEXTURE_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;

					parse_pattern_texture(&chunk);
				}
				break;
				case SIMPLEPIGMENT_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;
					parse_simple_pigment(&chunk);
				}
				break;
				case CHECKERPIGMENT_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;
					parse_checker_pigment(&chunk);
				}
				break;
				case BRICKPIGMENT_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;
					parse_brick_pigment(&chunk);
				}
				break;
				case HEXAGONPIGMENT_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;
					parse_hexagon_pigment(&chunk);
				}
				break;
				case BITMAPPIGMENT_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;
					parse_bitmap_pigment(&chunk);
				}
				break;
				case PATTERNPIGMENT_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;
					parse_pattern_pigment(&chunk);
				}
				break;

				case SIMPLENORMAL_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;
					parse_simple_normal(&chunk);
				}
				break;
				case CHECKERNORMAL_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;
					parse_checker_normal(&chunk);
				}
				break;
				case BRICKNORMAL_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;
					parse_brick_normal(&chunk);
				}
				break;
				case HEXAGONNORMAL_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;
					parse_hexagon_normal(&chunk);
				}
				break;
				case BITMAPNORMAL_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;
					parse_bitmap_normal(&chunk);
				}
				break;
				case PATTERNNORMAL_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;
					parse_pattern_normal(&chunk);
				}
				break;
				case SIMPLEFINISH_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;
					parse_simple_finish(&chunk);
				}
				break;

				case SIMPLEHALO_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;
					parse_simple_halo(&chunk);
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);
}

/* simple Texture */

void parse_simple_texture (CHUNK *mainchunk)
{
	CHUNK chunk;

	my_print( "%s ", get_token_str(TEXTURE_POB_TOKEN));PRINT_NEWLINE;
	PRINT_BEGIN;

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case PIGMENTREF_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(PIGMENT_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_obj_texture(&chunk);
					PRINT_END;
				}
				break;

				case NORMALREF_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(NORMAL_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_obj_texture(&chunk);
					PRINT_END;
				}
				break;

				case FINISHREF_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(FINISH_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					my_print("%s ", read_string()); // Texture name
					PRINT_NEWLINE;
					PRINT_END;
				}
				break;

				case HALOREF_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(HALO_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_obj_texture(&chunk);
					PRINT_END;
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);

	PRINT_END;
}

/* layered Texture */

void parse_layered_texture (CHUNK *mainchunk)
{
	CHUNK chunk;

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case SIMPLETEXTURE_POB_TOKEN:
				{
					read_string();
					parse_simple_texture(&chunk);
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);
}

/* mapped texture */

void parse_material_map_texture(CHUNK *mainchunk)
{
	CHUNK chunk;
	P_WORD once_flag;

	my_print( "%s ", get_token_str(TEXTURE_POB_TOKEN));PRINT_NEWLINE;
	PRINT_BEGIN;
	
	my_print( "%s ", get_token_str(MATERIAL_MAP_POB_TOKEN));PRINT_NEWLINE;
	PRINT_BEGIN;

	parse_image();
	
	once_flag = read_word();
	if(once_flag)
	{
		GET_TOKEN(ONCE_POB_TOKEN);
	}

	GET_ITOKEN(INTERPOLATE_POB_TOKEN);
	GET_ITOKEN(MAP_TYPE_POB_TOKEN);
	
	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case TEXTUREREF_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(TEXTURE_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_obj_texture(&chunk);
					PRINT_END;
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);
	
	PRINT_END;
	PRINT_END;
}

/* checker texture */

void parse_checker_texture(CHUNK *mainchunk)
{
	CHUNK chunk;

	my_print( "%s ", get_token_str(TEXTURE_POB_TOKEN));PRINT_NEWLINE;
	PRINT_BEGIN;
	my_print( "%s ", get_token_str(CHECKER_POB_TOKEN));PRINT_NEWLINE;

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case TEXTUREREF_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(TEXTURE_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_obj_texture(&chunk);
					PRINT_END;
				}
				break;
				case PATTERN_MOD_POB_TOKEN:
				{
					parse_pattern_mod(&chunk);
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);
	PRINT_END;
}

/* brick texture */

void parse_brick_texture(CHUNK *mainchunk)
{
	CHUNK chunk;
	P_DVECTOR brick_size;
	P_DOUBLE motar;


	my_print( "%s ", get_token_str(TEXTURE_POB_TOKEN));PRINT_NEWLINE;
	PRINT_BEGIN;
	my_print( "%s ", get_token_str(BRICK_POB_TOKEN));PRINT_NEWLINE;

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case TEXTUREREF_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(TEXTURE_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_obj_texture(&chunk);
					PRINT_END;
				}
				break;
				case BRICK_SIZE_POB_TOKEN:
				{
					GET_VTOKEN(BRICK_SIZE_POB_TOKEN);
				}
				break;
				case MORTAR_POB_TOKEN:
				{
					GET_DTOKEN(MORTAR_POB_TOKEN);
				}
				break;
				case PATTERN_MOD_POB_TOKEN:
				{
					parse_pattern_mod(&chunk);
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);
	PRINT_END;
}

/* hexagon texture */

void parse_hexagon_texture(CHUNK *mainchunk)
{
	CHUNK chunk;

	my_print( "%s ", get_token_str(TEXTURE_POB_TOKEN));PRINT_NEWLINE;
	PRINT_BEGIN;
	my_print( "%s ", get_token_str(HEXAGON_POB_TOKEN));PRINT_NEWLINE;

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case TEXTUREREF_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(TEXTURE_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_obj_texture(&chunk);
					PRINT_END;
				}
				break;
				case PATTERN_MOD_POB_TOKEN:
				{
					parse_pattern_mod(&chunk);
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);
	PRINT_END;
}

/* patterned texture */

void parse_pattern_texture(CHUNK *mainchunk)
{
	CHUNK chunk;
	P_WORD i;
	P_WORD count;
	P_DOUBLE index;
	char *txt_ref;

	my_print( "%s ", get_token_str(TEXTURE_POB_TOKEN));PRINT_NEWLINE;
	PRINT_BEGIN;
	
	parse_pattern();
	
	count = read_word();

	if(count)
	{
		my_print( "%s ", get_token_str(TEXTURE_MAP_POB_TOKEN));PRINT_NEWLINE;
		PRINT_BEGIN;
		for(i = 0;i < count;i++)
		{
			index = read_double();
			txt_ref = read_string();
			my_print( "[ ");
                        print_double(index);
			my_print( "%s ", txt_ref);
                        my_print( "] ");
                        PRINT_NEWLINE;
		}
		PRINT_END;
	}

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case PATTERN_MOD_POB_TOKEN:
				{
					parse_pattern_mod(&chunk);
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);

	PRINT_END;
}

/* simple pigment */

void parse_simple_pigment (CHUNK *mainchunk)
{
	CHUNK chunk;

	my_print( "%s ", get_token_str(PIGMENT_POB_TOKEN));PRINT_NEWLINE;
	PRINT_BEGIN;
	
	parse_color();PRINT_NEWLINE;

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case PATTERN_MOD_POB_TOKEN:
				{
					parse_pattern_mod(&chunk);
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);

	PRINT_END;
}

/* checker pigment */

void parse_checker_pigment (CHUNK *mainchunk)
{
	CHUNK chunk;
	P_WORD type;

	my_print( "%s ", get_token_str(PIGMENT_POB_TOKEN));PRINT_NEWLINE;
	PRINT_BEGIN;
	
	type = read_word();

	switch(type)
	{
		case POB_PIGMENT_COLORMAP: //color_type
		{
			my_print( "%s ", get_token_str(CHECKER_POB_TOKEN));PRINT_NEWLINE;

			parse_color();
			PRINT_COMMA;
			PRINT_NEWLINE;
			parse_color();
			PRINT_NEWLINE;
		}
		break;

		case POB_PIGMENT_PIGMENTMAP:
		{
			my_print( "%s ", get_token_str(CHECKER_POB_TOKEN));PRINT_NEWLINE;

			start_chunk (&chunk, mainchunk);
			my_print( "%s ", get_token_str(PIGMENT_POB_TOKEN));PRINT_NEWLINE;
			PRINT_BEGIN;
			parse_obj_texture(&chunk);
			PRINT_END;
			end_chunk (&chunk);

			start_chunk (&chunk, mainchunk);
			my_print( "%s ", get_token_str(PIGMENT_POB_TOKEN));PRINT_NEWLINE;
			PRINT_BEGIN;
			parse_obj_texture(&chunk);
			PRINT_END;
			end_chunk (&chunk);
		}
		break;
	}

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case PATTERN_MOD_POB_TOKEN:
				{
					parse_pattern_mod(&chunk);
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);

	PRINT_END;
}

/* brick pigment */

void parse_brick_pigment (CHUNK *mainchunk)
{
	CHUNK chunk;
	P_WORD type;
	P_DVECTOR Size;
	P_DOUBLE Mortar;

	my_print( "%s ", get_token_str(PIGMENT_POB_TOKEN));PRINT_NEWLINE;
	PRINT_BEGIN;
	
	type = read_word();

	switch(type)
	{
		case POB_PIGMENT_COLORMAP: //color_type
		{
			my_print( "%s ", get_token_str(BRICK_POB_TOKEN));PRINT_NEWLINE;
			read_vector(Size);
			Mortar = read_double();

			parse_color();
			PRINT_COMMA;
			PRINT_NEWLINE;
			parse_color();
			PRINT_NEWLINE;
			my_print( "%s ", get_token_str(BRICK_SIZE_POB_TOKEN));print_vector(Size);
			PRINT_NEWLINE;
			my_print( "%s ", get_token_str(MORTAR_POB_TOKEN));print_double(Mortar);
			PRINT_NEWLINE;
		}
		break;

		case POB_PIGMENT_PIGMENTMAP:
		{
			my_print( "%s ", get_token_str(BRICK_POB_TOKEN));PRINT_NEWLINE;

			read_vector(Size);
			Mortar = read_double();

			start_chunk (&chunk, mainchunk);
			my_print( "%s ", get_token_str(PIGMENT_POB_TOKEN));PRINT_NEWLINE;
			PRINT_BEGIN;
			parse_obj_texture(&chunk);
			PRINT_END;
			end_chunk (&chunk);

			start_chunk (&chunk, mainchunk);
			my_print( "%s ", get_token_str(PIGMENT_POB_TOKEN));PRINT_NEWLINE;
			PRINT_BEGIN;
			parse_obj_texture(&chunk);
			PRINT_END;
			end_chunk (&chunk);

			my_print( "%s ", get_token_str(BRICK_SIZE_POB_TOKEN));print_vector(Size);
			PRINT_NEWLINE;
			my_print( "%s ", get_token_str(MORTAR_POB_TOKEN));print_double(Mortar);
			PRINT_NEWLINE;
		}
		break;
	}

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case PATTERN_MOD_POB_TOKEN:
				{
					parse_pattern_mod(&chunk);
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);

	PRINT_END;
}

/* hexagon pigment */

void parse_hexagon_pigment (CHUNK *mainchunk)
{
	CHUNK chunk;
	P_WORD type;

	my_print( "%s ", get_token_str(PIGMENT_POB_TOKEN));PRINT_NEWLINE;
	PRINT_BEGIN;
	
	type = read_word();

	switch(type)
	{
		case POB_PIGMENT_COLORMAP: //color_type
		{
			my_print( "%s ", get_token_str(HEXAGON_POB_TOKEN));PRINT_NEWLINE;

			parse_color();
			PRINT_COMMA;
			PRINT_NEWLINE;
			parse_color();
			PRINT_COMMA;
			PRINT_NEWLINE;
			parse_color();
			PRINT_NEWLINE;
		}
		break;

		case POB_PIGMENT_PIGMENTMAP:
		{
			my_print( "%s ", get_token_str(HEXAGON_POB_TOKEN));PRINT_NEWLINE;

			start_chunk (&chunk, mainchunk);
			my_print( "%s ", get_token_str(PIGMENT_POB_TOKEN));PRINT_NEWLINE;
			PRINT_BEGIN;
			parse_obj_texture(&chunk);
			PRINT_END;
			end_chunk (&chunk);

			start_chunk (&chunk, mainchunk);
			my_print( "%s ", get_token_str(PIGMENT_POB_TOKEN));PRINT_NEWLINE;
			PRINT_BEGIN;
			parse_obj_texture(&chunk);
			PRINT_END;
			end_chunk (&chunk);

			start_chunk (&chunk, mainchunk);
			my_print( "%s ", get_token_str(PIGMENT_POB_TOKEN));PRINT_NEWLINE;
			PRINT_BEGIN;
			parse_obj_texture(&chunk);
			PRINT_END;
			end_chunk (&chunk);
		}
		break;
	}

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case PATTERN_MOD_POB_TOKEN:
				{
					parse_pattern_mod(&chunk);
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);

	PRINT_END;
}

/* bitmap pigment */

void parse_bitmap_pigment (CHUNK *mainchunk)
{
	CHUNK chunk;
	P_WORD type;
	P_WORD once_flag;

	my_print( "%s ", get_token_str(PIGMENT_POB_TOKEN));PRINT_NEWLINE;
	PRINT_BEGIN;
	
	type = read_word();

	switch(type)
	{
		case POB_PIGMENT_COLORMAP: //color_type
		{
			my_print( "%s ", get_token_str(IMAGE_MAP_POB_TOKEN));PRINT_NEWLINE;
			PRINT_BEGIN;

			parse_image();

			once_flag = read_word();
			if(once_flag)
			{
				GET_TOKEN(ONCE_POB_TOKEN);
			}

			GET_ITOKEN(INTERPOLATE_POB_TOKEN);
			GET_ITOKEN(MAP_TYPE_POB_TOKEN);

			PRINT_END;
		}
		break;
	}

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case PATTERN_MOD_POB_TOKEN:
				{
					parse_pattern_mod(&chunk);
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);

	PRINT_END;
}

/* patterned pigment */

void parse_pattern_pigment (CHUNK *mainchunk)
{
	CHUNK chunk;
	P_WORD type;
	P_WORD count;
	P_DOUBLE index;
	P_WORD i;


	my_print( "%s ", get_token_str(PIGMENT_POB_TOKEN));PRINT_NEWLINE;
	PRINT_BEGIN;
	
	type = read_word();

	switch(type)
	{
		case POB_PIGMENT_COLORMAP: //color_type
		{
			parse_pattern();

			count = read_word();

			if(count)
			{
       				my_print( "%s ", get_token_str(COLOR_MAP_POB_TOKEN));PRINT_NEWLINE;
       				PRINT_BEGIN;
				
				for(i = 0;i < count;i++)
				{
					index = read_double();
					my_print( "[ ");
                        		print_double(index);
					parse_color();
                        		my_print( "] ");
                        		PRINT_NEWLINE;
				}
				
				PRINT_END;
			}
		}
		break;

		case POB_PIGMENT_PIGMENTMAP:
		{
			parse_pattern();

			count = read_word();

			if(count)
			{
       				my_print( "%s ", get_token_str(PIGMENT_MAP_POB_TOKEN));PRINT_NEWLINE;
       				PRINT_BEGIN;
				
				for(i = 0;i < count;i++)
				{
					start_chunk (&chunk, mainchunk);
					index = read_double();
					end_chunk (&chunk);
					my_print( "[ ");
                        		print_double(index);

					start_chunk (&chunk, mainchunk);
					parse_obj_texture(&chunk);
					end_chunk (&chunk);

                        		my_print( "] ");
                        		PRINT_NEWLINE;
				}
				
				PRINT_END;
			}
		}
		break;
		case POB_PIGMENT_NOMAP: //without map
		{
			parse_pattern();
		}
		break;
	}

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case PATTERN_MOD_POB_TOKEN:
				{
					parse_pattern_mod(&chunk);
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);

	PRINT_END;
}

/* simple normal */

void parse_simple_normal (CHUNK *mainchunk)
{
	CHUNK chunk;
	P_DOUBLE amount;

	my_print( "%s ", get_token_str(NORMAL_POB_TOKEN));PRINT_NEWLINE;
	PRINT_BEGIN;

	amount = read_double();
	
	parse_pattern();

	my_print( "%s ", get_token_str(BUMP_SIZE_POB_TOKEN));print_double(amount);PRINT_NEWLINE;

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case PATTERN_MOD_POB_TOKEN:
				{
					parse_pattern_mod(&chunk);
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);

	PRINT_END;
}

/* checker normal */

void parse_checker_normal (CHUNK *mainchunk)
{
	CHUNK chunk;
	P_WORD type;
	P_DOUBLE amount;
	P_WORD blend_map;

	my_print( "%s ", get_token_str(NORMAL_POB_TOKEN));PRINT_NEWLINE;
	PRINT_BEGIN;
	
	amount = read_double();

	my_print( "%s ", get_token_str(CHECKER_POB_TOKEN));PRINT_NEWLINE;

	blend_map = read_word();
	if(blend_map == POB_NORMAL_NORMALMAP)
	{
	
		start_chunk (&chunk, mainchunk);
		my_print( "%s ", get_token_str(NORMAL_POB_TOKEN));PRINT_NEWLINE;
		PRINT_BEGIN;
		parse_obj_texture(&chunk);
		PRINT_END;
		end_chunk (&chunk);

		start_chunk (&chunk, mainchunk);
		my_print( "%s ", get_token_str(NORMAL_POB_TOKEN));PRINT_NEWLINE;
		PRINT_BEGIN;
		parse_obj_texture(&chunk);
		PRINT_END;
		end_chunk (&chunk);
	}

	my_print( "%s ", get_token_str(BUMP_SIZE_POB_TOKEN));print_double(amount);PRINT_NEWLINE;

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case PATTERN_MOD_POB_TOKEN:
				{
					parse_pattern_mod(&chunk);
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);

	PRINT_END;
}

/* brick normal */

void parse_brick_normal (CHUNK *mainchunk)
{
	CHUNK chunk;
	P_WORD type;
	P_DVECTOR Size;
	P_DOUBLE Mortar;
	P_DOUBLE amount;
	P_WORD blend_map;

	my_print( "%s ", get_token_str(NORMAL_POB_TOKEN));PRINT_NEWLINE;
	PRINT_BEGIN;
	
	my_print( "%s ", get_token_str(BRICK_POB_TOKEN));PRINT_NEWLINE;

	blend_map = read_word();

	read_vector(Size);
	Mortar = read_double();

	if(blend_map == POB_NORMAL_NORMALMAP)
	{
	
		start_chunk (&chunk, mainchunk);
		my_print( "%s ", get_token_str(NORMAL_POB_TOKEN));PRINT_NEWLINE;
		PRINT_BEGIN;
		parse_obj_texture(&chunk);
		PRINT_END;
		end_chunk (&chunk);

		start_chunk (&chunk, mainchunk);
		my_print( "%s ", get_token_str(NORMAL_POB_TOKEN));PRINT_NEWLINE;
		PRINT_BEGIN;
		parse_obj_texture(&chunk);
		PRINT_END;
		end_chunk (&chunk);
	}

	my_print( "%s ", get_token_str(BRICK_SIZE_POB_TOKEN));print_vector(Size);
	PRINT_NEWLINE;
	my_print( "%s ", get_token_str(MORTAR_POB_TOKEN));print_double(Mortar);
	PRINT_NEWLINE;

	my_print( "%s ", get_token_str(BUMP_SIZE_POB_TOKEN));print_double(amount);PRINT_NEWLINE;

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case PATTERN_MOD_POB_TOKEN:
				{
					parse_pattern_mod(&chunk);
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);

	PRINT_END;
}

/* hexagon normal */

void parse_hexagon_normal (CHUNK *mainchunk)
{
	CHUNK chunk;
	P_WORD type;
	P_DOUBLE amount;
	P_WORD blend_map;

	my_print( "%s ", get_token_str(NORMAL_POB_TOKEN));PRINT_NEWLINE;
	PRINT_BEGIN;
	
	amount = read_double();

	my_print( "%s ", get_token_str(HEXAGON_POB_TOKEN));PRINT_NEWLINE;

	blend_map = read_word();
	if(blend_map == POB_NORMAL_NORMALMAP)
	{
	
		start_chunk (&chunk, mainchunk);
		my_print( "%s ", get_token_str(NORMAL_POB_TOKEN));PRINT_NEWLINE;
		PRINT_BEGIN;
		parse_obj_texture(&chunk);
		PRINT_END;
		end_chunk (&chunk);

		start_chunk (&chunk, mainchunk);
		my_print( "%s ", get_token_str(NORMAL_POB_TOKEN));PRINT_NEWLINE;
		PRINT_BEGIN;
		parse_obj_texture(&chunk);
		PRINT_END;
		end_chunk (&chunk);

		start_chunk (&chunk, mainchunk);
		my_print( "%s ", get_token_str(NORMAL_POB_TOKEN));PRINT_NEWLINE;
		PRINT_BEGIN;
		parse_obj_texture(&chunk);
		PRINT_END;
		end_chunk (&chunk);
	}

	my_print( "%s ", get_token_str(BUMP_SIZE_POB_TOKEN));print_double(amount);PRINT_NEWLINE;

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case PATTERN_MOD_POB_TOKEN:
				{
					parse_pattern_mod(&chunk);
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);

	PRINT_END;
}

/* bitmap normal */

void parse_bitmap_normal (CHUNK *mainchunk)
{
	CHUNK chunk;
	P_WORD type;
	P_WORD once_flag;
	P_DOUBLE amount;

	my_print( "%s ", get_token_str(NORMAL_POB_TOKEN));PRINT_NEWLINE;
	PRINT_BEGIN;

	amount = read_double();
	
	my_print( "%s ", get_token_str(IMAGE_MAP_POB_TOKEN));PRINT_NEWLINE;
	PRINT_BEGIN;

	parse_image();

	once_flag = read_word();
	if(once_flag)
	{
		GET_TOKEN(ONCE_POB_TOKEN);
	}

	GET_ITOKEN(INTERPOLATE_POB_TOKEN);
	GET_ITOKEN(MAP_TYPE_POB_TOKEN);

	PRINT_END;

	my_print( "%s ", get_token_str(BUMP_SIZE_POB_TOKEN));print_double(amount);PRINT_NEWLINE;

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case PATTERN_MOD_POB_TOKEN:
				{
					parse_pattern_mod(&chunk);
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);

	PRINT_END;
}

/* patterned normal */

void parse_pattern_normal (CHUNK *mainchunk)
{
	CHUNK chunk;
	P_DOUBLE amount;
	P_WORD blend_map;
	P_WORD count;
	P_DOUBLE index;
	P_DVECTOR v;
	P_WORD i;

	my_print( "%s ", get_token_str(NORMAL_POB_TOKEN));PRINT_NEWLINE;
	PRINT_BEGIN;

	amount = read_double();
	
	parse_pattern();

	blend_map = read_word();

	switch(blend_map)
	{
		case POB_NORMAL_NOMAP: // no blend_map
		{
		}
		break;

		case POB_NORMAL_NORMALMAP: // blend_map
		{
			count = read_word();

			if(count)
			{
				my_print( "%s ", get_token_str(NORMAL_MAP_POB_TOKEN));PRINT_NEWLINE;
				PRINT_BEGIN;

				for(i = 0;i < count;i++)
				{
					start_chunk (&chunk, mainchunk);
					index = read_double();
					end_chunk (&chunk);
					my_print( "[ ");
					print_double(index);

					start_chunk (&chunk, mainchunk);
					parse_obj_texture(&chunk);
					end_chunk (&chunk);

					my_print( "] ");
					PRINT_NEWLINE;
				}

				PRINT_END;
			}
		}
		break;

		case POB_NORMAL_SLOPEMAP: // slope_map
		{
			count = read_word();

			if(count)
			{
				my_print( "%s ", get_token_str(SLOPE_MAP_POB_TOKEN));PRINT_NEWLINE;
				PRINT_BEGIN;

				for(i = 0;i < count;i++)
				{
					index = read_double();
					read_vector_uv (v);

					my_print( "[ ");
                        		print_double(index);
					print_uv_vector(v);
                        		my_print( "] ");
                        		PRINT_NEWLINE;
				}
				
				PRINT_END;
			}
		 }
		 break;
	}

	my_print( "%s ", get_token_str(BUMP_SIZE_POB_TOKEN));print_double(amount);PRINT_NEWLINE;

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case PATTERN_MOD_POB_TOKEN:
				{
					parse_pattern_mod(&chunk);
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);

	PRINT_END;
}

/* simple finish */

void parse_simple_finish (CHUNK *mainchunk)
{
	my_print( "%s ", get_token_str(FINISH_POB_TOKEN));PRINT_NEWLINE;
	PRINT_BEGIN;

	GET_CTOKEN(AMBIENT_POB_TOKEN);
	GET_DTOKEN(BRILLIANCE_POB_TOKEN);
	GET_DTOKEN(DIFFUSE_POB_TOKEN);
	GET_CTOKEN(REFLECTION_POB_TOKEN);
	GET_DTOKEN(REFRACTION_POB_TOKEN);
	GET_DTOKEN(IOR_POB_TOKEN);
	GET_DTOKEN(PHONG_POB_TOKEN);
	GET_DTOKEN(PHONG_SIZE_POB_TOKEN);
	GET_DTOKEN(SPECULAR_POB_TOKEN);
	GET_DTOKEN(ROUGHNESS_POB_TOKEN);
	GET_DTOKEN(METALLIC_POB_TOKEN);
	GET_DTOKEN(CAUSTICS_POB_TOKEN);
	GET_DTOKEN(CRAND_POB_TOKEN);

	my_print( "%s ", get_token_str(IRID_POB_TOKEN));PRINT_NEWLINE;
        PRINT_BEGIN;
	print_double( read_double() );PRINT_NEWLINE; // irid value
	GET_DTOKEN(THICKNESS_POB_TOKEN);
	GET_VTOKEN(TURBULENCE_POB_TOKEN);
	PRINT_END;

	GET_DTOKEN(FADE_DISTANCE_POB_TOKEN);
	GET_DTOKEN(FADE_POWER_POB_TOKEN);

	PRINT_END;
}

/* halo render type */

void parse_halo_render_type(void)
{
	P_WORD render_type;

	render_type = read_word();
	my_print( "%s ", get_token_str(render_type));PRINT_NEWLINE;
}

/* halo type */

void parse_halo_type(void)
{
	P_WORD halo_type;

	halo_type = read_word();
	if(halo_type != HALO_NO_HALO_POB_TOKEN)
		my_print( "%s ", get_token_str(halo_type));PRINT_NEWLINE;
}

/* halo mapping type */

void parse_halo_mapping_type(void)
{
	P_WORD mapping_type;

	mapping_type = read_word();
	my_print( "%s ", get_token_str(mapping_type));PRINT_NEWLINE;
}

void parse_simple_halo (CHUNK *mainchunk)
{
	P_WORD count_color_map;
	P_DOUBLE index;
	P_WORD i;
	P_WORD turb_active;

	my_print( "%s ", get_token_str(HALO_POB_TOKEN));PRINT_NEWLINE;
	PRINT_BEGIN;

	parse_halo_render_type();
	parse_halo_type();
	parse_halo_mapping_type();

	GET_DTOKEN(SAMPLES_POB_TOKEN);
	GET_ITOKEN(AA_LEVEL_POB_TOKEN);
	GET_DTOKEN(AA_THRESHOLD_POB_TOKEN);
	GET_DTOKEN(JITTER_POB_TOKEN);
	GET_ITOKEN(DUST_TYPE_POB_TOKEN);
	GET_DTOKEN(MAX_VALUE_POB_TOKEN);
	GET_DTOKEN(EXPONENT_POB_TOKEN);
	GET_DTOKEN(ECCENTRICITY_POB_TOKEN);
     	
	count_color_map = read_word();

	if(count_color_map)
	{
		my_print( "%s ", get_token_str(COLOR_MAP_POB_TOKEN));PRINT_NEWLINE;
       		PRINT_BEGIN;
				
		for(i = 0;i < count_color_map;i++)
		{
			index = read_double();
			my_print( "[ ");
                	print_double(index);
			parse_color();
                        my_print( "] ");
                        PRINT_NEWLINE;
		}
		PRINT_END;
	}

	turb_active = read_word();
	if(turb_active)
	{
		GET_VTOKEN(TURBULENCE_POB_TOKEN);
		GET_ITOKEN(OCTAVES_POB_TOKEN);
		GET_DTOKEN(OMEGA_POB_TOKEN);
		GET_DTOKEN(LAMBDA_POB_TOKEN);
	}

	GET_DTOKEN(FREQUENCY_POB_TOKEN);
	GET_DTOKEN(PHASE_POB_TOKEN);

	PRINT_END;
}

/* pattern type */

void parse_pattern(void)
{
	P_WORD pattern_token;

	pattern_token = read_word();
	
	switch(pattern_token)
	{
		case AGATE_POB_TOKEN :
		{
			GET_TOKEN(AGATE_POB_TOKEN);
			GET_DTOKEN(AGATE_TURB_POB_TOKEN);
		}
		break;

		case BOZO_POB_TOKEN :
		{
			GET_TOKEN(BOZO_POB_TOKEN);
		}
		break;
		case GRANITE_POB_TOKEN :
		{
			GET_TOKEN(GRANITE_POB_TOKEN);
		}
		break;
		case LEOPARD_POB_TOKEN :
		{
			GET_TOKEN(LEOPARD_POB_TOKEN);
		}
		break;
		case MARBLE_POB_TOKEN :
		{
			GET_TOKEN(MARBLE_POB_TOKEN);
		}
		break;
		case ONION_POB_TOKEN :
		{
			GET_TOKEN(ONION_POB_TOKEN);
		}
		break;
		case PATTERN1_POB_TOKEN :
		{
			GET_TOKEN(PATTERN1_POB_TOKEN);
		}
		break;
		case PATTERN2_POB_TOKEN :
		{
			GET_TOKEN(PATTERN2_POB_TOKEN);
		}
		break;
		case PATTERN3_POB_TOKEN :
		{
			GET_TOKEN(PATTERN3_POB_TOKEN);
		}
		break;
		case BUMPY1_POB_TOKEN :
		{
			GET_TOKEN(BUMPY1_POB_TOKEN);
		}
		break;
		case BUMPY2_POB_TOKEN :
		{
			GET_TOKEN(BUMPY2_POB_TOKEN);
		}
		break;
		case BUMPY3_POB_TOKEN :
		{
			GET_TOKEN(BUMPY3_POB_TOKEN);
		}
		break;
		case SPIRAL1_POB_TOKEN :
		{
			GET_DTOKEN(SPIRAL1_POB_TOKEN);
		}
		break;
		case SPIRAL2_POB_TOKEN :
		{
			GET_DTOKEN(SPIRAL2_POB_TOKEN);
		}
		break;
		case SPOTTED_POB_TOKEN :
		{
			GET_TOKEN(SPOTTED_POB_TOKEN);
		}
		break;
		case WOOD_POB_TOKEN :
		{
			GET_TOKEN(WOOD_POB_TOKEN);
		}
		break;
		case GRADIENT_POB_TOKEN :
		{
			GET_VTOKEN(GRADIENT_POB_TOKEN);
		}
		break;
		case RADIAL_POB_TOKEN :
		{
			GET_TOKEN(RADIAL_POB_TOKEN);
		}
		break;
		case CRACKLE_POB_TOKEN :
		{
			GET_TOKEN(CRACKLE_POB_TOKEN);
		}
		break;
		case WAVES_POB_TOKEN :
		{
			GET_TOKEN(WAVES_POB_TOKEN);
		}
		break;
		case RIPPLES_POB_TOKEN :
		{
			GET_TOKEN(RIPPLES_POB_TOKEN);
		}
		break;
		case WRINKLES_POB_TOKEN :
		{
			GET_TOKEN(WRINKLES_POB_TOKEN);
		}
		break;
		case BUMPS_POB_TOKEN :
		{
			GET_TOKEN(BUMPS_POB_TOKEN);
		}
		break;
		case DENTS_POB_TOKEN :
		{
			GET_TOKEN(DENTS_POB_TOKEN);
		}
		break;
		case QUILTED_POB_TOKEN :
		{
			GET_TOKEN(QUILTED_POB_TOKEN);
			GET_DTOKEN(CONTROL0_POB_TOKEN);
			GET_DTOKEN(CONTROL1_POB_TOKEN);
		}
		break;
		case AVERAGE_POB_TOKEN :
		{
			GET_TOKEN(AVERAGE_POB_TOKEN);
		}
		break;

	}

}

/* pattern modifier */

void parse_pattern_mod(CHUNK *mainchunk)
{
	CHUNK chunk;

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case WAVE_TYPE_POB_TOKEN:
				{
        				my_print( "%s ", get_token_str(read_word()));PRINT_NEWLINE;
				}
				break;
				case CLASSIC_TURB_WARP_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(WARP_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					GET_VTOKEN(TURBULENCE_POB_TOKEN);
					GET_ITOKEN(OCTAVES_POB_TOKEN);
					GET_DTOKEN(OMEGA_POB_TOKEN);
					GET_DTOKEN(LAMBDA_POB_TOKEN);
					PRINT_END;
				}
				break;
				case REPEAT_WARP_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(WARP_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
        				GET_VTOKEN(REPEAT_POB_TOKEN);
					GET_VTOKEN(OFFSET_POB_TOKEN);
					GET_VTOKEN(FLIP_POB_TOKEN);

					PRINT_END;
				}
				break;
				case BLACK_HOLE_WARP_POB_TOKEN:
				{
					P_DVECTOR v;
					P_DOUBLE d;
			
					my_print( "%s ", get_token_str(WARP_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
	      				my_print( "%s ", get_token_str(BLACK_HOLE_POB_TOKEN));PRINT_NEWLINE;
					read_vector(v);
					d = read_double();
					print_vector(v);PRINT_COMMA;print_double(d);PRINT_NEWLINE;
					GET_DTOKEN(STRENGTH_POB_TOKEN);
					GET_DTOKEN(FALLOFF_POB_TOKEN);

					if( read_word() ) // inverted
					{
						GET_TOKEN(INVERSE_POB_TOKEN)
					}

					GET_ITOKEN(TYPE_POB_TOKEN);

					if( read_word() ) // repeat
					{
						GET_VTOKEN(REPEAT_POB_TOKEN);
					}

					if( read_word() ) // Uncertain
					{
						GET_VTOKEN(TURBULENCE_POB_TOKEN);
					}

					PRINT_END;
				}
				break;
				case FREQUENCY_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(FREQUENCY_POB_TOKEN));
					print_double(read_double());
					PRINT_NEWLINE;
				}
				break;
				case PHASE_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(PHASE_POB_TOKEN));
					print_double(read_double());
					PRINT_NEWLINE;
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);
}


/* object section */

void parse_object_section (CHUNK *mainchunk)
{
	CHUNK chunk;

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case BICUBIC_PATCH_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(BICUBIC_PATCH_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_bicubic_patch(&chunk);
					PRINT_END;
				}
				break;
				case BLOB_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(BLOB_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_blob(&chunk);
					PRINT_END;
				}
				break;
				case BOX_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(BOX_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_box(&chunk);
					PRINT_END;
				}
				break;
				case CYLINDER_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(CYLINDER_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_cylinder(&chunk);
					PRINT_END;
				}
				break;
				case CONE_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(CONE_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_cone(&chunk);
					PRINT_END;
				}
				break;
				case UNION_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(UNION_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_group(&chunk);
					PRINT_END;
				}
				break;
				case MERGE_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(MERGE_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_group(&chunk);
					PRINT_END;
				}
				break;
				case INTERSECTION_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(INTERSECTION_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_group(&chunk);
					PRINT_END;
				}
				break;
				case DIFFERENCE_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(DIFFERENCE_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_group(&chunk);
					PRINT_END;
				}
				break;
				case DISC_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(DISC_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_disc(&chunk);
					PRINT_END;
				}
				break;
				case JULIA_FRACTAL_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(JULIA_FRACTAL_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_fractal(&chunk);
					PRINT_END;
				}
				break;
				case HEIGHT_FIELD_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(HEIGHT_FIELD_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_heightfield(&chunk);
					PRINT_END;
				}
				break;
				case LATHE_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(LATHE_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_lathe(&chunk);
					PRINT_END;
				}
				break;
				case MESH_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(MESH_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_mesh(&chunk);
					PRINT_END;
				}
				break;
				case PLANE_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(PLANE_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_plane(&chunk);
					PRINT_END;
				}
				break;
				case POLY_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(POLY_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_poly(&chunk);
					PRINT_END;
				}
				break;
				case POLYGON_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(POLYGON_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_polygon(&chunk);
					PRINT_END;
				}
				break;
				case PRISM_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(PRISM_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_prism(&chunk);
					PRINT_END;
				}
				break;
				case QUADRIC_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(QUADRIC_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_quadric(&chunk);
					PRINT_END;
				}
				break;
				case SOR_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(SOR_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_sor(&chunk);
					PRINT_END;
				}
				break;
				case SPHERE_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(SPHERE_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_sphere(&chunk);
					PRINT_END;
				}
				break;
				case SUPERELLIPSOID_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(SUPERELLIPSOID_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_superellipsoid(&chunk);
					PRINT_END;
				}
				break;
				case TORUS_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(TORUS_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_torus(&chunk);
					PRINT_END;
				}
				break;
				case TRIANGLE_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(TRIANGLE_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_triangle(&chunk);
					PRINT_END;
				}
				break;
				case SMOOTH_TRIANGLE_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(SMOOTH_TRIANGLE_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_smoothtriangle(&chunk);
					PRINT_END;
				}
				break;
				case TEXT_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(TEXT_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_text(&chunk);
					PRINT_END;
				}
				break;
				case LIGHT_SOURCE_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(LIGHT_SOURCE_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_light_source(&chunk);
					PRINT_END;
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);
}

/* camera */

void parse_camera (CHUNK *mainchunk)
{
	P_WORD type;
	P_WORD sub_type;
	P_DOUBLE used_Angle;
	P_DOUBLE focal_len;
	P_WORD blur_samples;
	P_DVECTOR focal_distance;
	
	CHUNK chunk;

	PRINT_NAME;
	
	type = read_word();
        my_print( "%s ", get_token_str(type));
	if(type == CYLINDER_POB_TOKEN)
	{
		sub_type = read_word();
		print_word(sub_type);
	}
	PRINT_NEWLINE;

	GET_VTOKEN(LOCATION_POB_TOKEN);
	GET_VTOKEN(LOOK_AT_POB_TOKEN);
	GET_VTOKEN(SKY_POB_TOKEN);
	GET_VTOKEN(UP_POB_TOKEN);
	GET_VTOKEN(DIRECTION_POB_TOKEN);
	GET_VTOKEN(RIGHT_POB_TOKEN);
	
	GET_DTOKEN(CONFIDENCE_POB_TOKEN);
	GET_DTOKEN(VARIANCE_POB_TOKEN);
	GET_DTOKEN(APERTURE_POB_TOKEN);
	
	blur_samples = read_word();
	if(blur_samples > 0)
	{
		my_print( "%s ", get_token_str(BLUR_SAMPLES_POB_TOKEN));
		print_word(blur_samples);
		PRINT_NEWLINE;
	}

	used_Angle = read_double();
	if(used_Angle > -1.0)
	{
		my_print( "%s ", get_token_str(ANGLE_POB_TOKEN));
		print_double(used_Angle);
		PRINT_NEWLINE;
	}

	focal_len = read_double();
	read_vector(focal_distance);
        if(focal_len > 0.0)
        {
		my_print( "%s ", get_token_str(FOCAL_POINT_POB_TOKEN));
		print_vector(focal_distance);
		PRINT_NEWLINE;
        }

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case NORMALREF_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(NORMAL_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_obj_texture(&chunk);
					PRINT_END;
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);
}

/* object modifier */

void parse_object_mod(CHUNK *mainchunk)
{
	CHUNK chunk;

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case FLAGS_POB_TOKEN:
				{
					P_WORD flags;
					flags = read_word();
					print_flags(flags);
				}
				break;

				case MATRIX_POB_TOKEN:
				{
					P_MATRIX m;
					read_matrix(m);
					print_matrix(m);
				}
				break;

				case TEXTUREREF_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(TEXTURE_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_obj_texture(&chunk);
					PRINT_END;
				}
				break;

				case BOUNDED_BY_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(BOUNDED_BY_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_object_section(&chunk);
					PRINT_END;
				}
				break;
				case CLIPPED_BY_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(CLIPPED_BY_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_object_section(&chunk);
					PRINT_END;
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);
}

/* bicubic patch */

void parse_bicubic_patch (CHUNK *mainchunk)
{
	CHUNK chunk;
	unsigned int i,j;
	P_DVECTOR tmp;

	PRINT_NAME;

	GET_ITOKEN(TYPE_POB_TOKEN);
	GET_DTOKEN(FLATNESS_POB_TOKEN);
	GET_ITOKEN(U_STEPS_POB_TOKEN);
	GET_ITOKEN(V_STEPS_POB_TOKEN);

	for (i = 0; i < 4; i++)
        {
                for (j = 0; j < 4; j++)
		{
			read_vector(tmp);
			print_vector(tmp);
                        if (!((i==3)&&(j==3)))
                                PRINT_COMMA;
                }
                PRINT_NEWLINE;
        }
        PRINT_NEWLINE;

	parse_object_mod(mainchunk);
}

/* blob sphere */

void parse_blob_sphere (CHUNK *mainchunk)
{
	P_DVECTOR v;
	P_DOUBLE radius;

	PRINT_NAME;
	read_vector (v);
	radius = read_double ();
	print_vector(v);
	PRINT_COMMA;
	print_double(radius);
	PRINT_COMMA;
	GET_DTOKEN(STRENGTH_POB_TOKEN);
	PRINT_NEWLINE;

	parse_object_mod(mainchunk);
}

/* blob cylinder */

void parse_blob_cylinder (CHUNK *mainchunk)
{
	P_DVECTOR v1;
	P_DVECTOR v2;
	P_DOUBLE radius;

	PRINT_NAME;
	read_vector (v1);
	read_vector (v2);
	radius = read_double ();
	print_vector(v1);
	PRINT_COMMA;
	print_vector(v2);
	PRINT_COMMA;
	print_double(radius);
	PRINT_COMMA;
	GET_DTOKEN(STRENGTH_POB_TOKEN);
	PRINT_NEWLINE;

	parse_object_mod(mainchunk);
}

/* blob components */

void parse_blob_components (CHUNK *mainchunk)
{
	CHUNK chunk;
	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case BLOBCYLINDER_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(CYLINDER_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_blob_cylinder(&chunk);
					PRINT_END;
				}
				break;
				case BLOBSPHERE_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(SPHERE_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_blob_sphere(&chunk);
					PRINT_END;
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);

}

/* blob */

void parse_blob (CHUNK *mainchunk)
{
	CHUNK chunk;

	PRINT_NAME;

	GET_DTOKEN(THRESHOLD_POB_TOKEN);
	read_word(); // component number
	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case BLOBCOMPONENT_POB_TOKEN:
				{
					parse_blob_components(&chunk);
				}
				break;

				case BLOBMOD_POB_TOKEN:
				{
					parse_object_mod(&chunk);
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);

}

/* box */

void parse_box (CHUNK *mainchunk)
{
	P_DVECTOR v1;
	P_DVECTOR v2;

	PRINT_NAME;

	read_vector (v1);
	read_vector (v2);

	print_vector(v1);
	PRINT_COMMA;
	print_vector(v2);
	PRINT_NEWLINE;

	parse_object_mod(mainchunk);
}

/* cylinder */

void parse_cylinder (CHUNK *mainchunk)
{
	P_DVECTOR v1;
	P_DVECTOR v2;
	P_DOUBLE radius;

	PRINT_NAME;
	read_vector (v1);
	read_vector (v2);
	radius = read_double ();

	print_vector(v1);
	PRINT_COMMA;
	print_vector(v2);
	PRINT_COMMA;
	print_double(radius);
	PRINT_NEWLINE;

	parse_object_mod(mainchunk);
}

/* cone */

void parse_cone (CHUNK *mainchunk)
{
	P_DVECTOR v1;
	P_DVECTOR v2;
	P_DOUBLE radius1;
	P_DOUBLE radius2;

	PRINT_NAME;
	read_vector (v1);
	radius1 = read_double ();
	read_vector (v2);
	radius2 = read_double ();

	print_vector(v1);
	PRINT_COMMA;
	print_double(radius1);
	PRINT_COMMA;
	print_vector(v2);
	PRINT_COMMA;
	print_double(radius2);
	PRINT_NEWLINE;

	parse_object_mod(mainchunk);
}

/* Union, Merge, Difference, Intersection */

void parse_group (CHUNK *mainchunk)
{
	CHUNK chunk;

	PRINT_NAME;

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case GROUP_POB_TOKEN:
				{
					parse_object_section (&chunk);
				}
				break;

				case GROUPMOD_POB_TOKEN:
				{
					parse_object_mod(&chunk);
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);
}

/* disc */

void parse_disc (CHUNK *mainchunk)
{
	P_DVECTOR v1;
	P_DVECTOR v2;
	P_DOUBLE radius1;
	P_DOUBLE radius2;

	PRINT_NAME;

	read_vector (v1);
	read_vector (v2);
	radius1 = read_double ();
	radius2 = read_double ();

	print_vector(v1);
	PRINT_COMMA;
	print_vector(v2);
	PRINT_COMMA;
	print_double(radius1);
	PRINT_COMMA;
	print_double(radius2);
	PRINT_NEWLINE;

	parse_object_mod(mainchunk);
}

static unsigned int fractal_sub_type[18] =
{
        EXP_POB_TOKEN,
        LOG_POB_TOKEN,
        SIN_POB_TOKEN,
        ASIN_POB_TOKEN,
        COS_POB_TOKEN,
        ACOS_POB_TOKEN,
        TAN_POB_TOKEN,
        ATAN_POB_TOKEN,
        SINH_POB_TOKEN,
        ASINH_POB_TOKEN,
        COSH_POB_TOKEN,
        ACOSH_POB_TOKEN,
        TANH_POB_TOKEN,
        ATANH_POB_TOKEN,
        PWR_POB_TOKEN,
        SQR_POB_TOKEN,
        CUBE_POB_TOKEN,
        RECIPROCAL_POB_TOKEN
};

static unsigned int fractal_algebra_type[2] =
{
        QUATERNION_POB_TOKEN,
        HYPERCOMPLEX_POB_TOKEN
};

/* fractal */

void parse_fractal (CHUNK *mainchunk)
{
	P_WORD algebra_type;	
	P_WORD sub_type;
	P_DOUBLE d_tmp[4];


	PRINT_NAME;
	
	d_tmp[0] = read_double();
	d_tmp[1] = read_double();
	d_tmp[2] = read_double();
	d_tmp[3] = read_double();

        my_print( "<%.6f,%.6f,%.6f,%.6f>", d_tmp[0],
                                           d_tmp[1],
                                           d_tmp[2],
                                           d_tmp[3]);
	PRINT_NEWLINE;

	algebra_type = read_word();
        my_print( "%s ", get_token_str(fractal_algebra_type[algebra_type]));
        PRINT_NEWLINE;
	
	sub_type = read_word();
        my_print( "%s ", get_token_str(fractal_sub_type[sub_type]));
	if (sub_type == 14) /* PWR_STYPE */
        {
                my_print( "(");
                print_double(read_double());
                PRINT_COMMA;                        
                print_double(read_double());
                my_print( ")");
        }
        PRINT_NEWLINE;

	GET_ITOKEN(MAX_ITERATION_POB_TOKEN);
	GET_DTOKEN(PRECISION_POB_TOKEN);

        my_print( "%s ", get_token_str(SLICE_POB_TOKEN));
	d_tmp[0] = read_double();
	d_tmp[1] = read_double();
	d_tmp[2] = read_double();
	d_tmp[3] = read_double();

        my_print( "<%.6f,%.6f,%.6f,%.6f>", d_tmp[0],
                                           d_tmp[1],
                                           d_tmp[2],
                                           d_tmp[3]);
        PRINT_COMMA;
        print_double(read_double());
        PRINT_NEWLINE;

	parse_object_mod(mainchunk);
}

/* color */

void parse_color(void)
{
	P_DOUBLE r,g,b,f,t;

	r = read_double();
	g = read_double();
	b = read_double();
	f = read_double();
	t = read_double();

        my_print( "%s ", get_token_str(COLOR_POB_TOKEN));

        my_print( "%s ", get_token_str(RGBFT_POB_TOKEN));
        my_print("<%.3f, %.3f, %.3f, %.3f, %.3f>",r,g,b,f,t);
}

/* image */

void parse_image(void)
{
	P_WORD type;
	char *name;

	type = read_word();
	name = read_string();

	my_print( "%s ", get_token_str(type));
	my_print( "\"%s\"",name);
	PRINT_NEWLINE;
	
}

/* height_field */

void parse_heightfield (CHUNK *mainchunk)
{

	PRINT_NAME;
	parse_image();
	GET_DTOKEN(WATER_LEVEL_POB_TOKEN);
	parse_object_mod(mainchunk);
}

static unsigned int lathe_spline_type[3] =
{
        LINEAR_SPLINE_POB_TOKEN,
        QUADRATIC_SPLINE_POB_TOKEN,
        CUBIC_SPLINE_POB_TOKEN
            
};

/* lathe */

void parse_lathe (CHUNK *mainchunk)
{
	P_WORD spline_type;
	P_WORD rNumber;
	P_WORD i;
	P_DVECTOR v;

	PRINT_NAME;
	spline_type = read_word();
        my_print( "%s ", get_token_str(lathe_spline_type[spline_type]));
        PRINT_NEWLINE;
	
	rNumber = read_word();
        print_word(rNumber);
        PRINT_COMMA;
        PRINT_NEWLINE;

        for(i = 0; i < rNumber;i++)
        {
		read_vector_uv (v);
                print_uv_vector(v);
                if(i != (rNumber - 1))
                        PRINT_COMMA;
                if((i % 2) == 1)
                        PRINT_NEWLINE;
        }
	PRINT_NEWLINE;

	parse_object_mod(mainchunk);
}

/* smoothed faces */

void parse_mesh_s_tri (CHUNK *mainchunk)
{
	CHUNK chunk;
	P_DVECTOR v;
		
	PRINT_NAME;
	
	read_vector(v);print_vector(v);PRINT_COMMA;
	read_vector(v);print_vector(v);PRINT_COMMA;
	read_vector(v);print_vector(v);PRINT_COMMA;
	read_vector(v);print_vector(v);PRINT_COMMA;
	read_vector(v);print_vector(v);PRINT_COMMA;
	read_vector(v);print_vector(v);PRINT_NEWLINE;

	/* texture */
	parse_object_mod(mainchunk);
}

/* faces */

void parse_mesh_tri (CHUNK *mainchunk)
{
	CHUNK chunk;
	P_DVECTOR v;
		
	PRINT_NAME;
	
	read_vector(v);print_vector(v);PRINT_COMMA;
	read_vector(v);print_vector(v);PRINT_COMMA;
	read_vector(v);print_vector(v);PRINT_NEWLINE;
	
	/* texture */
	parse_object_mod(mainchunk);
}

/* mesh data*/

void parse_mesh_data (CHUNK *mainchunk)
{
	CHUNK chunk;

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case SMOOTH_TRIANGLE_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(SMOOTH_TRIANGLE_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_mesh_s_tri(&chunk);
					PRINT_END;
				}
				break;

				case TRIANGLE_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(TRIANGLE_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_mesh_tri(&chunk);
					PRINT_END;
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);
}
	
/* mesh */

void parse_mesh (CHUNK *mainchunk)
{
	CHUNK chunk;

	PRINT_NAME;

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case MESHDATA_POB_TOKEN:
				{
					read_word(); // face number
					parse_mesh_data (&chunk);
				}
				break;

				case MESHMOD_POB_TOKEN:
				{
					parse_object_mod(&chunk);
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);
}

/* plane */

void parse_plane (CHUNK *mainchunk)
{
	P_DVECTOR v;
	P_DOUBLE dist;


	PRINT_NAME;
	read_vector (v);
	dist = read_double ();

	print_vector(v);
	PRINT_COMMA;
	print_double(dist);
	PRINT_NEWLINE;

	parse_object_mod(mainchunk);
}

#define term_counts(x) (((x)+1)*((x)+2)*((x)+3)/6)

/* poly */

void parse_poly (CHUNK *mainchunk)
{
	P_WORD order;
	P_WORD i;
	P_DOUBLE val;


	PRINT_NAME;
	
	order = read_word ();
        print_word(order);
        PRINT_COMMA;
        PRINT_NEWLINE;

        my_print( "<");
        for(i = 0; i < term_counts(order);i++)
        {
        	val = read_double();
		print_double(val);
                if(i != (term_counts(order) - 1))
                        PRINT_COMMA;
                if((i % 4) == 3)
                        PRINT_NEWLINE;                         
        }
        my_print( ">");
        PRINT_NEWLINE;

	parse_object_mod(mainchunk);
}

/* polygon */

void parse_polygon (CHUNK *mainchunk)
{
	P_WORD Number;
	P_WORD i;
	P_DVECTOR v;

	PRINT_NAME;
	
	Number = read_word();
        print_word(Number);
        PRINT_COMMA;
        PRINT_NEWLINE;

        for(i = 0; i < Number;i++)
        {
		read_vector_uv (v);
                print_uv_vector(v);
                if(i != (Number - 1))
                        PRINT_COMMA;
                if((i % 2) == 1)
                        PRINT_NEWLINE;
        }
        PRINT_NEWLINE;

	parse_object_mod(mainchunk);
}

static unsigned int prism_spline_type[3] =
{
        LINEAR_SPLINE_POB_TOKEN,
        QUADRATIC_SPLINE_POB_TOKEN,
        CUBIC_SPLINE_POB_TOKEN
            
};

static unsigned int prism_sweep_type[2] =
{
        LINEAR_SWEEP_POB_TOKEN,
        CONIC_SWEEP_POB_TOKEN
};

/* prism */

void parse_prism (CHUNK *mainchunk)
{
	P_WORD Spline_Type;
	P_WORD Sweep_Type;
	P_DOUBLE Height1;
	P_DOUBLE Height2;
	P_WORD rNumber;
	P_DVECTOR v;
	P_WORD i;

	PRINT_NAME;
	
	Spline_Type = read_word();
	my_print( "%s ", get_token_str(prism_spline_type[Spline_Type]));PRINT_NEWLINE;
	Sweep_Type = read_word();
	my_print( "%s ", get_token_str(prism_sweep_type[Sweep_Type]));PRINT_NEWLINE;

        Height1 = read_double();
	print_double(Height1);
        PRINT_COMMA;
        Height2 = read_double();
        print_double(Height2);
        PRINT_COMMA;
        PRINT_NEWLINE;
        
	rNumber = read_word();
	print_word(rNumber);
        PRINT_COMMA;
        PRINT_NEWLINE;

        for(i = 0; i < rNumber;i++)
        {
		read_vector_uv(v);
		print_uv_vector(v);
                if(i != (rNumber - 1))
                        PRINT_COMMA;
                if((i % 2) == 1)
                        PRINT_NEWLINE;
        }

        PRINT_NEWLINE;

	parse_object_mod(mainchunk);
}

/* quadric */

void parse_quadric (CHUNK *mainchunk)
{
	P_DVECTOR Square_Terms;
	P_DVECTOR Mixed_Terms;
	P_DVECTOR Terms;
	P_DOUBLE Constant;

	PRINT_NAME;
	read_vector(Square_Terms);
        print_vector(Square_Terms);
        PRINT_COMMA;
	read_vector(Mixed_Terms);
        print_vector(Mixed_Terms);
        PRINT_COMMA;
	read_vector(Terms);
        print_vector(Terms);
        PRINT_COMMA;
	Constant = read_double();
        print_double(Constant);
        PRINT_NEWLINE;

	parse_object_mod(mainchunk);
}

/* sor */

void parse_sor (CHUNK *mainchunk)
{
	P_WORD rNumber;
	P_WORD i;
	P_DVECTOR v;

	PRINT_NAME;

	rNumber = read_word();
        print_word(rNumber);
        PRINT_COMMA;
        PRINT_NEWLINE;

        for(i = 0; i < rNumber;i++)
        {
		read_vector_uv (v);
                print_uv_vector(v);
                if(i != (rNumber - 1))
                        PRINT_COMMA;
                if((i % 2) == 1)
                        PRINT_NEWLINE;
        }
	PRINT_NEWLINE;

	parse_object_mod(mainchunk);
}

/* sphere */

void parse_sphere (CHUNK *mainchunk)
{
	P_DVECTOR v;
	P_DOUBLE radius;


	PRINT_NAME;
	read_vector (v);
	radius = read_double ();

	print_vector(v);
	PRINT_COMMA;
	print_double(radius);
	PRINT_NEWLINE;

	parse_object_mod(mainchunk);
}

/* superelliopsoid */

void parse_superellipsoid (CHUNK *mainchunk)
{
	P_DOUBLE p1;
	P_DOUBLE p2;

	PRINT_NAME;

        my_print("< ");
        
	p1 = read_double();
	print_double(p1);
        PRINT_COMMA;
	p2 = read_double();
	print_double(p2);
        my_print(">");

        PRINT_NEWLINE;

	parse_object_mod(mainchunk);
}

/* torus */

void parse_torus (CHUNK *mainchunk)
{
	P_DOUBLE r1;
	P_DOUBLE r2;

	PRINT_NAME;

	r1 = read_double ();
	print_double(r1);
	PRINT_COMMA;
	r2 = read_double ();
	print_double(r2);
	PRINT_NEWLINE;

	parse_object_mod(mainchunk);
}

/* triangle */

void parse_triangle (CHUNK *mainchunk)
{
	P_DVECTOR v1;
	P_DVECTOR v2;
	P_DVECTOR v3;

	PRINT_NAME;

	read_vector(v1);
	print_vector(v1);
	PRINT_COMMA;
	read_vector(v2);
	print_vector(v2);
	PRINT_COMMA;
	read_vector(v3);
	print_vector(v3);
	PRINT_NEWLINE;

	parse_object_mod(mainchunk);
}

/* smoothed triangle */

void parse_smoothtriangle (CHUNK *mainchunk)
{
	P_DVECTOR v1;
	P_DVECTOR v2;
	P_DVECTOR v3;
	P_DVECTOR n1;
	P_DVECTOR n2;
	P_DVECTOR n3;

	PRINT_NAME;

	read_vector(v1);
	print_vector(v1);
	PRINT_COMMA;
	read_vector(n1);
	print_vector(n1);
	PRINT_COMMA;
	read_vector(v2);
	print_vector(v2);
	PRINT_COMMA;
	read_vector(n2);
	print_vector(n2);
	PRINT_COMMA;
	read_vector(v3);
	print_vector(v3);
	PRINT_COMMA;
	read_vector(n3);
	print_vector(n3);
	PRINT_NEWLINE;

	parse_object_mod(mainchunk);
}

/* text, ttf */

void parse_text (CHUNK *mainchunk)
{
	P_WORD type;
	char *filename;
	char *string;
	P_DOUBLE depth;
	P_DVECTOR offset;

	PRINT_NAME;

        type = read_word();
	my_print( "%s ", get_token_str(type));
	filename = read_string();
	my_print( "\"%s\"",filename);PRINT_COMMA;PRINT_NEWLINE;
	string = read_string();
        my_print( "\"%s\"",string);PRINT_COMMA;PRINT_NEWLINE;
	depth = read_double();
        print_double(depth);PRINT_COMMA;PRINT_NEWLINE;
	read_vector(offset);
	print_vector(offset);PRINT_NEWLINE;

	parse_object_mod(mainchunk);
}

static unsigned int light_type_token_id[3] =
{
    SPOTLIGHT_POB_TOKEN,
    SHADOWLESS_POB_TOKEN,
    CYLINDER_POB_TOKEN
};

/* looks like */

void parse_lookslike (CHUNK *mainchunk)
{
	CHUNK chunk;
	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case LOOKS_LIKE_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(LOOKS_LIKE_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_object_section (&chunk);
					PRINT_END;
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);

}

/* light source */

void parse_light_source (CHUNK *mainchunk)
{
	P_WORD Light_Type;
	P_WORD Area_Light;
	P_DVECTOR v;
	P_WORD i;
        P_WORD Jitter;
	P_WORD Track;

	PRINT_NAME;

	Light_Type = read_word();

	read_vector(v);
	print_vector(v);
        PRINT_NEWLINE;
        parse_color();
        PRINT_NEWLINE;

        if(Light_Type != POB_POINT_SOURCE)
	{
                my_print( "%s", get_token_str(light_type_token_id[Light_Type - 2]));
		PRINT_NEWLINE;
	}
                
        if ((Light_Type == POB_SPOT_SOURCE) || (Light_Type == POB_CYLINDER_SOURCE))
        {
                GET_VTOKEN(POINT_AT_POB_TOKEN);
		GET_DTOKEN(TIGHTNESS_POB_TOKEN);
		GET_DTOKEN(RADIUS_POB_TOKEN);
		GET_DTOKEN(FALLOFF_POB_TOKEN);
        }

	Area_Light = read_word();
	if(Area_Light == 1)
	{
		GET_TOKEN(AREA_LIGHT_POB_TOKEN);
		read_vector(v);print_vector(v);PRINT_COMMA;PRINT_NEWLINE;
		read_vector(v);print_vector(v);PRINT_COMMA;PRINT_NEWLINE;
		i = read_word();print_word(i);PRINT_COMMA;PRINT_NEWLINE;
		i = read_word();print_word(i);PRINT_NEWLINE;
	}

	GET_DTOKEN(FADE_DISTANCE_POB_TOKEN);
	GET_DTOKEN(FADE_POWER_POB_TOKEN);
	GET_DTOKEN(ATMOSPHERIC_ATTENUATION_POB_TOKEN);
	GET_DTOKEN(ATMOSPHERE_POB_TOKEN);
	GET_ITOKEN(ADAPTIVE_POB_TOKEN);

	Jitter = read_word();
	if(Jitter == 1)
		GET_TOKEN(JITTER_POB_TOKEN);

	Track = read_word();
	if(Track == 1)
		GET_TOKEN(TRACK_POB_TOKEN);

	parse_lookslike(mainchunk);
}

/* global section */

void parse_global_section (CHUNK *mainchunk)
{
	P_WORD gamma_correct;
	P_WORD Number_Of_Waves;
	P_WORD hf_gray_16;
	P_DOUBLE GammaFactor;

	my_print( "%s ", get_token_str(IRID_WAVELENGTH_POB_TOKEN));parse_color();PRINT_NEWLINE;

        gamma_correct = read_word();
	GammaFactor = read_double();
	if(gamma_correct)
	{
		my_print( "%s ", get_token_str(ASSUMED_GAMMA_POB_TOKEN));print_double(GammaFactor );PRINT_NEWLINE;
	}

        GET_ITOKEN(MAX_TRACE_LEVEL_POB_TOKEN);
	GET_ITOKEN(MAX_INTERSECTIONS_POB_TOKEN) ;
        GET_DTOKEN(ADC_BAILOUT_POB_TOKEN);

        Number_Of_Waves = read_word();
	if(Number_Of_Waves > 0)
        {
                my_print( "%s ", get_token_str(NUMBER_OF_WAVES_POB_TOKEN));print_word(Number_Of_Waves);PRINT_NEWLINE;
        }        
        
	my_print( "%s ", get_token_str(AMBIENT_LIGHT_POB_TOKEN));parse_color();PRINT_NEWLINE;

	hf_gray_16 = read_word();
        if(hf_gray_16)
	{
                GET_TOKEN(HF_GRAY_16_POB_TOKEN);
	}

	GET_TOKEN(RADIOSITY_POB_TOKEN);
        PRINT_BEGIN;
	GET_DTOKEN(BRIGHTNESS_POB_TOKEN);
	GET_ITOKEN(COUNT_POB_TOKEN);
	GET_DTOKEN(DISTANCE_MAXIMUM_POB_TOKEN);
	GET_DTOKEN(ERROR_BOUND_POB_TOKEN);
	GET_DTOKEN(GRAY_THRESHOLD_POB_TOKEN);
	GET_DTOKEN(LOW_ERROR_FACTOR_POB_TOKEN);
	GET_DTOKEN(MINIMUM_REUSE_POB_TOKEN);
	GET_ITOKEN(NEAREST_COUNT_POB_TOKEN);
	GET_ITOKEN(RECURSION_LIMIT_POB_TOKEN);
        PRINT_END;
}

/* global environment section */

void parse_global_environment (CHUNK *mainchunk)
{
	CHUNK chunk;
	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case RAINBOW_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(RAINBOW_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_rainbow (&chunk);
					PRINT_END;
				}
				break;
				case SKY_SPHERE_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(SKY_SPHERE_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_sky_sphere (&chunk);
					PRINT_END;
				}
				break;
				case SIMPLEPIGMENT_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;
					parse_simple_pigment(&chunk);
				}
				break;
				case CHECKERPIGMENT_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;
					parse_checker_pigment(&chunk);
				}
				break;
				case BRICKPIGMENT_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;
					parse_brick_pigment(&chunk);
				}
				break;
				case HEXAGONPIGMENT_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;
					parse_hexagon_pigment(&chunk);
				}
				break;
				case BITMAPPIGMENT_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;
					parse_bitmap_pigment(&chunk);
				}
				break;
				case PATTERNPIGMENT_POB_TOKEN:
				{
					char *name;
					name = read_string();
					my_print( "#%s ", get_token_str(DECLARE_POB_TOKEN)); my_print( "%s = ",name);PRINT_NEWLINE;
					parse_pattern_pigment(&chunk);
				}
				break;

				case FOG_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(FOG_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_fog (&chunk);
					PRINT_END;
				}
				break;
				case ATMOSPHERE_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(ATMOSPHERE_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_athmosphere (&chunk);
					PRINT_END;
				}
				break;
				case BACKGROUND_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(BACKGROUND_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					PRINT_NAME;
					parse_color ();PRINT_NEWLINE;
					PRINT_END;
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);

}

/* rainboy */

void parse_rainbow (CHUNK *mainchunk)
{
	P_WORD count;
	P_DOUBLE index;
	P_WORD i;

	PRINT_NAME;

        GET_VTOKEN(DIRECTION_POB_TOKEN);
        GET_DTOKEN(ANGLE_POB_TOKEN);
        GET_DTOKEN(DISTANCE_POB_TOKEN);
        GET_DTOKEN(JITTER_POB_TOKEN);
        GET_DTOKEN(WIDTH_POB_TOKEN);
        GET_VTOKEN(UP_POB_TOKEN);
        GET_DTOKEN(FALLOFF_ANGLE_POB_TOKEN);
        GET_DTOKEN(ARC_ANGLE_POB_TOKEN);

	count = read_word();

	if(count)
	{
		my_print( "%s ", get_token_str(COLOR_MAP_POB_TOKEN));PRINT_NEWLINE;
       		PRINT_BEGIN;
				
		for(i = 0;i < count;i++)
		{
			index = read_double();
			my_print( "[ ");
                	print_double(index);
			parse_color();
                	my_print( "] ");
                	PRINT_NEWLINE;
		}
		PRINT_END;
	}
}

/* sky_sphere */

void parse_sky_sphere (CHUNK *mainchunk)
{
	CHUNK chunk;

	PRINT_NAME;

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case PIGMENTREF_POB_TOKEN:
				{
					my_print( "%s ", get_token_str(PIGMENT_POB_TOKEN));PRINT_NEWLINE;
					PRINT_BEGIN;
					parse_obj_texture(&chunk);
					PRINT_END;
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);

	parse_object_mod(mainchunk);

}

/* fog */

void parse_fog (CHUNK *mainchunk)
{
	P_WORD Turb;
	PRINT_NAME;

        parse_color();PRINT_NEWLINE;

        GET_DTOKEN(DISTANCE_POB_TOKEN);
        GET_ITOKEN(FOG_TYPE_POB_TOKEN);
        GET_DTOKEN(FOG_ALT_POB_TOKEN);
        GET_DTOKEN(FOG_OFFSET_POB_TOKEN);
        GET_DTOKEN(TURB_DEPTH_POB_TOKEN);
	Turb = read_word();
	if(Turb)
        {
        	GET_VTOKEN(TURBULENCE_POB_TOKEN);
        	GET_ITOKEN(OCTAVES_POB_TOKEN);
        	GET_DTOKEN(OMEGA_POB_TOKEN);
        	GET_DTOKEN(LAMBDA_POB_TOKEN);
        }        
	GET_VTOKEN(UP_POB_TOKEN);
}

/* athmosphere */

void parse_athmosphere (CHUNK *mainchunk)
{
	PRINT_NAME;

        GET_ITOKEN(TYPE_POB_TOKEN);
	GET_DTOKEN(DISTANCE_POB_TOKEN);
	GET_ITOKEN(SAMPLES_POB_TOKEN);
	GET_DTOKEN(SCATTERING_POB_TOKEN);
	GET_ITOKEN(AA_LEVEL_POB_TOKEN);
	GET_DTOKEN(AA_THRESHOLD_POB_TOKEN);
	GET_DTOKEN(JITTER_POB_TOKEN);
	GET_DTOKEN(ECCENTRICITY_POB_TOKEN);

	parse_color();PRINT_NEWLINE;
}


/* object texture */

void parse_obj_texture(CHUNK *mainchunk)
{
	CHUNK chunk;

	my_print("%s ", read_string()); // Texture name
	PRINT_NEWLINE;

	do
	{
		start_chunk (&chunk, mainchunk);

		if (chunk.end <= mainchunk->end)
		{
			switch (chunk.tag)
			{
				case MATRIX_POB_TOKEN:
				{
					P_MATRIX m;
					read_matrix(m);
					print_matrix(m);
				}
				break;
			}
		}
		end_chunk (&chunk);
	}
	while (chunk.end <= mainchunk->end);

}

/********************** read datatypes stuff **************/

/* read simple byte */
P_BYTE read_byte( void )
{
    P_BYTE data;

    data = (fgetc (fptr) & 0xFF);

#ifdef __POB_DEBUG2
    if(dbg_fpt)
    {
	set_tab(dbg_fpt);
	fprintf(dbg_fpt, "Byte: 0x%02X\n", data);
    }
#endif // __POB_DEBUG2
    return data;
}

/* read word LOW/HIGH Byte order */
P_WORD read_word( void )
{
    P_WORD data;

    data = (fgetc (fptr) & 0xFF);
    data |= ((fgetc (fptr) & 0xFF) << 8);

#ifdef __POB_DEBUG2
    if(dbg_fpt)
    {
	set_tab(dbg_fpt);
	fprintf(dbg_fpt, "Word: 0x%04X\n", data);
    }
#endif // __POB_DEBUG2
    return data;
}

/* read dword LOW/HIGH Byte order */
P_DWORD read_dword( void )
{
    P_DWORD data;

    data = (fgetc (fptr) & 0xFF);
    data |= ((fgetc (fptr) & 0xFF) << 8);
    data |= ((fgetc (fptr) & 0xFF) << 16);
    data |= ((fgetc (fptr) & 0xFF) << 24);

#ifdef __POB_DEBUG2
    if(dbg_fpt)
    {
	set_tab(dbg_fpt);
	fprintf(dbg_fpt, "DWord: 0x%08lX\n", data);
    }
#endif // __POB_DEBUG2
    return data;
}

/* read string */
char *read_string( void )
{
	static char string[1024];
	P_WORD i;
	P_WORD len;

	len = (fgetc (fptr) & 0xFF);

	for (i = 0; i < len; i++)
		string[i] = (fgetc (fptr) & 0xFF);

	if(!(len & 1))
		fgetc (fptr);

	string[len] = 0x00;

#ifdef __POB_DEBUG2
	if(dbg_fpt)
	{
		set_tab(dbg_fpt);
		fprintf(dbg_fpt, "String: (%d) \"%s\"\n", len, string);
	}
#endif // __POB_DEBUG2
	return string;
}

/* read double ieee format base function */
P_DOUBLE read_double_ieee( void )
{
#ifdef __IEEE
    P_DOUBLE data;

    fread (&data, 8, 1, fptr);

    return data;
#else // __IEEE
    /*
	please put in here stuff for managing 8 byte IEEE double
    */
    return 0.0;
#endif // __IEEE
}

/* read double */
P_DOUBLE read_double( void )
{

    P_DOUBLE data;

    data = read_double_ieee();

#ifdef __POB_DEBUG2
    if(dbg_fpt)
    {
	set_tab(dbg_fpt);
	fprintf(dbg_fpt, "Double: %lf\n", data);
    }
#endif // __POB_DEBUG2
    return data;
}

/* read vector: 3 x double */
void read_vector (P_DVECTOR v)
{
    v[0] = read_double_ieee();
    v[1] = read_double_ieee();
    v[2] = read_double_ieee();

#ifdef __POB_DEBUG2
    if(dbg_fpt)
    {
	set_tab(dbg_fpt);
	fprintf(dbg_fpt, "DVector: <%lf,%lf,%lf>\n", v[0], v[1], v[2]);
    }
#endif // __POB_DEBUG2
}

/* read vector: 2 x double */
void read_vector_uv (P_DVECTOR v)
{
    v[0] = read_double_ieee();
    v[1] = read_double_ieee();
#ifdef __POB_DEBUG2
    if(dbg_fpt)
    {
	set_tab(dbg_fpt);
	fprintf(dbg_fpt, "DVector_UV: <%lf,%lf>\n", v[0], v[1]);
    }
#endif // __POB_DEBUG2
}

/* read matrix: 9 x double */
void read_matrix(P_MATRIX matrix)
{
	matrix[0][0] = read_double_ieee();
	matrix[0][1] = read_double_ieee();
	matrix[0][2] = read_double_ieee();
	matrix[0][3] = 0.0;

	matrix[1][0] = read_double_ieee();
	matrix[1][1] = read_double_ieee();
	matrix[1][2] = read_double_ieee();
	matrix[1][3] = 0.0;

	matrix[2][0] = read_double_ieee();
	matrix[2][1] = read_double_ieee();
	matrix[2][2] = read_double_ieee();
	matrix[2][3] = 0.0;

	matrix[3][0] = read_double_ieee();
	matrix[3][1] = read_double_ieee();
	matrix[3][2] = read_double_ieee();
	matrix[3][3] = 1.0;

#ifdef __POB_DEBUG2
	if(dbg_fpt)
	{
		set_tab(dbg_fpt);
		fprintf(dbg_fpt, "Matrix: %.6lf|%.6lf|%.6lf|%.6lf\n",
					matrix[0][0],
					matrix[0][1],
					matrix[0][2],
					matrix[0][3]);
		set_tab(dbg_fpt);
		fprintf(dbg_fpt, "        %.6lf|%.6lf|%.6lf|%.6lf\n",
					matrix[1][0],
					matrix[1][1],
					matrix[1][2],
					matrix[1][3]);
		set_tab(dbg_fpt);
		fprintf(dbg_fpt, "        %.6lf|%.6lf|%.6lf|%.6lf\n",
					matrix[2][0],
					matrix[2][1],
					matrix[2][2],
					matrix[2][3]);
		set_tab(dbg_fpt);
		fprintf(dbg_fpt, "        %.6lf|%.6lf|%.6lf|%.6lf\n",
					matrix[3][0],
					matrix[3][1],
					matrix[3][2],
					matrix[3][3]);
	}
#endif // __POB_DEBUG2
}

/********************** chunk stuff ******************************/

/* start a new chunk:read chunk header, check for eof */
void start_chunk (CHUNK *chunk, CHUNK *mainchunk)
{
    P_WORD tag;
    P_DWORD len;

    chunk->start  = ftell(fptr);	// store actual position

    if(chunk->start >= file_size)	// check for end of file
    {
	chunk->end = chunk->start + 1;  // set end over filesize
	chunk->tag = 0x0000;		// dummy tag
	chunk->length = 0L;		// no len
	return;
    }

    tag = (fgetc (fptr) & 0xFF);	// read in identifier (word)
    tag |= ((fgetc (fptr) & 0xFF) << 8);
    chunk->tag    = tag;		// store it

    len = (fgetc (fptr) & 0xFF);	// read in len (dword)
    len |= ((fgetc (fptr) & 0xFF) << 8);
    len |= ((fgetc (fptr) & 0xFF) << 16);
    len |= ((fgetc (fptr) & 0xFF) << 24);
    chunk->length = len;		// store it

    chunk->end    = chunk->start + chunk->length;	// calculate chunk end

#ifdef __POB_DEBUG1
    print_debug_chunk(chunk, mainchunk);
    inc_tab();
#endif // __POB_DEBUG1
}

/* end a chunk:jump to end of chunk */
void end_chunk (CHUNK *chunk)
{
    fseek (fptr, chunk->end, 0);	// jump to end of chunk
#ifdef __POB_DEBUG1
    dec_tab();
#endif // __POB_DEBUG1
}

/* evaluates file size */
P_DWORD filesize(FILE *stream)
{
   P_DWORD curpos, length;

   curpos = ftell(stream);
   fseek(stream, 0L, SEEK_END);
   length = ftell(stream);
   fseek(stream, curpos, SEEK_SET);
   return length;
}

/********************** print stuff ******************************/

/* base print routine, writes to stdout */
void my_print(char *format, ...)
{
	va_list marker;
	char vsbuffer[1024];

	va_start(marker, format);
	vsprintf(vsbuffer, format, marker);
	va_end(marker);

	fprintf(stdout, vsbuffer);
}

/* print vector (POV-Ray-style)*/
void print_vector(P_DVECTOR t)
{
	my_print("<%.6lf, %.6lf, %.6lf> ", t[0], t[1], t[2]);
}

/* print matrix (POV-Ray-style)*/
void print_matrix(P_MATRIX matrix)
{
	my_print( "%s ", get_token_str(MATRIX_POB_TOKEN));PRINT_NEWLINE;
	my_print( "\t<%.6lf,%.6lf,%.6lf,\n",
	matrix[0][0],
	matrix[0][1],
	matrix[0][2]);
	my_print( "\t %.6lf,%.6lf,%.6lf,\n",
	matrix[1][0],
	matrix[1][1],
	matrix[1][2]);
	my_print( "\t %.6lf,%.6lf,%.6lf,\n",
	matrix[2][0],
	matrix[2][1],
	matrix[2][2]);
	my_print( "\t %.6lf,%.6lf,%.6lf>",
	matrix[3][0],
	matrix[3][1],
	matrix[3][2]);
	PRINT_NEWLINE;
}

#define M_NO_SHADOW_FLAG    0x0001
//#define M_CLOSED_FLAG       0x0002
#define M_OPEN_FLAG       0x0002
#define M_INVERTED_FLAG     0x0004
#define M_SMOOTHED_FLAG     0x0008
//#define M_CYLINDER_FLAG     0x0010
//#define M_DEGENERATE_FLAG   0x0020
#define M_STURM_FLAG        0x0040
//#define M_OPAQUE_FLAG       0x0080
//#define M_MULTITEXTURE_FLAG 0x0100
//#define M_INFINITE_FLAG     0x0200
#define M_HIERARCHY_FLAG    0x0400
#define M_HOLLOW_FLAG       0x0800
//#define M_HOLLOW_SET_FLAG   0x1000

#define Test_Flag(flag_cont, Flag)     ((flag_cont) & (Flag))


/* print object flags (POV-Ray-style)*/
void print_flags(P_WORD flags)
{

	if(Test_Flag(flags, M_SMOOTHED_FLAG))
	{
		my_print( "%s ", get_token_str(SMOOTH_POB_TOKEN));PRINT_NEWLINE;
	}
	if(Test_Flag(flags, M_HIERARCHY_FLAG))
	{
		my_print( "%s ", get_token_str(HIERARCHY_POB_TOKEN));PRINT_NEWLINE;
	}
	if(Test_Flag(flags, M_STURM_FLAG))
	{
		my_print( "%s ", get_token_str(STURM_POB_TOKEN));PRINT_NEWLINE;
	}
	if(Test_Flag(flags, M_HOLLOW_FLAG))
	{
		my_print( "%s ", get_token_str(HOLLOW_POB_TOKEN));PRINT_NEWLINE;
	}
	if(Test_Flag(flags, M_INVERTED_FLAG))
	{
		my_print( "%s ", get_token_str(INVERSE_POB_TOKEN));PRINT_NEWLINE;
	}
	if(Test_Flag(flags, M_NO_SHADOW_FLAG))
	{
		my_print( "%s ", get_token_str(NO_SHADOW_POB_TOKEN));PRINT_NEWLINE;
	}

	// attention this flag has chaned it's meaning
	if(Test_Flag(flags, M_OPEN_FLAG))
	{
		my_print( "%s ", get_token_str(OPEN_POB_TOKEN));PRINT_NEWLINE;
	}
}

/* print uv vector (POV-Ray-style)*/
void print_uv_vector(P_DVECTOR t)
{
	my_print("<%.6lf, %.6lf> ", t[0], t[1]);
}

/* print double */
void print_double(P_DOUBLE t)
{
	my_print("%.6lf ", t);
}

/* print word */
void print_word(P_WORD t)
{
	my_print("%d ", t);
}

/* print string */
void print_string(char *t)
{
	my_print("\"%s\"", t);
}

/* print comment (POV-Ray-style)*/
void print_comment(char *t)
{
	my_print("// %s\n", t);
}

/********************** some debug stuff ******************************/

/* increment tab position */
void inc_tab( void )
{
	tab_size++;
}

/* decrement tab position */
void dec_tab( void )
{
	tab_size--;
}

/* print some tabs */
void set_tab( FILE *fp )
{
	unsigned int i;
	for(i = 0;i < tab_size;i++)
		fprintf(fp,"\t");
}

/* initialize debug output */
void start_debug(void)
{
#ifdef __POB_DEBUG1
	dbg_fpt = fopen(pob_log_file, "w");
	if(dbg_fpt == NULL)
	{
		fprintf(stderr, "Cannot open dbg.log\n");
		exit(1);
	}
#endif // __POB_DEBUG1
}

/* stops debug output */
void end_debug(void)
{
#ifdef __POB_DEBUG1
	fclose(dbg_fpt);
	dbg_fpt = NULL;
#endif // __POB_DEBUG1
}

/* print raw chunk data */
void print_data(CHUNK *chunk, FILE *dbg_fpt)
{
	P_DWORD idx;

	P_BYTE c;
	P_BYTE c_arr[16];
	P_WORD i;
	P_WORD add_len;

	idx = 0;

	set_tab(dbg_fpt);
	while(idx < chunk->length - 6L)
	{
		c = (fgetc (fptr) & 0xFF);
		c_arr[idx % 16] = c;

		fprintf(dbg_fpt, "%02X ", c);
		if((idx % 16) == 15)
		{
			for(i = 0;i < 16;i++)
			{
				if(isprint(c_arr[i]))
					fprintf(dbg_fpt, "%c", c_arr[i]);
				else
					fprintf(dbg_fpt, ".");
			}
			fprintf(dbg_fpt, "\n");
			if((idx +1) != (chunk->length - 6L))
				set_tab(dbg_fpt);
		}
		idx++;
	}

	add_len = 0;
	while((idx % 16) != 0 )
	{
		fprintf(dbg_fpt, "   ");
		idx++;
		add_len++;
	}

	if(add_len != 0)
	{
		for(i = 0;i < 16 - add_len;i++)
		{
			if(isprint(c_arr[i]))
				fprintf(dbg_fpt, "%c", c_arr[i]);
			else
				fprintf(dbg_fpt, ".");
		}
		fprintf(dbg_fpt, "\n");
	}
}

/* print debug info for chunk header */
void print_debug_chunk(CHUNK *chunk, CHUNK *mainchunk)
{
	P_DWORD akt_pos;

	if(!dbg_fpt)
		return;

	if (mainchunk)
	{
		if (chunk->end <= mainchunk->end)
		;
		else
		return;
	}

	akt_pos = ftell(fptr);

	fprintf(dbg_fpt, "\n");
	set_tab(dbg_fpt);
	fprintf(dbg_fpt, "identifier  : %04X ",chunk->tag);
	if(chunk->tag < LAST_POB_TOKEN)
		fprintf(dbg_fpt, "(%s)\n",get_token_str(chunk->tag));
	else
		fprintf(dbg_fpt, "(%s)\n",get_stoken_str(chunk->tag));

	set_tab(dbg_fpt);
	fprintf(dbg_fpt, "start       : %04lX\n",chunk->start);
	set_tab(dbg_fpt);
	fprintf(dbg_fpt, "end         : %04lX\n",chunk->end);
	set_tab(dbg_fpt);
	fprintf(dbg_fpt, "len         : %04lX\n",chunk->length);

#ifdef __POB_DEBUG3
	set_tab(dbg_fpt);
	fprintf(dbg_fpt, "raw data:\n");
	print_data(chunk, dbg_fpt);
#endif // __POB_DEBUG3

#ifdef __POB_DEBUG2
	set_tab(dbg_fpt);
	fprintf(dbg_fpt, "decoded data:\n");
#endif // __POB_DEBUG2
	fseek(fptr, akt_pos, SEEK_SET);
}
