/* @(#)dither.c 1.14   8/27/96 */

/******************************************************************************
* Threedom: a 3D polygon renderer                                             *
* (C) Copyright 1996 by Philip Stephens                                       *
* (C) Copyright 1996 by the IBM Corporation                                   *
* All Rights Reserved                                                         *
*                                                                             *
* Permission to use, copy, modify, and distribute this software and its       *
* documentation without fee for any non-commerical purpose is hereby granted, * 
* provided that the above copyright notice appears on all copies and that     *
* both that copyright notice and this permission notice appear in all         *
* supporting documentation.                                                   *
*                                                                             *
* NO REPRESENTATIONS ARE MADE ABOUT THE SUITABILITY OF THIS SOFTWARE FOR ANY  *
* PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.       *
* NEITHER PHILIP STEPHENS OR IBM SHALL BE LIABLE FOR ANY DAMAGES SUFFERED BY  *
* THE USE OF THIS SOFTWARE.                                                   *
******************************************************************************/

#include <stdio.h>
#include <stdlib.h>

#include "typedef.h"
#include "main.h"

/*
 * Dither arrays used to convert pixels to frame buffer pixels of the correct
 * brightness.
 */
fbpixel dither[4][FB_MAPS][PIXEL_COLOURS];

/*
 * Light array; used to select the approapiate dither arrays for a given
 * brightness level.
 */
fbpixel light[4][LIGHT_MAX + 1] = {
	{0, 1, 1, 2, 2, 2},
	{0, 0, 1, 0, 1, 2},
	{0, 0, 1, 0, 1, 2},
	{0, 1, 1, 2, 2, 2}
};

/**************************************************************************
 * Create the dither arrays, which are used to translate a frame buffer   *
 * pixel value to a device pixel, based upon a 2x2 dither square for both *
 * colour and brightness.  There are thus 16 arrays in total.		  *
 **************************************************************************/

void
create_dither_arrays(void)
{
	int index, i, j;
	int r, g, b;
	int red, green, blue;
	int dev_red_factor = FB_COLOURS / FB_REDS;
	int dev_green_factor = FB_COLOURS / FB_GREENS;
	int dev_blue_factor = FB_COLOURS / FB_BLUES;
	int fb_red_factor = FB_COLOURS / PIXEL_REDS;
	int fb_green_factor = FB_COLOURS / PIXEL_GREENS;
	int fb_blue_factor = FB_COLOURS / PIXEL_BLUES;
	int red_thresh[4], green_thresh[4], blue_thresh[4];

	/*
	 * Determine the RGB threshholds.
	 */
	for (i = 0; i < 4; i++) {
		red_thresh[i] = i * dev_red_factor / 4;
		green_thresh[i] = i * dev_green_factor / 4;
		blue_thresh[i] = i * dev_blue_factor / 4;
	}

	/*
	 * Generate the 16 dither mappings for each pixel.
	 */
	index = 0;
	for (r = 0; r < PIXEL_REDS; r++)
	    for (g = 0; g < PIXEL_GREENS; g++)
		for (b = 0; b < PIXEL_BLUES; b++) {
		    for (i = 0; i < 4; i++) {
			red = MYMIN(r * fb_red_factor / dev_red_factor,
				FB_REDS - 2);
	       		if (r * fb_red_factor - red * dev_red_factor >
			    red_thresh[i])
				red++;
	       		green = MYMIN(g * fb_green_factor / dev_green_factor,
				FB_GREENS - 2);
	       		if (g * fb_green_factor - green * dev_green_factor >
			    green_thresh[i])
				green++;
	       		blue = MYMIN(b * fb_blue_factor / dev_blue_factor,
				FB_BLUES - 2);
	       		if (b * fb_blue_factor - blue * dev_blue_factor >
			    blue_thresh[i])
				blue++;
			for (j = 0; j < FB_MAPS; j++)
	       			dither[i][j][index] = (fbpixel)((red * FB_REDS +
					green) * FB_GREENS + blue +
					j * FB_COLOURS_PER_MAP + FB_MAP_OFFSET);
	  	    }
		    index++;
		}
}

/***********************************************************
 * Render one perspective texture mapped row of a polygon. *
 ***********************************************************/

void
render_polygon_row(int sy, fixed left_sx, fixed right_sx,
		   fixed left_one_on_tz, fixed right_one_on_tz,
		   fixed left_u_on_tz, fixed right_u_on_tz,
		   fixed left_v_on_tz, fixed right_v_on_tz,
		   pixel *image_ptr, int width_mask, int height_mask,
		   int column_shift, int light_level)
{
	fixed sx, diff_sx;
	int start_sx, end_sx;
	fbpixel *fb_ptr;
	fixed *zb_ptr;
	fixed delta_sx;
	fixed delta_one_on_tz;
	fixed delta_u_on_tz;
	fixed delta_v_on_tz;
	fixed one_on_tz, u_on_tz, v_on_tz;
	fixed u, v;
	fbpixel *dither0, *dither1;
	int count;

	/*
	 * If the row is entirely off screen, nothing to render.
	 */
	if (INT(CEIL(left_sx)) >= window_width)
		return;
	if (right_sx < 0)
		return;

	/*
	 * Determine the delta values for the interpolants and the initial
	 * values.
	 */
	delta_sx = right_sx - left_sx;
	delta_one_on_tz = fdiv(right_one_on_tz - left_one_on_tz, delta_sx);
	delta_u_on_tz = fdiv(right_u_on_tz - left_u_on_tz, delta_sx);
	delta_v_on_tz = fdiv(right_v_on_tz - left_v_on_tz, delta_sx);
	one_on_tz = left_one_on_tz;
	u_on_tz = left_u_on_tz;
	v_on_tz = left_v_on_tz;

	/*
	 * Clamp left_sx and right_sx so they're on screen, and adjust the
	 * interpolants.
	 */
	sx = CEIL(left_sx);
	if (sx < 0)
		sx = 0;
	diff_sx = sx - left_sx;
	if (diff_sx > 0) {
		one_on_tz += fmul(delta_one_on_tz, diff_sx);
		u_on_tz += fmul(delta_u_on_tz, diff_sx);
		v_on_tz += fmul(delta_v_on_tz, diff_sx);
	}
	start_sx = INT(sx);
	end_sx = INT(right_sx);
	if (end_sx >= window_width)
		end_sx = window_width - 1;

	/*
	 * Choose the two dither arrays that are needed to dither this row,
	 * based upon whether the starting column and row are odd or even.
	 */
	if (sy & 1) {
		if (start_sx & 1) {
			dither0 = dither[3][light[3][light_level]];
			dither1 = dither[1][light[1][light_level]];
		} else {
			dither0 = dither[1][light[1][light_level]];
			dither1 = dither[3][light[3][light_level]];
		}
	} else {
		if (start_sx & 1) {
			dither0 = dither[2][light[2][light_level]];
			dither1 = dither[0][light[0][light_level]];
		} else {
			dither0 = dither[0][light[0][light_level]];
			dither1 = dither[2][light[2][light_level]];
		}
	}

	if (sy < 0 || sy >= window_height) {
		printf("sy out of range (%d)\n", sy);
		return;
	}
	if (start_sx < 0 || start_sx >= window_width) {
		printf("start sx out of range (%d)\n", start_sx);
		return;
	}
	if (end_sx < 0 || end_sx >= window_width) {
		printf("end sx out of range (%d)\n", end_sx);
		return;
	}
	
	/*
	 * Render the pixels across the row in groups of 32 until we've got
	 * less than 32 pixels left.
	 *
	 * We only compute u and v for every 32nd pixel; we interpolate u and
	 * v for the inbetween pixels.
	 */
	fb_ptr = fb_row[sy] + start_sx;
	zb_ptr = zb_row[sy] + start_sx;
	count = end_sx - start_sx + 1;
	u = fdiv(u_on_tz, one_on_tz);
	v = fdiv(v_on_tz, one_on_tz);
	while (count >= pixel_span) {
		fixed end_u, end_v, end_one_on_tz;
		fixed delta_u, delta_v;
		int i;

#ifdef	NO_FIXED_MATH
		u_on_tz += delta_u_on_tz * pixel_span;
		v_on_tz += delta_v_on_tz * pixel_span;
		end_one_on_tz = one_on_tz + (delta_one_on_tz * pixel_span);
		end_u = fdiv(u_on_tz, end_one_on_tz);
		end_v = fdiv(v_on_tz, end_one_on_tz);
		delta_u = (end_u - u) / pixel_span;
		delta_v = (end_v - v) / pixel_span;
#else
		u_on_tz += delta_u_on_tz << span_shift;
		v_on_tz += delta_v_on_tz << span_shift;
		end_one_on_tz = one_on_tz + (delta_one_on_tz << span_shift);
		end_u = fdiv(u_on_tz, end_one_on_tz);
		end_v = fdiv(v_on_tz, end_one_on_tz);
		delta_u = (end_u - u) >> span_shift;
		delta_v = (end_v - v) >> span_shift;
#endif

		for (i = 0; i < pixel_span; i += 2) {
			if (one_on_tz + z_factor > *zb_ptr) {
				*fb_ptr = dither0[*(image_ptr +
					((INT(u) & width_mask) << column_shift)
					+ (INT(v) & height_mask))];
				*zb_ptr = one_on_tz + z_factor;
			}
			fb_ptr++;
			zb_ptr++;
			one_on_tz += delta_one_on_tz;
			u += delta_u;
			v += delta_v;
			if (one_on_tz + z_factor > *zb_ptr) {
				*fb_ptr = dither1[*(image_ptr +
					((INT(u) & width_mask) << column_shift)
					+ (INT(v) & height_mask))];
				*zb_ptr = one_on_tz + z_factor;
			}
			fb_ptr++;
			zb_ptr++;
			one_on_tz += delta_one_on_tz;
			u += delta_u;
			v += delta_v;
		}

		count -= pixel_span;
		u = end_u;
		v = end_v;
	}

	/*
	 * If there are still pixels left, we compute u and v for the last
	 * pixel and interpolate u and v for the remaining inbetween pixels.
	 */
	if (count > 1) {
		fixed end_u, end_v, end_one_on_tz;
		fixed delta_u, delta_v;

		u_on_tz += delta_u_on_tz * count;
		v_on_tz += delta_v_on_tz * count;
		end_one_on_tz = one_on_tz + (delta_one_on_tz * count);
		end_u = fdiv(u_on_tz, end_one_on_tz);
		end_v = fdiv(v_on_tz, end_one_on_tz);
		delta_u = (end_u - u) / count;
		delta_v = (end_v - v) / count;
		
		while (count > 1) {
			if (one_on_tz + z_factor > *zb_ptr) {
				*fb_ptr = dither0[*(image_ptr +
					((INT(u) & width_mask) << column_shift)
					+ (INT(v) & height_mask))];
				*zb_ptr = one_on_tz + z_factor;
			}
			fb_ptr++;
			zb_ptr++;
			one_on_tz += delta_one_on_tz;
			u += delta_u;
			v += delta_v;
			if (one_on_tz + z_factor > *zb_ptr) {
				*fb_ptr = dither1[*(image_ptr +
					((INT(u) & width_mask) << column_shift)
					+ (INT(v) & height_mask))];
				*zb_ptr = one_on_tz + z_factor;
			}
			fb_ptr++;
			zb_ptr++;
			one_on_tz += delta_one_on_tz;
			u += delta_u;
			v += delta_v;
			count -= 2;
		}
	}

	/*
	 * This case handles an odd number of pixels in the row.
	 */
	if (count) {
		int last_u, last_v;

		last_u = INT(fdiv(u_on_tz, one_on_tz)) & width_mask;
		last_v = INT(fdiv(v_on_tz, one_on_tz)) & height_mask;
		if (one_on_tz + z_factor > *zb_ptr) {
			*fb_ptr = dither0[*(image_ptr +
				(last_u << column_shift) + last_v)];
			*zb_ptr = one_on_tz + z_factor;
		}
	}
}
